/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2012 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/

#ifndef ACTION_H
#define ACTION_H

// -- Core stuff
#include "CamiTKAPI.h"

#include <QSet>
#include <QWidget>
#include <QString>
#include <QAction>


namespace camitk {

class ActionExtension;

/**
 * Action class is an abstract class that enables you to build a action (generally on a component).
 * At least two classes have to be reimplemented to enable the action: ActionExtension + Action
 *
 * This is the list of attributes you need to consider when creating a new action
 * - extension : the ActionExtension class where to register your action;
 * - name: name of the action;
 * - description: tag used to describe the action (also used for tooltip and whatsThis of the corresponding QAction);
 * - component: the name of the component class on which this action can be applied or "" (default) for generic actions;
 * - family: families of actions allows to group different actions of the same kind under one name;
 * - tags: tags is a list of words used to define an action. These words can be used to find an action.
 * - widget: a default widget (instance of ActionWidget) is build by default (see below if this is not what you need);
 * - isEmbedded: this boolean defines if the gui widget is embedded in a given parent widget / action widget container (true by default)
 * - icon: the icon used for the visually distinguish the action (used by the corresponding QAction)
 *
 * An Action has a corresponding QAction, see getQAction(), that makes it easy to trigger an action from any Qt GUI (menus, toolbar,
 * push buttons...)
 * 
 * If the component class is defined (non empty string), an Action is applied on the currently selected components.
 * If there are no component defined (i.e. you specifies setComponent("")), it means that your action does not
 * need any inputs.
 * 
 * Two steps have to be considered when using an action:
 * - Step 1, trigger(): the action is either directly applied (if it does not have any GUI) or it's GUI is shown (using getWidget())
 * - Step 2, apply(): only the action algorithm is applied, i.e., the data are processed
 *
 * The targets can have changed between the time the action is first triggered and the time the action is applied.
 * getWidget() is always called when the targets are updated.
 * Therefore whenever getWidget() is called, you should make sure to update the the action GUI consequently.
 * getTargets() is always updated in trigger() and available.
 * 
 * \note
 * trigger() and apply() are public slots. They can be called either directly (classic C++ method invocation) or
 * by connecting them to a QWidget signal.
 *
 * When an action is triggered (e.g., by right clicking in the context menu), the following algorithm applies, see trigger():
 * - 1. Prepare targetComponents (available with getTargets()): only select the compatible components from the selected components
 * - 2. If the action is embedded, get the widget and show it in a parent/container (if parent is not specified, show it in the action viewer)
 * - 3. If the action in not embedded, show it as a dialog
 * - 4. If the action does not have any widget, directly call apply()
 *
 * This means that, if there is a widget, the action algorithm is controlled by the action widget, i.e. apply() is not
 * called by trigger() but should be called by one of the action widget's button.
 *
 * By default an action has a widget, instance of ActionWidget.
 * If ActionWidget does not correspond to what you need, just create a new class inheriting from QWidget, or 
 * even better, inheriting from ActionWidget.
 * 
 * These are the use cases for using the default behaviour (i.e. an instance of ActionWidget):
 * - your action has some parameters and you need the user to review the default or modify their values before the action is applied,
 * - or your action has no parameters but you still want the user to be applied only if/when the user click on an apply button.
 *
 * ActionWidget should be good enough in most of the cases.
 * The default widget contains a description, a reminder of the current target component names,
 * and an applyable/revertable ObjectController that allows you to edit/modify properties.
 * 
 * \note
 * The recommanded architecture is for the action widget to call the action's apply method.
 * The widget should only manage user interaction.
 *
 * Here are some notes about the rest of the properties:
 * 
 * extensionName is automatically given during the action registration in the ActionExtension.
 *
 * The component property determines on which type of component your action can be applied.
 * Generic actions are action that have an empty component name. Therefore generic actions
 * can be called to generate/synthetize data or initialize resources.
 *
 * You can add any number of tags using the method addTag().
 *
 * If ActionWidget is not what your need, a typical getWidget() method should use the lazy instanciation pattern to instanciate
 * MyVerySpecialActionWidget the first time it is called, and call the MyVerySpecialActionWidget instance's updateTargets() method
 * for any subsequent calls. Something like:
 * \code
 * QWidget *MyAction::getWidget() {
 *    // build or update the widget
 *    if (!myWidget)
 *        myWidget = new MyVerySpecialActionWidget(this);
 *    else
 *        // MyVerySpecialActionWidget should have a updateTargets() method
 *        myWidget->updateTargets();
 *
 *    return myWidget;
 * }
 * \endcode
 * But of course you can also use any kind of widget you like. ActionWidget is just defining
 * a default widget for an action.
 * If your action does not have any GUI/parameters, just override the getWidget() method in order to return NULL.
 *
 * By default the action's widget is embedded. If you do not want to embed your action's widget, use setEmbedded(false)
 * in the constructor.
 * When embedded, the parent widget has to be given at triggered time.
 * If there is no parent given for an embedded action, then the action is embedded in the ActionViewer by default.
 *
 * The method apply() must be implemented in your Action.
 *
 * \note at any moment, the selected components on which the action needs to be applied
 * are available by getTargets().
 * targetComponents is filtered so that it only contains compatible components (i.e.,
 * instances of getComponent()).
 *
 * For a simple example of an embedded action @see RenderingOption
 * For a simple example of a non-embedded action @see RigidTransform
 * For a simple example of an action with no widget (but with a GUI) @see ChangeColor
*/

class CAMITK_API Action : public QObject {
  Q_OBJECT

public:

  /// Default Constructor: the ActionExtension is needed
  Action(ActionExtension *);

  /// Destructor
  virtual ~Action();

  /// \enum ApplyStatus describes what happened during the application of an algorithm (i.e. results of the apply method)
  enum ApplyStatus {
    SUCCESS,    ///< everything went according to plan
    ERROR,      ///< an error occured (usually it means that the apply() was interrupted)
    WARNING,    ///< some (partial) error occured during the application of the algorithm
    ABORTED,    ///< the action was aborted before completion
    TRIGGERED,  ///< the action was triggered only, but not applied
  };

public slots:
  /** This method triggers the action.
   *  The parent widget is used if the action is embedded, see class description for more information about the algorithm.
   *  This method cannot be redefined in inherited class.
   */
  ApplyStatus trigger(QWidget * parent = NULL);

  /** This method is called when the action has to be applied on the target list (get the target lists using getTargets())
   *  \note it should never be empty!
   *  @return status of the apply method
   */
  virtual ApplyStatus apply() = 0;

public:

  /// @name Generic action getters
  /// These methods can not be redefined in subclasses.
  /// @{
  /** Get the corresponding QAction.
   *  The corresponding QAction has its triggered() signal connected to the trigger() slot of the action.
   *  It shares the action icon (as the QAction's icon) and name (as the QAction's text).
   *  It also use the descriptions of the action for the tooltip/whatsThis text.
   * 
   *  To add a shortcut, simply call getQAction()->setShortcut(..) in the action constructor.
   *  To make this shortcut available for any windows of the application, call getQAction()->setShortcutContext(Qt::ApplicationShortcut);
   */
  QAction *getQAction();
      
  /// the name of the component class that can be used by this action
  QString getName() const {
    return name;
  };

  /// the description of the action
  QString getDescription() const {
    return description;
  };

  /// the name of the component class that can be used by this action
  QString getComponent() const {
    return component;
  };

  /// the name of the family in which this action is associated
  QString getFamily() const {
    return family;
  };

  /// the name of the extension in the family in which this action is associated
  QString getExtensionName() const;

  /// the name of the tag called this action
  QStringList getTag() const {
    return tags;
  };

  /// argument use to know if the widget is embedded or not
  bool getEmbedded() const {
    return isEmbedded;
  };
  ///@}

  /// @name Method specific to an action.
  /// @{
  /** This method has to be redefined in your Action only if:
    * - you do not have any widget to control your action (i.e. getWidget() will have to return NULL),
    * - you do not use the default ActionWidget but another one.
    * 
    * In the second case, it is strongly recommanded to have a code similar to this:
    * \code
    * QWidget *MyAction::getWidget() {
    *    // build or update the widget
    *    if (!myWidget)
    *        myWidget = new MyVerySpecialActionWidget(this);
    *    else
    *        // MyVerySpecialActionWidget should have a updateTargets() method
    *        myWidget->updateTargets();
    *
    *    return myWidget;
    * }
    * \endcode
    * 
    * The updateTargets method in MyVerySpecialActionWidget is used in case the selection has changed since the 
    * last time the widget was shown (a change in the selection often means an updateTargets of the action's widget
    * fields).
    */
  virtual QWidget * getWidget();

  /// the icon to personalize the action (no icon by default)
  virtual QPixmap getIcon();
    
  /// the currently selected and valid (regarding the component property) components, for which this action is called
  const ComponentList getTargets() const;  
  ///@}

protected:
  /// @name Generic action attributes setters
  /// These methods can not be redefined in subclasses but have to used to ensure name/description unicity
  /// among CamiTK.
  /// @{
  /// the name of the component class that can be used by this action
  void setName(QString name);
  
  /// the description of the action
  void setDescription(QString description);
  
  /// the name of the component class that can be used by this action
  void setComponent(QString component);
  
  /// the name of the family in which this action is associated
  void setFamily(QString family);
  
  /// add a tag to the tags list of this action
  void addTag(QString tag);
  
  /// set the embedded property (an action is embedded by default, unless specified otherwise by explicitly calling this method with false)
  void setEmbedded(bool isEmbedded);
  
  /// set the Pixmap
  void setIcon(QPixmap);
  ///@}
  
  /// the action widget
  QWidget *actionWidget;
    
private:
  /// the name of the component class that can be used by this action
  QString name;

  /// the descriptionof the action
  QString description;

  /// the name of the component class that can be used by this action
  QString component;

  /// the name of the family in which this action is associated
  QString family;

  /// the name of the tag called this action
  QStringList tags;

  /// is the widget embedded or not
  bool isEmbedded;

  /// the extension in which this action is declared and registered
  ActionExtension *extension;

  /// the Action pixmap icon
  QPixmap icon;

  /// the corresponding QAction
  QAction *qAction;
    
  /// the currently selected and valid (regarding the component property) components, for which this action is called (private list, use getTargets() is subclasses)
  ComponentList targetComponents;
};

}

// -------------------- declare the interface for QPluginLoader --------------------
Q_DECLARE_INTERFACE(camitk::Action, "TIMC-IMAG.Action/2.1"); //TODO use svn version?

#endif // ACTION_H




