/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2012 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/

// -- Core stuff
#include "ExtensionManager.h"
#include "Core.h"
#include "Log.h"
#include "Application.h"
#include "Action.h"

// -- QT stuff
#include <QDir>
#include <QtGui>



namespace camitk {

// -------------------- autoload --------------------
void ExtensionManager::autoload() {
    autoloadComponentExtensions();
    autoloadActionExtensions();
}

// -------------------- getComponentExtensionMap --------------------
QMap< QString, ComponentExtension* > &  ExtensionManager::getComponentExtensionMap() {
    static QMap<QString, ComponentExtension*> componentExtensionMap;

    return componentExtensionMap;
}

// -------------------- getDataDirectoryComponentExtensionMap --------------------
QMap< QString, ComponentExtension* > & ExtensionManager::getDataDirectoryComponentExtensionMap() {
    static QMap<QString, ComponentExtension*> dataDirectoryComponentExtensionMap;

    return dataDirectoryComponentExtensionMap;
}

// -------------------- autoloadComponentExtensions --------------------
void ExtensionManager::autoloadComponentExtensions() {
    // In order to manage static instances, add code here to load the static instances using QPluginLoader::staticInstances()

    // get the executable path
    QDir extensionsDir(Core::getComponentDir());

    if (!extensionsDir.exists()) {
        QMessageBox::information(NULL, "ExtensionManager Autoload Error", "Cannot access to component extension directory:<br/><tt>" + extensionsDir.absolutePath() + "</tt><br/>Directory does not exist!");
    }
    else {
        // add component directory to library paths (needed especially for internal dependencies)
        Application::instance()->addLibraryPath(Core::getCoreLibDir());
        Application::instance()->addLibraryPath(extensionsDir.canonicalPath());
        CAMITK_DEBUG("ExtensionManager", "autoloadExtensions",
                     "corelib=" << Core::getCoreLibDir().toStdString() << endl
                     << "components=" << Core::getComponentDir().toStdString() << endl
                     << "application library paths=" << Application::instance()->libraryPaths().join(",").toStdString()
                    );

        // try to load all Components in the directory
        QStringList pluginFile;
        // linux: .so, windows: .dll, macOs: .dylib
        pluginFile << "*.so" << "*.dll" << "*.dylib";

        // loop to load component plugin, taking into account internal dependencies (i.e. dependency between
        // one component and another one.
        int i = 0;
        QStringList pluginFileNames = extensionsDir.entryList(pluginFile, QDir::Files);
        // if there is more than 5 dependency levels, then you have two choices:
        // - consider simplifying your component
        // - increase maxNumberOfTries
        int maxNumberOfTries = 5;
        int tryNr = 0;

        do {
            QMutableListIterator<QString> it(pluginFileNames);

            while (it.hasNext()) {
                QString extFileName = it.next();
                CAMITK_DEBUG("ExtensionManager", "autoloadExtensions", "try #" << tryNr << " for : " << extFileName.toStdString());

                if (loadComponentExtension(extensionsDir.absoluteFilePath(extFileName))) {
                    // this one is loaded, remove it from the list to load
                    it.remove();
                }
            }

            tryNr++;
        } while (tryNr < maxNumberOfTries && pluginFileNames.size() > 0);

        // manage loading errors
        if (pluginFileNames.size() > 0) {
            // get the messages from Qt
            QStringList errorStrings;
            foreach(QString fileName, pluginFileNames) {
                QPluginLoader pluginLoader(fileName);
                QObject *plugin = pluginLoader.instance();

                if (!plugin) {
                    errorStrings << QString("Plugin " + fileName + ", error: " + pluginLoader.errorString());
                }
                else {
                    errorStrings << QString("Plugin " + fileName + ", no error (this is impossible!)");
                }
            }
            QMessageBox::warning(NULL, "ExtensionManager Opening Error...",
                                 "AutoLoad plugin failed after " + QString::number(tryNr) + " tries for the following component(s):<ul><li>"
                                 + errorStrings.join("</li><li>")
                                 + "</li></ul>Component extension directory: <tt>"
                                 + Core::getComponentDir() + "</tt><br/>"
                                 + "List of library paths:<ul><li>"
                                 + Application::instance()->libraryPaths().join("</li><li>")
                                 + "</li></ul>");

        }
    }

    //-- user extensions
    QSettings & settings = Application::getSettings();
    settings.beginGroup ( "UserExtensions" );
    QStringList userComponents = settings.value("components", QVariant(QStringList())).toStringList();
    settings.endGroup();
    foreach(QString userComponentFile, userComponents) {
        loadComponentExtension(userComponentFile);
    }

}

// -------------------- loadComponentExtension --------------------
bool ExtensionManager::loadComponentExtension(QString fileName) {
    QPluginLoader pluginLoader(fileName);
    QObject *plugin = pluginLoader.instance();

    if (plugin) {
        ComponentExtension *cp = qobject_cast<ComponentExtension *> (plugin);

        if (cp) {
            cp->setLocation(fileName);
            // insert the ComponentExtension plugin in the application wide list

            if (cp->hasDataDirectory()) {
                getDataDirectoryComponentExtensionMap().insert(cp->getName(), cp);
            }
            else {
                // (cannot do that in the constructor because the virtual symbol table seems to be confused!)
                foreach(QString ext, cp->getFileExtensions()) {
                    getComponentExtensionMap().insert(ext, cp);
                }
            }

            return true;
        }
    }
    else {
        pluginLoader.unload(); // to make sure we could try again later
    }


    return false;
}

// -------------------- unloadComponentExtension --------------------
bool ExtensionManager::unloadComponentExtension(QString extOrName) {
    bool unloaded = false;
    // remove from the application wide list
    ComponentExtension * cp = getComponentExtensionMap().take(extOrName);

    if (cp != NULL) {
        // remove all the other extensions
        QMutableMapIterator<QString, ComponentExtension *> it(getComponentExtensionMap());

        while (it.hasNext()) {
            it.next();

            if (it.value() == cp) {
                getComponentExtensionMap().take(it.key());
            }
        }

        // open the plugin
        QPluginLoader pluginLoader(cp->getLocation());
        // delete the pointer
        delete cp;
        // try to unload
        unloaded = pluginLoader.unload();
    }
    else {
        cp = getDataDirectoryComponentExtensionMap().take(extOrName);

        if (cp != NULL) {
            // open the plugin
            QPluginLoader pluginLoader(cp->getLocation());
            // delete the pointer
            delete cp;
            // try to unload
            unloaded = pluginLoader.unload();
        }
        else {
            // look for the name in getComponentExtensionMap()
            QMutableMapIterator<QString, ComponentExtension *> it(getComponentExtensionMap());

            while (it.hasNext()) {
                it.next();

                if (it.value()->getName() == extOrName)
                    return unloadComponentExtension(it.value()->getName());
            }
        }
    }

    return unloaded;
}

// -------------------- getComponentExtension --------------------
const ComponentExtension * ExtensionManager::getComponentExtension(QString extOrName) {
    ComponentExtension *cp = getComponentExtensionMap().value(extOrName);

    if (!cp) {
        cp = getDataDirectoryComponentExtensionMap().value(extOrName);

        if (!cp) {
            // look for the name in getComponentExtensionMap()
            QMapIterator<QString, ComponentExtension *> it(getComponentExtensionMap());

            while (it.hasNext() && !cp) {
                it.next();

                if (it.value()->getName() == extOrName)
                    cp = it.value();
            }
        }
    }

    return cp;
}

// -------------------- getComponentExtensions --------------------
const QMap<QString, ComponentExtension*> & ExtensionManager::getComponentExtensions() {
    return getComponentExtensionMap();
}

// -------------------- getDataDirectoryComponents --------------------
const QMap<QString, ComponentExtension*> & ExtensionManager::getDataDirectoryComponents() {
    return getDataDirectoryComponentExtensionMap();
}

// -------------------- getFileExtensions --------------------
QStringList ExtensionManager::getFileExtensions() {
    return ExtensionManager::getComponentExtensions().keys();
}

// -------------------- getDataDirectoryExtNames --------------------
QStringList ExtensionManager::getDataDirectoryExtNames() {
    return ExtensionManager::getDataDirectoryComponents().keys();
}






// -------------------- getActionExtensionMap --------------------
QMap<QString, ActionExtension*>  &  ExtensionManager::getActionExtensionMap() {
    static QMap<QString, ActionExtension*> actionExtensionMap;

    return actionExtensionMap;
}

// -------------------- getActionExtensions --------------------
const QMap< QString, ActionExtension* >& ExtensionManager::getActionExtensions() {
    return getActionExtensionMap();
}


// -------------------- autoloadActionExtensions --------------------
void ExtensionManager::autoloadActionExtensions() {
    // TODO? may be try to first load the static instances using QPluginLoader::staticInstances() ?

    // get the executable path
    QDir actionsDir(Core::getActionDir());

    if (!actionsDir.exists()) {
        QMessageBox::information(NULL, "ExtensionManager Autoload Error", "Cannot access to actions directory:<br/><tt>" + actionsDir.absolutePath() + "</tt><br/>Directory does not exist!");
    }
    else {
        // try to load all Components in the directory
        QStringList extensionFile;
        extensionFile << "*.so" << "*.dll" << "*.dylib";

        // loop to load action extension, taking into account internal dependencies (i.e. dependency between
        // one action and another one.
        int i = 0;
        QStringList extensionFileNames = actionsDir.entryList(extensionFile, QDir::Files);
        // if there is more than 5 dependency levels, then you have two choices:
        // - consider simplifying your component
        // - increase maxNumberOfTries
        int maxNumberOfTries = 5;
        int tryNr = 0;

        do {
            i = 0;

            while (i < extensionFileNames.size()) {
                if (loadActionExtension(actionsDir.absoluteFilePath(extensionFileNames.at(i))))
                    // this one is loaded, remove it from the list to load
                    extensionFileNames.removeAt(i);
                else
                    i++;
            }

            tryNr++;
        }
        while (tryNr < maxNumberOfTries && extensionFileNames.size() > 0);

        if (extensionFileNames.size() > 0) {
            QMessageBox::warning(NULL, "ExtensionManager Opening Error...", "AutoLoad Extension failed after " + QString::number(tryNr) + " tries for the following action(s):<ul><li>" + extensionFileNames.join("</li><li>") + "</li></ul>Actions extension directory: <tt>" + Core::getActionDir() + "</tt>");
        }
    }

    //-- user extensions
    QSettings & settings = Application::getSettings();
    settings.beginGroup ( "UserExtensions" );
    QStringList userActions = settings.value("actions", QVariant(QStringList())).toStringList();
    settings.endGroup();
    foreach(QString userActionFile, userActions) {
        loadActionExtension(userActionFile);
    }

}


// -------------------- loadActionExtension --------------------
bool ExtensionManager::loadActionExtension(QString fileName) {
    QPluginLoader pluginLoader(fileName);
    QObject *extension = pluginLoader.instance();

    if (extension) {
        ActionExtension *ext = qobject_cast<ActionExtension *> (extension);

        if (ext) {
            //-- register the filename
            getActionExtensionMap().insert(fileName, ext);
            // initialize all actions
            ext->init();
            //-- register all actions
            Application::registerAllActions(ext);
            return true;
        }
    }
    else {
        CAMITK_ERROR("ExtensionManager", QString("loadAction(" + fileName + ")").toStdString(), "QPluginLoader error: " + pluginLoader.errorString().toStdString());
    }

    return false;
}


// -------------------- unloadActionExtension --------------------
bool ExtensionManager::unloadActionExtension(QString fileName) {
    if (getActionExtensionMap().contains(fileName)) {
        ActionExtension *ext = getActionExtensionMap().value(fileName);
        //-- unregister all actions
        foreach(Action *action, ext->getActions()) {
            getActionExtensionMap().remove(action->getName());
        }
        //-- unregister extension
        getActionExtensionMap().remove(fileName);
        //-- delete extensions (and all its actions)
        delete ext;
        return true;
    }
    else
        return false;
}

// -------------------- unloadAllActionExtensions --------------------
void ExtensionManager::unloadAllActionExtensions() {
    QList<QString> allExtensions = getActionExtensionMap().keys();

    while (!allExtensions.isEmpty())
        unloadActionExtension(allExtensions.takeFirst());
}

}
