/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2012 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/
#include <pml/Atom.h>
#include <pml/Cell.h>
#include <pml/PhysicalModel.h>

#include "LoadsEditor.h"
#include "LoadsManager.h"
#include "PMManagerDC.h"

#include "AtomDC.h"
#include "CellDC.h"

#include "LoadsAppliedTo.h"
#include "LoadsValueEvent.h"

#include <lml/Loads.h>
#include <lml/TranslationUnit.h>
#include <lml/RotationUnit.h>
#include <lml/ForceUnit.h>
#include <lml/PressureUnit.h>

#include <QRadioButton>


//---------------------- Constructor --------------------
LoadsEditor::LoadsEditor(LoadsManager * myLoadsManager, QWidget* parent) : QDialog(parent) {
  ui.setupUi(this);
  setModal(true);
  myLM = myLoadsManager;
  init();
}

LoadsEditor::LoadsEditor(std::vector <camitk::Component *> selectedDC, LoadsManager * myLoadsManager, QWidget* parent) : QDialog(parent) {
  ui.setupUi(this);

  myLM = myLoadsManager;
  init();

  // prepare a new load using the DCs
  createDCLoads(&selectedDC);
}

//---------------------- destructor --------------------
LoadsEditor::~LoadsEditor() {
  if (loads)
    delete loads;

  loads = NULL;
}

//---------------------- show --------------------
void LoadsEditor::show() {
  editedLoad = NULL;

  editLoad();

  QDialog::show();
}

//---------------------- init --------------------
void LoadsEditor::init() {
  editedLoad = NULL;
  loads = NULL;

  // copy the current load list from the LoadsManager
  updateLoads();

  // display load list
  updateLoadsTable();

  connect(ui.nullDisplCheckBox, SIGNAL(clicked()), this, SLOT(nullDisplacementClicked()));
  connect(ui.applyButton, SIGNAL(clicked()), this, SLOT(apply()));
  connect(ui.loadsTable, SIGNAL(itemActivated(QTableWidgetItem*)), this, SLOT(loadTableItemClicked(QTableWidgetItem*)));
  connect(ui.deleteValueEventButton, SIGNAL(clicked()), this, SLOT(deleteValueEvent()));
  connect(ui.newValueEventButton, SIGNAL(clicked()), this, SLOT(newValueEvent()));
  connect(ui.unitComboBox, SIGNAL(activated(const QString&)), this, SLOT(unitChanged()));
  connect(ui.x, SIGNAL(valueChanged()), this, SLOT(xChanged()));
  connect(ui.y, SIGNAL(valueChanged()), this, SLOT(yChanged()));
  connect(ui.z, SIGNAL(valueChanged()), this, SLOT(zChanged()));
  connect(ui.applyToButton, SIGNAL(clicked()), this, SLOT(applyToClicked()));
  connect(ui.applyToLineEdit, SIGNAL(textChanged(const QString&)), this, SLOT(applyToTextChanged(const QString&)));
  connect(ui.applyToLineEdit, SIGNAL(returnPressed()), this, SLOT(applyToReturnPressed()));
  connect(ui.deleteLoadButton, SIGNAL(clicked()), this, SLOT(deleteLoad()));
  connect(ui.newLoadButton, SIGNAL(clicked()), this, SLOT(newLoad()));
  connect(ui.resetButton, SIGNAL(clicked()), this, SLOT(reset()));
  connect(ui.valueEventTable, SIGNAL(itemActivated(QTableWidgetItem*)), this, SLOT(valueEventTableItemClicked(QTableWidgetItem*)));


}

//---------------------- createDCLoads --------------------
void LoadsEditor::createDCLoads(std::vector<camitk::Component *> *selectedDC) {
  // prepare the target list
  int atomIndex;
  AtomDC * adc;
  Atom * a;
  CellDC * cdc;
  Cell *c;
  TargetList targets;

  // loop on the selected DC and get all the atoms index

  for (unsigned int i = 0;i < selectedDC->size();i++) {
    if ((*selectedDC)[i]->isInstanceOf("AtomDC")) {
      adc = (AtomDC *)(*selectedDC)[i];
      a = adc->getAtom();
      atomIndex = a->getIndex();
      targets.add(atomIndex);
    }
    else if ((*selectedDC)[i]->isInstanceOf("CellDC")) {
      cdc = (CellDC *)(*selectedDC)[i];
      c = cdc->getCell();

      for (unsigned int j = 0; j < c->getNumberOfStructures(); j++) {
        a = (Atom *) c->getStructure(j);
        atomIndex = a->getIndex();
        targets.add(atomIndex);
      }
    }
  }

  // create a new load using current targerts
  Load *newL = createNewLoad();

  if (newL) {
    newL->setTargetList(targets);
    // add the copy to the list
    loads->addLoad(newL);
    editedLoad = newL;
    // edit it!
    editLoad();
  }
}

//---------------------- updateLoads --------------------
void LoadsEditor::updateLoads() {
  if (loads)
    delete loads;

  loads = new Loads();

  Loads *lml = myLM->getLoads();

  if (lml) {
    for (unsigned int i = 0; i < lml->numberOfLoads();i++) {
      Load *l = lml->getLoad(i);
      Load *copy;

      // instanciate the new load
      copy = Load::LoadFactory(l->getType());

      // copy everything
      copy->setUnit(l->getUnit());
      double x, y, z;
      l->getDirection(x, y, z);
      copy->setDirection(x, y, z);
      TargetList t(l->getTargetList().toString());
      copy->setTargetList(t);

      for (unsigned int j = 0; j < l->numberOfValueEvents(); j++) {
        ValueEvent *ve = l->getValueEvent(j);
        copy->addValueEvent(ve->getValue(), ve->getDate());
      }

      // add the copy to the list
      loads->addLoad(copy);
    }
  }
}

//---------------------- updateLoadsTable --------------------
void LoadsEditor::updateLoadsTable() {
  // erase all items
  ui.loadsTable->clearContents();
  ui.loadsTable->setSortingEnabled(false); // sorting by load type
  ui.loadsTable->setRowCount(loads->numberOfLoads());

  // complete the "added load" table
  for (unsigned int i = 0;i < loads->numberOfLoads();i++) {
    // get current load
    Load * currentLoad = loads->getLoad(i);

    // get type of current load
    ui.loadsTable->setItem(i, 0, new QTableWidgetItem(currentLoad->getType().c_str()));

    // get applied to of current load
    ui.loadsTable->setItem(i, 1, new QTableWidgetItem((currentLoad->getTargetList().toString().c_str() == "" ? "N/A" : currentLoad->getTargetList().toString().c_str())));

    // get the direction
    double x, y, z;
    currentLoad->getDirection(x, y, z);

    if (currentLoad->getType().c_str() == "Translation" &&
        currentLoad->getDirection().isXNull() &&
        currentLoad->getDirection().isYNull() &&
        currentLoad->getDirection().isZNull()) {//x==0.0 && y==0.0 && z==0.0) {
      ui.loadsTable->setItem(i, 2, new QTableWidgetItem("null displ."));
    }
    else {
      ui.loadsTable->setItem(i, 2, new QTableWidgetItem("(" + QString::number(x,'g',2) + "," + QString::number(y,'g', 2) + "," + QString::number(z, 'g', 2) + ")"));
    }

    // get the unit
    ui.loadsTable->setItem(i, 3, new QTableWidgetItem(currentLoad->getUnit().getUnitName().c_str()));

    // get first event of current load
    ValueEvent * ve = currentLoad->getValueEvent(0);

    // first event date
    ui.loadsTable->setItem(i, 4, new QTableWidgetItem((ve) ? QString::number(ve->getDate()) : "N/A"));

    // first event value
    ui.loadsTable->setItem(i, 5, new QTableWidgetItem((ve) ? QString::number(ve->getValue()) : "N/A"));

    if (editedLoad == currentLoad) {
      ui.loadsTable->setCurrentCell(i, 0);
    }

  }

  // sorting
  ui.loadsTable->setSortingEnabled(true);
  ui.loadsTable->sortByColumn(0, Qt::AscendingOrder); // sort by type
}

//---------------------- updateValueEventTable --------------------
void LoadsEditor::updateValueEventTable() {
  ui.valueEventTable->clearContents();
  ui.valueEventTable->setSortingEnabled(false);
  ui.valueEventTable->setRowCount(editedLoad->numberOfValueEvents());


  for (unsigned int i = 0; i < editedLoad->numberOfValueEvents(); i++) {
    ValueEvent *ve = editedLoad->getValueEvent(i);
    ui.valueEventTable->setItem(i, 0, new QTableWidgetItem(QString::number(ve->getDate())));
    ui.valueEventTable->setItem(i, 1, new QTableWidgetItem(QString::number(ve->getValue())));
    ui.valueEventTable->setCurrentCell(i, 0);
  }

  // sorting
  ui.valueEventTable->setSortingEnabled(true);
  ui.valueEventTable->sortByColumn(0, Qt::AscendingOrder); // sort by date
}


//---------------------- createNewLoad --------------------
Load* LoadsEditor::createNewLoad() {
  // create new dialog
  QDialog *dlg = new QDialog(this);
  dlg->setWindowTitle("Choose New Load Type...");

  QVBoxLayout * dialogLayout = new QVBoxLayout(dlg);

  // build the interface
  // load type
  QGroupBox *groupBox = new QGroupBox(tr("Choose load type"), dlg);

  //QButtonGroup *grp = new QButtonGroup(dlg);
  //grp->setExclusive(true);
  QRadioButton *bt1 = new QRadioButton("Translation");
  QRadioButton *bt2 = new QRadioButton("Force");
  QRadioButton *bt3 = new QRadioButton("Rotation");
  QRadioButton *bt4 = new QRadioButton("Pressure");
  bt1->setChecked(true); // default

  QVBoxLayout *vbox = new QVBoxLayout;
  vbox->addWidget(bt1);
  vbox->addWidget(bt2);
  vbox->addWidget(bt3);
  vbox->addWidget(bt4);
  vbox->addStretch(1);
  groupBox->setLayout(vbox);

  dialogLayout->addWidget(groupBox);

  // ok/cancel
  QHBoxLayout *hbox = new QHBoxLayout(/*vbox*/);
  QPushButton *ok = new QPushButton("Ok");
  QPushButton *cancel = new QPushButton("Cancel");
  ok->setDefault(true);
  hbox->addWidget(ok);
  hbox->addWidget(cancel);
  dialogLayout->addWidget(ok);
  dialogLayout->addWidget(cancel);
  dlg->setLayout(dialogLayout);
  dlg->connect(ok, SIGNAL(clicked()), dlg, SLOT(accept()));
  dlg->connect(cancel, SIGNAL(clicked()), dlg, SLOT(reject()));

  Load * l = NULL;

  if (dlg->exec() == QDialog::Accepted) {
    if (bt1->isChecked()) {
      l = Load::LoadFactory("Translation");
    }

    if (bt2->isChecked()) {
      l = Load::LoadFactory("Force");
    }

    if (bt3->isChecked()) {
      l = Load::LoadFactory("Rotation");
    }

    if (bt4->isChecked()) {
      l = Load::LoadFactory("Pressure");
    }

  }

  delete dlg;

  return l;
}

//---------------------- newLoad --------------------
void LoadsEditor::newLoad() {
  // first ask the user what can of load (translation, force, rotation, pressure)
  Load *newL = createNewLoad();

  if (newL) {
    // then insert it into load
    loads->addLoad(newL);
    editedLoad = newL;
    // edit it!
    editLoad();
  }
}

//---------------------- deleteLoad --------------------
void LoadsEditor::deleteLoad() {
  editedLoad = NULL;

  // delete current load
  int id = ui.loadsTable->currentRow();

  if (id >= 0) {
    loads->deleteLoad(id);

    // update the table
    updateLoadsTable();
  }
}

//---------------------- loadTableItemClicked --------------------
void LoadsEditor::loadTableItemClicked(QTableWidgetItem *) {
  // delete current load
  int id = ui.loadsTable->currentRow();

  if (id >= 0) {
    editedLoad = loads->getLoad(id);
    // update the load ui
    editLoad();
  }
}

//---------------------- editLoad --------------------
void LoadsEditor::editLoad() {
  ui.editGroupBox->setEnabled(editedLoad != NULL);

  if (editedLoad) {
    //-- update the GUI using editedLoad
    ui.editGroupBox->setTitle(QString("Edit %1").arg(editedLoad->getType().c_str()));

    // applyTo
    bgColor = ui.applyToLineEdit->palette().color(QPalette::Base);
    ui.applyToLineEdit->setEnabled(true);
    ui.applyToLineEdit->blockSignals(true);
    ui.applyToLineEdit->setText(editedLoad->getTargetList().toString().c_str());
    ui.applyToLineEdit->blockSignals(false);

    // direction
    double lx, ly, lz;
    editedLoad->getDirection(lx, ly, lz);
    ui.x->setEnabled(true);
    ui.y->setEnabled(true);
    ui.z->setEnabled(true);
    ui.x->setValue(lx);
    ui.y->setValue(ly);
    ui.z->setValue(lz);
    ui.nullDisplCheckBox->setEnabled(false);
    ui.nullDisplCheckBox->setChecked(false);

    // prepare the unit depending on the type
    std::string type = editedLoad->getType();
    ui.unitComboBox->clear();

    if (type == "Translation") {
      ui.unitComboBox->insertItem(0, "mm");
      ui.unitComboBox->insertItem(1, "µm");
      ui.unitComboBox->insertItem(2, "nm");
      ui.nullDisplCheckBox->setEnabled(true);
      updateND();
    }

    if (type == "Rotation") {
      ui.unitComboBox->insertItem(0, "radians");
      ui.unitComboBox->insertItem(1, "degrees");
    }

    if (type == "Force") {
      ui.unitComboBox->insertItem(0, "pN");
      ui.unitComboBox->insertItem(1, "N");
      ui.unitComboBox->insertItem(2, "kN");
    }

    if (type == "Pressure") {
      ui.unitComboBox->insertItem(0, "mmHg");
      ui.unitComboBox->insertItem(1, "kPa");
    }

    std::string u = editedLoad->getUnit().getUnitName();

    for (int i = 0; i < ui.unitComboBox->count(); i++) {
      if (u.c_str() == ui.unitComboBox->itemText(i))
        ui.unitComboBox->setCurrentIndex(i);
    }

    // update value events
    updateValueEventTable();

  }

  // update the table
  updateLoadsTable();
}

//---------------------- updateND --------------------
void LoadsEditor::updateND() {
  Direction ld = editedLoad->getDirection();
  bool disableSliders = (ld.isXNull() && ld.isYNull() && ld.isZNull());
  ui.x->setEnabled(!disableSliders);
  ui.y->setEnabled(!disableSliders);
  ui.z->setEnabled(!disableSliders);

  ui.nullDisplCheckBox->setChecked(disableSliders);

  /// \todo do something to set the "unspecified" flag
}

//---------------------- nullDisplacementClicked --------------------
void LoadsEditor::nullDisplacementClicked() {
  if (ui.nullDisplCheckBox->isChecked()) {
    Direction ld;
    ld.setNullX();
    ld.setNullY();
    ld.setNullZ();

    editedLoad->setDirection(ld);

    updateND();

    ui.x->setValue(0.0);
    ui.y->setValue(0.0);
    ui.z->setValue(0.0);

    // check box is now checked
    ui.x->setEnabled(false);
    ui.y->setEnabled(false);
    ui.z->setEnabled(false);
  }
  else {
    // check box is now not checked
    ui.x->setEnabled(true);
    ui.y->setEnabled(true);
    ui.z->setEnabled(true);
  }

  // update the table
  updateLoadsTable();
}

//---------------------- applyToTextChanged --------------------
void LoadsEditor::applyToTextChanged(const QString&) {
  QPalette palette;
  palette.setColor(QPalette::Base, QColor(255, 220, 168));
  ui.applyToLineEdit->setPalette(palette);
}

//---------------------- applyToReturnPressed --------------------
void LoadsEditor::applyToReturnPressed() {
  // check if the new loads are possibles
  PhysicalModel *pm = myLM->getPMManagerDC()->getPhysicalModel();
  TargetList ok;
  TargetList input(ui.applyToLineEdit->text().toStdString());

  for (unsigned int i = 0;i < input.getNumberOfTargets();i++) {
    int id = input.getIndexedTarget(i);

    if (pm->getAtom(id))
      ok.add(id);
  }

  // use the new value
  editedLoad->setTargetList(ok);

  // display the interpreted value
  bool alreadyBlocked = ui.applyToLineEdit->signalsBlocked();
  if (!alreadyBlocked)
    ui.applyToLineEdit->blockSignals(true);
  ui.applyToLineEdit->setText(editedLoad->getTargetList().toString().c_str());
  if (!alreadyBlocked)
    ui.applyToLineEdit->blockSignals(false);

  // reset the bgd color
  QPalette palette;
  palette.setColor(QPalette::Base, bgColor);
  ui.applyToLineEdit->setPalette(palette);

  // update the table
  updateLoadsTable();
}

//---------------------- applyToClicked --------------------
void LoadsEditor::applyToClicked() {
  // open the applied to dialog
  LoadsAppliedTo *appliedToDialog = new LoadsAppliedTo(editedLoad->getTargetList(), this);
  appliedToDialog->show();
}

//---------------------- xChanged --------------------
void LoadsEditor::xChanged() {
  double lx, ly, lz;
  editedLoad->getDirection(lx, ly, lz);
  lx = ui.x->getValue();
  editedLoad->setDirection(lx, ly, lz);

  // update null displ
  updateND();

  // update the table
  updateLoadsTable();
}

//---------------------- yChanged --------------------
void LoadsEditor::yChanged() {
  double lx, ly, lz;
  editedLoad->getDirection(lx, ly, lz);
  ly = ui.y->getValue();
  editedLoad->setDirection(lx, ly, lz);

  // update null displ
  updateND();

  // update the table
  updateLoadsTable();
}

//---------------------- zChanged --------------------
void LoadsEditor::zChanged() {
  double lx, ly, lz;
  editedLoad->getDirection(lx, ly, lz);
  lz = ui.z->getValue();
  editedLoad->setDirection(lx, ly, lz);

  // update null displ
  updateND();

  // update the table
  updateLoadsTable();
}

//---------------------- unitChanged --------------------
void LoadsEditor::unitChanged() {
  // set the correct unit
  if ((ui.unitComboBox->currentText()) == "mm") {
    editedLoad->setUnit(TranslationUnit::MM);
  } else if ((ui.unitComboBox->currentText()) == "µm") {
    editedLoad->setUnit(TranslationUnit::MICRO_M);
  } else if ((ui.unitComboBox->currentText()) == "nm") {
    editedLoad->setUnit(TranslationUnit::NM);
  } else if ((ui.unitComboBox->currentText()) == "radians") {
    editedLoad->setUnit(RotationUnit::RAD);
  } else if ((ui.unitComboBox->currentText()) == "degrees") {
    editedLoad->setUnit(RotationUnit::DEG);
  } else if ((ui.unitComboBox->currentText()) == "pN") {
    editedLoad->setUnit(ForceUnit::PN);
  } else if ((ui.unitComboBox->currentText()) == "N") {
    editedLoad->setUnit(ForceUnit::N);
  } else if ((ui.unitComboBox->currentText()) == "kN") {
    editedLoad->setUnit(ForceUnit::KN);
  } else if ((ui.unitComboBox->currentText()) == "kPa") {
    editedLoad->setUnit(PressureUnit::KPA);
  } else if ((ui.unitComboBox->currentText()) == "mmHg") {
    editedLoad->setUnit(PressureUnit::MMHG);
  }

  // update the table
  updateLoadsTable();
}

//---------------------- newValueEvent --------------------
void LoadsEditor::newValueEvent() {
  // open the valueevent dialog
  LoadsValueEvent *valueEventDialog = new LoadsValueEvent(this);
  if (valueEventDialog->exec()==QDialog::Accepted) {
    // add the value event
    editedLoad->addValueEvent(valueEventDialog->getValue(), valueEventDialog->getDate());

    // refresh the display
    updateValueEventTable();
  }

  delete valueEventDialog;
}

//---------------------- deleteValueEvent --------------------
void LoadsEditor::deleteValueEvent() {
  // delete the current row
  int id = ui.valueEventTable->currentRow();
  if (id >= 0) {
    delete ui.valueEventTable->takeItem(id, 0);
    delete ui.valueEventTable->takeItem(id, 1);

    // recreate the list of value event
    std::vector <ValueEvent *> newL;

    for (int i = 0; i < ui.valueEventTable->rowCount(); i++) {
      double date = ui.valueEventTable->item(i, 0)->text().toDouble();
      double value = ui.valueEventTable->item(i, 1)->text().toDouble();
      newL.push_back(new ValueEvent(value, date));
    }

    // replace old list with this one
    editedLoad->setAllEvents(newL);

    // refresh the display
    updateValueEventTable();
  }
}

//---------------------- valueEventTableItemClicked --------------------
void LoadsEditor::valueEventTableItemClicked(QTableWidgetItem *) {
  // get the current value event
  int id = ui.valueEventTable->currentRow();

  if (id >= 0) {
    ValueEvent *ve = editedLoad->getValueEvent(id);

    // open the value event dialog
    LoadsValueEvent *valueEventDialog = new LoadsValueEvent(this, ve->getDate(), ve->getValue());
    if (valueEventDialog->exec()==QDialog::Accepted) {
      // add the value event
      ve->setDate(valueEventDialog->getDate());
      ve->setValue(valueEventDialog->getValue());

      // refresh the display
      updateValueEventTable();
    }

    delete valueEventDialog;
  }

}

//---------------------- reset --------------------
void LoadsEditor::reset() {
  init();
}

//---------------------- reject --------------------
void LoadsEditor::reject() {
  // nothing to be done
  QDialog::reject();
}

//---------------------- apply --------------------
void LoadsEditor::apply() {
  // first clean the old ones
  myLM->deleteAllLoads();

  // copy loads back to myLM
  for (unsigned int i = 0; i < loads->numberOfLoads(); i++) {
    myLM->addLoad(loads->getLoad(i));
  }

  myLM->updateLoadsDisplay();
}
