/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2012 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/

#include <pml/PhysicalModel.h>

#include "PMManagerDC.h"
#include "PMManagerDCPopup.h"
#include "StructuralComponentDC.h"
#include "CellDC.h"
#include "AtomDC.h"
#include "MultiComponentDC.h"
#include "LoadsManager.h"
#include "AtomDCWidget.h"

#include <Application.h>
#include <InteractiveViewer.h>

#include <vtkUnstructuredGrid.h>
#include <vtkHexahedron.h>
#include <vtkWedge.h>
#include <vtkQuad.h>

#include <QFileInfo>

using namespace camitk;

// -------------------- default constructor --------------------
PMManagerDC::PMManagerDC(const QString &fileName) throw(AbortException) : MeshComponent(fileName) {
    myPM = NULL;
    initialBoundingRadius = 0.0;
    myLoadsManager = NULL;
    myAtomDCWidget = NULL;

    // create the Physical model DCs
    try {

        if (!QFileInfo(myFileName).exists()) {
            throw AbortException("Cannot Open File:" + myFileName.toStdString() + ": file does not exist!");
        }

        //-- create the PhysicalModel
        // directly instantiate physical model
        myPM = new PhysicalModel(myFileName.toStdString().c_str());

        if (!myPM || myPM->getAtoms() == NULL) {
            throw AbortException("Cannot Open PML, either not a valid PML document or no atoms in this PML (for file:" + myFileName.toStdString() + ")");
        }

        computeBoundingRadius();

        // -- initialize fields
        myPopupMenu = NULL;

        setName(myPM->getName().c_str());

        // build all the sub DCs
        buildPhysicalModelDCs();

        // add it in the InteractiveViewer
        setVisibility(InteractiveViewer::get3DViewer(), true);

        // -- create the Loads Manager
        myLoadsManager = new LoadsManager(this);

        //-- init fake representation
        initRepresentation();

        //--reset modification flag
        setModified(false);
    }
    catch (AbortException & e) {
        // has to do that, because the PMManagerDC constructor
        // inherits from Component constructor, which register the object
        // as a member of the DataManager, i.e. myDCs.size() is equal to 1
        // although the DC is not completely initialized!
        // delete all the DCs (the Manager DC included)
        while (getChildren().size() > 0) {
            Component *dc = getChildren().last();
            removeChild(dc);
        }

        childrenComponent.clear();

        // re-throw the exception
        throw e;
    }
}

// --------------------- constructor ----------------------------
PMManagerDC::PMManagerDC(PhysicalModel *pm, const QString &fileName) : MeshComponent(fileName) {
    myPM = pm;
    initialBoundingRadius = 0.0;
    myLoadsManager = NULL;
    myAtomDCWidget = NULL;

    computeBoundingRadius();

    // -- initialize fields
    myPopupMenu = NULL;

    setName(myPM->getName().c_str());

    // build all the sub DCs
    buildPhysicalModelDCs();

    // add it in the InteractiveViewer
    setVisibility(InteractiveViewer::get3DViewer(), true);

    // -- create the Loads Manager
    myLoadsManager = new LoadsManager(this);

    //-- init fake representation
    initRepresentation();

    //--reset modification flag
    setModified(false);
}


// ---------------------- destructor ----------------------------
PMManagerDC::~PMManagerDC() {
    delete myPopupMenu;
    myPopupMenu = NULL;

    delete myAtomDCWidget;
    myAtomDCWidget = NULL;

    delete myLoadsManager;
    myLoadsManager = NULL;

    // remove all the adc myself
    for (unsigned int i = 0;i < myPM->getAtoms()->getNumberOfStructures(); i++) {
        AtomDC * adc = getDC(dynamic_cast<Atom*>(myPM->getAtoms()->getStructure(i)));
        delete adc;
    }

    delete myPM;
    myPM = NULL;
}

// -------------------- initRepresentation --------------------
void PMManagerDC::initRepresentation() {
    myGeometry = new Geometry(getName(), vtkSmartPointer<vtkUnstructuredGrid>::New(), InterfaceGeometry::None);
}

// -------------------- progressOneStep --------------------
void PMManagerDC::progressOneStep() {
    nrOfDoneSteps++;
    float done = (100.0 * ((float)nrOfDoneSteps) / (float)nrOfSteps);

    if (done > 99.0)
        nrOfDoneSteps--;

    Application::setProgressBarValue(50.0 + done*0.5);
}


// -------------------- buildPhysicalModelDCs --------------------
void PMManagerDC::buildPhysicalModelDCs() {
    nrOfDoneSteps = 0;
    nrOfSteps =  1 + myPM->getNumberOfCells();

    // create and add the atoms SC
    StructuralComponent *sc = myPM->getAtoms();

    if (sc)
        new StructuralComponentDC(this, this, sc);

    // create and add the exclusive components
    if (myPM->getNumberOfExclusiveComponents() > 0)
        new MultiComponentDC(this, this, myPM->getExclusiveComponents());

    // create and add the informative component
    if (myPM->getNumberOfInformativeComponents() > 0)
        new MultiComponentDC(this, this, myPM->getInformativeComponents());
}

// -------------------- createPointData --------------------
void PMManagerDC::createPointData() {
    // ask the exclusive and informative component to create the point data
    foreach(Component *child, getChildren()) {
        if (child->isInstanceOf("MultiComponentDC")) {
            dynamic_cast<MultiComponentDC *>(child)->createPointData();
        }
    }
}

// -------------------- destroyPointData --------------------
void PMManagerDC::destroyPointData() {
    // destroy all point data
    foreach(Component *child, getChildren()) {
        if (child->isInstanceOf("MultiComponentDC")) {
            dynamic_cast<MultiComponentDC *>(child)->destroyPointData();
        }
    }
}

// -------------------- setName --------------------
void PMManagerDC::setName(const QString & n) {
    myPM->setName(n.toStdString());
    Component::setName(n);
}

// -------------------- getPopupMenu --------------------
QMenu * PMManagerDC::getPopupMenu(QWidget* parent) {
    if (!myPopupMenu) {
        myPopupMenu = new PMManagerDCPopup(this, parent);
    }

    dynamic_cast<PMManagerDCPopup *>(myPopupMenu)->updateMenuActions();

    return myPopupMenu;
}

//------------------------ getPixmap ---------------------
#include "physicalmodel_20x20.xpm"
QPixmap * PMManagerDC::myPixmap = NULL;
QPixmap PMManagerDC::getIcon() {
    if (!myPixmap) {
        myPixmap = new QPixmap(physicalmodel_20x20);
    }

    return (*myPixmap);
}

// ---------------------- getDC ----------------------------
MultiComponentDC * PMManagerDC::getDC(MultiComponent *mc) {
    if (!mc)
        return NULL;

    std::ComponentDCMapIterator result = myMCDCMap.find(dynamic_cast< ::Component *>(mc));

    return (result == myMCDCMap.end()) ? NULL : dynamic_cast<MultiComponentDC *>(result->second);
}

StructuralComponentDC * PMManagerDC::getDC(StructuralComponent *sc) {
    if (!sc)
        return NULL;

    std::ComponentDCMapIterator result = mySCDCMap.find(dynamic_cast< ::Component *>(sc));

    return (result == mySCDCMap.end()) ? NULL : dynamic_cast<StructuralComponentDC *>(result->second);
}

// ---------------------- static getModifiedFlag ----------------------------
bool PMManagerDC::getModified() const {
    return (modifiedFlag || myLoadsManager->isModified());
}

// -------------------- Cell / DC Map : addCellDCPair --------------------
void PMManagerDC::addCellDCPair(std::ComponentDCPair p) {
    // cout << "add a new pair <" << p.first << "," << p.second << ">" << endl;
    myCDCMap.insert(p);
}

// -------------------- Cell / DC Map : : getDC --------------------
CellDC * PMManagerDC::getDC(Cell *sc) {
    /* Component *c;
     c = (StructuralComponent *) sc;
     cout << "search for " << sc << ": " << (Component *) sc << " / " << c << endl;
     */
    std::ComponentDCMapIterator result = myCDCMap.find(dynamic_cast< ::Component *>(sc));
    return (result == myCDCMap.end()) ? NULL : dynamic_cast<CellDC *>(result->second);
}

/* TODO CLEANUP when the class PMManagerDC is updated to CamiTK 2.0 in order to inherits from Mesh and
   that Mesh manages export properly
   => fix export

// ---------------------- vtkToPhysicalModel ----------------------------
PhysicalModel * PMManagerDC::geometryToPhysicalModel(Geometry *geom) {
  PhysicalModel * myPM = NULL;

  if (!geom) {
    return myPM;
  }

  // first step: initialize
  vtkSmartPointer<vtkPointSet> ds = geom->getPointSet();

  //  unsigned int nrOfPoints = ds->GetNumberOfPoints();
  unsigned int nrOfCells = ds->GetNumberOfCells();

  // this part of the method extract the atoms from the Geometry
  vtkSmartPointer<vtkPoints> thePoints = vtkSmartPointer<vtkPoints>::New();

  thePoints->DeepCopy(ds->GetPoints());

  // create a new physical model
  myPM = new PhysicalModel();

  // create the structural components for the atoms
  myPM->setAtoms(vtkToPhysicalModel(myPM, thePoints, "The atoms"));

  // create the unique structural components containing all the vtkCell, this is an exclusive component
  StructuralComponent *sc;

  sc = new StructuralComponent(myPM, "The cells (VTK import)");

  // fill in this new structure
  Cell *c;

  for (unsigned int i = 0; i < nrOfCells; i++) {
    // create a cell for each cell
    c = (Cell *) vtkToPhysicalModel(myPM, ds->GetCell(i), myPM->getAtoms());
    // insert the cell in the structure
    sc->addStructure(c, false);
  }

  // create the exclusive multi-component
  MultiComponent * exclusiveComponents = new MultiComponent(myPM);

  // insert all the cells (i.e. the sc component)
  exclusiveComponents->addSubComponent(sc);

  // insert this exclusive component into the pm
  myPM->setExclusiveComponents(exclusiveComponents);

  return myPM;
}

// ---------------------- vtkToPhysicalModel ----------------------------
PhysicalModel * PMManagerDC::vtkToPhysicalModel(const QString& vtkFileName) {
  // do that step by step to avoid memory leak of the Geometry
  Geometry * o = VtkMeshUtil::meshToGeometry(NULL, vtkFileName.toStdString());
  PhysicalModel * myPM = geometryToPhysicalModel(o);
  delete o;
  return myPM;
}

// --------------- vtkToPhysicalModel -------------------
StructuralComponent * PMManagerDC::vtkToPhysicalModel(PhysicalModel *pm, vtkSmartPointer<vtkPoints> thePoints, const QString& n) {
  Atom *a;
  double pos[3];
  StructuralComponent *sc;

  // instanciate the structural component
  sc = new StructuralComponent(pm, n.toStdString());

  // create the atom structures

  for (int i = 0; i < thePoints->GetNumberOfPoints(); i++) {
    thePoints->GetPoint(i, pos);
    a = new Atom(pm, pos);
    sc->addStructure(a, false);
  }

  return sc;
}

// --------------- vtkToPhysicalModel -------------------
StructuralComponent *PMManagerDC::vtkToPhysicalModel(PhysicalModel *pm, vtkSmartPointer<vtkCell> theCell, StructuralComponent *atoms) {
  Cell *c;
  int type;
  StructureProperties::GeometricType cellType = StructureProperties::INVALID;
  unsigned int ptId;
  Atom *a;

  // get the type
  type = theCell->GetCellType();
  // create a new cell of this type

  switch (type) {
    case VTK_TETRA:
      cellType = StructureProperties::TETRAHEDRON;
      break;
    case VTK_HEXAHEDRON:
      cellType = StructureProperties::HEXAHEDRON;
      break;
    case VTK_WEDGE:
      cellType = StructureProperties::WEDGE;
      break;
    case VTK_POLY_LINE:
      cellType = StructureProperties::POLY_LINE;
      break;
    case VTK_POLY_VERTEX:
      cellType = StructureProperties::POLY_VERTEX;
      break;
    case VTK_TRIANGLE:
      cellType = StructureProperties::TRIANGLE;
      break;
    case VTK_QUAD:
      cellType = StructureProperties::QUAD;
      break;
  }

  c = new Cell(pm, cellType);

  // fill-in the cell "structures"

  for (int i = 0; i < theCell->GetNumberOfPoints(); i++) {
    // get the index of the point
    ptId = theCell->GetPointId(i);
    // get the corresponding atom
    a = (Atom *) atoms->getStructure(ptId);
    // set the corresponding atom as a structure composing the cell
    c->addStructure(a, false);
  }

  return c;
}

// --------------- physicalModelToVtk -------------------
bool PMManagerDC::physicalModelToVtk(PhysicalModel *myPM, const QString& fName) {
  // build a big structural component containing all the exclusive components' cell
  unsigned int totalNumberOfCells = myPM->getExclusiveComponents()->getNumberOfCells();

  // instanciate the structural component
  StructuralComponent allTheCells(myPM, myPM->getName());

  // create all the cells
  unsigned int i;

  for (i = 0; i < totalNumberOfCells; i++) {
    allTheCells.addStructure(myPM->getExclusiveComponents()->getCell(i), false);
  }

  // create a new unstructured grid from allTheCellsDC
  vtkUnstructuredGrid * uGrid = StructuralComponentDC::structuralComponentToVtk(&allTheCells);

  // save this in the vtk file
  VtkMeshUtil::saveUnstructuredGridToFile(uGrid, fName.toStdString(), myPM->getName());

  return true; // !!!
}

// --------------- physicalModelToVtkAllNodes -------------------
bool PMManagerDC::physicalModelToVtkAllNodes(PhysicalModel *myPM, const QString& fName) {
  unsigned int i, j;
  double pos[3];

  // Directly write all the atoms and cell, even if some atoms are not used
  // this is ABSOLUTELY REQUIRED to export the PM for Ansys, and to use the informative components

  // insert ALL the atoms
  vtkSmartPointer<vtkPoints> points = vtkSmartPointer<vtkPoints>::New();
  points->Allocate(myPM->getNumberOfAtoms(), myPM->getNumberOfAtoms());
  points->SetNumberOfPoints(myPM->getNumberOfAtoms());

  for (i = 0; i < myPM->getNumberOfAtoms(); i++) {
    myPM->getAtom(i)->getPosition(pos);
    points->InsertPoint(i, pos[0], pos[1], pos[2]);
  }


  // create the VTK unstructured grid
  vtkSmartPointer<vtkUnstructuredGrid> ugrid = vtkSmartPointer<vtkUnstructuredGrid>::New();

  ugrid->Allocate(myPM->getNumberOfCells(), myPM->getNumberOfCells());

  ugrid->SetPoints(points);



  // insert all the cells
  // tant pis p�ur les mempry leak, c'est minuit et je suis press� de partir...
  //vtkTetra * tetra = NULL;
  vtkSmartPointer<vtkHexahedron> hex = NULL;

  vtkSmartPointer<vtkWedge> wedge = NULL;

  //vtkPolyLine * polyLine = NULL;
  //vtkPolyVertex * polyVertex = NULL;
  //vtkTriangle * triangle = NULL;
  vtkSmartPointer<vtkQuad> quad = NULL;

  // for (i=0; i<myPM->getNumberOfCells(); i++)
  for (i = 0; i < myPM->getExclusiveComponent(0)->getNumberOfCells(); i++) {
    //Cell * c = (Cell *) myPM->getCell(i);
    Cell * c = (Cell *) myPM->getExclusiveComponent(0)->getCell(i);

    // if (c->getType() == StructureProperties::HEXAHEDRON)
    int type = (int) c->getType();

    if (type == StructureProperties::HEXAHEDRON) {
      hex = vtkSmartPointer<vtkHexahedron>::New();

      for (j = 0; j < c->getNumberOfStructures(); j++) {
        Atom *a = (Atom *)(c->getStructure(j));
        hex->GetPointIds()->SetId(j, a->getIndex());
      }

      ugrid->InsertNextCell(hex->GetCellType(), hex->GetPointIds());
    } else if (type == StructureProperties::WEDGE) {
      wedge = vtkSmartPointer<vtkWedge>::New();

      for (j = 0; j < c->getNumberOfStructures(); j++) {
        Atom *a = (Atom *)(c->getStructure(j));
        wedge->GetPointIds()->SetId(j, a->getIndex());
      }

      ugrid->InsertNextCell(wedge->GetCellType(), wedge->GetPointIds());
    } else if (type == StructureProperties::QUAD) {
      quad = vtkSmartPointer<vtkQuad>::New();

      for (j = 0; j < c->getNumberOfStructures(); j++) {
        Atom *a = (Atom *)(c->getStructure(j));
        quad->GetPointIds()->SetId(j, a->getIndex());
      }

      ugrid->InsertNextCell(quad->GetCellType(), quad->GetPointIds());
    }


  }

  / *
  for (i=0; i<myPM->getNumberOfCells(); i++)
  {
   Cell * c = (Cell *) myPM->getCell(i);

   // create a new cell of this type
   switch (c->getType())
   {
    case StructureProperties::TETRAHEDRON:
  //    tetra = vtkTetra::New();
  //    for (j=0; j<c->getNumberOfStructures(); c++)
  //     tetra->GetPointIds()->SetId(j, c->getStructure(j)->getIndex());
  //    ugrid->InsertNextCell(tetra->GetCellType(), tetra->GetPointIds());


     break;
    case StructureProperties::HEXAHEDRON:
     hex = vtkHexahedron::New();
     for (j=0; j<c->getNumberOfStructures(); c++)
      hex->GetPointIds()->SetId(j, c->getStructure(j)->getIndex());
     ugrid->InsertNextCell(hex->GetCellType(), hex->GetPointIds());

     break;
    case StructureProperties::WEDGE:
     wedge = vtkWedge::New();
     for (j=0; j<c->getNumberOfStructures(); c++)
      wedge->GetPointIds()->SetId(j, c->getStructure(j)->getIndex());
     ugrid->InsertNextCell(wedge->GetCellType(), wedge->GetPointIds());

     break;
    case StructureProperties::POLY_LINE:
     polyLine = vtkPolyLine::New();
     for (j=0; j<c->getNumberOfStructures(); c++)
      polyLine->GetPointIds()->SetId(j, c->getStructure(j)->getIndex());
     ugrid->InsertNextCell(polyLine->GetCellType(),
  polyLine->GetPointIds());

     break;
    case StructureProperties::POLY_VERTEX:
     polyVertex = vtkPolyVertex::New();
     for (j=0; j<c->getNumberOfStructures(); c++)
      polyVertex->GetPointIds()->SetId(j,
  c->getStructure(j)->getIndex());
     ugrid->InsertNextCell(polyVertex->GetCellType(),
  polyVertex->GetPointIds());

     break;
    case StructureProperties::TRIANGLE:
     triangle = vtkTriangle::New();
     for (j=0; j<c->getNumberOfStructures(); c++)
      triangle->GetPointIds()->SetId(j, c->getStructure(j)->getIndex());
     ugrid->InsertNextCell(triangle->GetCellType(),
  triangle->GetPointIds());

     break;
    case StructureProperties::QUAD:
     quad = vtkQuad::New();
     for (j=0; j<c->getNumberOfStructures(); c++)
      quad->GetPointIds()->SetId(j, c->getStructure(j)->getIndex());
     ugrid->InsertNextCell(quad->GetCellType(), quad->GetPointIds());

     break;

    default:
     break;
   }
  * /

  // save this in the vtk file
  VtkMeshUtil::saveUnstructuredGridToFile(ugrid, fName.toStdString(), myPM->getName());

  // if (tetra)  tetra->Delete();
  //if (hex)  hex->Delete();
  //if (wedge)  wedge->Delete();
  //if (polyLine) polyLine->Delete();
  //if (polyVertex) polyVertex->Delete();
  //if (triangle) triangle->Delete();


  return true; // !!!
}

*/

// -------------------- getAtomDCWidget --------------------
QWidget * PMManagerDC::getAtomDCWidget(AtomDC *adc, QWidget *parent) {

    // create if needed
    if (!myAtomDCWidget && adc) {
        myAtomDCWidget = new AtomDCWidget(adc, parent);
    }

    // update the properties
    if (myAtomDCWidget) {
        myAtomDCWidget->updateProperties(adc);
    }

    // return the ptr
    return myAtomDCWidget;
}

// -------------------- toPMRenderingMode --------------------
::RenderingMode::Mode PMManagerDC::toPMRenderingMode(InterfaceGeometry::RenderingModes renderingModes) {
    ::RenderingMode converted;

    if (renderingModes & InterfaceGeometry::Surface) {
        converted.setVisible(::RenderingMode::SURFACE, true);
    }

    if (renderingModes & InterfaceGeometry::Wireframe) {
        converted.setVisible(::RenderingMode::WIREFRAME, true);
    }

    if (renderingModes & InterfaceGeometry::Points) {
        converted.setVisible(::RenderingMode::POINTS, true);
    }

    return converted.getMode();
}

// -------------------- toDCRenderingMode --------------------
InterfaceGeometry::RenderingModes PMManagerDC::toDCRenderingMode(::RenderingMode::Mode renderingMode) {
    InterfaceGeometry::RenderingModes converted;

    switch (renderingMode)  {
    case ::RenderingMode::NONE:
        converted = InterfaceGeometry::None;
        break;
    case ::RenderingMode::POINTS:
        converted = InterfaceGeometry::Points;
        break;
    case ::RenderingMode::POINTS_AND_SURFACE:
        converted = InterfaceGeometry::Points | InterfaceGeometry::Surface;
        break;
    case ::RenderingMode::SURFACE:
        converted = InterfaceGeometry::Surface;
        break;
    case ::RenderingMode::WIREFRAME_AND_SURFACE_AND_POINTS:
        converted = InterfaceGeometry::Points | InterfaceGeometry::Surface | InterfaceGeometry::Wireframe;
        break;
    case ::RenderingMode::WIREFRAME_AND_SURFACE:
        converted = InterfaceGeometry::Surface | InterfaceGeometry::Wireframe;
        break;
    case ::RenderingMode::WIREFRAME_AND_POINTS:
        converted = InterfaceGeometry::Points | InterfaceGeometry::Wireframe;
        break;
    case ::RenderingMode::WIREFRAME:
        converted = InterfaceGeometry::Wireframe;
        break;
    }
    return converted;
}

// --------------- computeBoundingRadius -------------------
void PMManagerDC::computeBoundingRadius() {
    // compute the bounding radius
    float minX, maxX, minY, maxY, minZ, maxZ;
    minX = minY = minZ = 999999999.9f; //MAXFLOAT;
    maxY = maxX = maxZ = -999999999.9f; //MINFLOAT;
    Atom *a;
    double pos[3];

    StructuralComponent *theAtoms;
    theAtoms = myPM->getAtoms();

    for (unsigned int i = 0;i < theAtoms->getNumberOfStructures();i++) {
        a = (Atom *) theAtoms->getStructure(i);
        a->getPosition(pos);

        if (pos[0] < minX)
            minX = pos[0];

        if (pos[0] > maxX)
            maxX = pos[0];

        if (pos[1] < minY)
            minY = pos[1];

        if (pos[1] > maxY)
            maxY = pos[1];

        if (pos[2] < minZ)
            minZ = pos[2];

        if (pos[2] > maxZ)
            maxZ = pos[2];
    }

    double xLength, yLength, zLength;

    xLength = fabs(maxX - minX);
    yLength = fabs(maxY - minY);
    zLength = fabs(maxZ - minZ);
    initialBoundingRadius = sqrt(xLength * xLength + yLength * yLength + zLength * zLength) / 2.0;
}

