# FindOpenCV.cmake file using CMake's Module Mode
# The search starts by looking if the package is installed on the system.
# If not, it looks for a copy of the package somewhere on the disk using system or user environement variables.
# SET the <package>_USE_FILE
SET( OpenCV_USE_FILE ${CMAKE_MODULE_PATH}/UseOpenCV.cmake )

if ( WIN32 )

###############################################################################
#			FIND OpenCVConfig.cmake
###############################################################################

# Check for file OpenCVConfig.cmake in specified environement variables OPENCV_DIR and PATH
IF( WIN32 )
	FIND_PATH(OpenCV_DIR OpenCVConfig.cmake PATHS ENV OPENCV_DIR ENV PATH )
ENDIF( WIN32 )

IF( UNIX )
	FIND_PATH(OpenCV_DIR OpenCVConfig.cmake PATHS /usr/share/opencv /usr/lib/opencv )
ENDIF( UNIX )

IF( OpenCV_DIR )
	# Read the CMake  file provided by OpenCV
	INCLUDE( ${OpenCV_DIR}/OpenCVConfig.cmake )
ENDIF( OpenCV_DIR )

###############################################################################
#			VERSION CHECKING
###############################################################################

# If version specified by user else using minimum version 2.0.0
IF (NOT OpenCV_MIN_VERSION)
	SET(OpenCV_MIN_VERSION "2.0.0")
ENDIF (NOT OpenCV_MIN_VERSION)

STRING(REGEX MATCH "^[0-9]+\\.[0-9]+\\.[0-9]+" req_OpenCV_major_vers "${OpenCV_MIN_VERSION}")
# Check version string correctness
IF (NOT req_OpenCV_major_vers)
	MESSAGE( FATAL_ERROR "Invalid OpenCV version STRING given: \"${OpenCV_MIN_VERSION}\", expected e.g. \"0.0.3\"")
ENDIF (NOT req_OpenCV_major_vers)

# Parsing the different version information : major minor patch
STRING(REGEX REPLACE "^([0-9]+)\\.[0-9]+\\.[0-9]+" "\\1" req_OpenCV_major_vers "${OpenCV_MIN_VERSION}")
STRING(REGEX REPLACE "^[0-9]+\\.([0-9]+)\\.[0-9]+" "\\1" req_OpenCV_minor_vers "${OpenCV_MIN_VERSION}")
STRING(REGEX REPLACE "^[0-9]+\\.[0-9]+\\.([0-9]+)" "\\1" req_OpenCV_patch_vers "${OpenCV_MIN_VERSION}")

# If version specified bu user in find_package command
IF( OpenCV_FIND_VERSION )
	SET( OpenCV_MIN_VERSION ${OpenCV_FIND_VERSION} )
	SET( req_OpenCV_major_vers ${OpenCV_FIND_VERSION_MAJOR} )
	SET( req_OpenCV_minor_vers ${OpenCV_FIND_VERSION_MINOR} )
	SET( req_OpenCV_patch_vers ${OpenCV_FIND_VERSION_PATCH} )
ENDIF( OpenCV_FIND_VERSION )

# Check if the minimum version restriction is statisfied 
IF(req_OpenCV_major_vers LESS 2)
      MESSAGE( FATAL_ERROR "Invalid OpenCV version STRING given: \"${OpenCV_MIN_VERSION}\", major version 2 is required, e.g. \"2.0.0\"")
ENDIF( req_OpenCV_major_vers LESS 2)


# Compute the mathematical version numbers for comparision
MATH(EXPR req_vers "${req_OpenCV_major_vers}*10000 + ${req_OpenCV_minor_vers}*100 + ${req_OpenCV_patch_vers}")
MATH(EXPR found_vers "${OpenCV_VERSION_MAJOR}*10000 + ${OpenCV_VERSION_MINOR}*100 + ${OpenCV_VERSION_PATCH}")

# Support for EXACT query.
IF( OpenCV_FIND_VERSION_EXACT )
  IF(found_vers EQUAL req_vers)
    SET( OpenCV_EXACT_FOUND TRUE )
  ELSE(found_vers EQUAL req_vers)
    SET( OpenCV_EXACT_FOUND FALSE )
    IF (found_vers LESS req_vers)
      SET(OpenCV_INSTALLED_VERSION_TOO_OLD TRUE)
    ELSE (found_vers LESS req_vers)
      SET(OpenCV_INSTALLED_VERSION_TOO_NEW TRUE)
    ENDIF (found_vers LESS req_vers)
  ENDIF(found_vers EQUAL req_vers)
ELSE( OpenCV_FIND_VERSION_EXACT )
  IF (found_vers LESS req_vers)
    SET(OpenCV_EXACT_FOUND FALSE)
    SET(OpenCV_INSTALLED_VERSION_TOO_OLD TRUE)
  ELSE (found_vers LESS req_vers)
    SET(OpenCV_EXACT_FOUND TRUE)
  ENDIF (found_vers LESS req_vers)
ENDIF( OpenCV_FIND_VERSION_EXACT )

###############################################################################
#			FIND MODULES
###############################################################################

# Assuming the the package is found
IF( OpenCV_EXACT_FOUND )
	SET( OpenCV_FOUND TRUE )
ENDIF( OpenCV_EXACT_FOUND )

# Support for REQUIRED query
IF( OpenCV_FOUND )
	MESSAGE( STATUS "Found OpenCV-Version ${OpenCV_VERSION} (using ${OpenCV_DIR})" )
ELSE( OpenCV_FOUND )
	IF(OpenCV_FIND_REQUIRED)
		MESSAGE( SEND_ERROR "Can't find OpenCV package. Please install the package on the system OR use a HDD copy of the package" )
	ENDIF(OpenCV_FIND_REQUIRED)
ENDIF( OpenCV_FOUND )

endif()

if ( UNIX )
  find_path(OpenCV_DIR include/opencv/cv.h
    /usr
    /usr/local
    DOC "The root of an installed open-cv installation"
  )


endif()
