#
# CamiTK Macros
#


# MACRO get_subdirectories 
#
# usage:
# get_subdirectories(Name)
#
# get the names of all the subdirectories that contains a CMakeLists.txt
macro(get_subdirectories Name)
  set(Name)
  file(GLOB LS *)
  foreach(FILENAME ${LS})
    if(IS_DIRECTORY ${FILENAME} AND EXISTS ${FILENAME}/CMakeLists.txt)
      get_directory_name(${FILENAME} DIRNAME)
      # ignore .svn
      if (NOT ${DIRNAME} STREQUAL ".svn")
          set(${Name} ${${Name}} ${DIRNAME})
      endif()
    endif()
  endforeach()
endmacro()

# MACRO gather_headers_and_sources 
#
# usage:
# gather_headers_and_sources(Name [BaseDirectory])
#
# find all headers, sources, including the qt ui, moc and resources
# and create two variables from it: ${Name}_HEADERS ${Name}_SOURCES, and define all needed commands for qt
# if BaseDirectory is specified, the glob recurse starts in directory DIR
#

macro(gather_headers_and_sources Name)
  # gather all possible C++ and Qt sources
  if (${ARGC} EQUAL 1)
    file(GLOB_RECURSE HEADERS *.h)
    file(GLOB_RECURSE SRCS *.cpp *.c)
    file(GLOB_RECURSE File_UI *.ui)
    file(GLOB_RECURSE File_QRC *.qrc)
  else()
    # if an optional parameter is used, gather everything from BaseDirectory
    file(GLOB_RECURSE HEADERS ${ARGV1}/*.h )
    file(GLOB_RECURSE SRCS ${ARGV1}/*.cpp *.c)
    file(GLOB_RECURSE File_UI ${ARGV1}/*.ui )
    file(GLOB_RECURSE File_QRC ${ARGV1}/*.qrc )
  endif()

  # manage Qt ui
  qt4_wrap_ui (UI ${File_UI})
  
  # manage Qt resources
  qt4_add_resources(QRC ${File_QRC})

  # find Q_OBJECT derived class
  foreach(HEADER ${HEADERS})
    file(READ ${HEADER} stream)
    if(stream MATCHES "Q_OBJECT")
      set(MOC_SOURCES ${MOC_SOURCES} ${HEADER})
    endif(stream MATCHES "Q_OBJECT")
  endforeach(HEADER)

  # manage Qt inherited sources
  qt4_wrap_cpp(QT_SRCS ${MOC_SOURCES})

  # name all headers
  set (${Name}_HEADERS
      ${HEADERS}
      ${UI}
      ${QRC}
  )

  # name all sources
  set (${Name}_SOURCES
      ${HEADERS}
      ${UI}
      ${QRC}
      ${SRCS}
      ${QT_SRCS}
  )
endmacro ()

# MACRO parse_arguments
#
# usage, see: http://www.cmake.org/Wiki/CMakeMacroParseArguments
macro(parse_arguments prefix arg_names option_names)
  set(DEFAULT_ARGS)
  foreach(arg_name ${arg_names})    
    set(${prefix}_${arg_name})
  endforeach()
  foreach(option ${option_names})
    set(${prefix}_${option} FALSE)
  endforeach()

  set(current_arg_name DEFAULT_ARGS)
  set(current_arg_list)
  foreach(arg ${ARGN})            
    set(larg_names ${arg_names})    
    list(FIND larg_names "${arg}" is_arg_name)                   
    if (is_arg_name GREATER -1)
      set(${prefix}_${current_arg_name} ${current_arg_list})
      set(current_arg_name ${arg})
      set(current_arg_list)
    else()
      set(loption_names ${option_names})    
      list(FIND loption_names "${arg}" is_option)            
      if (is_option GREATER -1)
       set(${prefix}_${arg} TRUE)
      else()
       set(current_arg_list ${current_arg_list} ${arg})
      endif()
    endif()
  endforeach()
  set(${prefix}_${current_arg_name} ${current_arg_list})
endmacro()

# MACRO get_directory_name
#
# usage:
# directory_name(FullPathDirectory VariableName)
#
# set the variable VariableName to the name of the last directory of FullPathDirectory
# set the variable VariableName_CMAKE to be the same but in uppercase

macro(get_directory_name FullPathDirectory VariableName)
  string (REGEX REPLACE ".*/([^/]*)$" "\\1" ${VariableName} ${FullPathDirectory})

  # set the variable name to uppercase
  string(TOUPPER ${${VariableName}} ${VariableName}_CMAKE)  
endmacro()

# MACRO component_extension
#
# Simplify writing component extension CMakeLists.txt
# 
# usage:
# component_extension([DEFAULT]
#                     [NEEDS_ITK]
#                     [NEEDS_LIBXML2]
#                     [NEEDS_OPENCV]
#                     [NEEDS_IGSTK]
#                     [NEEDS_TOOL tool1 tool2 ...]
#                     [NEEDS_COMPONENT_EXTENSION component1 component2 ...]
#                     [INCLUDE_DIRECTORIES dir1 dir2 ...]
#                     [DEFINES flag1 flag2 ...]
#                     [EXTERNAL_SOURCES file1 file2 ...]
#                     [LIBRARIES lib1 lib2 ...]
#                     [HEADERS_TO_INSTALL header1.h header2.h ...]
#                     [INSTALL_ALL_HEADERS]
#                    )
#
# The name of the component is automatically the name of the directory from where this macro
# is called.
#
# DEFAULT                   = means this is a default component to be compiled automatically
# NEEDS_ITK                 = add this if your component needs itk. 
#                             Do not forget to add the needed list of itk libraries in the LIBRARIES parameter
# NEEDS_LIBXML2             = add this if your component needs libxml2 
# NEEDS_XSD                 = add this if your action needs Codesynthesis xsd cxx (xml schema compiler)
# NEEDS_OPENCV              = add this if your component needs OpenCV
# NEEDS_IGSTK               = add this if your component needs IgsTK
# NEEDS_TOOL                = list of needed camitk tools
# NEEDS_COMPONENT_EXTENSION = list of needed component extensions
# INCLUDE_DIRECTORIES       = additional include directories
# DEFINES                   = list of define flags to add at compilation time
# EXTERNAL_SOURCES          = list of extra source/headers files (external to the current directory)
#                             that needed to be added to the SOURCES variable. 
#                             Note: EXTERNAL_SOURCES are not installed
# LIBRARIES                 = libraries to add to the link command
# HEADERS_TO_INSTALL        = list of headers to install, if present this will automatically
#                             create an "install-COMPONENT_NAMEcomponent" target, that can be used
#                             anywhere else to manage dependencies to this component.
#                             The headers are installed ${CAMITK_INCLUDE_DIR}
#                             when the target "install-COMPONENT_NAMEcomponent" is called.
# INSTALL_ALL_HEADERS       = install all of the headers (this is the lazy solution, please consider
#                             making a list and using HEADERS_TO_INSTALL parameter instead!   
macro(component_extension)

  get_directory_name(${CMAKE_CURRENT_SOURCE_DIR} COMPONENT_NAME)

  parse_arguments(${COMPONENT_NAME_CMAKE} 
    "NEEDS_TOOL;NEEDS_COMPONENT_EXTENSION;INCLUDE_DIRECTORIES;LIBRARIES;HEADERS_TO_INSTALL;DEFINES;EXTERNAL_SOURCES"  # possible lists
    "DEFAULT;NEEDS_ITK;NEEDS_LIBXML2;NEEDS_XSD;NEEDS_OPENCV;NEEDS_IGSTK;INSTALL_ALL_HEADERS" # possible options
    ${ARGN}
  )

  # if it is the first cmake run, create the component variable with a correct initial value
  if(NOT CAMITK_COMPONENT_${COMPONENT_NAME_CMAKE}_INTERNAL)
    message(STATUS "Checking ${COMPONENT_NAME}")
    # add option to enable/disable this component extension
    set(COMPONENT_${COMPONENT_NAME_CMAKE} ${${COMPONENT_NAME_CMAKE}_DEFAULT} CACHE BOOL "Build component extension ${COMPONENT_NAME}")
    set(CAMITK_COMPONENT_${COMPONENT_NAME_CMAKE}_INTERNAL TRUE CACHE BOOL "Is variable COMPONENT_${COMPONENT_NAME} already created?")
    mark_as_advanced(CAMITK_COMPONENT_${COMPONENT_NAME_CMAKE}_INTERNAL)
  endif()

  message(STATUS "Building component extension ${COMPONENT_NAME}: ${COMPONENT_${COMPONENT_NAME_CMAKE}}") 

  # if this extension is enabled, do everything needed
  # otherwise... do nothing
  if (COMPONENT_${COMPONENT_NAME_CMAKE})   
    # components are compiled as shared object 
    set(CMAKE_LIBRARY_OUTPUT_DIRECTORY ${CMAKE_LIBRARY_OUTPUT_DIRECTORY}/components) 

    # components are compiled as shared object (for dll plateform put it in BIN)
    set(CMAKE_RUNTIME_OUTPUT_DIRECTORY ${CMAKE_RUNTIME_OUTPUT_DIRECTORY}/components) 

    # include directories
    include_directories(${CAMITK_INCLUDE_DIRECTORIES})
    include_directories(${CMAKE_CURRENT_BINARY_DIR})
    include_directories(${CMAKE_CURRENT_SOURCE_DIR})
    include_directories(${${COMPONENT_NAME_CMAKE}_INCLUDE_DIRECTORIES})

    # check definitions
    if(${COMPONENT_NAME_CMAKE}_DEFINES)
      foreach (FLAG ${${COMPONENT_NAME_CMAKE}_DEFINES})
        add_definitions(-D${FLAG})
      endforeach()
    endif()

	# Looking for ITK
	set(ITK_LIBRARIES "")
    if(${COMPONENT_NAME_CMAKE}_NEEDS_ITK)
      # ITK is required
      find_package(ITK REQUIRED)
      include(${ITK_USE_FILE})
	else()
		set(ITK_LIBRARIES "")
    endif()

   # Looking for LibXML2.0
    set(LIBXML2_LIBRARIES)
    if(${COMPONENT_NAME_CMAKE}_NEEDS_LIBXML2) 
      # LibXml2 is required
      find_package(Xml2)
      if (LIBXML2_FOUND)
        add_definitions(${LIBXML2_DEFINITIONS})
        include_directories(${LIBXML2_INCLUDE_DIR})      
      else()
        # most probably win32 or crosscompiling
        message(STATUS "${COMPONENT_NAME}: libxml2 required")
      endif()
    endif()
	
    # Looking for XercesC
    set(XERCESC_LIBRARIES)
    if(${EXTENSION_NAME_CMAKE}_NEEDS_XERCESC)
      # XercesC is required
      find_package(XercesC REQUIRED)
      if (XERCESC_FOUND)
        include_directories(${XERCESC_INCLUDE_DIR})
      else()
        # most probably win32 or crosscompiling
        message(STATUS "${EXTENSION_NAME}: xerces-c required")
      endif()
    endif()

    # Looking for codesynthesis XSD CXX
    if(${EXTENSION_NAME_CMAKE}_NEEDS_XSD)
      # XercesC is required
      find_package(XercesC REQUIRED)
      if (XERCESC_FOUND)
        include_directories(${XERCESC_INCLUDE_DIR})
        find_package(XSD REQUIRED)
        include_directories(${XSD_INCLUDE_DIR})
	  else()
        # most probably win32 or crosscompiling
        message(STATUS "${EXTENSION_NAME}: xerces-c required because of XSD cxx, please set XERCESC_INCLUDE_DIR")
      endif()
    endif()

    set(OpenCV_LIBRARIES "")
    if(${COMPONENT_NAME_CMAKE}_NEEDS_OPENCV) 
		# OpenCV is required
		find_package(OpenCV 2 REQUIRED )
#		include( ${OpenCV_USE_FILE} )
	else(${COMPONENT_NAME_CMAKE}_NEEDS_OPENCV)
		set(OpenCV_LIBRARIES "")
	endif()

    set(IGSTK_LIBRARIES "")
    if(${COMPONENT_NAME_CMAKE}_NEEDS_IGSTK)
		find_package(IGSTK REQUIRED)
		include(${IGSTK_USE_FILE})	
	else(${COMPONENT_NAME_CMAKE}_NEEDS_IGSTK)
		set(IGSTK_LIBRARIES "")
	endif()

    # check for intra-component extension
    if(${COMPONENT_NAME_CMAKE}_NEEDS_COMPONENT_EXTENSION) 
      foreach(COMPONENT_NEEDED ${${COMPONENT_NAME_CMAKE}_NEEDS_COMPONENT_EXTENSION})
        string(TOUPPER ${COMPONENT_NEEDED} ${COMPONENT_NEEDED}_INTERNAL)
        set(COMPONENT_${${COMPONENT_NEEDED}_INTERNAL} ON CACHE BOOL "Required by component extension ${COMPONENT_NAME}" FORCE )
        set(COMPONENT_EXTENSION_LIBRARIES ${COMPONENT_EXTENSION_LIBRARIES} ${COMPONENT_NEEDED})
        include_directories(${CAMITK_INCLUDE_DIR}/components/${COMPONENT_NEEDED})
      endforeach()
    endif()

    # check for needed tool 
    set(TOOL_LIBRARIES)
    if(${COMPONENT_NAME_CMAKE}_NEEDS_TOOL) 
      foreach(TOOL_NEEDED ${${COMPONENT_NAME_CMAKE}_NEEDS_TOOL})
        string(TOUPPER ${TOOL_NEEDED} ${TOOL_NEEDED}_INTERNAL)
        set(TOOL_${${TOOL_NEEDED}_INTERNAL} ON CACHE BOOL "Required by component extension ${COMPONENT_NAME}" FORCE )
        set(TOOL_LIBRARIES ${TOOL_LIBRARIES} ${TOOL_NEEDED})
        include_directories(${CAMITK_INCLUDE_DIR}/${TOOL_NEEDED})
      endforeach()
    endif()

    # check for target name
    set(COMPONENT_TARGET_NAME component-${COMPONENT_NAME})

    # check for libs in all known camitk dir + components subdir (beware of the plateform bin or lib?)
    link_directories( ${CAMITK_LIB_DIRECTORIES}
                      ${CAMITK_BIN_DIR} 
                      ${CAMITK_BIN_DIR}/components 
                      ${CAMITK_INSTALL_ROOT}/lib/${CAMITK_SHORT_VERSION_STRING}/components
                      ${CAMITK_INSTALL_ROOT}/bin/components
                      ${CMAKE_CURRENT_BINARY_DIR}/../../bin/components
    )

    # get all headers, sources and do what is needed for Qt 
    # one need to do this just before the add_library so that all defines, include directories and link directories
    # are set properly (gather_headers_and_sources include the call to Qt moc and uic)
    gather_headers_and_sources(${COMPONENT_NAME_CMAKE})
    
    # check for external sources
    if(${COMPONENT_NAME_CMAKE}_EXTERNAL_SOURCES)
            set(${COMPONENT_NAME_CMAKE}_SOURCES ${${COMPONENT_NAME_CMAKE}_SOURCES} ${${COMPONENT_NAME_CMAKE}_EXTERNAL_SOURCES})
    endif()

    # build the component extension module (library)
    add_library(${COMPONENT_TARGET_NAME} SHARED ${${COMPONENT_NAME_CMAKE}_SOURCES})

    # dependences to component and specific libraries
    # A component has to be linked with ${CAMITK_CORE_LIB}. As CAMTIK_CORE_LIB is always a DLL/shared lib, there should
    # be no symbol loaded multiple times
    target_link_libraries(${COMPONENT_TARGET_NAME} ${CAMITK_LIBRARIES} ${CAMITK_CORE_LIB} ${ITK_LIBRARIES} ${LIBXML2_LIBRARIES} ${OpenCV_LIBRARIES} ${IGSTK_LIBRARIES} ${TOOL_LIBRARIES} ${COMPONENT_EXTENSION_LIBRARIES} ${${COMPONENT_NAME_CMAKE}_LIBRARIES})

    # dependences to core (only set for internal build)
    if (CAMITK_INTERNAL_BUILD)
      add_dependencies(${COMPONENT_TARGET_NAME} ${CAMITK_CORE_LIB})
    endif()

    # target properties (outputname and remove soname)
    set_property(TARGET ${COMPONENT_TARGET_NAME} PROPERTY NO_SONAME 1) 
    set_target_properties(${COMPONENT_TARGET_NAME} 
                          PROPERTIES OUTPUT_NAME ${COMPONENT_NAME}
    )

    # tool dependencies
    if(${COMPONENT_NAME_CMAKE}_NEEDS_TOOL) 
      foreach(TOOL_NEEDED ${${COMPONENT_NAME_CMAKE}_NEEDS_TOOL})
        add_dependencies(${COMPONENT_TARGET_NAME} ${TOOL_NEEDED})
      endforeach()
    endif()

    # other component extension dependencies
    if(${COMPONENT_NAME_CMAKE}_NEEDS_COMPONENT_EXTENSION) 
      foreach(COMPONENT_NEEDED ${${COMPONENT_NAME_CMAKE}_NEEDS_COMPONENT_EXTENSION})
        string(TOUPPER ${COMPONENT_NEEDED} COMPONENT_NEEDED_CMAKE)
        # if this is not true, then the dependencies is from an external build, do not use add_dependencies
        if (CAMITK_COMPONENT_${COMPONENT_NEEDED_CMAKE}_INTERNAL)
          add_dependencies(${COMPONENT_TARGET_NAME} component-${COMPONENT_NEEDED})
        endif()
      endforeach()
    endif()

    # generate an install target for headers
    if(${COMPONENT_NAME_CMAKE}_HEADERS_TO_INSTALL)
      export_headers(${${COMPONENT_NAME_CMAKE}_HEADERS_TO_INSTALL} COMPONENT ${COMPONENT_TARGET_NAME} GROUP components)      
    endif()

    # lazy you!
    if(${COMPONENT_NAME_CMAKE}_INSTALL_ALL_HEADERS)
      export_headers(${${COMPONENT_NAME_CMAKE}_HEADERS} COMPONENT ${COMPONENT_TARGET_NAME} GROUP components)   
    endif()

    # lib installation
    install(TARGETS ${COMPONENT_TARGET_NAME}
            RUNTIME DESTINATION bin/components
            LIBRARY DESTINATION lib/${CAMITK_SHORT_VERSION_STRING}/components
            ARCHIVE DESTINATION lib/${CAMITK_SHORT_VERSION_STRING}/components
            COMPONENT ${COMPONENT_TARGET_NAME}
    )

    # testdata files installation
    if (EXISTS ${CMAKE_CURRENT_SOURCE_DIR}/testdata)
      # Check test data dir directory
      if (NOT EXISTS ${CAMITK_TESTDATA_DIR})
        add_custom_command(TARGET ${COMPONENT_TARGET_NAME}
                          POST_BUILD
                          COMMAND ${CMAKE_COMMAND} -E make_directory ${CAMITK_TESTDATA_DIR}
                          COMMENT "Creating build-time share directory for test data ${CAMITK_TESTDATA_DIR}"
                          VERBATIM
        )
      endif()

      # At build time, copy the files to test data directory
      add_custom_command(TARGET ${COMPONENT_TARGET_NAME}
                        POST_BUILD
                        COMMAND ${CMAKE_COMMAND} -E copy_directory testdata ${CAMITK_TESTDATA_DIR}
                        COMMENT "Installing build-time test data for ${COMPONENT_NAME}"
                        WORKING_DIRECTORY ${CMAKE_CURRENT_SOURCE_DIR}
                        VERBATIM
      )
      

      # during installation, copy the files to install directory
	  set (TESTDATA_DEST_DIR)
	  if ( MSVC )
		# Do NOT add ${CAMITK_SHORT_VERSION_STRING} to the testdata path for Win32 MSVC !
		set (TESTDATA_DEST_DIR share/testdata)
	  else ()
		set (TESTDATA_DEST_DIR share/${CAMITK_SHORT_VERSION_STRING}/testdata)
	  endif ()
      install(DIRECTORY testdata/
			  #DESTINATION share/testdata
              #DESTINATION share/${CAMITK_SHORT_VERSION_STRING}/testdata
			  DESTINATION ${TESTDATA_DEST_DIR}
              COMPONENT ${COMPONENT_TARGET_NAME}
              PATTERN ".svn" EXCLUDE
              PATTERN "*~" EXCLUDE
      )
    endif()

  endif()

endmacro()







# MACRO action_extension
#
# Simplify writing action extension CMakeLists.txt
# 
# usage:
# action_extension([DEFAULT]
#                     [NEEDS_ITK]
#                     [NEEDS_LIBXML2]
#					  [NEEDS_OPENCV]
#					  [NEEDS_IGSTK]
#                     [NEEDS_TOOL tool1 tool2 ...]
#                     [NEEDS_COMPONENT_EXTENSION component1 component2 ...]
#                     [NEEDS_ACTION_EXTENSION action1 action2 ...]
#                     [INCLUDE_DIRECTORIES dir1 dir2 ...]
#                     [DEFINES flag1 flag2 ...]
#                     [EXTERNAL_SOURCES file1 file2 ...]
#                     [LIBRARIES lib1 lib2 ...]
#                     [HEADERS_TO_INSTALL header1.h header2.h ...]
#                     [INSTALL_ALL_HEADERS]
#                    )
#
# The name of the action is automatically the name of the directory from where this macro
# is called.
#
# DEFAULT                   = means this is a default action to be compiled automatically
# NEEDS_ITK                 = add this if your action needs itk. 
#                             Do not forget to add the needed list of itk libraries in the LIBRARIES parameter
# NEEDS_LIBXML2             = add this if your action needs libxml2
# NEEDS_XSD                 = add this if your action needs Codesynthesis xsd cxx (xml schema compiler)
# NEEDS_OPENCV              = add this if your action needs OpenCV
# NEEDS_IGSTK               = add this if your action needs IgsTK
# NEEDS_TOOL                = list of needed camitk tools
# NEEDS_COMPONENT_EXTENSION = list of needed component extensions
# NEEDS_ACTION_EXTENSION    = list of needed action extensions
# INCLUDE_DIRECTORIES       = additional include directories
# DEFINES                   = list of define flags to add at compilation time
# EXTERNAL_SOURCES          = list of extra source/headers files (external to the current directory)
#                             that needed to be added to the SOURCES variable. 
#                             Note: EXTERNAL_SOURCES are not installed
# LIBRARIES                 = libraries to add to the link command
# HEADERS_TO_INSTALL        = list of headers to install, if present this will automatically
#                             create an "install-COMPONENT_NAMEcomponent" target, that can be used
#                             anywhere else to manage dependencies to this component.
#                             The headers are installed ${CAMITK_INCLUDE_DIR}
#                             when the target "install-COMPONENT_NAMEcomponent" is called.
# INSTALL_ALL_HEADERS       = install all of the headers (this is the lazy solution, please consider
#                             making a list and using HEADERS_TO_INSTALL parameter instead!   
macro(action_extension)
  get_directory_name(${CMAKE_CURRENT_SOURCE_DIR} ACTION_NAME)
  
  parse_arguments(${ACTION_NAME_CMAKE} 
    "NEEDS_TOOL;NEEDS_COMPONENT_EXTENSION;NEEDS_ACTION_EXTENSION;INCLUDE_DIRECTORIES;LIBRARIES;HEADERS_TO_INSTALL;DEFINES;EXTERNAL_SOURCES"  # possible lists
    "DEFAULT;NEEDS_ITK;NEEDS_LIBXML2;NEEDS_XSD;NEEDS_OPENCV;NEEDS_IGSTK;INSTALL_ALL_HEADERS" # possible options
    ${ARGN}
  )

  # if it is the first cmake run, create the action variable with a correct initial value
  if(NOT CAMITK_ACTION_${ACTION_NAME_CMAKE}_INTERNAL)
    message(STATUS "Checking ${ACTION_NAME}")
    # add option to enable/disable this action extension
    set(ACTION_${ACTION_NAME_CMAKE} ${${ACTION_NAME_CMAKE}_DEFAULT} CACHE BOOL "Build action extension ${ACTION_NAME}")
    set(CAMITK_ACTION_${ACTION_NAME_CMAKE}_INTERNAL TRUE CACHE BOOL "Is variable ACTION_${ACTION_NAME} already created?")
    mark_as_advanced(CAMITK_ACTION_${ACTION_NAME_CMAKE}_INTERNAL)
  endif()

  message(STATUS "Building action extension ${ACTION_NAME}: ${ACTION_${ACTION_NAME_CMAKE}}") 

  # if this extension is enabled, do everything needed
  # otherwise... do nothing
  if (ACTION_${ACTION_NAME_CMAKE})   
    # actions are compiled as shared object 
    set(CMAKE_LIBRARY_OUTPUT_DIRECTORY ${CMAKE_LIBRARY_OUTPUT_DIRECTORY}/actions) 

    # actions are compiled as shared object (for dll plateform put it in BIN)
    set(CMAKE_RUNTIME_OUTPUT_DIRECTORY ${CMAKE_RUNTIME_OUTPUT_DIRECTORY}/actions) 

    # include directories
    include_directories(${CAMITK_INCLUDE_DIRECTORIES})
    include_directories(${CMAKE_CURRENT_BINARY_DIR})
    include_directories(${CMAKE_CURRENT_SOURCE_DIR})
    include_directories(${${ACTION_NAME_CMAKE}_INCLUDE_DIRECTORIES})

    # check definitions
    if(${ACTION_NAME_CMAKE}_DEFINES)
      foreach (FLAG ${${ACTION_NAME_CMAKE}_DEFINES})
        add_definitions(-D${FLAG})
      endforeach()
    endif()

    # check for needed tool
    set(TOOL_LIBRARIES)
    if(${ACTION_NAME_CMAKE}_NEEDS_TOOL)
      foreach(TOOL_NEEDED ${${ACTION_NAME_CMAKE}_NEEDS_TOOL})
        string(TOUPPER ${TOOL_NEEDED} ${TOOL_NEEDED}_INTERNAL)
        set(TOOL_${${TOOL_NEEDED}_INTERNAL} ON CACHE BOOL "Required by action extension ${ACTION_NAME}" FORCE )
        set(TOOL_LIBRARIES ${TOOL_LIBRARIES} ${TOOL_NEEDED})
        include_directories(${CAMITK_INCLUDE_DIR}/${TOOL_NEEDED})
      endforeach()
    endif()

 	# Looking for ITK
	set(ITK_LIBRARIES "")
   if(${ACTION_NAME_CMAKE}_NEEDS_ITK)
      # ITK is required
      find_package(ITK REQUIRED)
      include(${ITK_USE_FILE})
	else()
		set(ITK_LIBRARIES "")
    endif()

   # Looking for LibXML2.0
    set(LIBXML2_LIBRARIES)
    if(${ACTION_NAME_CMAKE}_NEEDS_LIBXML2) 
      # LibXml2 is required
      find_package(Xml2)
      if (LIBXML2_FOUND)
        add_definitions(${LIBXML2_DEFINITIONS})
        include_directories(${LIBXML2_INCLUDE_DIR})      
      else()
        # most probably win32 or crosscompiling
        message(STATUS "${ACTION_NAME}: libxml2 required")
      endif()
    endif()
	
    set(OpenCV_LIBRARIES "")
    if(${ACTION_NAME_CMAKE}_NEEDS_OPENCV) 
		# OpenCV is required
		find_package( OpenCV REQUIRED )
	else ( )
		set(OpenCV_LIBRARIES "")
	endif()

    set(IGSTK_LIBRARIES "")
    if(${ACTION_NAME_CMAKE}_NEEDS_IGSTK)
		find_package(IGSTK REQUIRED)
		include(${IGSTK_USE_FILE})	
	else()
		set(IGSTK_LIBRARIES "")
	endif()

    # Looking for XercesC
    set(XERCESC_LIBRARIES)
    if(${ACTION_NAME_CMAKE}_NEEDS_XERCESC)
      # XercesC is required
      find_package(XercesC REQUIRED)
      if (XERCESC_FOUND)
        include_directories(${XERCESC_INCLUDE_DIR})
      else()
        # most probably win32 or crosscompiling
        message(STATUS "${ACTION_NAME}: xerces-c required")
      endif()
    endif()

    # Looking for codesynthesis XSD CXX
    if(${ACTION_NAME_CMAKE}_NEEDS_XSD)
      # XercesC is required
      find_package(XercesC REQUIRED)
      if (XERCESC_FOUND)
        include_directories(${XERCESC_INCLUDE_DIR})
        find_package(XSD REQUIRED)
        include_directories(${XSD_INCLUDE_DIR})
      else()
        # most probably win32 or crosscompiling
        message(STATUS "${ACTION_NAME}: xerces-c required because of XSD cxx, please set XERCESC_INCLUDE_DIR")
      endif()
    endif()

    set(COMPONENT_EXTENSION_LIBRARIES)
    # check for intra-action extension
    if(${ACTION_NAME_CMAKE}_NEEDS_COMPONENT_EXTENSION) 
      foreach(COMPONENT_NEEDED ${${ACTION_NAME_CMAKE}_NEEDS_COMPONENT_EXTENSION})
        string(TOUPPER ${COMPONENT_NEEDED} ${COMPONENT_NEEDED}_INTERNAL)
	set(COMPONENT_${${COMPONENT_NEEDED}_INTERNAL} ON CACHE BOOL "Required by action ${ACTION_NAME}" FORCE )
        set(COMPONENT_EXTENSION_LIBRARIES ${COMPONENT_EXTENSION_LIBRARIES} ${COMPONENT_NEEDED})
 	include_directories(${CAMITK_INCLUDE_DIR}/components/${COMPONENT_NEEDED})
      endforeach()
    endif()

    set(ACTION_EXTENSION_LIBRARIES)
    # check for intra-action extension
    if(${ACTION_NAME_CMAKE}_NEEDS_ACTION_EXTENSION) 
      foreach(ACTION_NEEDED ${${ACTION_NAME_CMAKE}_NEEDS_ACTION_EXTENSION})
        string(TOUPPER ${ACTION_NEEDED} ${ACTION_NEEDED}_INTERNAL)
	set(ACTION_${${ACTION_NEEDED}_INTERNAL} ON CACHE BOOL "Required by action ${ACTION_NAME}" FORCE )
        set(ACTION_EXTENSION_LIBRARIES ${ACTION_EXTENSION_LIBRARIES} ${ACTION_NEEDED})
 	include_directories(${CAMITK_INCLUDE_DIR}/actions/${ACTION_NEEDED})
      endforeach()
    endif()


    # check for target name
    set(ACTION_TARGET_NAME action-${ACTION_NAME})

    # check for libs in all known camitk dir + actions subdir (beware of the plateform bin or lib?)
    link_directories( ${CAMITK_LIB_DIRECTORIES}
                      ${CAMITK_BIN_DIR} 
                      ${CAMITK_BIN_DIR}/components
                      ${CAMITK_BIN_DIR}/actions 
                      ${CAMITK_INSTALL_ROOT}/lib/${CAMITK_SHORT_VERSION_STRING}/actions
                      ${CAMITK_INSTALL_ROOT}/lib/${CAMITK_SHORT_VERSION_STRING}/components
                      ${CAMITK_INSTALL_ROOT}/bin/components
                      ${CAMITK_INSTALL_ROOT}/bin/actions
                      ${CMAKE_CURRENT_BINARY_DIR}/../../bin/components
                      ${CMAKE_CURRENT_BINARY_DIR}/../../bin/actions
    )

    # get all headers, sources and do what is needed for Qt 
    # one need to do this just before the add_library so that all defines, include directories and link directories
    # are set properly (gather_headers_and_sources include the call to Qt moc and uic)
    gather_headers_and_sources(${ACTION_NAME_CMAKE})
	
    # check for external sources
    if(${ACTION_NAME_CMAKE}_EXTERNAL_SOURCES)
            set(${ACTION_NAME_CMAKE}_SOURCES ${${ACTION_NAME_CMAKE}_SOURCES} ${${ACTION_NAME_CMAKE}_EXTERNAL_SOURCES})
    endif()

    # build the action extension module (library)
    add_library(${ACTION_TARGET_NAME} SHARED ${${ACTION_NAME_CMAKE}_SOURCES})

    # dependences to action and specific libraries
    # An action has to be linked with ${CAMITK_CORE_LIB}. As CAMTIK_CORE_LIB is always a DLL/shared lib, there should
    # be no symbol loaded multiple times
    target_link_libraries(${ACTION_TARGET_NAME} ${CAMITK_LIBRARIES} ${CAMITK_CORE_LIB} ${COMPONENT_EXTENSION_LIBRARIES} ${ITK_LIBRARIES} ${LIBXML2_LIBRARIES} ${OpenCV_LIBRARIES} ${IGSTK_LIBRARIES} ${XERCESC_LIBRARY} ${TOOL_LIBRARIES}  ${${ACTION_NAME_CMAKE}_LIBRARIES})

    # dependences to core (only set for internal build)
    if (CAMITK_INTERNAL_BUILD)
      add_dependencies(${ACTION_TARGET_NAME} ${CAMITK_CORE_LIB})
    endif()

    # target properties (outputname and remove soname)
    set_property(TARGET ${ACTION_TARGET_NAME} PROPERTY NO_SONAME 1)    
    set_target_properties(${ACTION_TARGET_NAME} 
                          PROPERTIES OUTPUT_NAME ${ACTION_NAME}
    )

    # tool dependencies
    if(${ACTION_NAME_CMAKE}_NEEDS_TOOL) 
      foreach(TOOL_NEEDED ${${ACTION_NAME_CMAKE}_NEEDS_TOOL})
        add_dependencies(${ACTION_TARGET_NAME} ${TOOL_NEEDED})
      endforeach()
    endif()

    # other action extension dependencies
    if(${ACTION_NAME_CMAKE}_NEEDS_COMPONENT_EXTENSION) 
      foreach(COMPONENT_NEEDED ${${ACTION_NAME_CMAKE}_NEEDS_COMPONENT_EXTENSION})
        string(TOUPPER ${COMPONENT_NEEDED} COMPONENT_NEEDED_CMAKE)
	# if this is not true, then the dependencies is from an external build, do not use add_dependencies
        if (CAMITK_COMPONENT_${COMPONENT_NEEDED_CMAKE}_INTERNAL)
	  add_dependencies(${ACTION_TARGET_NAME} component-${COMPONENT_NEEDED})
        endif()
      endforeach()
    endif()

    if(${ACTION_NAME_CMAKE}_NEEDS_ACTION_EXTENSION) 
      foreach(ACTION_NEEDED ${${ACTION_NAME_CMAKE}_NEEDS_ACTION_EXTENSION})
        string(TOUPPER ${ACTION_NEEDED} ACTION_NEEDED_CMAKE)
	# if this is not true, then the dependencies is from an external build, do not use add_dependencies
        if (CAMITK_ACTION_${ACTION_NEEDED_CMAKE}_INTERNAL)
	  add_dependencies(${ACTION_TARGET_NAME} action-${ACTION_NEEDED})
        endif()
      endforeach()
    endif()

    # generate an install target for headers
    if(${ACTION_NAME_CMAKE}_HEADERS_TO_INSTALL)
      export_headers(${${ACTION_NAME_CMAKE}_HEADERS_TO_INSTALL} COMPONENT ${ACTION_TARGET_NAME} GROUP actions)
    endif()

    # lazy you!
    if(${ACTION_NAME_CMAKE}_INSTALL_ALL_HEADERS)
      export_headers(${${ACTION_NAME_CMAKE}_HEADERS} COMPONENT ${ACTION_TARGET_NAME} GROUP actions)
    endif()

    # lib installation
    install(TARGETS ${ACTION_TARGET_NAME}
            RUNTIME DESTINATION bin/actions
            LIBRARY DESTINATION lib/${CAMITK_SHORT_VERSION_STRING}/actions
            ARCHIVE DESTINATION lib/${CAMITK_SHORT_VERSION_STRING}/actions
            COMPONENT ${ACTION_TARGET_NAME}
    )
  endif()

endmacro()








# MACRO application_extension
#
# Simplify writing application extension CMakeLists.txt
#
# usage:
# application_extension([DEFAULT])
#
# The name of the application is automatically the name of the directory from where this macro
# is called prefixed with "camitk-"
#
# DEFAULT = means this is a default application to be compiled automatically
macro(application_extension)

  get_directory_name(${CMAKE_CURRENT_SOURCE_DIR} APPLICATION_NAME)

  parse_arguments(${APPLICATION_NAME_CMAKE} 
    "NEEDS_TOOL"  # possible lists
    "DEFAULT;NEEDS_XSD;NEEDS_XERCESC" # possible options
    ${ARGN}
  )

  # if it is the first cmake run, create the application variable with a correct initial value
  if(NOT CAMITK_APPLICATION_${APPLICATION_NAME_CMAKE}_INTERNAL)
    message(STATUS "Checking ${APPLICATION_NAME}")
    # add option to enable/disable this application extension
    set(APPLICATION_${APPLICATION_NAME_CMAKE} ${${APPLICATION_NAME_CMAKE}_DEFAULT} CACHE BOOL "Build application ${APPLICATION_NAME}")
    set(CAMITK_APPLICATION_${APPLICATION_NAME_CMAKE}_INTERNAL TRUE CACHE BOOL "Is variable APPLICATION_${APPLICATION_NAME} already created?")
    mark_as_advanced(CAMITK_APPLICATION_${APPLICATION_NAME_CMAKE}_INTERNAL)
  endif()

  message(STATUS "Building application ${APPLICATION_NAME}: ${APPLICATION_${APPLICATION_NAME_CMAKE}}") 

  # if this extension is enabled, do everything needed
  # otherwise... do nothing
  if (APPLICATION_${APPLICATION_NAME_CMAKE})
    # check for needed tool
    set(TOOL_LIBRARIES)
    if(${APPLICATION_NAME_CMAKE}_NEEDS_TOOL)
      foreach(TOOL_NEEDED ${${APPLICATION_NAME_CMAKE}_NEEDS_TOOL})
        message(STATUS "${TOOL_NEEDED} required by application ${APPLICATION_NAME}")
        string(TOUPPER ${TOOL_NEEDED} ${TOOL_NEEDED}_INTERNAL)
        set(TOOL_${${TOOL_NEEDED}_INTERNAL} ON CACHE BOOL "Required by application ${APPLICATION_NAME}" FORCE )
        set(TOOL_LIBRARIES ${TOOL_LIBRARIES} ${TOOL_NEEDED})
        include_directories(${CAMITK_INCLUDE_DIR}/${TOOL_NEEDED})
      endforeach()
    endif()

    # Looking for XercesC
    set(XERCESC_LIBRARIES)
    if(${APPLICATION_NAME_CMAKE}_NEEDS_XERCESC)
      # XercesC is required
      find_package(XercesC REQUIRED)
      if (XERCESC_FOUND)
        include_directories(${XERCESC_INCLUDE_DIR})
      else()
        # most probably win32 or crosscompiling
        message(STATUS "${APPLICATION_NAME}: xerces-c required")
      endif()
    endif()

    # Looking for codesynthesis XSD CXX
    if(${APPLICATION_NAME_CMAKE}_NEEDS_XSD)
      # XercesC is required
      find_package(XercesC REQUIRED)
      if (XERCESC_FOUND)
        include_directories(${XERCESC_INCLUDE_DIR})
        find_package(XSD REQUIRED)
        include_directories(${XSD_INCLUDE_DIR})
      else()
        # most probably win32 or crosscompiling
        message(STATUS "${APPLICATION_NAME}: xerces-c required because of XSD cxx, please set XERCESC_INCLUDE_DIR")
      endif()
    endif()

    # configure compiler
    include_directories(${CAMITK_INCLUDE_DIRECTORIES} ${CAMITK_INCLUDE_DIR}/${CAMITK_CORE_LIB} ${CAMITK_INCLUDE_DIR}/component ${CMAKE_CURRENT_BINARY_DIR})

    # configure linker
    link_directories(${CAMITK_LIB_DIRECTORIES})

    # check for target name
    set(APPLICATION_TARGET_NAME application-${APPLICATION_NAME})

    # get all headers, sources and do what is needed for Qt 
    # one need to do this just before the add_library so that all defines, include directories and link directories
    # are set properly (gather_headers_and_sources include the call to Qt moc and uic)
    gather_headers_and_sources(${APPLICATION_NAME})

    # add the executable target
    add_executable(${APPLICATION_TARGET_NAME} ${${APPLICATION_NAME}_SOURCES})

    target_link_libraries(${APPLICATION_TARGET_NAME} ${CAMITK_LIBRARIES} ${CAMITK_CORE_LIB} ${TOOL_LIBRARIES} ${XERCESC_LIBRARY} )

    # target properties
    set_target_properties(${APPLICATION_TARGET_NAME} 
                          PROPERTIES OUTPUT_NAME camitk-${APPLICATION_NAME}
    )

    # tool dependencies
    if(${APPLICATION_TARGET_NAME}_NEEDS_TOOL)
      foreach(TOOL_NEEDED ${${APPLICATION_TARGET_NAME}_NEEDS_TOOL})
        add_dependencies(${APPLICATION_TARGET_NAME} ${TOOL_NEEDED})
      endforeach()
    endif()

    # dependences to core (only set for internal build)
    if (CAMITK_INTERNAL_BUILD)
	add_dependencies(${APPLICATION_TARGET_NAME} ${CAMITK_CORE_LIB})
    endif()
	
    # installation
    install(TARGETS ${APPLICATION_TARGET_NAME}
            RUNTIME DESTINATION bin
            COMPONENT ${APPLICATION_TARGET_NAME}
    )
  endif()

# For Microsoft Visual C++, sets the default application to the corresponding project
# (i.e. launches imp when we you click on "Debug" or "Start Without Debugging" button on Visual)
# In addition, but not the least, sets the environment to the debug dll directory for VTK (and ITK)
#     to solve the dll incompatibility between debug and relase version of QVTK.dll and ITKCommon.dll
#if ( MSVC )
#	create_target_launcher(
#		${APPLICATION_TARGET_NAME}
#		RUNTIME_LIBRARY_DIRS "%VTK_DIR%/bin/$(OutDir)" "%ITK_DIR%/bin/$(OutDir)"
#	)		
#endif( MSVC )
  
endmacro()

# MACRO export_headers
#
# Duplicate headers installation:
# - one is used at compiled time and puts everything in 
#   ${CAMITK_BUILD_DIR}/include/${GroupName}/${ComponentName}/${SubdirName} 
# - the other one is used at installation time and puts everything in 
#   ${CAMITK_INSTALL_ROOT}/include/${GroupName}/${ComponentName}/${SubdirName}
#
# usage:
# export_headers(HeaderFile1.h HeaderFile2.h ...
#                COMPONENT ComponentName
#                [GROUP GroupName]
#                [SUBDIRECTORY SubdirName]
# )
# 
# FILES        = list of files to install
# COMPONENT    = name of the component to use. This is also the include subdirectory name
#                used for copying the file
# GROUP        = the name of the group this install should be using group will be
#                prepend to the component name.
# SUBDIRECTORY = optional subdirectory to use in ${CAMITK_INCLUDE_DIR}/include/${ComponentName}
macro(export_headers)

  parse_arguments(EXPORT_HEADER 
    "COMPONENT;SUBDIRECTORY;GROUP"  # possible lists
    "" # possible options
    ${ARGN}
  )

  # special case for components: remove prefix for the destination directory
  string(REGEX REPLACE "^component-|^action-" "" EXPORT_HEADER_DESTINATION ${EXPORT_HEADER_COMPONENT})
  
  # Check group
  if(EXPORT_HEADER_GROUP)
    # check directory
    if (NOT EXISTS ${CAMITK_INCLUDE_DIR}/${EXPORT_HEADER_GROUP})
      add_custom_command(TARGET ${EXPORT_HEADER_COMPONENT}
                        POST_BUILD
                        COMMAND ${CMAKE_COMMAND} -E make_directory ${CAMITK_INCLUDE_DIR}/${EXPORT_HEADER_GROUP}
                        COMMENT "Creating build-time group include dir ${CAMITK_INCLUDE_DIR}/${EXPORT_HEADER_GROUP}"
                        VERBATIM
      )
    endif()
    set(EXPORT_HEADER_DESTINATION ${EXPORT_HEADER_GROUP}/${EXPORT_HEADER_DESTINATION})
  endif()

  # check that the directory exists otherwise create it
  if (NOT EXISTS ${CAMITK_INCLUDE_DIR}/${EXPORT_HEADER_DESTINATION})
    add_custom_command(TARGET ${EXPORT_HEADER_COMPONENT}
                      POST_BUILD
                      COMMAND ${CMAKE_COMMAND} -E make_directory ${CAMITK_INCLUDE_DIR}/${EXPORT_HEADER_DESTINATION}
                      COMMENT "Creating build-time include dir ${CAMITK_INCLUDE_DIR}/${EXPORT_HEADER_DESTINATION}"
                      VERBATIM
    )
  endif()
  
  # check the SUBDIRECTORY parameter
  if(EXPORT_HEADER_SUBDIRECTORY)
    set(EXPORT_HEADER_DESTINATION ${EXPORT_HEADER_DESTINATION}/${EXPORT_HEADER_SUBDIRECTORY})
    # create subdirectory if it does not exists
    if (NOT EXISTS ${CAMITK_INCLUDE_DIR}/${EXPORT_HEADER_DESTINATION})
      add_custom_command(TARGET ${EXPORT_HEADER_COMPONENT}
                        POST_BUILD
                        COMMAND ${CMAKE_COMMAND} -E make_directory ${CAMITK_INCLUDE_DIR}/${EXPORT_HEADER_DESTINATION}
                        COMMENT "Creating build-time include subdir ${CAMITK_INCLUDE_DIR}/${EXPORT_HEADER_DESTINATION}"
                        VERBATIM
      )
    endif()
  endif()
  

  # at build time, copy the files to build directory include files when the target is built
  foreach(HEADER ${EXPORT_HEADER_DEFAULT_ARGS})
    # check if file name is relative or not
    set(FILE_TO_COPY ${HEADER})
   
   # copy after build, only if there was a change
    add_custom_command(TARGET ${EXPORT_HEADER_COMPONENT}
                       POST_BUILD
                       COMMAND ${CMAKE_COMMAND} -E copy_if_different ${FILE_TO_COPY} ${CAMITK_INCLUDE_DIR}/${EXPORT_HEADER_DESTINATION}/
                       COMMENT "Installing build-time header ${HEADER}"
                       WORKING_DIRECTORY ${CMAKE_CURRENT_SOURCE_DIR}
                       VERBATIM
    )
  endforeach()

  # during installation, copy the files to install directory
  install(FILES ${EXPORT_HEADER_DEFAULT_ARGS} 
          DESTINATION include/${CAMITK_SHORT_VERSION_STRING}/${EXPORT_HEADER_DESTINATION}/
          COMPONENT ${EXPORT_HEADER_COMPONENT}
  )

endmacro()

# TODO write a viewer_extension macro in CamiTK

