/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2012 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/


#include "Tetrahedralize.h"
#include <MeshComponent.h>
#include <Log.h>
#include <Application.h>

#include <VtkMeshUtil.h>

#include <vtkPolyData.h>
#include <vtkCell.h>
#include <vtkTetra.h>
#include <vtkCellArray.h>
#include <vtkUnstructuredGrid.h>

#include "tetgen.h" // Defined tetgenio, tetrahedralize().




// -------------------- Tetrahedralize --------------------
Tetrahedralize::Tetrahedralize(ActionExtension* extension) : Action(extension) {
    this->setName("Tetrahedralize");
    this->setDescription("Generates a Delaunay tetrahedralization from the mesh points, msh, sofa scn, pml and corresponding lml");
    this->setComponent("MeshComponent");
    this->setFamily("Mesh Processing");
    this->addTag("Tetrahedralize");
    
    // command line parameter see documentation for more information
    // http://wias-berlin.de/software/tetgen/files/tetgen-manual.pdf
    setProperty("Tetgen Parameters","pq1.414V"); 
}


// --------------- apply -------------------
Action::ApplyStatus Tetrahedralize::apply() {
    CAMITK_INFO("Tetrahedralize", "apply", "Tetrahedalizing " << getTargets().last()->getName().toStdString());

    // set waiting cursor and status bar
    QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
    Application::showStatusBarMessage("Tetrahedalizing...");
    Application::resetProgressBar();

    // use the last target
    MeshComponent *targetMesh = dynamic_cast<MeshComponent*>(getTargets().last());

    //-- check if this is a polydata and if this is an unstructured grid, extract the surface
    vtkSmartPointer<vtkPolyData> targetPolyData = VtkMeshUtil::vtkPointSetToVtkPolyData(targetMesh->getPointSet());
    
    //-- build the tetgen data structures using just the points
    tetgenio in;
    // All indices start from 1.
    in.firstnumber = 1;

    in.numberofpoints = targetPolyData->GetPoints()->GetNumberOfPoints();
    in.pointlist = new REAL[in.numberofpoints * 3];
    int index = 0;
    double xyz[3];
    for ( int i = 0; i < targetPolyData->GetPoints()->GetNumberOfPoints(); i++ ) {
        targetPolyData->GetPoints()->GetPoint ( i, xyz );
        in.pointlist[index++] = xyz[0];
        in.pointlist[index++] = xyz[1];
        in.pointlist[index++] = xyz[2];
    }

    // generates all facets, use only triangles
    unsigned nrOfTriangles = 0;
    for (int i = 0;i < targetPolyData->GetNumberOfCells(); i++) {
        if (targetPolyData->GetCell(i)->GetNumberOfPoints() == 3)
            nrOfTriangles++;
    }

    in.numberoffacets = nrOfTriangles;
    in.facetlist = new tetgenio::facet[in.numberoffacets];
    //in.facetmarkerlist = new int[in.numberoffacets];

    // generates tetgen triangle facets
    for (int i = 0, facetId = 0; i < targetPolyData->GetNumberOfCells(); i++) {
        vtkCell *c = targetPolyData->GetCell(i);
        if (c->GetNumberOfPoints() == 3) {
            //in.facetmarkerlist[facetId] = 1; // set boundary marker (but why?)
            tetgenio::facet *f = &in.facetlist[facetId];
            f->numberofpolygons = 1;
            f->polygonlist = new tetgenio::polygon[f->numberofpolygons];
            f->numberofholes = 0;
            f->holelist = NULL;
            tetgenio::polygon *p = &f->polygonlist[0];
            p->numberofvertices = 3;
            p->vertexlist = new int[p->numberofvertices];
            p->vertexlist[0] = in.firstnumber + c->GetPointId(0);
            p->vertexlist[1] = in.firstnumber + c->GetPointId(1);
            p->vertexlist[2] = in.firstnumber + c->GetPointId(2);
            // next time, process next facet
            facetId++;
        }
    }
    // save input mesh
    //in.save_poly("tet");
    //in.save_faces("tet");
    //in.save_nodes("tet");

    //in.numberoffacets = 0;

    //-- Tetrahedralize the PLC
    // Tetrahedralize the PLC. Switches are chosen to read a PLC (p),
    //   do quality mesh generation (q) with a specified quality bound
    //   (1.414), and apply a maximum volume constraint (a0.1).
    // NOTE (warning) with the q option tetgen will add some vertex, the point list has to be updated!
    //tetrahedralize("pq1.414a0.1", &in, &out);
    tetgenio firstPass, secondPass, out;
    tetrahedralize(property("Tetgen Parameters").toString().toAscii().data(), &in, &out);
    /*
    tetrahedralize("-rq1.414", &firstPass, &secondPass);
    tetrahedralize("-ra0.5", &secondPass, &out);
    */
    
    // save Output elements
    //out.save_elements ((char *) QString(QFileInfo(myComponent->getFileName()).completeBaseName() + "_out").toStdString().c_str());
    vtkSmartPointer<vtkPoints> thePoints = vtkSmartPointer<vtkPoints>::New();
    for (int i = 0;i < out.numberofpoints;i++) {
        thePoints->InsertNextPoint(out.pointlist[i*3], out.pointlist[i*3+1], out.pointlist[i*3+2]);
    }

    //-- export to VTK
    // import the cells
    vtkSmartPointer<vtkCellArray> theCells = vtkSmartPointer<vtkCellArray>::New();
    for (int i = 0; i < out.numberoftetrahedra; i++) {
        vtkSmartPointer<vtkCell> c = vtkSmartPointer<vtkTetra>::New();

        /* Vtk/Pml tetrahedron                   Tetgen tetrahedron
        *       3                                          3
        *     /| \                                        /| \
        *    / |  \                                      / |  \
        *   1..|...\ 2                                  0..|...\ 2
        *   \  |   /                      vs            \  |   /
        *    \ |  /                                      \ |  /
        *     \|/                                         \|/
        *      0                                           1
        * => 1 and 0 are inversed
        */
        c->GetPointIds()->SetId(0, out.tetrahedronlist[i * out.numberofcorners + 1] - in.firstnumber);
        c->GetPointIds()->SetId(1, out.tetrahedronlist[i * out.numberofcorners + 0] - in.firstnumber);
        c->GetPointIds()->SetId(2, out.tetrahedronlist[i * out.numberofcorners + 2] - in.firstnumber);
        c->GetPointIds()->SetId(3, out.tetrahedronlist[i * out.numberofcorners + 3] - in.firstnumber);
        theCells->InsertNextCell(c);
    }

    // generates new point set (unstructured grid)
    vtkSmartPointer<vtkUnstructuredGrid> newPointSet = vtkSmartPointer<vtkUnstructuredGrid>::New();
    newPointSet->Allocate(theCells->GetNumberOfCells());
    newPointSet->SetCells(VTK_TETRA, theCells);
    newPointSet->SetPoints(thePoints);
    newPointSet->Update();

    //-- Create a new mesh Component    
    MeshComponent * result = new MeshComponent(newPointSet, targetMesh->getName() + " tetrahedralized");
    Application::refresh();    

    // restore the normal cursor and progress bar
    Application::resetProgressBar();
    QApplication::restoreOverrideCursor();
    return SUCCESS;
}
