# Copyright (C) 2006 Canonical Ltd
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

"""Test the ability to convert CVS history into a bzr project."""

import errno
import os
import subprocess

from bzrlib import (
    errors,
    inventory,
    osutils,
    revision,
    tests,
    trace,
    )

import cvsps.importer
from cvsps import errors as cvsps_errors


class TestCVSPSHelper(tests.TestCaseInTempDir):
    """Create a CVS project, and convert it to a bzr one"""

    def setUp(self):
        super(TestCVSPSHelper, self).setUp()
        self.cvs_working_dir = None
        self.setup_cvs_repo()

    def setup_cvs_repo(self):
        """Create a cvs repo for use with the test."""
        self.cvs_root = osutils.abspath('cvs_root')
        self.cvs('init')
        self.assertPathExists('cvs_root/CVSROOT')

    def run_command(self, cmd, working_dir=None, stdin='', returncode=0):
        """Run a generic command.

        :param cmd: The argument list of the command to run, starting with the
            program name.
        :param working_dir: Set the working directory of the child process,
            defaults to self.cvs_working_dir. None => current working dir.
        :param stdin: The text to pass to the process.
        :param returncode: The expected return code, defaults to 0, supplying
            None will disable return code checking.
        :return: (stdout, stderr)
        """
        trace.mutter('running command %s in %s', cmd, working_dir)
        p = subprocess.Popen(cmd,
                             stdin=subprocess.PIPE,
                             stdout=subprocess.PIPE,
                             stderr=subprocess.PIPE,
                             cwd=working_dir,
                            )
        out, err = p.communicate(stdin)
        if returncode is not None:
            self.assertEqual(returncode, p.returncode,
                'command %s failed. Expected return code %s not %s.'
                '\nOut: %s\n'
                '\nErr: %s\n'
                % (cmd, returncode, p.returncode, out, err))
        return out, err

    def cvs(self, *args, **kwargs):
        """Run a cvs command.

        General arguments are just passed to 'cvs', keyword arguments modify
        the handling.
        """
        cwd = kwargs.get('working_dir', self.cvs_working_dir)
        stdin = kwargs.get('stdin', '')
        returncode = kwargs.get('returncode', 0)
        cmd = ['cvs', '-d', self.cvs_root] + list(args)

        try:
            return self.run_command(cmd, working_dir=cwd, stdin=stdin,
                                    returncode=returncode)
        except OSError, e:
            if e.errno in (errno.ENOENT,):
                raise cvsps_errors.MissingProgram(cmd, e)
            raise

    def cvsps(self, module='.', cvsroot=None):
        """Run 'cvsps'.

        This uses --cvs-direct to handle connecting to the cvs repository. It
        also manually supplies the '--root' and repository, so that it properly
        strips paths. In testing, cvsps has really weird ideas about what
        paths need to be stripped.

        :param module: The sub-module to process
        :param cvsroot: The root of the repository. If not supplied,
            self.cvs_root is used
        """
        if cvsroot is None:
            cvsroot = self.cvs_root
        controller = cvsps.importer.CVSPSController(cvs_root=cvsroot,
                                                    cvs_module=module)
        controller.create_cvsps_dump('test.dump')
        fp = open('test.dump', 'rb')
        try:
            return fp.read()
        finally:
            fp.close()

    def create_checkout(self, module='.', target_dir='checkout',
                        set_as_default=True,
                        extra_args=[]):
        """Create a cvs checkout.

        Checkout the supplied module, into the target directory.
        If set_as_default is true, set the cvs working directory to the
        checkout.
        """
        os.mkdir(target_dir)
        self.cvs('checkout', module, working_dir=target_dir, *extra_args)
        if set_as_default:
            self.cvs_working_dir = target_dir


class TestMissingPrograms(TestCVSPSHelper):
    """Test what errors are raised when programs are missing."""

    def setUp(self):
        # Set the PATH env to just the current dir, which prevents all other
        # programs from being found.
        super(TestMissingPrograms, self).setUp()
        old_path = osutils.set_or_unset_env('PATH', '.')
        def restore():
            osutils.set_or_unset_env('PATH', old_path)
        self.addCleanup(restore)

    def test_missing_cvs(self):
        self.assertRaises(cvsps_errors.MissingProgram,
                          self.cvs, 'init')

    def test_missing_cvsps(self):
        self.assertRaises(cvsps_errors.MissingProgram,
                          self.cvsps)


class TestImportWithoutCVS(tests.TestCaseWithTransport):
    """Test Importer for things that don't require a CVS repo."""

    def test_create_repository(self):
        """Test that we can create a repository if it doesn't exist."""
        # The target directory doesn't exist
        importer = cvsps.importer.Importer(cvsroot='cvsroot',
                                           cvs_module='module',
                                           output_base='new_repo',
                                          )
        repo = importer.open_or_create_bzr_repo()
        self.assertEndsWith(repo.bzrdir.root_transport.base,
                            'new_repo/bzr/module/')
        # The repository should be created with the shared flag set.
        self.assertTrue(repo.is_shared())

    def test_create_root_repository(self):
        """Test that we can create a repository if it doesn't exist."""
        # The target directory doesn't exist
        importer = cvsps.importer.Importer(cvsroot='cvsroot',
                                           cvs_module='.',
                                           output_base='new_repo',
                                          )
        repo = importer.open_or_create_bzr_repo()
        self.assertEndsWith(repo.bzrdir.root_transport.base,
                            'new_repo/bzr/')
        # The repository should be created with the shared flag set.
        self.assertTrue(repo.is_shared())

    def test_open_repository(self):
        """Test that we can open an existing repository."""
        # Create a repository with a single entry, so we can check that
        # the repository is opened later.
        tree = self.make_branch_and_tree('bzr')
        self.build_tree(['bzr/a'])
        tree.add('a')
        tree.commit('initial', rev_id='test-revid-xxx')

        importer = cvsps.importer.Importer(cvsroot='cvsroot',
                                           cvs_module='.',
                                           output_base='.',
                                          )

        repo = importer.open_or_create_bzr_repo()
        self.assertEqual(tree.branch.repository.bzrdir.root_transport.base,
                         repo.bzrdir.root_transport.base)
        self.assertTrue(repo.has_revision('test-revid-xxx'))


class TestImport(TestCVSPSHelper):
    """Test basic conversion"""

    def test_init(self):
        self.create_checkout()
        self.build_tree(['checkout/a'])
        self.cvs('add', 'a')
        self.cvs('commit', '-m', 'init')
        self.assertPathExists(osutils.pathjoin(self.cvs_root, 'a,v'))

        # Test that we can run 'cvsps'
        out = self.cvsps()

        # Creation of the CVSROOT
        self.assertContainsRe(out, 'PatchSet 1 \n')
        self.assertContainsRe(out, 'Date: ')
        self.assertContainsRe(out, 'Author: ')
        # It should have properly stripped the first part of the path
        self.assertContainsRe(out, '\\s+CVSROOT/modules:INITIAL->1.1')
        # Adding 'a'
        self.assertContainsRe(out, 'PatchSet 2 \n')
        self.assertContainsRe(out, '\\s+a:')


class TestCVSPSMapFile(tests.TestCaseInTempDir):
    """Test the CVS Map file support"""

    def test_read_nothing(self):
        """Create a MapFile with no associated file."""
        # It shouldn't fail to open, but it should have no data.
        map_f = cvsps.importer.MapFile('no_such_path.map', 'module')
        self.assertEqual({}, map_f._map)

    def test_create_empty_map(self):
        """Create a new (empty) map file"""
        map_f = cvsps.importer.MapFile('new.map', 'module')
        map_f.write()

        self.check_file_contents('new.map',
            (map_f.HEADER + '\n'
             + 'cvs module: module\n'
            ))

    def test_with_contents(self):
        """Create a map file with contents."""
        map_f = cvsps.importer.MapFile('new.map', 'module')

        map_f.add(1, u'revid-xxyy')
        map_f.add(3, u'revid-\xb5')

        map_f.write()
        self.check_file_contents('new.map',
            (map_f.HEADER + '\n'
             + 'cvs module: module\n'
             + '1 revid-xxyy\n'
             + '3 revid-\xc2\xb5\n'))

        # If we re-open the file, we should get the correct contents.
        del map_f
        map_f = cvsps.importer.MapFile('new.map', 'module')
        self.assertEqual(u'revid-xxyy', map_f.get(1))
        self.assertEqual(u'revid-\xb5', map_f.get(3))
        self.assertEqual(None, map_f.get(2))

    def test_empty_file(self):
        """Using an empty file should just succeed."""
        open('new.map', 'wb').close()
        map_f = cvsps.importer.MapFile('new.map', 'module')

    def test_invalid_header(self):
        """A bad header should raise an exception."""
        open('new.map', 'wb').write('# bogus header')
        self.assertRaises(cvsps.errors.InvalidMapFile,
                          cvsps.importer.MapFile, 'new.map', 'module')

    def test_invalid_module_line(self):
        """A bad module line will raise InvalidMapFile."""
        f = open('new.map', 'wb')
        try:
            f.write(cvsps.importer.MapFile.HEADER)
            f.write('\nfoobar\n')
        finally:
            f.close()
        self.assertRaises(cvsps.errors.InvalidMapFile,
                          cvsps.importer.MapFile, 'new.map', 'module')

    def test_incorrect_module(self):
        """If the module name doesn't match, it is an error"""
        f = open('new.map', 'wb')
        try:
            f.write(cvsps.importer.MapFile.HEADER)
            f.write('\ncvs module: wrong/module\n')
        finally:
            f.close()
        self.assertRaises(cvsps.errors.InvalidMapModule,
                          cvsps.importer.MapFile, 'new.map', 'module')


class TestMinimalTree(tests.TestCase):
    """Tests for how MinimalTree handles adding/removing entries."""

    def assertInvEqual(self, expected, inv):
        """Make sure that the inventory has the right shape."""
        actual = [path for path, ie in inv.iter_entries_by_dir()]
        self.assertEqual(expected, actual)

    def test__create_leading_dirs_none(self):
        min_tree = cvsps.importer.MinimalTree(inventory.Inventory())
        self.assertInvEqual([''], min_tree._inventory)

        # No directories should be created
        parent = min_tree._create_leading_dirs('foo')
        self.assertInvEqual([''], min_tree._inventory)
        self.assertEqual({}, min_tree._modified_paths)
        self.assertEqual(min_tree._inventory.root.file_id, parent.file_id)

    def test__create_leading_dirs_subdir_exists(self):
        inv = inventory.Inventory()
        inv.add(inventory.make_entry('directory', 'foo',
                     parent_id=inv.root.file_id, file_id='foo-id'))
        min_tree = cvsps.importer.MinimalTree(inv)
        self.assertInvEqual(['', 'foo'], min_tree._inventory)

        # No directories should be created
        parent = min_tree._create_leading_dirs('foo/bar')
        self.assertInvEqual(['', 'foo'], min_tree._inventory)
        self.assertEqual({}, min_tree._modified_paths)
        self.assertEqual('foo', parent.name)

    def test__create_leading_dirs_one(self):
        min_tree = cvsps.importer.MinimalTree(inventory.Inventory())
        self.assertInvEqual([''], min_tree._inventory)

        # A single level should work
        parent = min_tree._create_leading_dirs('bar/foo')
        self.assertInvEqual(['', 'bar'], min_tree._inventory)
        self.assertEqual({'bar':('add', 'directory')}, min_tree._modified_paths)
        self.assertEqual('bar', parent.name)

    def test__create_leading_dirs_deep(self):
        min_tree = cvsps.importer.MinimalTree(inventory.Inventory())
        self.assertInvEqual([''], min_tree._inventory)

        # As should something deeply nested
        parent = min_tree._create_leading_dirs('baz/biz/boz/foo')
        self.assertInvEqual(['', 'baz', 'baz/biz', 'baz/biz/boz'],
                            min_tree._inventory)
        self.assertEqual({'baz':('add', 'directory'),
                          'baz/biz':('add', 'directory'),
                          'baz/biz/boz':('add', 'directory'),
                         }, min_tree._modified_paths)
        self.assertEqual('boz', parent.name)

    def test__create_leading_dirs_existing(self):
        inv = inventory.Inventory()
        inv.add(inventory.make_entry('directory', 'foo',
                    parent_id=inv.root.file_id, file_id='foo-id'))
        inv.add(inventory.make_entry('directory', 'bar', parent_id='foo-id',
                                     file_id='bar-id'))
        min_tree = cvsps.importer.MinimalTree(inv)
        self.assertInvEqual(['', 'foo', 'foo/bar'], min_tree._inventory)

        # As should something deeply nested
        parent = min_tree._create_leading_dirs('foo/bar/biz/baz')
        self.assertInvEqual(['', 'foo', 'foo/bar', 'foo/bar/biz'],
                            min_tree._inventory)
        self.assertEqual({'foo/bar/biz':('add', 'directory'),
                         }, min_tree._modified_paths)
        self.assertEqual('biz', parent.name)

    def test_set_text_existing(self):
        inv = inventory.Inventory()
        inv.add(inventory.make_entry('file', 'foo',
                    parent_id=inv.root.file_id, file_id='foo-id'))
        min_tree = cvsps.importer.MinimalTree(inv)

        min_tree.set_text('foo', 'text for foo\n')
        self.assertInvEqual(['', 'foo'], min_tree._inventory)
        self.assertEqual({'foo':('update', 'file'),
                         }, min_tree._modified_paths)
        self.assertEqual({'foo':'text for foo\n'}, min_tree._texts)

    def test_set_text_missing(self):
        min_tree = cvsps.importer.MinimalTree(inventory.Inventory())
        min_tree.set_text('foo', 'text for foo\n')
        self.assertInvEqual(['', 'foo'], min_tree._inventory)
        self.assertEqual({'foo':('add', 'file'),
                         }, min_tree._modified_paths)
        self.assertEqual({'foo':'text for foo\n'}, min_tree._texts)

    def test_set_text_missing_parent(self):
        min_tree = cvsps.importer.MinimalTree(inventory.Inventory())
        min_tree.set_text('foo/bar', 'text for bar\n')
        self.assertInvEqual(['', 'foo', 'foo/bar'], min_tree._inventory)
        self.assertEqual({'foo/bar':('add', 'file'),
                          'foo':('add', 'directory'),
                         }, min_tree._modified_paths)
        self.assertEqual({'foo/bar':'text for bar\n'}, min_tree._texts)

    def test_set_text_subdir(self):
        inv = inventory.Inventory()
        inv.add(inventory.make_entry('directory', 'foo',
                    parent_id=inv.root.file_id, file_id='foo-id'))
        min_tree = cvsps.importer.MinimalTree(inv)
        min_tree.set_text('foo/bar', 'text for bar\n')
        self.assertInvEqual(['', 'foo', 'foo/bar'], min_tree._inventory)
        self.assertEqual({'foo/bar':('add', 'file'),
                         }, min_tree._modified_paths)
        self.assertEqual({'foo/bar':'text for bar\n'}, min_tree._texts)

    def test_set_text_executable(self):
        min_tree = cvsps.importer.MinimalTree(inventory.Inventory())
        min_tree.set_text('foo/bar', 'text for bar\n', executable=True)
        self.assertInvEqual(['', 'foo', 'foo/bar'], min_tree._inventory)
        ie = min_tree._inventory[min_tree._inventory.path2id('foo/bar')]
        self.assertTrue(ie.executable)

    def test_set_text_non_executable(self):
        min_tree = cvsps.importer.MinimalTree(inventory.Inventory())
        min_tree.set_text('foo/bar', 'text for bar\n', executable=False)
        self.assertInvEqual(['', 'foo', 'foo/bar'], min_tree._inventory)
        ie = min_tree._inventory[min_tree._inventory.path2id('foo/bar')]
        self.assertEqual(False, ie.executable)

    def test_remove_file_missing(self):
        min_tree = cvsps.importer.MinimalTree(inventory.Inventory())
        self.assertRaises(errors.NoSuchFile, min_tree.remove_file, 'foo')

    def test_remove_file(self):
        inv = inventory.Inventory()
        inv.add(inventory.make_entry('file', 'foo',
                    parent_id=inv.root.file_id, file_id='foo-id'))
        min_tree = cvsps.importer.MinimalTree(inv)

        min_tree.remove_file('foo')
        self.assertInvEqual([''], min_tree._inventory)
        self.assertEqual({'foo':('remove', 'file')}, min_tree._modified_paths)
        self.assertEqual(set(['foo']), min_tree._removed_paths)

    def test_resolve_removed_nothing(self):
        min_tree = cvsps.importer.MinimalTree(inventory.Inventory())
        self.assertInvEqual([''], min_tree._inventory)
        min_tree.resolve_removed()
        self.assertInvEqual([''], min_tree._inventory)
        self.assertEqual({}, min_tree._modified_paths)
        self.assertEqual(set(), min_tree._removed_paths)

    def test_resolve_removed_root(self):
        """resolve_removed() shouldn't remove the root."""
        inv = inventory.Inventory()
        inv.add(inventory.make_entry('file', 'foo',
                    parent_id=inv.root.file_id, file_id='foo-id'))
        min_tree = cvsps.importer.MinimalTree(inv)
        min_tree.remove_file('foo')
        min_tree.resolve_removed()
        self.assertInvEqual([''], min_tree._inventory)
        self.assertEqual({'foo':('remove', 'file')}, min_tree._modified_paths)
        self.assertEqual(set(['foo']), min_tree._removed_paths)

    def test_resolve_removed_single(self):
        """A single empty dir should be removed."""
        inv = inventory.Inventory()
        inv.add(inventory.make_entry('directory', 'foo',
                    parent_id=inv.root.file_id, file_id='foo-id'))
        inv.add(inventory.make_entry('file', 'bar', parent_id='foo-id',
                                     file_id='bar-id'))
        min_tree = cvsps.importer.MinimalTree(inv)
        min_tree.remove_file('foo/bar')
        min_tree.resolve_removed()
        self.assertInvEqual([''], min_tree._inventory)
        self.assertEqual({'foo':('remove', 'directory'),
                          'foo/bar':('remove', 'file'),
                         }, min_tree._modified_paths)
        self.assertEqual(set(['foo', 'foo/bar']), min_tree._removed_paths)

    def test_resolve_removed_deep(self):
        """Removing directories can cascade."""
        inv = inventory.Inventory()
        inv.add(inventory.make_entry('directory', 'foo',
                    parent_id=inv.root.file_id, file_id='foo-id'))
        inv.add(inventory.make_entry('directory', 'bar', parent_id='foo-id',
                                     file_id='bar-id'))
        inv.add(inventory.make_entry('file', 'baz', parent_id='bar-id',
                                     file_id='baz-id'))
        min_tree = cvsps.importer.MinimalTree(inv)
        min_tree.remove_file('foo/bar/baz')
        min_tree.resolve_removed()
        self.assertInvEqual([''], min_tree._inventory)
        self.assertEqual({'foo':('remove', 'directory'),
                          'foo/bar':('remove', 'directory'),
                          'foo/bar/baz':('remove', 'file'),
                         }, min_tree._modified_paths)
        self.assertEqual(set(['foo', 'foo/bar', 'foo/bar/baz']),
                         min_tree._removed_paths)

    def test_resolve_removed_after_remove_and_add(self):
        """Even if a dir is empty at some point, if it has something when
        finished, it shouldn't be removed.
        """
        inv = inventory.Inventory()
        inv.add(inventory.make_entry('directory', 'foo',
                    parent_id=inv.root.file_id, file_id='foo-id'))
        inv.add(inventory.make_entry('file', 'bar', parent_id='foo-id',
                                     file_id='bar-id'))
        min_tree = cvsps.importer.MinimalTree(inv)
        min_tree.remove_file('foo/bar')
        min_tree.set_text('foo/baz', 'text for baz\n')
        min_tree.resolve_removed()
        # import pdb; pdb.set_trace()
        self.assertInvEqual(['', 'foo', 'foo/baz'], min_tree._inventory)
        self.assertEqual({'foo/bar':('remove', 'file'),
                          'foo/baz':('add', 'file'),
                         }, min_tree._modified_paths)
        self.assertEqual(set(['foo/bar']),
                         min_tree._removed_paths)
        self.assertEqual({'foo/baz':'text for baz\n'}, min_tree._texts)

    def test_get_text(self):
        """get_text returns the text given by set_text, and fails otherwise."""
        min_tree = cvsps.importer.MinimalTree(inventory.Inventory())
        self.assertRaises(KeyError, min_tree.get_text, 'foo')

        min_tree.set_text('foo', 'txt for foo\n')
        self.assertEqual('txt for foo\n', min_tree.get_text('foo'))

    def test_get_file(self):
        """CommitBuilder uses ie.snapshot, which requires get_file."""
        min_tree = cvsps.importer.MinimalTree(inventory.Inventory())
        min_tree.set_text('foo', 'txt\nfor\nfoo\n')

        foo_id = min_tree._inventory.path2id('foo')
        self.assertEqual(['txt\n', 'for\n', 'foo\n'],
                         min_tree.get_file(foo_id).readlines())

    def test_last_revision(self):
        min_tree = cvsps.importer.MinimalTree(inventory.Inventory())
        self.assertEqual(None, min_tree.last_revision())

        min_tree = cvsps.importer.MinimalTree(inventory.Inventory(),
                                              revision_id='test-rev-id')
        self.assertEqual('test-rev-id', min_tree.last_revision())

class TestImporterNoDisk(tests.TestCase):
    """Tests for Importer that don't require the filesystem."""

    def assertSanitized(self, expected, module):
        """Check that a module is sanitized correctly."""
        sanitized = cvsps.importer.Importer.sanitize_module(module)
        self.assertEqual(expected, sanitized)

    def test_sanitize_module(self):
        # Module '.' needs to be handled specially
        self.assertSanitized('ROOT', '.')
        self.assertSanitized('test', 'test')
        self.assertSanitized('test', '.test')
        self.assertSanitized('test_foo_bar', 'test/foo/bar')


class TestMinimalTreeWithBranch(tests.TestCaseWithTransport):
    """Test functionality that requires a real branch."""

    def assertInvEqual(self, expected, inv):
        """Make sure that the inventory has the right shape."""
        actual = [path for path, ie in inv.iter_entries_by_dir()]
        self.assertEqual(expected, actual)

    def test_create_from_branch(self):
        tree = self.make_branch_and_tree('tree')
        self.build_tree(['tree/a'])
        tree.add('a')
        tree.commit('first', rev_id='rev-1')
        min_tree = cvsps.importer.MinimalTree.create_from_branch(tree.branch)
        self.assertEqual('rev-1', min_tree.last_revision())
        self.assertInvEqual(['', 'a'], min_tree._inventory)

    def test_commit_from_empty(self):
        """Test that we can create the first commit."""
        tree = self.make_branch_and_tree('tree')

        min_tree = cvsps.importer.MinimalTree.create_from_branch(tree.branch)
        self.assertEqual(revision.NULL_REVISION, min_tree.last_revision())
        min_tree.set_text('a', 'text for a\n')
        min_tree.set_text('b/c', 'text for c\n')
        self.assertInvEqual(['', 'a', 'b', 'b/c'], min_tree._inventory)
        self.assertInvEqual([], min_tree._base_inventory)

        tree.branch.lock_write()
        try:
            new_rev = min_tree.commit(branch=tree.branch,
                            branch_name='tree',
                            message=u'new a and b/c',
                            timestamp=1164906100.026212,
                            timezone=0,
                            committer='Joe Foo <joe@foo.com>',
                            revision_id='rev-1',
                            )
        finally:
            tree.branch.unlock()
        self.assertEqual('rev-1', new_rev)
        self.assertEqual('rev-1', min_tree.last_revision())

        self.assertInvEqual(['', 'a', 'b', 'b/c'], min_tree._inventory)
        self.assertInvEqual(['', 'a', 'b', 'b/c'], min_tree._base_inventory)

        # And the branch itself should be updated
        self.assertEqual((1, 'rev-1'), tree.branch.last_revision_info())
        rev_tree = tree.branch.repository.revision_tree('rev-1')
        self.assertInvEqual(['', 'a', 'b', 'b/c'], rev_tree)

    def test_commit(self):
        """This is sort of a catch-all test.

        It makes sure that MinimalTree.commit() does the right thing. That
        after a commit MinimalTree is ready for creating another new tree,
        without having to re-open it from the branch, etc.
        """

        tree = self.make_branch_and_tree('tree')
        self.build_tree(['tree/a', 'tree/b/', 'tree/b/c'])
        tree.add(['a', 'b', 'b/c'])
        tree.commit('first', rev_id='rev-1')
        tree.branch.lock_write()
        self.addCleanup(tree.branch.unlock)

        min_tree = cvsps.importer.MinimalTree.create_from_branch(tree.branch)
        self.assertEqual('rev-1', min_tree.last_revision())
        self.assertInvEqual(['', 'a', 'b', 'b/c'], min_tree._inventory)
        self.assertInvEqual(['', 'a', 'b', 'b/c'], min_tree._base_inventory)

        # Now modify the minimal tree
        min_tree.set_text('a', 'new text for a\n')
        min_tree.set_text('d', 'text for d\n', executable=True)
        self.assertInvEqual(['', 'a', 'b', 'd', 'b/c'], min_tree._inventory)
        # _base_inventory should not be modified
        self.assertInvEqual(['', 'a', 'b', 'b/c'], min_tree._base_inventory)

        self.assertEqual('new text for a\n', min_tree.get_text('a'))
        self.assertEqual('text for d\n', min_tree.get_text('d'))

        new_rev = min_tree.commit(branch=tree.branch,
                        branch_name='tree',
                        message=u'new a and d',
                        timestamp=1164906336.026212,
                        timezone=0,
                        committer='Joe Foo <joe@foo.com>',
                        revision_id='rev-2',
                        )
        # After commit, the MinimalTree should be updated and ready for a new
        # commit.
        self.assertEqual('rev-2', new_rev)
        self.assertEqual('rev-2', min_tree.last_revision())

        self.assertInvEqual(['', 'a', 'b', 'd', 'b/c'], min_tree._inventory)
        self.assertInvEqual(['', 'a', 'b', 'd', 'b/c'],
                            min_tree._base_inventory)
        self.assertEqual({}, min_tree._modified_paths)
        self.assertEqual(set(), min_tree._removed_paths)
        self.assertEqual({}, min_tree._texts)
        self.assertEqual({}, min_tree._executable)

        # And the branch itself should be updated
        self.assertEqual((2, 'rev-2'), tree.branch.last_revision_info())
        rev_tree = tree.branch.repository.revision_tree('rev-2')
        self.assertInvEqual(['', 'a', 'b', 'd', 'b/c'], rev_tree)

        a_file_id = rev_tree.path2id('a')
        self.assertEqual('new text for a\n',
                         rev_tree.get_file_text(a_file_id))
        self.assertEqual(False, rev_tree.is_executable(a_file_id))

        d_file_id = rev_tree.path2id('d')
        self.assertEqual('text for d\n',
                         rev_tree.get_file_text(d_file_id))
        self.assertTrue(rev_tree.is_executable(d_file_id))

        # Now lets build up another commit
        min_tree.set_text('d', 'new text for d\n', executable=False)
        min_tree.remove_file('b/c')
        new_rev = min_tree.commit(branch=tree.branch,
                        branch_name='tree',
                        message=u'new d, deleted b/c (and b)',
                        timestamp=1164906400.026212,
                        timezone=0,
                        committer='Joe Bar <joe@foo.com>',
                        revision_id='rev-3',
                        )
        self.assertEqual('rev-3', new_rev)
        self.assertEqual('rev-3', min_tree.last_revision())

        self.assertInvEqual(['', 'a', 'd'], min_tree._inventory)
        self.assertInvEqual(['', 'a', 'd'], min_tree._base_inventory)

        self.assertEqual({}, min_tree._modified_paths)
        self.assertEqual(set(), min_tree._removed_paths)
        self.assertEqual({}, min_tree._texts)
        self.assertEqual({}, min_tree._executable)

        self.assertEqual((3, 'rev-3'), tree.branch.last_revision_info())
        rev_tree = tree.branch.repository.revision_tree('rev-3')
        self.assertInvEqual(['', 'a', 'd'], rev_tree)

        self.assertEqual('new text for d\n',
                         rev_tree.get_file_text(d_file_id))
        self.assertFalse(rev_tree.is_executable(d_file_id))


class TestCVSUpdater(TestCVSPSHelper):
    """Test that CVSUpdater returns valid paths."""

    def setup_basic_tree(self):
        """create a few files, and a couple checkins."""
        self.create_checkout()
        # Create the file 'a', and a subdirectory with another file
        self.build_tree(['checkout/a', 'checkout/sub/', 'checkout/sub/b'])
        self.cvs('add', 'a', 'sub', 'sub/b')
        self.cvs('commit', '-m', 'add a and sub/b')

        self.cvs('tag', '-b', 'branch')
        self.cvs('update', '-r', 'branch')

        self.build_tree(['checkout/c', 'checkout/sub/d'])
        self.cvs('add', 'c', 'sub/d')
        self.cvs('commit', '-m', 'add c and sub/d in branch')

    def assertFilesEqual(self, expected, actual):
        """Read both files, and make sure the contents are identical."""
        f = open(expected, 'rb')
        try:
            exp_txt = f.read()
        finally:
            f.close()

        f = open(actual, 'rb')
        try:
            act_txt = f.read()
        finally:
            f.close()

        self.assertEqualDiff(exp_txt, act_txt)

    def test_real_cvs(self):
        """This is written as a big test, because spawning cvs is very slow.

        cvs tends to sleep for 1s before doing any work (or maybe just when
        finished?). Either way, it means we want to avoid doing it for lots of
        tests.

        So this test is longer than we would generally prefer.
        """
        self.setup_basic_tree()
        a_control = osutils.pathjoin(self.cvs_root, 'a,v')
        a_attic_control = osutils.pathjoin(self.cvs_root, 'Attic', 'a,v')
        self.assertPathExists(a_control)
        self.assertPathDoesNotExist(a_attic_control)

        b_control = osutils.pathjoin(self.cvs_root, 'sub', 'b,v')
        b_attic_control = osutils.pathjoin(self.cvs_root, 'sub', 'Attic', 'b,v')
        self.assertPathExists(b_control)
        self.assertPathDoesNotExist(b_attic_control)

        c_control = osutils.pathjoin(self.cvs_root, 'c,v')
        c_attic_control = osutils.pathjoin(self.cvs_root, 'Attic', 'c,v')
        self.assertPathDoesNotExist(c_control)
        self.assertPathExists(c_attic_control)

        d_control = osutils.pathjoin(self.cvs_root, 'sub', 'd,v')
        d_attic_control = osutils.pathjoin(self.cvs_root, 'sub', 'Attic', 'd,v')
        self.assertPathDoesNotExist(d_control)
        self.assertPathExists(d_attic_control)

        updater = cvsps.importer.CVSUpdater(self.cvs_root, '.')
        
        self.assertEqual(a_control, updater._get_rcs_filename('a'))
        self.assertEqual(b_control, updater._get_rcs_filename('sub/b'))
        self.assertEqual(c_attic_control, updater._get_rcs_filename('c'))
        self.assertEqual(d_attic_control, updater._get_rcs_filename('sub/d'))

        # Now start creating test files in the output
        self.build_tree(['test_out/'])
        self.assertEqual((['a'], []),
                         updater.update_file('test_out', 'a', '1.1'))
        self.assertFilesEqual('checkout/a', 'test_out/a')

        self.assertEqual((['sub', 'sub/b'], []),
                         updater.update_file('test_out', 'sub/b', '1.1'))
        self.assertFilesEqual('checkout/sub/b', 'test_out/sub/b')

        self.assertEqual((['c'], []),
                         updater.update_file('test_out', 'c', '1.1.2.1'))
        self.assertFilesEqual('checkout/c', 'test_out/c')

        self.assertEqual((['sub/d'], []),
                         updater.update_file('test_out', 'sub/d', '1.1.2.1'))
        self.assertFilesEqual('checkout/sub/d', 'test_out/sub/d')

        # For 'c' and 'd' the files have been deleted in the main branch
        self.assertEqual(([], ['c']),
                         updater.update_file('test_out', 'c', '1.1(DEAD)'))
        self.assertPathDoesNotExist('test_out/c')

        # If the subdir is empty, it will be removed
        os.remove('test_out/sub/b')
        self.assertEqual(([], ['sub/d', 'sub']),
                         updater.update_file('test_out', 'sub/d', '1.1(DEAD)'))
        self.assertPathDoesNotExist('test_out/sub/d')
        self.assertPathDoesNotExist('test_out/sub')

    def test__is_executable(self):
        if not osutils.supports_executable():
            # executable bits aren't tracked here
            return

        self.build_tree(['666', '660', '777', '700', '670', '667'])
        os.chmod('666', 0666)
        os.chmod('660', 0660)
        os.chmod('777', 0777)
        os.chmod('700', 0700)
        os.chmod('670', 0670)
        os.chmod('667', 0667)

        is_exec = cvsps.importer.CVSUpdater._is_executable
        self.failIf(is_exec('666'))
        self.failIf(is_exec('660'))
        self.failUnless(is_exec('777'))
        self.failUnless(is_exec('700'))
        self.failUnless(is_exec('670'))
        self.failUnless(is_exec('667'))

    def test__remove_file(self):
        # Removing a missing file shouldn't fail
        updater = cvsps.importer.CVSUpdater('.', '.')
        self.assertEqual(['foo'], updater._remove_file('.', 'foo'))
        self.assertEqual(['foo'], updater._remove_file('path', 'foo'))
        self.assertEqual(['foo/bar'], updater._remove_file('.', 'foo/bar'))
        self.assertEqual(['foo/bar'], updater._remove_file('path', 'foo/bar'))

        # Now with a real file, it should remove it, as well as empty
        # containing directories
        self.build_tree(['file', 'dir/', 'dir/file1', 'dir/file2'])
        self.assertEqual(['file'], updater._remove_file('.', 'file'))
        self.assertPathDoesNotExist('file')

        self.assertEqual(['dir/file2'], updater._remove_file('.', 'dir/file2'))
        self.assertPathDoesNotExist('dir/file2')

        self.assertEqual(['dir/file1', 'dir'],
                         updater._remove_file('.', 'dir/file1'))
        self.assertPathDoesNotExist('dir/file1')
        self.assertPathDoesNotExist('dir')

        # But if we only supply the path, it shouldn't remove the dir
        self.build_tree(['dir/', 'dir/file'])
        self.assertEqual(['file'], updater._remove_file('dir', 'file'))
        self.assertPathDoesNotExist('dir/file')
        self.assertPathExists('dir')

    def test__prepare_target(self):
        updater = cvsps.importer.CVSUpdater('.', '.')
        self.assertEqual(('./file', []),
                         updater._prepare_target('.', 'file'))

        self.assertEqual(('./dir1/file1', ['dir1']),
                         updater._prepare_target('.', 'dir1/file1'))
        self.assertPathExists('dir1')

        self.assertEqual(('./dir2/subdir1/file1', ['dir2', 'dir2/subdir1']),
                         updater._prepare_target('.', 'dir2/subdir1/file1'))
        self.assertPathExists('dir2')
        self.assertPathExists('dir2/subdir1')

        # And it should only operate in relative paths
        self.assertEqual(('dir2/subdir1/file2', []),
                         updater._prepare_target('dir2/subdir1', 'file2'))

        self.assertEqual(('dir2/subdir2/file1', ['subdir2']),
                         updater._prepare_target('dir2', 'subdir2/file1'))
        self.assertPathExists('dir2/subdir2')
