/*
 * Folder list widget
 *
 * Copyright (C) 2003  Enrico Zini <enrico@debian.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#pragma implementation

#include "FolderList.h"

#include <buffy/config/Config.h>

#include <map>
#include <iostream>

#include <sigc++/sigc++.h>
#include <gtkmm/treemodel.h>
#include <gtkmm/treemodelfilter.h>
#include <gtkmm/main.h>

#include "Environment.h"

using namespace std;
using namespace buffy;

static const int VF_ALWAYS		= 0x0001;
static const int VF_IMPORTANT	= 0x0002;
static const int VF_NONEMPTY	= 0x0004;
static const int VF_NEVER		= 0x0008;


bool FolderList::on_event(GdkEvent* e)
{
	if (e->type == GDK_BUTTON_PRESS && e->button.button > 1)
	{
		Gtk::TreeModel::Path path;
		Gtk::TreeViewColumn* column;
		int cell_x, cell_y;
		if (folderList.get_path_at_pos(
				(int)e->button.x, (int)e->button.y,
				path, column,
				cell_x, cell_y))
		{
			// Clicked on an item
			debug("Cell %d, %d\n", cell_x, cell_y);

			Gtk::TreeModel::Row row = *sortedModel->get_iter(path);
			MailFolder folder = row[folderListModelColumns.folder];
			verbose("Clicked on %s\n", folder.path().c_str());
			config::Config& conf(Environment::get().config());
			bool forceview = conf.folder(folder).forceview();
			bool forcehide = conf.folder(folder).forcehide();
			cerr << "foo0" << endl;

			folderList.get_selection()->select(row);

			Gtk::CheckMenuItem* item;

			itemPopup.items().clear();
			itemPopup.items().push_back(Gtk::Menu_Helpers::CheckMenuElem("View always"));
			item = dynamic_cast<Gtk::CheckMenuItem*>(&itemPopup.items().back());
			item->set_active(forceview);
			item->signal_toggled().connect(
						sigc::bind< MailFolder >(
							sigc::mem_fun(*this, &FolderList::toggle_view_always), folder));

			cerr << "foo1" << endl;

			itemPopup.items().push_back(Gtk::Menu_Helpers::CheckMenuElem("Hide always"));
			item = dynamic_cast<Gtk::CheckMenuItem*>(&itemPopup.items().back());
			item->set_active(forcehide);
			item->signal_toggled().connect(
						sigc::bind< MailFolder >(
							sigc::mem_fun(*this, &FolderList::toggle_hide_always), folder));

#if 0
			itemPopup.items().push_back(Gtk::Menu_Helpers::MenuElem("Select this tag set",
						sigc::bind< TagSet >(
							sigc::mem_fun(*this, &ItemList::do_signal_select_tagset), ts)));
			itemPopup.items().push_back(Gtk::Menu_Helpers::MenuElem("Select this tag set in the other panel",
						sigc::bind< TagSet >(
							sigc::mem_fun(*this, &ItemList::do_signal_select_tagset_other_panel), ts)));
#endif
			itemPopup.popup(e->button.button, e->button.time);

			cerr << "popup" << endl;

			return true;
		} else {
			// Clicked outside
			warning("itemList.get_path_at_pos failed\n");
			return false;
		}
	}
#if 0
			Glib::RefPtr<Gtk::TreeSelection> sel = itemList.get_selection();
			int rows = sel->count_selected_rows();

			if (rows == 1)
			{
				itemPopup.items().push_back(Gtk::Menu_Helpers::MenuElem("Select this tag set",
							sigc::bind< TagSet >(
								sigc::mem_fun(*this, &ItemList::do_signal_select_tagset), ts)));
				itemPopup.items().push_back(Gtk::Menu_Helpers::MenuElem("Select this tag set in the other panel",
							sigc::bind< TagSet >(
								sigc::mem_fun(*this, &ItemList::do_signal_select_tagset_other_panel), ts)));
			}
			else
			{
				itemPopup.items().push_back(Gtk::Menu_Helpers::MenuElem("Merge",
							sigc::mem_fun(*this, &ItemList::do_signal_request_tagset_merge)));
				itemPopup.items().push_back(Gtk::Menu_Helpers::MenuElem("Intersect",
							sigc::mem_fun(*this, &ItemList::do_signal_request_tagset_intersect)));
			}
			itemPopup.items().push_back(Gtk::Menu_Helpers::MenuElem("Copy to other panel",
						sigc::mem_fun(*this, &ItemList::do_signal_request_item_copy)));
			itemPopup.items().push_back(Gtk::Menu_Helpers::MenuElem("Move to other panel",
						sigc::mem_fun(*this, &ItemList::do_signal_request_item_move)));

			TagMenu<TagcollDocument<string> >* addMenu = new TagMenu<TagcollDocument<string> >();
			addMenu->set_manage();
			addMenu->populateUnselected(doc, ts);
			string stag = tag;
			addMenu->signal_selected().connect(
						sigc::mem_fun(*this, &ItemList::on_add_tag));
			itemPopup.items().push_back(Gtk::Menu_Helpers::MenuElem("_Add...", *addMenu));
	
			if (!ts.empty())
			{
				itemPopup.items().push_back(Gtk::Menu_Helpers::SeparatorElem());
				for (TagSet::const_iterator i = ts.begin();
						i != ts.end(); i++)
					itemPopup.items().push_back(Gtk::Menu_Helpers::MenuElem("Remove " + i->fullname(),
						sigc::bind<Tag>(
							sigc::mem_fun(*this, &ItemList::on_remove_tag), *i)));
			}

			itemPopup.popup(e->button.button, e->button.time);
			//printf("Menu finished\n");

			//delete addMenu;
			return true;
		} else {
			// Clicked outside
			warning("itemList.get_path_at_pos failed\n");
			return false;
		}
	}
#endif
	return false;
}

void FolderList::toggle_hide_always(const MailFolder& folder)
{
// TODO: set the per-list configuration
// TODO: honor per-list configuration when filtering
	config::Config& conf(Environment::get().config());
	config::FolderNode fc = conf.folder(folder);
	fc.setForceHide(!fc.forcehide());
	configureFilter(conf);
}

void FolderList::toggle_view_always(const MailFolder& folder)
{
// TODO: set the per-list configuration
// TODO: honor per-list configuration when filtering
	config::Config& conf(Environment::get().config());
	config::FolderNode fc = conf.folder(folder);
	//cerr << "Currently: \"" << folder.get("forceview") << "\"" << endl;
	fc.setForceView(!fc.forceview());
	configureFilter(conf);
}


void FolderList::on_row_activated(const Gtk::TreeModel::Path& path, Gtk::TreeViewColumn*)
{
	if(const Gtk::TreeIter iter = folderList.get_model()->get_iter(path))
	{
		Gtk::TreeModel::Row row = *iter;
		MailFolder folder = row[folderListModelColumns.folder];
		signal_open_folder().emit(folder);
	}
}

bool FolderList::filter_row(const Gtk::TreeModel::const_iterator& iter)
{
	// Hide empty folders
	Gtk::TreeModel::Row row = *iter;

	MailFolder f = row[folderListModelColumns.folder];
	if (!f)
		return false;

	int flags = row[folderListModelColumns.view_flags];
	
	if (flags & VF_ALWAYS)
		return true;

	if (flags & VF_NEVER)
		return false;

	if (row[folderListModelColumns.msg_unread] > 0)
		return true;

	if (flags & VF_IMPORTANT && row[folderListModelColumns.msg_flagged] > 0)
		return true;

	if (flags & VF_NONEMPTY && row[folderListModelColumns.msg_total] > 0)
		return true;

	return false;
}


FolderList::FolderList()
{
	// Add the TreeView, inside a ScrolledWindow
	scrolledFolderList.add(folderList);

	// Only show the scrollbars when they are necessary:
	scrolledFolderList.set_policy(Gtk::POLICY_AUTOMATIC, Gtk::POLICY_AUTOMATIC);

	pack_start(scrolledFolderList, Gtk::PACK_EXPAND_WIDGET);

	// Create the Tree model
	folderListModel = Gtk::ListStore::create(folderListModelColumns);

	filteredModel = Gtk::TreeModelFilter::create(
			folderListModel,
			folderListModel->get_path(folderListModel->children().begin()));
	filteredModel->set_visible_func(sigc::mem_fun(*this, &FolderList::filter_row));

	sortedModel = Gtk::TreeModelSort::create(
			filteredModel);

	//folderList.set_model(folderListModel);
	//folderList.set_model(filteredModel);
	folderList.set_model(sortedModel);


	// Add the view columns
	Gtk::TreeViewColumn* column;
	int count;

	count = folderList.append_column("Name", folderListModelColumns.name);
	column = folderList.get_column(count - 1);
	column->set_sort_column(folderListModelColumns.name);
	
	count = folderList.append_column("New", folderListModelColumns.msg_new);
	column = folderList.get_column(count - 1);
	column->set_sort_column(folderListModelColumns.msg_new);

	count = folderList.append_column("Unread", folderListModelColumns.msg_unread);
	column = folderList.get_column(count - 1);
	column->set_sort_column(folderListModelColumns.msg_unread);

	count = folderList.append_column("Total", folderListModelColumns.msg_total);
	column = folderList.get_column(count - 1);
	column->set_sort_column(folderListModelColumns.msg_total);

	count = folderList.append_column("Flagged", folderListModelColumns.msg_flagged);
	column = folderList.get_column(count - 1);
	column->set_sort_column(folderListModelColumns.msg_flagged);

	// Restore sort order if previously saved
	config::Config& conf = Environment::get().config();
	config::Node prefs(conf.application("buffy"));
	if (prefs.getBool("sorted"))
		sortedModel->set_sort_column_id(prefs.getInt("sort_col_id"), (Gtk::SortType)prefs.getInt("sort_order"));

	Glib::RefPtr<Gtk::TreeSelection> folderListSelection = folderList.get_selection();
	folderListSelection->set_mode(Gtk::SELECTION_SINGLE);
	
	folderList.add_events(Gdk::BUTTON_PRESS_MASK);
	folderList.signal_event().connect(sigc::mem_fun(*this, &FolderList::on_event));
	folderList.signal_row_activated().connect(sigc::mem_fun(*this, &FolderList::on_row_activated));

	sortedModel->signal_sort_column_changed().connect(sigc::mem_fun(*this, &FolderList::on_sort_column_changed));
}

void FolderList::on_sort_column_changed()
{
	int id;
	Gtk::SortType order;
	bool res = sortedModel->get_sort_column_id(id, order);
	//fprintf(stderr, "sort column changed id %d order %d res %d\n", id, order, res);

	config::Config& conf = Environment::get().config();
	config::Node prefs = conf.application("buffy");

	prefs.setBool("sorted", res);
	prefs.setInt("sort_col_id", id);
	prefs.setInt("sort_order", order);
}

void FolderList::consume(MailFolder& folder)
{
	Gtk::TreeModel::Row row = *(folderListModel->append());
	row[folderListModelColumns.folder] = folder;
	row[folderListModelColumns.name] = folder.name();

	//int msg_total, msg_unread, msg_new, msg_flagged;
	//folder.computeStatistics(&msg_total, &msg_unread, &msg_new, &msg_flagged);
	if (folder.changed())
		folder.updateStatistics();

	row[folderListModelColumns.msg_new] = folder.getMsgNew();
	row[folderListModelColumns.msg_unread] = folder.getMsgUnread();
	row[folderListModelColumns.msg_total] = folder.getMsgTotal();
	row[folderListModelColumns.msg_flagged] = folder.getMsgFlagged();

	config::Config& conf(Environment::get().config());
	int flags = 
		(conf.view().empty() ? VF_ALWAYS : 0) |
		(conf.view().read() ? VF_NONEMPTY : 0) |
		(conf.view().important() ? VF_IMPORTANT : 0);
	config::FolderNode fc = conf.folder(folder);
	if (fc.forceview())
		flags |= VF_ALWAYS;
	if (fc.forcehide())
		flags |= VF_NEVER;
	row[folderListModelColumns.view_flags] = flags;

	while (Gtk::Main::events_pending())
		Gtk::Main::iteration(false);
}

void FolderList::clear()
{
	folderListModel->clear();
}
	
void FolderList::do_update()
{
	debug("FolderList::do_update\n");
	for (Gtk::TreeModel::Children::const_iterator i = folderListModel->children().begin();
			i != folderListModel->children().end(); i++)
	{
		Gtk::TreeModel::Row row = *i;
		MailFolder folder = row[folderListModelColumns.folder];
		//row[folderListModelColumns.name] = folder.name();

		if (folder.changed())
			folder.updateStatistics();

		row[folderListModelColumns.msg_new] = folder.getMsgNew();
		row[folderListModelColumns.msg_unread] = folder.getMsgUnread();
		row[folderListModelColumns.msg_total] = folder.getMsgTotal();
		row[folderListModelColumns.msg_flagged] = folder.getMsgFlagged();
	}
}

bool FolderList::do_timer_tick()
{
	do_update();
	return true;
}

void FolderList::configureFilter(config::Config& conf)
{
	int baseFlags = 
		(conf.view().empty() ? VF_ALWAYS : 0) |
		(conf.view().read() ? VF_NONEMPTY : 0) |
		(conf.view().important() ? VF_IMPORTANT : 0);

	for (Gtk::TreeModel::Children::const_iterator i = folderListModel->children().begin();
			i != folderListModel->children().end(); i++)
	{
		int flags = baseFlags;
		Gtk::TreeModel::Row row = *i;
		MailFolder folder = row[folderListModelColumns.folder];
		config::FolderNode fc(conf.folder(folder));
		if (fc.forceview())
			flags |= VF_ALWAYS;
		if (fc.forcehide())
			flags |= VF_NEVER;

		row[folderListModelColumns.view_flags] = flags;
	}

	filteredModel->refilter();
}

// vim:set ts=4 sw=4:
