/********************************************************************
 * Copyright (C) 2005, 2006 Piotr Pszczolkowski
 *-------------------------------------------------------------------
 * This file is part of BSCommander (Beesoft Commander).
 *
 * BsC is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * BsC is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with BsC; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *******************************************************************/

/*------- include files:
-------------------------------------------------------------------*/
#include "Workspace.h"
#include "ViewWindow.h"
#include "LeftView.h"
#include "RightView.h"
#include "Copier.h"
#include "Packer.h"
#include "ViewTableItem.h"
#include "CmpFiles.h"
#include "CmpDirs.h"
#include "DirSynch.h"
#include "DateTime.h"
#include "DiffViewer.h"
#include "Configurator.h"
#include "FileJoiner.h"
#include "FTPCopyFrom.h"
#include "FTPCopyTo.h"
#include "MainWindow.h"
#include <qsplitter.h>
#include <qlayout.h>
#include <qinputdialog.h>
#include <qvaluelist.h>
#include <qmessagebox.h>
#include <qpixmap.h>

/*------- local constants:
-------------------------------------------------------------------*/
const int     Workspace::HANDLE_WIDTH        = 2;
const QString Workspace::CMP_FILES_ICON      = "cmpfiles.png";
const QString Workspace::CMP_FILES_KOMPARE   = "&kompare"; 
const QString Workspace::CMP_FILES_DIFF      = "&diff"; 
const QString Workspace::KOMPARE_PRG_NAME    = "kompare";
//...................................................................
const QString Workspace::CMP_FILES_CAPTION   = QT_TR_NOOP( "Files comparing" );
const QString Workspace::CMP_FILES_MSG       = QT_TR_NOOP( "You would like to compare these two files:\n\n1) %1\n2) %2\n " );
const QString Workspace::CMP_FILES_NEG_INFO  = QT_TR_NOOP( "Files are not identical (not the same size)." );
const QString Workspace::CMP_FILES_IOERROR   = QT_TR_NOOP( "Can't open/read a file." );
const QString Workspace::CMP_FILES_BREAK     = QT_TR_NOOP( "Operation has been broken by user." );
const QString Workspace::CMP_FILES_EQUAL     = QT_TR_NOOP( "Both files are identical." );
const QString Workspace::CMP_FILES_NOT_EQUAL = QT_TR_NOOP( "These files are not identical." );
const QString Workspace::FilesJoining        = QT_TR_NOOP( "Files joing" );  
const QString Workspace::EnterFileName       = QT_TR_NOOP( "Enter destination file name:" ); 

 
//*******************************************************************
// Workspace
//*******************************************************************
Workspace::Workspace( QWidget* const in_parent, MainWindow* const in_main_window )
: QFrame             ( in_parent )
, d_main_window      ( in_main_window )
, d_left_view_window ( 0 )
, d_right_view_window( 0 )
, d_splitter         ( new QSplitter( QSplitter::Horizontal, this ))
, d_viewL            ( new LeftView( d_splitter, this ))
, d_viewR            ( new RightView( d_splitter, this ))
, d_ftp_copy_from    ( 0 )
, d_ftp_copy_to      ( 0 )
{
	QVBoxLayout* const main_layout = new QVBoxLayout( this );
	main_layout->addWidget( d_splitter );
	d_splitter->setHandleWidth( HANDLE_WIDTH );
	d_splitter->setOpaqueResize();
		
	QValueList<int> vlist;
	vlist.push_back( 1 );
	vlist.push_back( 1 );
	d_splitter->setSizes( vlist );
}
// end of Workspace

//###################################################################
//#                                                                 #
//#                          P U B L I C                            #
//#                                                                 #
//###################################################################

//*******************************************************************
// set_left_view_window                                       PUBLIC
//-------------------------------------------------------------------
// Funckja, za pomoca ktorej lewy panel (LeftView) melduje
// wskaznik do swojego widoku (ViewWindow).
//*******************************************************************
void Workspace::set_left_view_window( ViewWindow* const in_view )
{
	d_left_view_window = in_view;
}
// end of set_left_view_window

//*******************************************************************
// set_right_view_window                                      PUBLIC
//-------------------------------------------------------------------
// Funkcja, za pomoca ktorej prawy panel (LeftView) melduje
// wskaznik do swojego wyniku (ViewWindow).
//*******************************************************************
void Workspace::set_right_view_window( ViewWindow* const in_view )
{
	d_right_view_window = in_view;
}
// end of set_right_view_window


//###################################################################
//#                                                                 #
//#                         P R I V A T E                           #
//#                                                                 #
//###################################################################

//*******************************************************************
// show                                            PRIVATE inherited
//*******************************************************************
void Workspace::show()
{
	QFrame::show();
	d_viewL->set_focus();
}
// end of show

//*******************************************************************
// keyPressEvent                                             PRIVATE
//-------------------------------------------------------------------
// Na poziomie 'Workspace' przechwytujemy i obslugujemy klawisze,
// ktore dotycza operacji wymagajacych dzialan na obu panelach.
// (Panele nie wiedza nic o sobie wzajemnie.)
//*******************************************************************
void Workspace::keyPressEvent( QKeyEvent* e )
{
	bool accepted = FALSE;

	if( Qt::ControlButton == e->state() ) {
		switch( e->key() ) {
			case Qt::Key_O:
				e->accept();
				accepted = TRUE;
				ctrl_O();
				break;
		}
	}
	else {
		switch( e->key() ) {
			case Qt::Key_Escape:
				emit stop();
				break;
			case Qt::Key_F5:	// copy
				e->accept();
				accepted = TRUE;
				slotF5();
				break;
			case Qt::Key_F9:	// pack
				e->accept();
				accepted = TRUE;
				slotF9();
				break;
		}
	}
	
	if( FALSE == accepted ) {
		QFrame::keyPressEvent( e );
	}
}
// end of keyPressEvent

//*******************************************************************
// slot_switch_panels                                   PRIVATE slot
//-------------------------------------------------------------------
// Zamiana miejscami obu paneli.
//*******************************************************************
void Workspace::slot_switch_panels()
{
	static bool switched = FALSE;

	if( FALSE == switched ) {
		d_splitter->moveToFirst( (QWidget*)d_viewR );
		switched = TRUE;
	}
	else {
		d_splitter->moveToFirst( (QWidget*)d_viewL );
		switched = FALSE;
	}
	d_splitter->refresh();
}
// end of slot_switch_panels

//*******************************************************************
// ctrl_O                                                    PRIVATE
//*******************************************************************
void Workspace::ctrl_O()
{
	ViewWindow* src_view =  0;
	ViewWindow* dst_view =  0;

	if( d_viewL->has_focus() ) {
		src_view =  d_left_view_window;
		dst_view =  d_right_view_window;
	}
	else if( d_viewR->has_focus() ) {
		src_view =  d_right_view_window;
		dst_view =  d_left_view_window;
	}

	if( src_view && dst_view ) {
		if( d_left_view_window->is_ftp_dir() || d_right_view_window->is_ftp_dir() ) {
			// to do
		}
		else {
			const QString src_dir = src_view->get_dir();
			const QString src_fname = src_view->get_current_fname();
			dst_view->disp_this_location( src_dir, src_fname );
		}
	}
}
// end of ctrl_O

//*******************************************************************
// copy                                                      PRIVATE
//-------------------------------------------------------------------
// Kopiowanie zaznaczonych plikow/katalogow pomiedzy panelami.
//*******************************************************************
void Workspace::copy( const int in_direction )
{
    ViewWindow* src_view = 0;
    ViewWindow* dst_view = 0;

    switch( in_direction ) {
		case LEFT_TO_RIGHT:
			src_view = d_left_view_window;
			dst_view = d_right_view_window;
			break;
		case RIGHT_TO_LEFT:
			src_view = d_right_view_window;
			dst_view = d_left_view_window;
			break;
	}
	
	if( src_view && dst_view ) {
		const ViewTable::SelectedItems items = src_view->selections();

		if( !items.empty() ) {
			// FTP - kopiowanie z serwera ftp na dysk --------------------------
            if( src_view->is_ftp_dir() ) {
                if( d_ftp_copy_from ) delete d_ftp_copy_from;
                d_ftp_copy_from = new FTPCopyFrom( this, src_view->get_fsi(), items, dst_view->get_dir() );
                if( d_ftp_copy_from ) {
                    connect( d_ftp_copy_from, SIGNAL( done() ), this, SLOT( after_ftp_copy() ));
                    d_ftp_copy_from->exec();
                }
            }
            // FTP - kopiowanie z dysku na serwer ------------------------------
            else if( dst_view->is_ftp_dir() ) {
                if( d_ftp_copy_to ) delete d_ftp_copy_to;
                d_ftp_copy_to = new FTPCopyTo( this, dst_view->get_fsi(), items, src_view->get_dir() );
                if( d_ftp_copy_to ) {
                    connect( d_ftp_copy_to, SIGNAL( done() ), this, SLOT( after_ftp_copy() ));
                    d_ftp_copy_to->exec();
                }
            }
			// LFS - kopiowanie w obrebie dysku --------------------------------
			else {
                QString dir = dst_view->get_dir();
				Copier* copier = new Copier( this, items, dir );
				if( copier ) {
					copier->exec();
					delete copier;
					copier = 0;
				}
                src_view->update_view( ViewWindow::REMOVE_SELECTION );
                dst_view->update_view( ViewWindow::STAY_SELECTION   );
			}
		}
	}
}
// end of copy

//*******************************************************************
// after_ftp_copy                                       PRIVATE slot
//*******************************************************************
void Workspace::after_ftp_copy()
{
    if( d_ftp_copy_from ) {
        disconnect( d_ftp_copy_from, SIGNAL( done() ), this, SLOT( after_ftp_copy() ));
        delete d_ftp_copy_from;
        d_ftp_copy_from = 0;
    }
    if( d_ftp_copy_to ) {
        disconnect( d_ftp_copy_to, SIGNAL( done() ), this, SLOT( after_ftp_copy() ));
        delete d_ftp_copy_to;
        d_ftp_copy_to = 0;
    }

	d_left_view_window->update_view( ViewWindow::REMOVE_SELECTION );
	d_right_view_window->update_view( ViewWindow::STAY_SELECTION );
}
// end of after_ftp_copy

//*******************************************************************
// pack                                                      PRIVATE
//-------------------------------------------------------------------
// Pakowanie zaznaczonych plikow/katalogow.
//*******************************************************************
void Workspace::pack( const int in_direction )
{
	ViewWindow* src_view = 0;
	ViewWindow* dst_view = 0;

	switch( in_direction ) {
		case LEFT_TO_RIGHT:
			src_view = d_left_view_window;
			dst_view = d_right_view_window;
			break;
		case RIGHT_TO_LEFT:
			src_view = d_right_view_window;
			dst_view = d_left_view_window;
			break;
	}
	
	if( src_view && dst_view ) {
		const ViewTable::SelectedItems& items = src_view->selections();

		// FTP
		if( src_view->is_ftp_dir() || dst_view->is_ftp_dir() ) {
		}
		// LFS
		else {
			if( FALSE == items.empty() ) {
				Packer* packer = new Packer( this, items, dst_view->get_dir() );
				if( packer ) {
					packer->exec();
					delete packer;
					packer = 0;
				}
			}
			src_view->update_view( ViewWindow::REMOVE_SELECTION );
			dst_view->update_view( ViewWindow::STAY_SELECTION );
		}
	}
}
// end of pack

//*******************************************************************
// join_files                                                PRIVATE
//-------------------------------------------------------------------
// Obsluga rzadania laczenia plikow tekstowych.
//*******************************************************************
void Workspace::join_files( const int in_direction )
{
    ViewWindow* src_view = 0;
    ViewWindow* dst_view = 0;

	switch( in_direction ) {
		case LEFT_TO_RIGHT:
			src_view = d_left_view_window;
			dst_view = d_right_view_window;
			break;
		case RIGHT_TO_LEFT:
			src_view = d_right_view_window;
			dst_view = d_left_view_window;
			break;
	}
	
	if( src_view && dst_view ) {
		const ViewTable::SelectedItems& items = src_view->selections();

		// FTP
		if( src_view->is_ftp_dir() || dst_view->is_ftp_dir() ) {
		}
		// LFS
		else {
			if( !items.empty() ) {
			    bool ok;
                const QString fname = QInputDialog::getText(
                        tr(FilesJoining), tr(EnterFileName), QLineEdit::Normal,
                        QString::null, &ok, this );
                if ( ok && !fname.isEmpty() ) {
                    FileJoiner* fj = new FileJoiner( items, dst_view->get_dir(), fname );
                    if( fj ) {
                        fj->run();
                        delete fj;
                        fj = 0;
                    }
                    src_view->update_view( ViewWindow::REMOVE_SELECTION );
                    dst_view->update_view( ViewWindow::STAY_SELECTION );
                }
            }
		}
	}

}
// end of slot_join_files



//###################################################################
//#                                                                 #
//#                           S L O T S                             #
//#                                                                 #
//###################################################################



//*******************************************************************
// lang_changed                                         PRIVATE slot
//*******************************************************************
void Workspace::lang_changed()
{
	emit update_lang();
}
// end of lang_changed

//*******************************************************************
// selection_changed                                    PRIVATE slot
//*******************************************************************
void Workspace::selection_changed()
{
	emit update_selection();
}
// end of selection_changed

//*******************************************************************
// configure                                            PRIVATE slot
//*******************************************************************
void Workspace::configure()
{
	Configurator* cfg = new Configurator( this );
	if( cfg ) {
		connect( cfg, SIGNAL( looks_refresh() ), this, SIGNAL( looks_refresh() ));
		cfg->exec();
		disconnect( cfg, SIGNAL( looks_refresh() ), this, SIGNAL( looks_refresh() ));
		delete cfg;
		cfg = 0;
	}
}
// end of configure

//*******************************************************************
// cmp_dirs                                             PRIVATE slot
//*******************************************************************
void Workspace::cmp_dirs()
{
	if( d_left_view_window->is_ftp_dir() || d_right_view_window->is_ftp_dir() ) {
		// to do
	}
	else {
		const ViewTableItem* const lp_item = d_left_view_window->current_item();
		const ViewTableItem* const rp_item = d_right_view_window->current_item();
		
		if( lp_item && rp_item ) {
			if( lp_item->is_dir() && ( FALSE == lp_item->is_parent_dir() )) {
				if( rp_item->is_dir() && ( FALSE == rp_item->is_parent_dir() )) {
					CmpDirs* cd = new CmpDirs( this, lp_item->path(), rp_item->path() );
					if( cd ) {
						cd->exec();
						delete cd;
						cd = 0;
					}
				}
			}
		}
	}
}
// end of cmp_dirs

//*******************************************************************
// dir_synch                                            PRIVATE slot
//*******************************************************************
void Workspace::dir_synch()
{
	if( d_left_view_window->is_ftp_dir() || d_right_view_window->is_ftp_dir() ) {
		// to do
	}
	else {
		const ViewTableItem* const lp_item = d_left_view_window->current_item();
		const ViewTableItem* const rp_item = d_right_view_window->current_item();
		
		if( lp_item && rp_item ) {
			if( lp_item->is_dir() && ( FALSE == lp_item->is_parent_dir() )) {
				if( rp_item->is_dir() && ( FALSE == rp_item->is_parent_dir() )) {
					DirSynch* ds = new DirSynch( this, lp_item->path(), rp_item->path() );
					if( ds ) {
						ds->exec();
						delete ds;
						ds = 0;
					}
					//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
				}
			}
		}
	}}
// end of dir_synch

//*******************************************************************
// cmp_files                                            PRIVATE slot
//*******************************************************************
void Workspace::cmp_files()
{
	bool call_kompare = FALSE;
	bool call_diff    = FALSE;
	
	if( d_left_view_window->is_ftp_dir() || d_right_view_window->is_ftp_dir() ) {
		// to do
	}
	else {
		const ViewTableItem* const lp_item = d_left_view_window->current_item();
		const ViewTableItem* const rp_item = d_right_view_window->current_item();
		const QString lpath = lp_item->path();
		const QString rpath = rp_item->path();
		
		if( lp_item && rp_item ) {
			if(( FALSE == lp_item->is_dir() ) && ( FALSE == rp_item->is_dir() )) {
				QMessageBox mb( tr(CMP_FILES_CAPTION),
					tr(CMP_FILES_MSG).arg( lpath ).arg( rpath ),
        			QMessageBox::NoIcon,
        			QMessageBox::Yes | QMessageBox::Default,
        			QMessageBox::No  | QMessageBox::Escape,
        			QMessageBox::NoButton,
        			this );
        		mb.setIconPixmap( QPixmap::fromMimeSource( CMP_FILES_ICON ) );
        		
		   	if( QMessageBox::Yes == mb.exec() ) {
		   		// niezgodnosc rozmiarow
		   		if( lp_item->size() != rp_item->size() ) {
		   			const int retval =	QMessageBox::information(	this,
		   											tr(CMP_FILES_CAPTION), tr(CMP_FILES_NEG_INFO),
		   											CMP_FILES_DIFF, CMP_FILES_KOMPARE,
		   											tr(Shared::OkBtnLabel) );
		   			     if( 1 == retval ) call_kompare = TRUE;
		   			else if( 0 == retval ) call_diff    = TRUE;
		   		}
		   		// sprawdz
		   		else {
		   			CmpFiles* cf = new CmpFiles( this, lpath, rpath );
		   			connect( this, SIGNAL( stop() ), cf, SLOT( stop() ));
		   			CmpFiles::CmpResult retval = cf->compare();
		   			disconnect( this, SIGNAL( stop() ), cf, SLOT( stop() ));
		   			delete cf;
		   			cf = 0;

						QString msg;
						switch( retval ) {
							case CmpFiles::IOError:
								msg = CMP_FILES_IOERROR;
								break;
							case CmpFiles::Break:
								msg = CMP_FILES_BREAK;
								break;
							case CmpFiles::Equal:
								msg = CMP_FILES_EQUAL;
								break;
							case CmpFiles::NotEqual:
								msg = CMP_FILES_NOT_EQUAL;
								break;
						}

						if( CmpFiles::NotEqual == retval ) {
							const int retval =	QMessageBox::information(	this,
		   												tr(CMP_FILES_CAPTION), tr(msg),
		   												CMP_FILES_DIFF, CMP_FILES_KOMPARE,
		   												tr(Shared::OkBtnLabel) );
			   			     if( 1 == retval ) call_kompare = TRUE;
			   			else if( 0 == retval ) call_diff    = TRUE;
						}
						else {
							QMessageBox::information(	this, tr(CMP_FILES_CAPTION), tr(msg), tr(Shared::OkBtnLabel) );
						}
		   		}
		   	}
		   	if( call_kompare ){
		   		const QString cmd = KOMPARE_PRG_NAME + Shared::Spc + lpath + Shared::Spc + rpath;
		   		const int retval = system( cmd.local8Bit() );
		   		if( retval ) {
		   		}
		   	}
		   	else if( call_diff ) {
		   		DiffViewer* dv = new DiffViewer( this, lpath, rpath );
		   		if( dv ) {
		   			dv->exec();
		   			delete dv;
		   			dv = 0;
		   		}
		   	}
			}
		}
	}
}
// end of cmp_files

//*******************************************************************
// date_time                                            PRIVATE slot
//*******************************************************************
void Workspace::date_time()
{
	ViewWindow* src_view =  0;
	ViewWindow* dst_view =  0;

	if( d_viewL->has_focus() ) {
		src_view =  d_left_view_window;
		dst_view =  d_right_view_window;
	}
	else {
		src_view =  d_right_view_window;
		dst_view =  d_left_view_window;
	}
	
	if( d_left_view_window->is_ftp_dir() || d_right_view_window->is_ftp_dir() ) {
		// to do
	}
	else {
		const ViewTable::SelectedItems& items = src_view->selections();
		
		if( FALSE == items.empty() ) {
			ViewTableItem* const muster = ( dst_view ) ? dst_view->current_item() : 0;
		
			DateTime* dt = new DateTime( this, src_view->selections(), muster );
			if( dt ) {
				if( QDialog::Accepted == dt->exec() ) {
					src_view->update_view( ViewWindow::STAY_SELECTION );
				}
				delete dt;
				dt = 0;
			}
		}
	}
}
// end of date_time

//*******************************************************************
// slot_tree                                            PRIVATE slot
//********************************************************************
void Workspace::slot_tree()
{
	     if( TRUE == d_viewL->has_focus() ) emit tree_left();
	else if( TRUE == d_viewR->has_focus() ) emit tree_right();
}
// end of slot tree

//*******************************************************************
// select                                               PRIVATE slot
//*******************************************************************
void Workspace::select()
{
	     if( TRUE == d_viewL->has_focus() ) emit select_left();
	else if( TRUE == d_viewR->has_focus() ) emit select_right();
}
// end of select

//*******************************************************************
// unselect                                          PRIVATE private
//*******************************************************************
void Workspace::unselect()
{
	     if( TRUE == d_viewL->has_focus() ) emit unselect_left();
	else if( TRUE == d_viewR->has_focus() ) emit unselect_right();
}
// end of unselect

//*******************************************************************
// revers                                            PRIVATE private
//*******************************************************************
void Workspace::revers()
{
	     if( TRUE == d_viewL->has_focus() ) emit revers_left();
	else if( TRUE == d_viewR->has_focus() ) emit revers_right();
}
// end of revers

//*******************************************************************
// slot_ftp_start                                       slot PRIVATE
//*******************************************************************
void Workspace::slot_ftp_start()
{
	     if( TRUE == d_viewL->has_focus() ) emit ftp_start_left();
	else if( TRUE == d_viewR->has_focus() ) emit ftp_start_right();
}
// end of slot_ftp_start

//*******************************************************************
// slot_ftp_end                                         slot PRIVATE
//*******************************************************************
void Workspace::slot_ftp_end()
{
	emit ftp_end();
}
// end of slot_ftp_end

//*******************************************************************
// slot_ftp_rejected                                    PRIVATE slot
//*******************************************************************
void Workspace::slot_ftp_rejected()
{
    d_main_window->ftp_end_handle();
}
// end of slot_ftp_end

//*******************************************************************
// slotF2
//*******************************************************************
void Workspace::slotF2()
{
	     if( TRUE == d_viewL->has_focus() ) emit F2ToLeft();
	else if( TRUE == d_viewR->has_focus() ) emit F2ToRight();
}
// end of slotF2

//*******************************************************************
// slotF3
//*******************************************************************
void Workspace::slotF3()
{
	     if( TRUE == d_viewL->has_focus() ) emit F3ToLeft();
	else if( TRUE == d_viewR->has_focus() ) emit F3ToRight();
}
// end of slotF3

//*******************************************************************
// slotF4
//*******************************************************************
void Workspace::slotF4()
{
	     if( TRUE == d_viewL->has_focus() ) emit F4ToLeft();
	else if( TRUE == d_viewR->has_focus() ) emit F4ToRight();
}
// end of slotF4

//*******************************************************************
// slotF5
//*******************************************************************
void Workspace::slotF5()
{
	     if( TRUE == d_viewL->has_focus() ) copy( LEFT_TO_RIGHT );
	else if( TRUE == d_viewR->has_focus() ) copy( RIGHT_TO_LEFT );
}
// end of slotF5

//**********************************************s*********************
// slotF6
//*******************************************************************
void Workspace::slotF6()
{
	     if( TRUE == d_viewL->has_focus() ) emit F6ToLeft();
	else if( TRUE == d_viewR->has_focus() ) emit F6ToRight();
}
// end of slotF6

//*******************************************************************
// slotF7
//*******************************************************************
void Workspace::slotF7()
{
	     if( TRUE == d_viewL->has_focus() ) emit F7ToLeft();
	else if( TRUE == d_viewR->has_focus() ) emit F7ToRight();
}
// end of slotF7

//*******************************************************************
// slotF8
//*******************************************************************
void Workspace::slotF8()
{
	     if( TRUE == d_viewL->has_focus() ) emit F8ToLeft();
	else if( TRUE == d_viewR->has_focus() ) emit F8ToRight();
}
// end of slotF8

//*******************************************************************
// slotF9                                               PRIVATE slot
//*******************************************************************
void Workspace::slotF9()
{
	     if( TRUE == d_viewL->has_focus() ) pack( LEFT_TO_RIGHT );
	else if( TRUE == d_viewR->has_focus() ) pack( RIGHT_TO_LEFT );
}
// end of slotF5

//*******************************************************************
// slot_md5_create                                      PRIVATE slot
//*******************************************************************
void Workspace::slot_md5_create()
{
	     if( d_viewL->has_focus() ) emit md5_create_left();
	else if( d_viewR->has_focus() ) emit md5_create_right();
}
// end of slot_md5_create

//*******************************************************************
// slot_md5_check                                       PRIVATE slot
//*******************************************************************
void Workspace::slot_md5_check()
{
	     if( d_viewL->has_focus() ) emit md5_check_left();
	else if( d_viewR->has_focus() ) emit md5_check_right();
}
// end of slot_md5_create

//*******************************************************************
// slot_join_files                                      PRIVATE slot
//*******************************************************************
void Workspace::slot_join_files()
{
	     if( TRUE == d_viewL->has_focus() ) join_files( LEFT_TO_RIGHT );
	else if( TRUE == d_viewR->has_focus() ) join_files( RIGHT_TO_LEFT );
}
// end of slot_join_files

//*******************************************************************
// disconnect_signals                                   PRIVATE slot
//*******************************************************************
void Workspace::disconnect_signals()
{
	disconnect();
	d_viewL->reconnect();
	d_viewR->reconnect();	
}
// end of disconnect_signals

//*******************************************************************
// slot_tab_new                                         PRIVATE slot
//*******************************************************************
void Workspace::slot_tab_new()
{
	     if( d_viewL->has_focus() ) emit tab_new_left();
	else if( d_viewR->has_focus() ) emit tab_new_right();
}
// end of slot_tab_new

//*******************************************************************
// slot_tab_remove                                      PRIVATE slot
//*******************************************************************
void Workspace::slot_tab_remove()
{
	     if( d_viewL->has_focus() ) emit tab_remove_left();
	else if( d_viewR->has_focus() ) emit tab_remove_right();
}
// end of slot_tab_remove
