/********************************************************************
 * Copyright (C) 2005, 2006 Piotr Pszczolkowski
 *-------------------------------------------------------------------
 * This file is part of BSCommander (Beesoft Commander).
 *
 * BSCommander is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * BSCommander is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with BsC; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *******************************************************************/

/*------- include files:
-------------------------------------------------------------------*/
#include "Limits.h"
#include "Shared.h"
#include "SystemCall.h"
#include <qfile.h>
#include <qdir.h>
#include <qapplication.h>
#include <iostream>
#include <qmessagebox.h>
#include <qobject.h>

#include <sys/types.h>
#include <pwd.h>

using namespace std;


/*------- local constants:
-------------------------------------------------------------------*/
const QString Limits::FilePath     = "/usr/share/bsc/limits.bsc.txt";
const QString Limits::AcceptPrefix = "accept:";
const QString Limits::BlockPrefix  = "block:";
const QString Limits::HomePrefix   = "~/";
const QString Limits::CommentMark  = "#";
const QString Limits::MsgCaption   = QT_TR_NOOP( "Change directory" );
const QString Limits::MsgInfo      = QT_TR_NOOP( "Directory %1 is not accessible.\nThis directory is blocked,\ncontact please with your administrator" );

/*------- local static variables:
-------------------------------------------------------------------*/
Limits* Limits::d_instance = 0;


//*******************************************************************
// Limits                                                CONSTRUCTOR
//*******************************************************************
Limits::Limits()
: QObject()
, d_apply( FALSE )
{
	refresh();
}
// end of Limits

//*******************************************************************
// instance                                                   PUBLIC
//*******************************************************************
Limits* Limits::instance()
{
	if( 0 == d_instance ) {
		d_instance = new Limits;
	}
	return d_instance;
}
// end of instance

//*******************************************************************
// refresh                                                    PUBLIC
//*******************************************************************
void Limits::refresh()
{
	d_apply = FALSE;
	d_accepted.clear();
	d_blocked.clear();

	struct passwd* info = getpwuid( getuid() );
	const QString user_name( info->pw_name );

//	if( FALSE == Shared::is_user_root() ) {
	if( "root" != user_name ) {
		if( QFile::exists( FilePath ) ) {
			QFile file( FilePath );
	
			if( file.open( IO_ReadOnly ) ) {
				QString txt;
				while( FALSE == file.atEnd() ) {
					if( file.readLine( txt, 1024 ) != -1 ) {
						if( Shared::remove_white_chars( txt ) ) {
							if( FALSE == txt.isEmpty() ) {
								if( FALSE == txt.startsWith( CommentMark ) ) {
									if( txt.startsWith( AcceptPrefix ) ) {
										check_home_dir( txt.remove( AcceptPrefix ) );
										d_accepted.push_back( txt );
									}
									else if( txt.startsWith( BlockPrefix ) ) {
										check_home_dir( txt.remove( BlockPrefix ) );
										d_blocked.push_back( txt );
									}
								}
							}
						}
					}
				}
				file.close();
				d_apply = ( ( FALSE == d_accepted.empty() ) || ( FALSE == d_blocked.empty() ));
			}
		}
		vector<QString>( d_accepted ).swap( d_accepted );
		vector<QString>( d_blocked ).swap( d_blocked );
	}
}
// end of refresh

//*******************************************************************
// accepted                                                   PUBLIC
//*******************************************************************
bool Limits::accepted( const QString& in_path )
{
	bool result = TRUE;

	if( d_apply ) {
		if( FALSE == d_blocked.empty() ) {
			vector<QString>::const_iterator it = d_blocked.begin();
			while( it != d_blocked.end() ) {
				if( in_path.startsWith( *it ) ) {
					result = FALSE;
					break;
				}
				++it;
			}
		}

		if( result ) {
			if( FALSE == d_accepted.empty() ) {
				result = FALSE;
				vector<QString>::const_iterator it = d_accepted.begin();
				while( it != d_accepted.end() ) {
					if( in_path.startsWith( *it ) ) {
						result = TRUE;
						break;
					}
					++it;
				}
			}
		}
	}
	
	if( FALSE == result ) {
		QMessageBox::information(	0,
											tr(MsgCaption),
											tr(MsgInfo).arg( in_path),
											QMessageBox::Ok );
	}

	return result;
}
// end of accepted

//*******************************************************************
// check_home_dir                                            PRIVATE
//-------------------------------------------------------------------
// Zamiana symbolicznego oznaczenia katalogu domowego (~/) na pelna
// wersje (np: /home/piotr).
//*******************************************************************
void Limits::check_home_dir( QString& inout_dir )
{
	if( inout_dir.startsWith( HomePrefix ) ) {
		inout_dir.remove( HomePrefix );
		if( inout_dir.isEmpty() ) inout_dir = QDir::homeDirPath();
		else inout_dir.insert( 0, QDir::homeDirPath() + "/" );
	}
}
// end of check_home_dir
