/********************************************************************
 * Copyright (C) 2005 Piotr Pszczolkowski
 *-------------------------------------------------------------------
 * This file is part of BsC (Beesoft Commander).
 *
 * BsC is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * BsC is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with BsC; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *******************************************************************/

/*------- include files:
-------------------------------------------------------------------*/
#include "FtpCopier.h"
#include "Shared.h"
#include "InfoField.h"
#include "Config.h"
#include <qlayout.h>
#include <qpushbutton.h>
#include <qlabel.h>
#include <qprogressbar.h>
#include <qgroupbox.h>
#include <qmessagebox.h>
#include <qfileinfo.h>
using namespace std;

/*------- local constants:
-------------------------------------------------------------------*/

const QString FtpCopier::CantCopyDir      = tr( "\"%1\" is a directory.\nCan't copy a directory." );


//*******************************************************************
// FtpCopier                                             CONSTRUCTOR
//*******************************************************************
FtpCopier::FtpCopier(	QWidget* const in_parent,
								const FtpServerInfo& in_fsi, 
								const ViewTable::SelectedItems& in_items,
								const QString& in_dir,
								const bool in_src_is_ftp )
: FTPCopyDialog( in_parent )
, d_fsi        ( in_fsi )
, d_dir        ( in_dir )
, d_items      ( in_items )
, d_src_is_ftp ( in_src_is_ftp )
, d_fm         ( font() )
, d_ftp        ( this )
, d_idx        ( 0 )
, d_ftp_cmd    ( FTP_CMD_NONE )
{
}
// end of FtpCopier


//*******************************************************************
// polish                                          PRIVATE inherited
//*******************************************************************
/*
void FtpCopier::polish()
{
	Shared::polish_width( this, 40 );
	state_changed( QFtp::Unconnected );
}
*/
// end of polish


//*******************************************************************
// get_lfs_info                                              PRIVATE
//*******************************************************************
bool FtpCopier::get_lfs_info( const QString& in_name, int& in_size, bool& in_isdir )
{
	int retval = FALSE;
	
	QFileInfo fi( in_name );
	if( fi.exists() ) {
		in_size  = fi.size();
		in_isdir = fi.isDir();
		retval   = TRUE;
	}
	return retval;
}
// end of get_lfs_size

//*******************************************************************
// next_file                                                 PRIVATE
//*******************************************************************
bool FtpCopier::next_file( QString& out_src, QString& out_dst )
{
	bool retval = FALSE;
	
    reset_progress();
    
	if( FALSE == d_items.empty() ) {
		if( d_idx < d_items.size() ) {
			const ViewTableItem* const item = d_items[d_idx];
					
			// *** FTP -> LFS ***
			if( d_src_is_ftp ) {
				out_src = item->name();
				if( item->is_dir() ) {
//					QMessageBox::information( this, tr(FtpCopyCaption), tr(CantCopyDir).arg( out_src ) );
				}
				else {
					out_dst = Shared::make_path( d_dir, item->name() );
					set_total_progress( item->size() );
					display_info( Shared::make_path( d_fsi.dir(), out_src ), out_dst );
					retval = TRUE;
				}
			}
			// *** LFS -> FTP
			else {
				out_src = item->path();
				if( item->is_dir() ) {
//					QMessageBox::information( this, tr(FtpCopyCaption), tr(CantCopyDir).arg( out_src ) );
				}
				else {
					out_dst = item->name();
					set_total_progress( item->size() );
					display_info( out_src, Shared::make_path( d_fsi.dir(), out_dst ));
					retval = TRUE;
				}
			}
		}
		else {
			d_ftp_cmd = FTP_CMD_CLOSE;
			d_ftp.close();
		}
	}
	
	return retval;
}
// end of display_next

//*******************************************************************
// slot_left_btn                                        PRIVATE slot
//*******************************************************************
/*
void FtpCopier::slot_left_btn()
{
//	d_left_btn->hide();
//	d_right_btn->setText( Shared::BreakBtnLabel );
//	ftp_connect();
}
*/
// end of slot_left_btn

//*******************************************************************
// slot_right_btn                                       PRIVATE slot
//*******************************************************************
/*
void FtpCopier::slot_right_btn()
{
	( QFtp::Unconnected == d_ftp.state() ) ? reject() : ftp_abort();
}
*/
// end of slot_right_btn

//*******************************************************************
// ftp_connect                                               PRIVATE
//*******************************************************************
void FtpCopier::ftp_connect()
{
	d_ftp_cmd = FTP_CMD_LOGIN;	
	d_ftp.connectToHost( d_fsi.addr() );
	d_ftp.login( d_fsi.user(), d_fsi.pass() );
}
// end of ftp_connect

//*******************************************************************
// ftp_cd                                                    PRIVATE
//*******************************************************************
void FtpCopier::ftp_cd( const QString& in_path )
{
	d_ftp_cmd = FTP_CMD_CD;
	d_ftp.cd( in_path );
}
// end of ftp_cd

//*******************************************************************
// ftp_get                                                   PRIVATE
//*******************************************************************
void FtpCopier::ftp_get()
{
	d_ftp_cmd = FTP_CMD_GET;
	QString src, dst;	

	if( next_file( src, dst ) ) {
		d_current_file.setName( dst );
		if( FALSE == d_current_file.open( IO_WriteOnly ) ) {
            write_open_error( dst );
			done( FALSE );
		}
		else {
			d_ftp.get( src, &d_current_file );
		}
	}
	else {
		done( FALSE );
	}
}
// end of ftp_get

//*******************************************************************
// ftp_put                                                   PRIVATE
//*******************************************************************
void FtpCopier::ftp_put()
{
	d_ftp_cmd = FTP_CMD_PUT;
	QString src, dst;
	
	if( next_file( src, dst ) ) {
		d_current_file.setName( src );
		if( FALSE == d_current_file.open( IO_ReadOnly ) ) {
            read_open_error( src );
			done( FALSE );
		}
		else {
			d_ftp.put( &d_current_file, dst );
		}
	}
	else {
		done( FALSE );
	}
}
// end of ftp_put

//*******************************************************************
// ftp_abort                                                 PRIVATE
//*******************************************************************
void FtpCopier::ftp_abort()
{
	d_ftp_cmd = FTP_CMD_ABORT;
	d_ftp.abort();
}
// end of ftp_abort

//*******************************************************************
// ftp_close                                                PRIVATE
//*******************************************************************
void FtpCopier::ftp_close()
{
	d_ftp_cmd = FTP_CMD_CLOSE;
	d_ftp.close();
}
// end of ftp_close

//*******************************************************************
// state_changed                                        PRIVATE slot
//*******************************************************************
void FtpCopier::state_changed( int in_state )
{
	switch( in_state ) {
		case QFtp::Unconnected:
			set_status( tr(Shared::FTP_Unconnected) );
			break;
		case QFtp::HostLookup:
			set_status( tr(Shared::FTP_HostLookup) );
			break;
		case QFtp::Connecting:
			set_status( tr(Shared::FTP_Connecting) );
			break;
		case QFtp::Connected:
			set_status( tr(Shared::FTP_Connected) );
			break;
		case QFtp::LoggedIn:
			set_status( tr(Shared::FTP_LoggedIn) );
			break;
		case QFtp::Closing:
			set_status( tr(Shared::FTP_Closing) );
			break;
		default:
			set_status( tr(Shared::FTP_Unknown) );
	}
}
// end of state_changed

//*******************************************************************
// transfer_progress                                    PRIVATE slot
//*******************************************************************
void FtpCopier::transfer_progress( int in_done, int )
{
    set_progress( in_done );
}
// end of transfer_progress

//*******************************************************************
// done                                                 PRIVATE slot
//*******************************************************************
void FtpCopier::done( bool in_error )
{
	if( in_error ) {
		if( FTP_CMD_ABORT == d_ftp_cmd ) {
			ftp_close();
		}
		else {
			if( d_ftp_cmd != FTP_CMD_CLOSE ) {
				//QMessageBox::critical( this, tr(FtpCopyCaption), d_ftp.errorString() );
				d_ftp.close();
			}
			reject();
		}
	}
	else {
		switch( d_ftp_cmd ) {
			case FTP_CMD_LOGIN:
				ftp_cd( d_fsi.dir() );
				break;
			case FTP_CMD_ABORT:
				ftp_close();
				break;
			case FTP_CMD_CLOSE:
				d_current_file.close();
				reject();
				break;
			case FTP_CMD_CD:
				d_src_is_ftp ? ftp_get() : ftp_put();
				break;
			case FTP_CMD_PUT:
				d_current_file.close();
				++d_idx;
				ftp_put();
				break;
			case FTP_CMD_GET:
				d_current_file.close();
				++d_idx;
				ftp_get();
				break;
			default:
				d_ftp_cmd = FTP_CMD_NONE;
		}
	}
}
// end of done
