/******************************************************************
 * Copyright (C) 2005, 2006 Piotr Pszczolkowski
 *-------------------------------------------------------------------
 * This file is part of BSCommander (Beesoft Commander).
 *
 * BSCommander is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * BSCommander is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with BsC; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *******************************************************************/

/*------- include files:
-------------------------------------------------------------------*/
#include "FTPDeleter.h"
#include "PleaseWait.h"
#include "Events.h"
#include "Shared.h"
#include <qftp.h>
#include <qapplication.h>
#include <qmessagebox.h>

/*------- local constants:
-------------------------------------------------------------------*/
const char* const FTPDeleter::DeleteCaption = QT_TR_NOOP( "Deleting file(s)/directory(s)" );


//*******************************************************************
// FTPDeleter                                            CONSTRUCTOR
//*******************************************************************
FTPDeleter::FTPDeleter( const FtpServerInfo& in_fsi,
                        const ViewTable::SelectedItems& in_items,
                        PleaseWait* const in_please_wait )
: QObject       ()
, d_ftp         ( new QFtp( 0 ) )
, d_fsi         ( in_fsi )
, d_please_wait ( in_please_wait )
, d_dir_name    ( "" )
, d_path        ( "" )
, d_cmd         ( CMD_NONE )
, d_pwd_cmd_id  ( -1 )
, d_first_cd    ( TRUE )
{
    ViewTable::SelectedItems::const_iterator it = in_items.begin();
    while( it != in_items.end() ) {
        if( (*it)->is_dir() ) d_dirs_on_dir.push ( (*it)->path() );
        else                  d_files_on_dir.push( (*it)->path() );
        ++it;
    }
    connect( d_ftp, SIGNAL( done( bool )), this, SLOT( done( bool )));
	connect( d_ftp, SIGNAL( listInfo( const QUrlInfo& )), this, SLOT( item( const QUrlInfo& )));
    connect( d_ftp, SIGNAL( commandFinished( int, bool )), this, SLOT( raw_cmd_finish( int, bool )));
    connect( d_ftp, SIGNAL( rawCommandReply( int, const QString& )), this, SLOT( raw_cmd_reply( int, const QString& )));
}
// end of FTPDeleter

//*******************************************************************
// ~FTPDeleter                                            DESTRUCTOR
//*******************************************************************
FTPDeleter::~FTPDeleter()
{
    d_ftp->abort();
    delete d_ftp;
}
// end of ~FTPDeleter

//*******************************************************************
// run                                                        PUBLIC
//-------------------------------------------------------------------
// Polaczenie sie z serwerem i zalogowanie.
//*******************************************************************
void FTPDeleter::run()
{
    d_cmd = CMD_LOGIN;
    d_ftp->connectToHost( d_fsi.addr(), d_fsi.port() );
    d_ftp->login( d_fsi.user(), d_fsi.pass() );
}
// end of run


//###################################################################
//#                                                                 #
//#                       P R I V A T E                             #
//#                                                                 #
//###################################################################


//*******************************************************************
// doit                                                       PUBLIC
//-------------------------------------------------------------------
// Funkcja zarzadzajaca usuwaniem plikow i katalogow.
//*******************************************************************
void FTPDeleter::doit()
{
    if( Shared::d_break ) {
        close();
    }
    else {
        // Najpierw usuwamy konsekwetnie wszystkie pliki
        // az do oproznienia stosu.
        if( !d_files_on_dir.empty() ) {
            remove();
        }
        // Jesli wszystkie pliki aktualnego katalogu sa juz usuniete
        // zabieramy sie do rekursywnego usuwania katalogo.
        else if( !d_dirs_on_dir.empty() ) {
            // Jezeli aktualny katalog znajduje na szczycie stosu
            // oznacza to, ze usunelismy cala jego zawartosc.
            // Pozostaje go tylko usunac.
            if( d_dir_name == d_dirs_on_dir.top() ) {
                cd_up();
            }
            else {
                // Jeszcze do usuniecia sa podkatalogi.
                const QString fname = d_dirs_on_dir.top();
                if( fname.isEmpty() || !Shared::is_regular_file( fname )) {
                    d_dirs_on_dir.pop();
                    doit();
                }
                else {
                    d_dirs.push( d_dir_name );
                    d_dir_name = fname;
                    cd( d_dir_name );
                }
            }
        }
        // Wszystko usuniete.
        else {
            close();
        }
    }
}
// end of doit

//*******************************************************************
// display                                                   PRIVATE
//-------------------------------------------------------------------
// Wyslanie do wyswietlenia informacji o usuwanym pliku.
//*******************************************************************
void FTPDeleter::display( const QString& in_fname )
{
    if( d_please_wait ) {
        BscEvent* event = new BscEvent( DeleteFileEvent );
        if( event ) {
            QString fpath = d_path;
            if( fpath.endsWith( "/" ) ) fpath += in_fname;
            else fpath += ( "/" + in_fname );
            
            event->m_message = Shared::tr(Shared::DeletingFileMsg).arg( fpath );
            QApplication::postEvent( d_please_wait, event );
            Shared::idle();
        }
    }
}
// end of display

//*******************************************************************
// list                                                      PRIVATE
//*******************************************************************
void FTPDeleter::list()
{
    d_cmd = CMD_LIST;
    d_ftp->list();
}
// end of list

//*******************************************************************
// rmdir                                                     PRIVATE
//*******************************************************************
void FTPDeleter::rmdir()
{
    d_cmd = CMD_RMDIR;
    d_ftp->rmdir( d_dir_name );
}
// end of rmdir

//*******************************************************************
// remove                                                    PRIVATE
//*******************************************************************
void FTPDeleter::remove()
{
    if( Shared::d_break ) {
        close();
    }
    else {
        const QString fname = d_files_on_dir.top();
        d_files_on_dir.pop();
        display( fname );

        d_cmd = CMD_REMOVE;
        d_ftp->remove( fname );
    }
}
// end of remove

//*******************************************************************
// cd_up                                                     PRIVATE
//*******************************************************************
void FTPDeleter::cd_up()
{
    d_cmd = CMD_CDUP;
    d_ftp->cd( ".." );
}
// end of cd_up

//*******************************************************************
// cd                                                        PRIVATE
//*******************************************************************
void FTPDeleter::cd( const QString& in_path )
{
    d_cmd = CMD_CD;
    d_ftp->cd( in_path );
}
// end of cd

//*******************************************************************
// pwd                                                   PRIVATE
//*******************************************************************
void FTPDeleter::pwd()
{
    d_cmd = CMD_NONE;
    d_pwd_cmd_id = d_ftp->rawCommand( "PWD" );
}
// end of ftp_pwd
//*******************************************************************
// close                                                     PRIVATE
//*******************************************************************
void FTPDeleter::close()
{
    d_cmd = CMD_CLOSE;
    d_ftp->close();
}
// end of close

//###################################################################
//#                                                                 #
//#                         S L O T S                               #
//#                                                                 #
//###################################################################

//*******************************************************************
// done                                                 PRIVATE slot
//*******************************************************************
void FTPDeleter::done( bool in_error )
{
    if( in_error ) {
        if( d_cmd != CMD_CLOSE ) {
            QMessageBox::critical( 0, tr(DeleteCaption), d_ftp->errorString() );
            if( d_cmd != CMD_CLOSE ) close();
            else emit done();
        }
    }
	else {
        switch( d_cmd ) {
            case CMD_LOGIN:
                cd( d_fsi.dir()  );
                break;
            case CMD_CD:
                pwd();
                break;
            case CMD_REMOVE:
                d_cmd = CMD_NONE;
                doit();
                break;
            case CMD_LIST:
                d_cmd = CMD_NONE;
                doit();
                break;
            case CMD_CDUP:
                rmdir();
                break;
            case CMD_RMDIR:
                d_dirs_on_dir.pop();
                if( d_dirs.empty() ) {
                    d_dir_name = "";
                }
                else {
                    d_dir_name = d_dirs.top();
                    d_dirs.pop();
                }
                d_cmd = CMD_NONE;
                doit();
                break;
            case CMD_CLOSE:
                d_cmd = CMD_NONE;
                emit done();
        }
    }
}
// end of done

//*******************************************************************
// item                                                 PRIVATE slot
//-------------------------------------------------------------------
// Asynchroniczny odbior informacji o plikach na serwerze FTP.
//*******************************************************************
void FTPDeleter::item( const QUrlInfo& in_item )
{
    if( CMD_LIST == d_cmd ) {
        if( !in_item.isSymLink() ) {
                 if( in_item.isDir()  ) d_dirs_on_dir.push( in_item.name() );
            else if( in_item.isFile() ) d_files_on_dir.push( in_item.name() );
        }
    }
}
// end of item

//*******************************************************************
// raw_cmd_finish                                       PRIVATE slot
//*******************************************************************
void FTPDeleter::raw_cmd_finish( int, bool in_error )
{
    if( in_error ) close();
}
// end of raw_list_finish

//*******************************************************************
// raw_cmd_reply                                        PRIVATE slot
//-------------------------------------------------------------------
// Jako ospowiedz otrzymujemy tekst np. "/test/a" is current directory.
// Jak widac nazwa aktualnego katalogu jest zawarta w cudzyslowiu.
//*******************************************************************
void FTPDeleter::raw_cmd_reply( int, const QString& in_detail )
{
    if( d_pwd_cmd_id != -1 ) {
        d_path = "";
        
        const char DELIMITER = '\"';    // cudzyslow
        const int idx_start = in_detail.find( DELIMITER );
        if( idx_start != -1 ) {
            const int idx_end = in_detail.find( DELIMITER, idx_start + 1 );
            if( idx_end != -1 ) {
                d_path = in_detail.mid( idx_start + 1, idx_end - idx_start - 1 );
			}
		}
		
		d_pwd_cmd_id = -1;
        if( d_first_cd ) {
            d_cmd = CMD_NONE;
            d_first_cd = FALSE;
            doit();
        }
        else {
            list();
        }
	}
}
// end of ftp_cmd_reply
