/******************************************************************
 * Copyright (C) 2005, 2006 Piotr Pszczolkowski
 *-------------------------------------------------------------------
 * This file is part of BSCommander (Beesoft Commander).
 *
 * BSCommander is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * BSCommander is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with BsC; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *******************************************************************/

/*------- include files:
-------------------------------------------------------------------*/
#include "FTPCopyFrom.h"
#include <qmessagebox.h>
#include <qpushbutton.h>

//*******************************************************************
// FTPCopyFrom                                           CONSTRUCTOR
//*******************************************************************
FTPCopyFrom::FTPCopyFrom(   QWidget* const in_parent,
                            const FtpServerInfo& in_fsi,
                            const ViewTable::SelectedItems& in_items,
                            const QString& in_dst_dir )
: CopyDialog   ( in_parent, TRUE )
, d_fsi        ( in_fsi )
, d_path       ( "" )
, d_dir_name   ( "" )
, d_cmd        ( CMD_NONE )
, d_pwd_cmd_id ( -1 )
, d_first_cd   ( TRUE ) 
, d_running    ( FALSE )
, d_break      ( FALSE )
{
    d_dir.cd( in_dst_dir );

    ViewTable::SelectedItems::const_iterator it = in_items.begin();
    while( it != in_items.end() ) {
        if( (*it)->is_dir() ) d_dirs_on_dir.push ( (*it)->path() );
        else                  d_files_on_dir.push( (*it)->path() );
        ++it;
    }

    connect( d_left_btn, SIGNAL( clicked() ), this, SLOT( run() ));
    connect( d_right_btn, SIGNAL( clicked() ), this, SLOT( cancel() ));
    connect( &d_ftp, SIGNAL( dataTransferProgress( int, int)), this, SLOT( transfer_progress( int, int )));
    connect( &d_ftp, SIGNAL( done( bool )), this, SLOT( done( bool )));
	connect( &d_ftp, SIGNAL( listInfo( const QUrlInfo& )), this, SLOT( item( const QUrlInfo& )));
    connect( &d_ftp, SIGNAL( commandFinished( int, bool )), this, SLOT( raw_cmd_finish( int, bool )));
    connect( &d_ftp, SIGNAL( rawCommandReply( int, const QString& )), this, SLOT( raw_cmd_reply( int, const QString& )));
}
// end of FTPCopyFrom

//*******************************************************************
// ~FTPCopyFrom                                           DESTRUCTOR
//*******************************************************************
FTPCopyFrom::~FTPCopyFrom()
{
    d_ftp.abort();
}
// end of ~FTPCopyFrom


//###################################################################
//#                                                                 #
//#                      P R I V A T E                              #
//#                                                                 #
//###################################################################


//*******************************************************************
// doit                                                      PRIVATE
//*******************************************************************
void FTPCopyFrom::doit()
{
    Shared::idle();

    if( d_break ) {
        close();
    }
    else {
        // Najpierw kopiujemy konsekwetnie wszystkie pliki
        // az do oproznienia stosu.
        if( !d_files_on_dir.empty() ) {
            get();
        }
        // Jesli wszystkie pliki aktualnego katalogu sa juz przekopiowane
        // zabieramy sie do rekursywnego przechodzenia przez katalogogi.
        else if( !d_dirs_on_dir.empty() ) {
            // Jezeli aktualny katalog znajduje na szczycie stosu
            // oznacza to, ze przekopiowalismy cala jego zawartosc.
            if( d_dir_name == d_dirs_on_dir.top() ) {
                cd_up();
            }
            else {
                // Sa jeszcze do przekopiowania podkatalogi.
                const QString fname = d_dirs_on_dir.top();
                if( fname.isEmpty() || !Shared::is_regular_file( fname )) {
                    d_dirs_on_dir.pop();
                    doit();
                }
                else {
                    d_dirs.push( d_dir_name );
                    d_dir_name = fname;
                    cd( d_dir_name );
                }
            }
        }
        // Wszystko usuniete.
        else {
            close();
        }
    }
}
// end of doit

//*******************************************************************
// get                                                       PRIVATE
//-------------------------------------------------------------------
// Inicjacja kopiowania pliku z serwera.
// Jesli nie mozna utworzyc docelowego pliku wyswietlany jest
// stosowny komunikat i kontynuawane jest kopiowanie nastepnych
// plikow.
//*******************************************************************
void FTPCopyFrom::get()
{
    Shared::idle();

    if( d_break ) {
        close();
    }
    else {
        const QString fname = d_files_on_dir.top();
        d_files_on_dir.pop();

        QString lfs_fpath = d_dir.absPath();
        if( !lfs_fpath.endsWith( "/" ) ) lfs_fpath += "/";
        lfs_fpath += fname;
        d_file.setName( lfs_fpath );

        // Jesli na dysku uda sie otworzyc plik
        // wysylamy do serwera polecenie kopiowania.
        if( d_file.open( IO_WriteOnly ) ) {
            QString ftp_fpath = d_path;
            if( !ftp_fpath.endsWith( "/" ) ) ftp_fpath += "/";
            ftp_fpath += fname;
            display_info(  ftp_fpath, lfs_fpath );
        
            d_cmd = CMD_GET;
            d_ftp.get( fname, &d_file );
        }
        // W przeciwnym przypadku wyswietlamy informacje o bledzie
        // i kontynuujemy kopiowanie pozostalych plikow.
        // UWAGA: informacje o tym pliku zostala juz zdjeta ze stosu.
        else {
            write_open_error( lfs_fpath );
            doit();
        }
    }
}
// end of get

//*******************************************************************
// list                                                      PRIVATE
//*******************************************************************
void FTPCopyFrom::list()
{
    d_cmd = CMD_LIST;
    d_ftp.list();
}
// end of list

//*******************************************************************
// cd_up                                                     PRIVATE
//*******************************************************************
void FTPCopyFrom::cd_up()
{
    d_dir.cdUp();
    d_cmd = CMD_CDUP;
    d_ftp.cd( ".." );
}
// end of cd_up

//*******************************************************************
// cd                                                        PRIVATE
//*******************************************************************
void FTPCopyFrom::cd( const QString& in_dir_name )
{
    bool is_cd_ok = TRUE;
    if( !d_first_cd ) {
        d_dir.mkdir( in_dir_name, FALSE );
        is_cd_ok = d_dir.cd( in_dir_name, FALSE );
        if( is_cd_ok ) {
            is_cd_ok = Shared::is_dir_ok( d_dir.absPath() );
        }
    }

    d_cmd = ( is_cd_ok ) ? CMD_CD : CMD_TMPCD;
    d_ftp.cd( in_dir_name );
}
// end of cd

//*******************************************************************
// pwd                                                   PRIVATE
//*******************************************************************
void FTPCopyFrom::pwd()
{
    d_cmd = CMD_NONE;
    d_pwd_cmd_id = d_ftp.rawCommand( "PWD" );
}
// end of ftp_pwd

//*******************************************************************
// close                                                     PRIVATE
//*******************************************************************
void FTPCopyFrom::close()
{
    d_cmd = CMD_CLOSE;
    d_ftp.close();
}
// end of close


//###################################################################
//#                                                                 #
//#                          S L O T S                              #
//#                                                                 #
//###################################################################


//*******************************************************************
// run                                                  PRIVATE slot
//-------------------------------------------------------------------
// Funkcja jest uruchamiana po nacisnieciu lewego przycisku
// w dialogu 'FTPCopyDialog'.
//*******************************************************************
void FTPCopyFrom::run()
{
    d_cmd = CMD_LOGIN;
    d_ftp.connectToHost( d_fsi.addr(), d_fsi.port() );
    d_ftp.login( d_fsi.user(), d_fsi.pass() );
}
// end of run

//*******************************************************************
// cancel                                               PRIVATE slot
//*******************************************************************
void FTPCopyFrom::cancel()
{
    if( d_running ) {
        d_break = TRUE;
    }
    else {
        accept();
        emit done();
    }
}
// end of run

//*******************************************************************
// transfer_progress                                    PRIVATE slot
//*******************************************************************
void FTPCopyFrom::transfer_progress( const int in_done, const int in_total )
{
    set_progress( in_done, in_total );
}
// end of transfer_progress

//*******************************************************************
// done                                                 PRIVATE slot
//*******************************************************************
void FTPCopyFrom::done( bool in_error )
{
    if( in_error ) {
        if( d_cmd != CMD_CLOSE ) {
            QMessageBox::critical( this, tr( d_caption ), d_ftp.errorString() );
            close();
        }
        else {
            reject();
            emit done();
        }
    }
	else {
        switch( d_cmd ) {
            case CMD_LOGIN:
                d_running = TRUE;
                runned();
                cd( d_fsi.dir()  );
                break;
            case CMD_CD:
                pwd();
                break;
            case CMD_TMPCD:
                cd_up();
                break;
            case CMD_LIST:
                d_cmd = CMD_NONE;
                doit();
                break;
            case CMD_CDUP:
                d_dirs_on_dir.pop();
                if( d_dirs.empty() ) {
                    d_dir_name = "";
                }
                else {
                    d_dir_name = d_dirs.top();
                    d_dirs.pop();
                }
                d_cmd = CMD_NONE;
                doit();
                break;
            case CMD_GET:
                d_file.close();
                d_cmd = CMD_NONE;
                doit();
                break;
            case CMD_CLOSE:
                d_cmd = CMD_NONE;
                accept();
                emit done();
        }
    }
}
// end of done

//*******************************************************************
// item                                                 PRIVATE slot
//-------------------------------------------------------------------
// Asynchroniczny odbior informacji o plikach na serwerze FTP.
//*******************************************************************
void FTPCopyFrom::item( const QUrlInfo& in_item )
{
    if( CMD_LIST == d_cmd ) {
             if( in_item.isDir()  ) d_dirs_on_dir.push( in_item.name() );
        else if( in_item.isFile() ) d_files_on_dir.push( in_item.name() );
    }
}
// end of item

//*******************************************************************
// raw_cmd_finish                                       PRIVATE slot
//*******************************************************************
void FTPCopyFrom::raw_cmd_finish( int, bool in_error )
{
    if( in_error ) {
        QMessageBox::critical( this, tr( d_caption ), d_ftp.errorString() );
        close();
    }
}
// end of raw_list_finish

//*******************************************************************
// raw_cmd_reply                                        PRIVATE slot
//-------------------------------------------------------------------
// Jako ospowiedz otrzymujemy tekst np. "/test/a" is current directory.
// Jak widac nazwa aktualnego katalogu jest zawarta w cudzyslowiu.
//*******************************************************************
void FTPCopyFrom::raw_cmd_reply( int, const QString& in_detail )
{
    if( d_pwd_cmd_id != -1 ) {
        d_path = "";
        
        const QChar DELIMITER = '\"';    // cudzyslow
        const int idx_start = in_detail.find( DELIMITER );
        if( idx_start != -1 ) {
            const int idx_end = in_detail.find( DELIMITER, idx_start + 1 );
            if( idx_end != -1 ) {
                d_path = in_detail.mid( idx_start + 1, idx_end - idx_start - 1 );
			}
		}

		d_pwd_cmd_id = -1;
        if( d_first_cd ) {
            d_cmd = CMD_NONE;
            d_first_cd = FALSE;
            doit();
        }
        else {
            list();
        }
	}
}
// end of ftp_cmd_reply
