
/*
 *  Diverse Bristol midi routines.
 *  Copyright (c) by Nick Copeland <nick.copeland@ntlworld.com> 1996,2002
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/*
 * This code should open the midi device and read data from it.
 *
 * All requests will be dispatched to the relevant handlers, with support for
 * OSS (/dev/midi*), ALSA (hw:0,0) and as of release 0.8 also the ALSA sequencer
 * (with client.port config).
 *
 * All events will be converted into a bristol native message format for
 * independency from the driver.
 */

//#define DEBUG

#include <unistd.h>

#include "bristolmessages.h"
#include "bristol.h"
#include "bristolmidi.h"

bristolMidiMain bmidi;

/*
 * This is the generic code. Calls the desired (configured) drivers.
 */
bristolMidiOpen(char *dev, int flags, int chan, int msgs, void *callback,
void *param)
{
	int handle, devnum;

#ifdef DEBUG
	printf("bristolMidiOpen(%s, %x)\n", dev, flags);
#endif

	initMidiLib();

	/*
	 * The value of -1 indicates all channels, otherwise we are only interested
	 * in messages on the given channel.
	 */
	if ((chan < -1) || (chan >= 16))
	{
		/*
		 * If this is a TCP connection then we can specify a channel greater
		 * than 1024 - it is interpreted as a TCP port, all channels.
		 */
		if ((chan < 1024)
			|| ((flags & BRISTOL_CONNMASK & BRISTOL_CONN_TCP) == 0))
			return(BRISTOL_MIDI_CHANNEL);
	}

#ifdef DEBUG
	printf("bristolMidiOpen(%s, %i, %i, %i)\n", dev, flags, chan, msgs);
#endif

	/*
	 * Make sure we have a handle available
	 */
	if ((handle = bristolMidiFindFreeHandle()) < 0)
		return(handle);

	bmidi.handle[handle].handle = handle; /* That looks kind of wierd! */
	bmidi.handle[handle].state = BRISTOL_MIDI_OK;
	bmidi.handle[handle].channel = chan;
	bmidi.handle[handle].dev = -1;
	bmidi.handle[handle].flags = BRISTOL_MIDI_OK;
	bmidi.handle[handle].messagemask = msgs;

	/*
	 * We should then check to see if this dev is open, and if so link
	 * another handle to it. We should support multiple destinations of the
	 * same midi information.
	 */
	if (((flags & BRISTOL_CONN_FORCE) == 0)
		&& (devnum = bristolMidiFindDev(dev)) >= 0)
	{
		/*
		 * We have the device open, configure the new handle, and return it.
		 */
		bmidi.dev[devnum].handleCount++;
		bmidi.handle[handle].dev = devnum;
		bmidi.handle[handle].callback = callback;
		bmidi.handle[handle].param = param;
		bmidi.handle[handle].flags = bmidi.dev[devnum].flags;

		printf("reusing connection %x\n", bmidi.dev[devnum].flags);

		return(handle);
	}

	/*
	 * Now try and find a device that is free.
	 */
	if ((devnum = bristolMidiFindDev(NULL)) < 0)
		return(devnum);

	switch (flags & BRISTOL_CONNMASK) {
		case BRISTOL_CONN_TCP:
			if (handle != bristolMidiTCPOpen(dev, flags,
				chan, msgs, callback, param, devnum, handle))
				return(BRISTOL_MIDI_DRIVER);
			bmidi.handle[handle].channel = -1;
			break;
		case BRISTOL_CONN_MIDI:
			if (handle != bristolMidiALSAOpen(dev, flags,
					chan, msgs, callback, param, devnum, handle))
				return(BRISTOL_MIDI_DRIVER);
			break;
		case BRISTOL_CONN_OSSMIDI:
			if (handle != bristolMidiOSSOpen(dev, flags,
				chan, msgs, callback, param, devnum, handle))
				return(BRISTOL_MIDI_DRIVER);
			break;
		case BRISTOL_CONN_SEQ:
			if (handle != bristolMidiSeqOpen(dev, flags,
				chan, msgs, callback, param, devnum, handle))
				return(BRISTOL_MIDI_DRIVER);
			break;
		case BRISTOL_CONN_UNIX:
		default:
			printf("conn type %x not supported\n", flags & BRISTOL_CONNMASK);
			return(BRISTOL_MIDI_DRIVER);
			break;
	}

	sprintf(bmidi.dev[devnum].name, "%s\0", dev);
	bmidi.dev[devnum].state = BRISTOL_MIDI_OK;
	bmidi.dev[devnum].handleCount = 1;
	bmidi.dev[devnum].bufindex = 0;
	bmidi.dev[devnum].bufcount = 0;
	bmidi.handle[handle].dev = devnum;
	bmidi.handle[handle].callback = callback;
	bmidi.handle[handle].param = param;

//printf("returning handle %i, flags %x\n", handle, bmidi.dev[devnum].flags);
	return(handle);
}

bristolMidiClose(int handle)
{
#ifdef DEBUG
	printf("bristolMidiClose()\n");
#endif

	/*
	 * We close a handle, not a device. We need to clean out the handle, and if
	 * no other handles have this device open then close the physical device as
	 * well.
	 */
#warning - close handle, and then device if no longer in use.
	if (bristolMidiSanity(handle) < 0)
		return(bristolMidiSanity(handle));

	switch (bmidi.dev[bmidi.handle[handle].dev].flags & BRISTOL_CONNMASK) {
		case BRISTOL_CONN_OSSMIDI:
			return(bristolMidiOSSClose(handle));
		case BRISTOL_CONN_MIDI:
			return(bristolMidiALSAClose(handle));
		case BRISTOL_CONN_SEQ:
			return(bristolMidiSeqClose(handle));
	}

	return(BRISTOL_MIDI_DRIVER);
}

bristolMidiDevRead(int dev, bristolMidiMsg *msg)
{
#ifdef DEBUG
	printf("bristolMidiRead(%i)\n", dev);
#endif

	if (bristolMidiDevSanity(dev) < 0)
		return(bristolMidiDevSanity(dev));

	/*
	 * ALSARead handles all the TCP interface as well. It should be reworked to
	 * manage OSS raw interfaces as well.....
	 */
	switch (bmidi.dev[dev].flags & BRISTOL_CONNMASK) {
		case BRISTOL_CONN_MIDI:
		case BRISTOL_CONN_TCP:
		case BRISTOL_CONN_OSSMIDI:
			return(bristolMidiALSARead(dev, msg));
		case BRISTOL_CONN_SEQ:
			return(bristolMidiSeqRead(dev, msg));
	}

	return(BRISTOL_MIDI_DRIVER);
}

bristolMidiRead(int handle, bristolMidiMsg *msg)
{
#ifdef DEBUG
	printf("bristolMidiRead(%i): %i/%i\n", handle,
		bmidi.handle[handle].dev, bmidi.dev[bmidi.handle[handle].dev].fd);
#endif

	if (bristolMidiSanity(handle) < 0)
		return(bristolMidiSanity(handle));

	msg->command = 0xff;

	if (bmidi.handle[handle].callback == NULL)
	{
		while (msg->command == 0xff)
		{
//printf("reading type %x\n", bmidi.dev[bmidi.handle[handle].dev].flags);
			switch (bmidi.dev[bmidi.handle[handle].dev].flags
				& BRISTOL_CONNMASK)
			{
				case BRISTOL_CONN_MIDI:
				case BRISTOL_CONN_TCP:
				case BRISTOL_CONN_OSSMIDI:
					if (bristolMidiALSARead(bmidi.handle[handle].dev, msg) < 0)
						return(-1);
					break;
				case BRISTOL_CONN_SEQ:
					if (bristolMidiSeqRead(bmidi.handle[handle].dev, msg) < 0)
						return(-1);
					break;
			}
		}

		return(BRISTOL_MIDI_OK);
	} else {
		switch (bmidi.dev[handle].flags & BRISTOL_CONNMASK) {
			case BRISTOL_CONN_MIDI:
			case BRISTOL_CONN_TCP:
			case BRISTOL_CONN_OSSMIDI:
				return(bristolMidiALSARead(bmidi.handle[handle].dev, msg));
			case BRISTOL_CONN_SEQ:
				return(bristolMidiSeqRead(bmidi.handle[handle].dev, msg));
		}
	}

	return(BRISTOL_MIDI_DRIVER);
}

bristolMidiTerminate()
{
	/*
	 * We need this since the library never actually returns when we are
	 * using the midiCheck routines. We are going to close all handles.
	 */
	bmidi.flags = BRISTOL_MIDI_TERMINATE;

	if (bmidi.dev[0].flags & BRISTOL_CONTROL_SOCKET)
	{
		/*
		 * We should send an all_notes_off and terminate the socket layer with
		 * a REQSTOP
		 */
		close(bmidi.dev[0].fd);
		unlink(BRISTOL_SOCKNAME);
	}
}

bristolMidiWrite(int dev, bristolMsg *msg, int size)
{
	int result, i;
	unsigned char byte;

	if (bristolMidiDevSanity(dev) < 0)
		return(bristolMidiDevSanity(dev));

#ifdef DEBUG
	printf("bristolMidiWrite %i/%i, %i\n", dev, bmidi.dev[dev].fd, size);
#endif

	byte = MIDI_SYSEX;
	bristolPhysWrite(bmidi.dev[dev].fd, &byte, 1);

	bristolPhysWrite(bmidi.dev[dev].fd, (unsigned char *) msg, size);

	byte = MIDI_EOS;
	bristolPhysWrite(bmidi.dev[dev].fd, &byte, 1);
}

bristolMidiControl(int handle, int channel, int operator, int controller,
	int value)
{
	unsigned char comm = 0xb0, ctr, val;

//	printf("Midi Control %i %i %i %i\n", channel, operator, controller, value);

	comm |= channel;
	ctr = controller;
	val = value & 0x7f;

	bristolPhysWrite(bmidi.dev[bmidi.handle[handle].dev].fd, &comm, 1);
	bristolPhysWrite(bmidi.dev[bmidi.handle[handle].dev].fd, &ctr, 1);
	bristolPhysWrite(bmidi.dev[bmidi.handle[handle].dev].fd, &val, 1);
}

bristolPitchEvent(int handle, int op, int channel, int key)
{
	unsigned char comm = 0xe0, lsb, msb;

//	printf("pitch event %i %i %i\n", op, channel, key);

	comm |= channel;
	lsb = key & 0x7f;
	msb = (key >> 7) & 0x7f;

//	printf("Sending %i %i %i on %i\n", comm, msb, lsb,
//		bmidi.dev[bmidi.handle[handle].dev].fd);

	bristolPhysWrite(bmidi.dev[bmidi.handle[handle].dev].fd, &comm, 1);
	bristolPhysWrite(bmidi.dev[bmidi.handle[handle].dev].fd, &lsb, 1);
	bristolPhysWrite(bmidi.dev[bmidi.handle[handle].dev].fd, &msb, 1);
}

bristolKeyEvent(int handle, int op, int channel, int key)
{
	unsigned char comm, velocity = 64;

// 	printf("key event %i %i over %i\n", channel, key,
//bmidi.dev[bmidi.handle[handle].dev].fd);

	if (bristolMidiSanity(handle) < 0)
		return(bristolMidiSanity(handle));

	if (op == BRISTOL_EVENT_KEYON)
		comm = MIDI_NOTE_ON | channel;
	else
		comm = MIDI_NOTE_OFF | channel;

	bristolPhysWrite(bmidi.dev[bmidi.handle[handle].dev].fd, &comm, 1);
	bristolPhysWrite(bmidi.dev[bmidi.handle[handle].dev].fd, &key, 1);
	bristolPhysWrite(bmidi.dev[bmidi.handle[handle].dev].fd, &velocity, 1);
}

bristolMidiSendMsg(int handle, int channel, int operator, int controller,
	int value)
{
	bristolMsg msg;

#ifdef DEBUG
	printf("bristolMidiSendMsg(%i, %i, %i, %i, %i)\n",
		handle, channel, operator, controller, value);
#endif

	/*
	 * Key on and off events are NOT sent as system messages, but as normal
	 * key events on the control channel. As such they require separate formats.
	 */
	if (operator == BRISTOL_EVENT_PITCH)
	{
		bristolPitchEvent(bmidi.handle[handle].dev, operator, channel, value);
		return;
	}
	if ((operator == BRISTOL_EVENT_KEYON) || (operator == BRISTOL_EVENT_KEYOFF))
	{
		bristolKeyEvent(bmidi.handle[handle].dev, operator,
			channel, value & BRISTOL_PARAMMASK);
		return;
	}
	/*
	 * We are going to send this through to the control port as a basic format
	 * message. The parameter is not going to be scaled, just given as a value
	 * from 0 to 127. The receiver can decide how to scale the value.
	 */
	msg.SysID = 83;
	msg.L = 76;
	msg.a = 97;
	msg.b = 98;

	msg.msgLen = sizeof(bristolMsg);
	msg.msgType = MSG_TYPE_PARAM;

	msg.channel = channel;
	msg.from = handle;
	msg.operator = operator;
	msg.controller = controller;
	msg.valueLSB = value & 0x007f;
	msg.valueMSB = (value >> 7) & 0x007f;

	bristolMidiWrite(bmidi.handle[handle].dev, &msg, sizeof(bristolMsg));
}

