//  BMP
//  Copyright (C) 2005-2007 BMP development.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License Version 2
//  as published by the Free Software Foundation.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//  --
//
//  The BMPx project hereby grants permission for non-GPL compatible GStreamer
//  plugins to be used and distributed together with GStreamer and BMPx. This
//  permission is above and beyond the permissions granted by the GPL license
//  BMPx is covered by.

#include "banner-image.hh"

namespace Bmp
{
  namespace UI
  {
    BannerImage::BannerImage (Glib::RefPtr<Gdk::Pixbuf> const& left,
                              Glib::RefPtr<Gdk::Pixbuf> const& right,
                              Glib::RefPtr<Gdk::Pixbuf> const& center)
      : m_left (left),
        m_right (right),
        m_center (center)
    {
      set_flags (Gtk::NO_WINDOW);

      m_max_height = std::max (std::max (m_left->get_height (), m_right->get_height ()), m_center->get_height ());
    }

    BannerImage::~BannerImage ()
    {}

    bool
    BannerImage::on_expose_event (GdkEventExpose * event)
    {
      // FIXME: I think the calculations can be simplified somewhat - descender

      if (!is_drawable ())
        return false;

      Gtk::Allocation const& allocation = get_allocation ();

      int x      = allocation.get_x ();
      int y      = allocation.get_y ();
      int width  = allocation.get_width ();
      int height = allocation.get_height ();

      int left_width   = m_left->get_width ();
      int right_width  = m_right->get_width ();
      int center_width = m_center->get_width ();

      // centre images vertically
      if (height > m_max_height)
        y += (height - m_max_height) / 2;

      Glib::RefPtr<Gdk::Window> window = get_window ();
      Glib::RefPtr<Gdk::GC>     gc     = Gdk::GC::create (window);

      // left portion
      window->draw_pixbuf (gc, m_left, 0, 0, x, y,
                           std::min (left_width, width),
                           std::min (m_left->get_height (), height),
                           Gdk::RGB_DITHER_NONE, 0, 0);
      x += left_width;

      // central portion
      int central_portion_width = std::max (width - left_width - right_width, 0);
      if (central_portion_width > 0)
        {
          int central_portion_height = std::min (m_center->get_height (), height);

          int n_complete_tiles = central_portion_width / center_width;

          for (int i = 0; i < n_complete_tiles; i++)
            {
              window->draw_pixbuf (gc, m_center, 0, 0, x, y,
                                   -1, central_portion_height,
                                   Gdk::RGB_DITHER_NONE, 0, 0);

              x += center_width;
            }

          int central_portion_width_left = std::max (central_portion_width - n_complete_tiles * center_width, 0);
          if (central_portion_width_left > 0)
            {
              window->draw_pixbuf (gc, m_center, 0, 0, x, y,
                                   central_portion_width_left, central_portion_height,
                                   Gdk::RGB_DITHER_NONE, 0, 0);

              x += central_portion_width_left;
            }
        }

      // right portion
      int right_portion_width = width - left_width - central_portion_width;
      if (right_portion_width > 0)
        {
          window->draw_pixbuf (gc, m_right, 0, 0, x, y,
                               right_portion_width, std::min (m_right->get_height (), height),
                               Gdk::RGB_DITHER_NONE, 0, 0);
        }

      return false;
    }

    void
    BannerImage::on_size_request (Gtk::Requisition * requisition)
    {
      // NOTE: To allow arbitrarily small sizes? Just remove this function - descender

      requisition->width  = m_left->get_width () + m_right->get_width ();
      requisition->height = m_max_height;
    }

  } // UI

} // Bmp
