/*
*
*  A2DPD - Bluetooth A2DP daemon for Linux
*
*  Copyright (C) 2006-2007  Frédéric DALLEAU <frederic.dalleau@palmsource.com>
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include "a2dpd_ipc.h"
#include <unistd.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <syslog.h>
#include <errno.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/poll.h>
#include <sys/un.h>
#include <sys/stat.h>
 #include <sys/wait.h> 
#include <netinet/in.h>
#include <arpa/inet.h>

#include "a2dpd_protocol.h"

#define DEFAULTENABLEINTERNET 0
#define DEFAULTIP    "127.0.0.1"
#define DEFAULTPORT  21453
#define BROADCASTIP  "127.0.0.255"

#define ABSTRACT_SOCKET

#ifdef ABSTRACT_SOCKET
#define AFUNIX_PATH  "\0/tmp/a2dpd"
#else
#define AFUNIX_PATH  "/tmp/a2dpd"
#endif

static int g_enable_internet_socket = DEFAULTENABLEINTERNET;
static char g_addr [256] = DEFAULTIP;
static char g_bcst [256] = BROADCASTIP;
static int g_port = DEFAULTPORT;
static int g_server_socket = -1;

void init_ipc(int inet, char* addr, char* bcst, int port)
{
	g_enable_internet_socket = inet;
	if(addr && addr[0]) strncpy(g_addr, addr, sizeof(g_addr)); g_addr[sizeof(g_addr)-1]=0;
	if(bcst && bcst[0]) strncpy(g_bcst, bcst, sizeof(g_bcst)); g_bcst[sizeof(g_bcst)-1]=0;
	if(port) g_port = port;
	DBG("Selected IPC: %s, addr=%s, bcst=%s, port=%d",
		g_enable_internet_socket?"inet":"unix",
		g_addr,
		g_bcst,
		g_port);
}

void close_socket(int *sockfd)
{
        if(*sockfd>0)
        {
        	DBG2("Closing %d", *sockfd);
        	(void)shutdown(*sockfd, SHUT_RDWR);
        	(void)close(*sockfd);
        }
        *sockfd = -1;
}

void close_server_socket_on_fork()
{
	DBG("Closing server socket %d", g_server_socket);
        if(g_server_socket>0)
        {
        	if(g_enable_internet_socket) {
        		(void)close(g_server_socket);
        	} else {
        		(void)close(g_server_socket);
		}
        }
}

void close_server_socket(int *sockfd)
{
        if(*sockfd>0)
        {
        	if(g_enable_internet_socket) {
			(void)shutdown(*sockfd, SHUT_RDWR);
			(void)close(*sockfd);
        	} else {
			(void)shutdown(*sockfd, SHUT_RDWR);
			(void)close(*sockfd);
#ifndef ABSTRACT_SOCKET
			(void)unlink(AFUNIX_PATH);
#endif
		}
        }
        *sockfd = -1;
        g_server_socket = -1;
}

int make_udp_socket()
{
        int sockfd = socket(PF_INET, SOCK_DGRAM, 0);
        if(sockfd>0)
        {
                int broadcast=1;
                if(setsockopt(sockfd, SOL_SOCKET, SO_BROADCAST, &broadcast, (int)sizeof(broadcast)) == 0)
                {
                        struct sockaddr_in peer_addr;
                        peer_addr.sin_family = AF_INET;
                        peer_addr.sin_port = htons(g_port);
                        peer_addr.sin_addr.s_addr = inet_addr(g_bcst);
                        // Connect on a datagram socket simulate recvfrom with the address specified
                        (void)bind(sockfd, (struct sockaddr *)&peer_addr, (int)sizeof(peer_addr));
                        if(connect(sockfd, (struct sockaddr*)&peer_addr, (int)sizeof(peer_addr)) == 0)
                        {
                        }
                        else
                        {
                                (void)close(sockfd);
                                sockfd=-1;
                        }
               }
               else
               {
                       (void)close(sockfd);
                       sockfd=-1;
               }
        }
        return sockfd;
}

int make_client_socket()
{
	int sockfd = -1;

	if(g_enable_internet_socket)
	{
		sockfd = socket(PF_INET, SOCK_STREAM, 0);
		if(sockfd>0)
		{
			struct sockaddr_in peer_addr;
			peer_addr.sin_family = AF_INET;
			peer_addr.sin_port = htons(g_port);
			peer_addr.sin_addr.s_addr = inet_addr(g_addr);
			if(connect(sockfd, (struct sockaddr*)&peer_addr, (int)sizeof(peer_addr))<0)
				close_socket(&sockfd);
		}
	} else {
		sockfd = socket(AF_UNIX, SOCK_STREAM, 0);
		if(sockfd>0)
		{
			struct sockaddr_un peer_addr;
			int peer_addr_size = (int)(sizeof(peer_addr.sun_family)+sizeof(AFUNIX_PATH));
#ifdef ABSTRACT_SOCKET
			peer_addr_size--;
#endif
			memset(&peer_addr, 0, sizeof(peer_addr));
			peer_addr.sun_family = AF_UNIX;
			memcpy(peer_addr.sun_path, AFUNIX_PATH, sizeof(AFUNIX_PATH));
			if(connect(sockfd, (struct sockaddr*)&peer_addr, peer_addr_size)<0)
				close_socket(&sockfd);
		}
	}
	return sockfd;
}

int make_server_socket()
{
	int sockfd = -1;

	DBG("");
	if(g_enable_internet_socket) {
		struct sockaddr_in my_addr;

		sockfd = socket(PF_INET, SOCK_STREAM, 0);
		memset(&my_addr, 0, sizeof(my_addr));
		my_addr.sin_family = AF_INET;
		my_addr.sin_port = htons(g_port);
		my_addr.sin_addr.s_addr = INADDR_ANY;

		if(sockfd>0) {
			int on = 1;
			(void)setsockopt(sockfd, SOL_SOCKET, SO_REUSEADDR, &on, (int)sizeof(on));
			if(bind(sockfd, (struct sockaddr *)&my_addr, (int)sizeof(struct sockaddr))==0)
			{
				if(listen(sockfd, 0)==0)
				{
					// No error
				}
				else
				{
					(void)close(sockfd);
					sockfd = -3;
				}
			}
			else
			{
				(void)close(sockfd);
				sockfd = -2;
			}
		}
	} else {
		struct sockaddr_un my_addr;
		int my_addr_size = (int)(sizeof(my_addr.sun_family)+sizeof(AFUNIX_PATH));
#ifdef ABSTRACT_SOCKET
		my_addr_size --;
#else
		(void)unlink(AFUNIX_PATH);
#endif
		sockfd = socket(AF_UNIX, SOCK_STREAM, 0);
		memset(&my_addr, 0, sizeof(my_addr));
		my_addr.sun_family = AF_UNIX;
		memcpy(my_addr.sun_path, AFUNIX_PATH, sizeof(AFUNIX_PATH));

		if(sockfd>0)
		{
			int on = 1;
			(void)setsockopt(sockfd, SOL_SOCKET, SO_REUSEADDR, &on, (int)sizeof(on));
			if(bind(sockfd, (struct sockaddr *)&my_addr, my_addr_size)==0)
			{
				if(listen(sockfd, 0)==0)
				{
#ifndef ABSTRACT_SOCKET
					(void)chmod(AFUNIX_PATH, 0777);
#endif
					// No error
				}
				else
				{
					(void)close(sockfd);
					sockfd = -3;
				}
			} else {
				(void)close(sockfd);
				sockfd = -2;
			}
		}
	}

	g_server_socket = sockfd;

        return sockfd;
}

void add_fd_to_poll(struct pollinfo* pollinfos, int fd, int events, int timeout, fnpollfd_cb pollfd_cb, void* param1, void* param2)
{
	if(pollinfos->pollfdcount < pollinfos->maxfd) {
		if(fd >= 0) {
			pollinfos->pollfds[pollinfos->pollfdcount].fd=fd;
			pollinfos->pollfds[pollinfos->pollfdcount].events=events;
			pollinfos->pollfds[pollinfos->pollfdcount].revents=0;
			pollinfos->pollfds_cb[pollinfos->pollfdcount].pollfd_cb=pollfd_cb;
			pollinfos->pollfds_cb[pollinfos->pollfdcount].param1=param1;
			pollinfos->pollfds_cb[pollinfos->pollfdcount].param2=param2;

			if (timeout != -1)
				pollinfos->polltimeout = (pollinfos->polltimeout == -1)?timeout:min(pollinfos->polltimeout, timeout);
			/* else keep value in struct as it can only be less than infinite */
			pollinfos->pollfdcount++;
		} else {
			DBG("Invalid socket to poll (fd=%d, count=%d)", fd, pollinfos->pollfdcount);
		}
	} else {
		DBG("Too many sockets to poll");
	}
}

int poll_accept(int sockfd, int timeout)
{
	int result = 0;
	int ipoll = -1;
	struct pollfd pollfds[1];
	pollfds[0].fd = sockfd;
	pollfds[0].events = POLLIN;
	pollfds[0].revents = 0;

	ipoll = poll(pollfds, (nfds_t)1, timeout);

	if((ipoll>0) && (pollfds[0].revents & POLLIN)) {
		result = 1;
	} else if(ipoll==0) {
		errno = EAGAIN;
	}

	return result;
}

int poll_error(int sockfd, int timeout)
{
	int result = 0;
	int ipoll = -1;
	struct pollfd pollfds[1];
	pollfds[0].fd = sockfd;
	pollfds[0].events = 0;
	pollfds[0].revents = 0;

	//DBG("Polling %d", sockfd);
	ipoll = poll(pollfds, (nfds_t)ARRAY_SIZE(pollfds), timeout);
	// if(pollfds[0].revents)
	//	DBG("Polled %d = %d %08X", sockfd, ipoll, pollfds[0].revents);
	if((ipoll>0) && (pollfds[0].revents & (POLLERR | POLLHUP | POLLNVAL))) {
		DBG("%s detected on socket %d", (pollfds[0].revents & POLLERR) ? "POLLERR" : (pollfds[0].revents & POLLHUP) ? "POLLHUP" : "Other?(POLLNVAL)", sockfd);
		result = 1;
	} else if(ipoll==0) {
		errno = EAGAIN;
	}

	return result;
}

int poll_out(int sockfd, int timeout)
{
	int result = 0;
	int ipoll = -1;
	struct pollfd pollfds[1];
	pollfds[0].fd = sockfd;
	pollfds[0].events = POLLOUT ;
	pollfds[0].revents = 0;

	ipoll = poll(pollfds, (nfds_t)1, timeout);

	if((ipoll>0) && (pollfds[0].revents & POLLOUT)) {
		result = 1;
	} else if(ipoll==0) {
		errno = EAGAIN;
	}

	return result;
}


int accept_socket(int sockfd)
{
	int new_fd = -1;

	if(poll_accept(sockfd, -1))
	{
		// Block until connections
		if(g_enable_internet_socket)
		{
			struct sockaddr_in peer_addr;
			size_t sin_size = sizeof(peer_addr);
			new_fd = accept(sockfd, (struct sockaddr *)&peer_addr, &sin_size);
		} else {
			struct sockaddr_un peer_addr;
			size_t sin_size = sizeof(peer_addr);
			new_fd = accept(sockfd, (struct sockaddr *)&peer_addr, &sin_size);
		}
	}
        return new_fd;
}

void setup_socket(int sockfd)
{
	if(g_enable_internet_socket)
	{
		// Timeouts
		struct timeval t = { 1, 0 };
		(void)setsockopt( sockfd, SOL_SOCKET, SO_SNDTIMEO, &t, (int)sizeof(t));
		(void)setsockopt( sockfd, SOL_SOCKET, SO_RCVTIMEO, &t, (int)sizeof(t));
	} else {
		// Timeouts
		struct timeval t = { 1, 0 };
		(void)setsockopt( sockfd, SOL_SOCKET, SO_SNDTIMEO, &t, (int)sizeof(t));
		(void)setsockopt( sockfd, SOL_SOCKET, SO_RCVTIMEO, &t, (int)sizeof(t));
	}
}

ssize_t send_socket(int sockfd, void* buffer, size_t size)
{
        ssize_t result = -1;
        ssize_t ioffset = 0;
        while(sockfd>0 && ioffset<(ssize_t)size)
        {
                result=send(sockfd, ((char*)buffer)+ioffset, size-ioffset, MSG_NOSIGNAL);
                if(result>0)
                {
                        ioffset += result;
                        if(result != (ssize_t)size)
                        	DBG("split frame sent (%d/%d)", (int)result, (int)size);
                }
                else
                {
                        break;
                }
        }
        return result;
}

ssize_t recv_socket(int sockfd, void* buffer, size_t size)
{
        ssize_t received = 0;
        while((buffer!=0) && (received<(ssize_t)size))
        {
                ssize_t result = recv(sockfd, buffer+received, size-received, MSG_NOSIGNAL);
                if(result>0)
                {
                        received += result;
                        if(result != (ssize_t)size)
                        	DBG("split frame received (%d/%d)", (int)result, (int)size);
                }
                else
                {
                        received=result;
                        break;
                }
        }
        return received;
}

//
// Utility fonctions
//
void async_run_process(char* cmd, int wait)
{
        char command[256];
        char* argv[8];
        int i = 0, status = 0;
        strncpy(command, cmd, sizeof(command));
        DBG("Command is %s", command);
        command[sizeof(command)-1]= '\0';
        memset(argv, 0, sizeof(argv));
        argv[0] = command;

        // Keep NULL at the end of the argv table
        while((i < ARRAY_SIZE(argv)-2) && (argv[i] != NULL)) {
		// End current word
		argv[i+1] = strchr(argv[i], ' ');
		// Begin next word if exist
		while((argv[i+1]!=NULL) && *argv[i+1]==' ') {
			*argv[i+1] = '\0';
			argv[i+1]++;
		}
		i++;
        }

	for(i=0; i < ARRAY_SIZE(argv); i++) {
		DBG("argv[%d] = %s", i, argv[i]?argv[i]:"NULL");
		if(!argv[i]) break;
	}

        if((cmd!=NULL) && (cmd[0]!='\0'))
        {
	        pid_t pid = fork();
                switch(pid)
                {
                        case 0:
                                // Children process
                                close_server_socket_on_fork();

                                // Replace children with new process
                                i = execvp(command, argv);
                                DBG("execlp failed %s=%d (errno=%d:%s)", cmd, i, errno, strerror(errno));
                                break;
                        case -1:
                                // failed
                                DBG("Fork %s failed", cmd);
                                break;
                        default:
                                DBG("Forked %s", cmd);
                                // Parent wait for completion if asked
                                if(wait) {
                                	DBG("Waiting pid=%d", pid);
					waitpid(pid, &status, 0);
                                	DBG("Wait returned %d", status);
                                }
                                break;
                }
        }
}


void startup_a2dpd_upon_request()
{
	int running = 0;
	pid_t pid = 0;

	// Read pid
	FILE* fp = fopen(PIDFILE, "r");
	if ((fscanf(fp, "%d", &pid) != 1)) {
		// Check a2dpd runs
		if((pid!=0)&&kill(pid,SIGUSR1)==0){
			// Already running
			DBG("A2DPD is already running");
			running = 1;
		}
	}

	// Start a2dpd if not running
	if(!running)
	{
		DBG("A2DPD is not running");
		async_run_process("a2dpd -d", 1);
	}
}

void get_config_filename(char* filename, size_t buffersize)
{
	// This strange construction allows to pass splint
	char* home_path = "/";
	if(getenv("HOME")) {
		home_path = getenv("HOME");
	}
	if(home_path != NULL) {
		(void)snprintf(filename, buffersize, "%s/%s", home_path, ".a2dprc");
		filename[buffersize-1] = '\0';
	}
}

void read_config_string(char* filename, char* section, char* key, char* returnbuffer, size_t buffersize, char* defvalue)
{
        int found=0, error=0;
        FILE* hFile = fopen(filename, "rt");
	returnbuffer[0] = '\0';
        //DBG("read_config_string: reading %s", filename);
        if(hFile)
        {
                // search section
                while(!error && !found && !feof(hFile))
                {
                        char buffer[256], szsection[256];
                        if(fgets(buffer, (int)sizeof(buffer), hFile) == NULL)
                        {
                                error=1;
                                break;
                        }

        //DBG("read_config_string: read section %s", buffer);
                        if(sscanf(buffer, "[%s]", szsection)==1)
                        {
                                szsection[strlen(szsection)-1]='\0';
        //DBG("read_config_string: scanned %s", szsection);
                                // Found section
                                if(!strcasecmp(section, szsection))
                                {
                                        // key search loop
                                        while(!error && !found && !feof(hFile))
                                        {
                                                char szkey[256], szvalue[256];
                                                if(fgets(buffer, (int)sizeof(buffer), hFile) == NULL)
                                                {
                                                        error=1;
                                                        break;
                                                }
        //DBG("read_config_string: read key %s", buffer);
                                                // Another section name will exit the key search loop
                                                if(sscanf(buffer, "[%s]", szsection)==1)
                                                {
                                                        break;
                                                }
                                                // A key name
                                                if(sscanf(buffer, "%[^=]=%[^\n]", szkey, szvalue)>1)
                                                {
        //DBG("read_config_string: read %s '%s' '%s'", buffer, szkey, szvalue);
                                                        // Found key
                                                        if(!strcasecmp(key, szkey))
                                                        {
        //DBG("read_config_string: buffer=%s", buffer);
                                                                strncpy(returnbuffer, szvalue, buffersize);
                                                                returnbuffer[buffersize-1]='\0';
                                                                found = 1;
                                                        }
                                                }
                                        }
                                }
                        }
                }
                (void)fclose(hFile);
        }

        // Put default value
        if(!found)
        {
                strncpy(returnbuffer, defvalue, buffersize);
                returnbuffer[buffersize-1]='\0';
        }
        //DBG("%s [%s] '%s'='%s'", __FUNCTION__, section, key, returnbuffer);
}

int read_config_int(char* filename, char* section, char* key, int defvalue)
{
        char def[32]="";
        char result[512]="";
        (void)snprintf(def, sizeof(def), "%d", defvalue);
        read_config_string(filename, section, key, result, sizeof(result), def);
        return (atoi(result));
}
