/*
 * Copyright (c) 2005, 2006 Jeremy Erickson
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
 * CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include "mainact.h"
#include "verseact.h"
#include "aboutact.h"
#include "verse.h"
#include "filterallowall.h"
#include "filtercategory.h"
#include "watchermainact.h"
#include "filtersearchquery.h"
#include "filterquery.h"
#include "versecollection.h"
#include "preferencesact.h"
#include "bibleplugin.h"
#include "biblepluginmeta.h"
#include "qt_types.h"
#include "settings.h"
#include <qsplitter.h>
#include <qpushbutton.h>
#include <qtabwidget.h>
#include <qapplication.h>
#include <qmap.h>
#include <qstringlist.h>
#include <qcombobox.h>
#include <qlineedit.h>
#include <qmessagebox.h>
#include <qinputdialog.h>
#include <qdom.h>
#include <qtextstream.h>
#include <qclipboard.h>
#include <list>
#include <iostream>
using namespace bmemcore;
namespace bmemui
{

MainAct::MainAct(QWidget *parent, const char *name)
:MainUI(parent, name), currWidget(NULL), mVerses(new VerseCollection())
{
#if QT_VERSION >= 0x040000
    mCategoryNoticeLabel->setWordWrap(true);
#endif
    bool worked = true;
    BiblePluginMeta meta = PreferencesAct::getMetaStatic(worked);
    mPlugin = new BiblePlugin(meta);
    mFilterComboBox->insertItem(tr("All Verses"), 0);
    mFilterComboBox->insertItem(tr("From Search"), 1);
    mCategoryListBox->clear();
    mFileFilters = tr("Bible Verse Collection files (*.bvc)\n"
                      "All Files (*)");
    setCurrentFileName("");
    mOpenForComboBox->insertItem(tr("View"),VerseAct::MODE_VIEW);
    mOpenForComboBox->insertItem(tr("Edit"),VerseAct::MODE_EDIT);
    mOpenForComboBox->insertItem(tr("Quiz"),VerseAct::MODE_QUIZ);
    mVerseListBox->clear();
    mSearchFilter = new FilterAllowAll();
    mMainSplitter->setResizeMode(mBigNavFrame, QSplitter::KeepSize);
    //mSearchQuery = new FilterSearchQuery();
    mSearchQuery = "true";
    //mSearchQuery->setMatches(FilterSearchQuery::MATCHES_ANY);
    qApp->installEventFilter(this);
    connect(editMenu, SIGNAL(aboutToShow()), this, 
            SLOT(editMenu_aboutToShow()));
    connect( qApp, SIGNAL( lastWindowClosed() ), qApp, SLOT( quit() ) );
    if (Settings::getOpenLastFile() && QFile::exists(Settings::getLastFile())){
        openFile(Settings::getLastFile());
    }
}

void MainAct::mToggleButton_clicked()
{
    if (mNavFrame->isVisible())
    {
        mNavFrame->hide();
        mToggleButton->setText("");
        mMainSplitter->setResizeMode(mBigNavFrame, QSplitter::FollowSizeHint);
        mMainSplitter->refresh();
    }
    else
    {
        mNavFrame->show();
        mToggleButton->setText("");
        mMainSplitter->setResizeMode(mBigNavFrame, QSplitter::KeepSize);
    }
}

void MainAct::mAddButton_clicked()
{
    Verse* newVerse = mVerses->addNewVerse();
    newVerse->setTranslation(Settings::getDefaultTranslation());
    WatcherMainAct* newWatcher = new WatcherMainAct(*this, *newVerse,
            mSearchFilter);
    openVerse(newVerse, VerseAct::MODE_EDIT);
}

void MainAct::fileExitAction_activated()
{
    //if (promptSave())
    //{
        qApp->closeAllWindows();
    //}
}

void MainAct::closeEvent(QCloseEvent* evt){
    if (promptSave()){
        Settings::setLastFile(mCurrentFileName);
        evt->accept();
    }
    else{
        evt->ignore();
    }
}

void MainAct::helpAboutAction_activated()
{
    AboutAct(this).exec();
}

void MainAct::setCurrentFileName(const QString &newFile)
{
    mCurrentFileName = newFile;
    QString shownName = newFile;
    if (shownName == "")
    {
        shownName = tr("Untitled");
    }
    setCaption(tr("BibleMemorizer") + " - " + shownName);
}

bool MainAct::promptSave()
{
    if (mVerses->hasChanged())
    {
        int choice = QMessageBox::warning(this, tr("BibleMemorizer"),
            tr("The current collection has been modified.\n"
            "Do you want to save the changes?"),
            QMessageBox::Yes | QMessageBox::Default,
            QMessageBox::No,
            QMessageBox::Cancel | QMessageBox::Escape);
        if (choice == QMessageBox::Yes)
        {
            return save();
        }
        else if (choice == QMessageBox::Cancel)
        {
            return false;
        }
    }
    return true;
}

bool MainAct::save()
{
     if (mCurrentFileName == "")
     {
        return saveAs();
     }
     else
     {
        mVerses->saveFile(mCurrentFileName);
        return true;
     }
}

bool MainAct::saveAs()
{
    QString fileName = "";
    bool keepGoing = true;
    while (keepGoing)
    {
        QString passName = mCurrentFileName;
        if (passName == "")
            passName = ".";
        fileName = BQFileDialog::getSaveFileName(passName, mFileFilters, this);
        if (fileName.isEmpty())
            return false;
        if (!fileName.endsWith(".bvc", false)){
            fileName.append(".bvc");
        }
        if (QFile::exists(fileName)){
            int choice = QMessageBox::warning(this, tr("BibleMemorizer"),
                tr("The selected file already exists.\n"
                "Do you want to overwrite it?"),
                QMessageBox::Yes | QMessageBox::Default,
                QMessageBox::No,
                QMessageBox::Cancel | QMessageBox::Escape); 
            if (choice == QMessageBox::Yes)
            {
                keepGoing = false;
            }
            else if (choice == QMessageBox::No)
            {
                //do nothing, we will keep going.
            }
            else if (choice = QMessageBox::Cancel)
            {
                return false;
            }
        }
        else
        {
            keepGoing = false;
        }
    }
    if (!fileName.isEmpty())
        mVerses->saveFile(fileName);
    return true;
}

void MainAct::openFile(const QString &fileName)
{
    delete mVerses;
    mVerses = new VerseCollection(fileName);
    mCategoryListBox->clear();
    mCategoryListBox->insertStringList(mVerses->getCategories());
    mFilterComboBox->clear();
    mFilterComboBox->insertItem(tr("All Verses"), 0);
    mFilterComboBox->insertItem(tr("From Search"), 1);
    mFilterComboBox->insertStringList(mVerses->getCategories());
    const std::list<Verse*>& verseList = mVerses->getVerses();
    std::list<Verse*>::const_iterator it = verseList.begin();
    while (it != verseList.end())
    {
        WatcherMainAct* newWatcher = new WatcherMainAct(*this, *(*it), 
                mSearchFilter);
        it++;
    }
    setCurrentFileName(fileName);
}

void MainAct::associateListItem(BQListBoxItem* listItem, Verse* verse)
{
    mListItemMap[listItem] = verse;
    //New Code
    bool placed = false;
    int index = 0;
    for (index = 0; index < mVerseListBox->count(); index++)
    {
        if (verse->compareTo(*mListItemMap[mVerseListBox->item(index)]) < 0){
            break;
        }
    }
    mVerseListBox->insertItem(listItem, index);
    mVerseListBox->setCurrentItem(listItem);
}


void MainAct::dissociateListItem(BQListBoxItem* listItem)
{
    mListItemMap.erase(listItem);
    delete listItem;
}

void MainAct::addWatcher(WatcherMainAct* watcher)
{
    mWatchers.push_back(watcher);
}

void MainAct::removeWatcher(WatcherMainAct* watcher)
{
    mWatchers.remove(watcher);
}

void MainAct::mOpenButton_clicked()
{
    if (mVerseListBox->selectedItem() != 0)
    {
        Verse* verse = mListItemMap[mVerseListBox->selectedItem()];
        openVerse(verse, static_cast<VerseAct::PageType>
                (mOpenForComboBox->currentItem()));
    }
}

void MainAct::mVerseListBox_selectionChanged()
{
    if (mVerseListBox->selectedItem() != 0)
    {
        mOpenButton->setEnabled(true);
        mRemoveButton->setEnabled(true);
    }
    else
    {
        mOpenButton->setEnabled(false);
        mRemoveButton->setEnabled(true);
    }
}

void MainAct::fileOpenAction_activated()
{
    if (promptSave())
    {
        QString fileName = BQFileDialog::getOpenFileName(".", mFileFilters, 
                this);
        openFile(fileName);
    }
}

void MainAct::fileSaveAction_activated()
{
    save();
}

void MainAct::fileSaveAsAction_activated()
{
    saveAs();
}

void MainAct::mRemoveButton_clicked()
{
    int choice = QMessageBox::warning(this, tr("BibleMemorizer"),
        tr("Are you sure you want to remove this verse?"),
        QMessageBox::Yes | QMessageBox::Default,
        QMessageBox::No | QMessageBox::Escape);
    if (choice == QMessageBox::Yes)
    {
        Verse* verse = mListItemMap[mVerseListBox->selectedItem()];
        bool removed = mVerses->removeVerse(verse);
        if (!removed)
            delete verse;
    }
}

void MainAct::fileNewAction_activated()
{
    if (promptSave())
    {
        delete mVerses;
        mVerses = new VerseCollection();
        mCategoryListBox->clear();
        mFilterComboBox->clear();
        mFilterComboBox->insertItem(tr("All Verses"), 0);
        mFilterComboBox->insertItem(tr("From Search"), 1);
        setCurrentFileName("");
    }
}

void MainAct::helpContentsAction_activated()
{
    //mAssistantClient.showPage("book1.htm");

}

void MainAct::mAddCategoryButton_clicked()
{
    bool ok;
    QString newCat = QInputDialog::getText(
            tr("BibleMemorizer"), 
            tr("Enter the name of the new category:"),
            QLineEdit::Normal, QString::null, &ok, this );
    newCat = newCat.simplifyWhiteSpace();
    if (ok && !newCat.isEmpty() && !mVerses->containsCategory(newCat) && 
            !newCat.contains(';')) 
    {
        mCategoryListBox->insertItem(newCat);
        mFilterComboBox->insertItem(newCat);
        for (int i = 0; i < mTabWidget->count(); i++)
        {
            QWidget *nextPage = mTabWidget->page(i);
            if (VerseAct* nextWidget = dynamic_cast<VerseAct*>(nextPage))
            {
                nextWidget->addCategory(newCat);
            }
        }
        mVerses->addCategory(newCat);
    }
    else if (mVerses->containsCategory(newCat))
    {
        QMessageBox::information(this, tr("BibleMemorizer"), 
                tr("You have already created that category"));
        mCategoryListBox->setSelected(mCategoryListBox->findItem(newCat,
                B_CaseSensitive | B_ExactMatch), true);
    }
    else if (newCat.contains(';'))
    {
        QMessageBox::information(this, tr("BibleMemorizer"), 
                tr("Semicolons are used internally and cannot be in "
                "category names."));
    }
}

void MainAct::mRenameCategoryButton_clicked()
{
    bool ok;
    QString oldCat = mCategoryListBox->selectedItem()->text();
    QString newCat = QInputDialog::getText(
            tr("BibleMemorizer"), 
            tr("Enter the new name for this category:"),
            QLineEdit::Normal, QString::null, &ok, this );
    newCat = newCat.simplifyWhiteSpace();
    if (ok && !newCat.isEmpty() && !mVerses->containsCategory(newCat) && 
            !newCat.contains(';')) 
    {
        int index = mCategoryListBox->index(mCategoryListBox->selectedItem());
        delete mCategoryListBox->selectedItem();
        mCategoryListBox->insertItem(newCat, index);
        for (int i = 0; i < mFilterComboBox->count(); i++)
        {
            if (mFilterComboBox->text(i) == oldCat)
            {
                mFilterComboBox->changeItem(newCat, i);
                break;
            }
        }
        for (int i = 0; i < mTabWidget->count(); i++)
        {
            QWidget *nextPage = mTabWidget->page(i);
            if (VerseAct* nextWidget = dynamic_cast<VerseAct*>(nextPage))
            {
                nextWidget->renameCategory(oldCat, newCat);
            }
        }
        mVerses->renameCategory(oldCat, newCat, true);
    }
    else if (mVerses->containsCategory(newCat))
    {
        QMessageBox::information(this, tr("BibleMemorizer"), 
                tr("You have already created that category"));
        mCategoryListBox->setSelected(mCategoryListBox->findItem(newCat, 
                B_CaseSensitive | B_ExactMatch), true);
    }
    else if (newCat.contains(';'))
    {
        QMessageBox::information(this, tr("BibleMemorizer"), 
                tr("Semicolons are used internally and cannot be in "
                "category names."));
    }
}

void MainAct::mRemoveCategoryButton_clicked()
{
    QString oldCat = mCategoryListBox->selectedItem()->text();
    int choice = QMessageBox::warning(this, tr("BibleMemorizer"),
        tr("Are you sure you want to remove this category?\n\n"
           "No verses will be deleted, but the category will be\n"
           "removed from all category lists."),
        QMessageBox::Yes | QMessageBox::Default,
        QMessageBox::No | QMessageBox::Escape);
    if (choice == QMessageBox::Yes)
    {
        delete mCategoryListBox->selectedItem();
        for (int i = 0; i < mFilterComboBox->count(); i++)
        {
            if (mFilterComboBox->text(i) == oldCat)
            {
                mFilterComboBox->removeItem(i);
                break;
            }
        }
        for (int i = 0; i < mTabWidget->count(); i++)
        {
            QWidget *nextPage = mTabWidget->page(i);
            if (VerseAct* nextWidget = dynamic_cast<VerseAct*>(nextPage))
            {
                nextWidget->removeCategory(oldCat);
            }
        }
        mVerses->removeCategory(oldCat, true);
    }
}

MainAct::~MainAct()
{
}

void MainAct::mCategoryListBox_selectionChanged()
{
    if (mCategoryListBox->selectedItem() != 0)
    {
        mRenameCategoryButton->setEnabled(true);
        mRemoveCategoryButton->setEnabled(true);
    }
    else
    {
        mRenameCategoryButton->setEnabled(false);
        mRemoveCategoryButton->setEnabled(false);
    }
}

void MainAct::mFilterComboBox_activated(int newVal)
{
    //mListItemMap.clear();
    //if (mSearchFilter != mSearchQuery)
        delete mSearchFilter;
    switch (newVal)
    {
    case 0:
        //mSearchFilter = new FilterAllowAll();
        mSearchFilter = new FilterQuery("true");
        break;
    case 1:
        //mSearchFilter = mSearchQuery;
        mSearchFilter = new FilterQuery(mSearchQuery);
        break;
    default:
        //mSearchFilter = new FilterCategory(mFilterComboBox->text(newVal));
        QString catName = mFilterComboBox->text(newVal);
        //Escape characters that need to be escaped.
        catName.replace('\\', "\\\\");
        catName.replace('"', "\\\"");
        mSearchFilter = new FilterQuery("categories contains \"" + 
                                    mFilterComboBox->text(newVal) + "\"");
        break;
    }
    std::list<WatcherMainAct*>::const_iterator it = mWatchers.begin();
    while (it != mWatchers.end())
    {
        (*it)->newFilter(mSearchFilter);
        it++;
    }
}

void MainAct::mSearchButton_clicked()
{
//    (new SearchAct(*this, this))->show();
    bool ok;
    QString newQuery = QInputDialog::getText(tr("New Query"),
                        tr("Enter a new search query:\n\n"
                        "Examples:\ntext contains \"Jesus Christ\"\n"
                        "(reference contains \"3:16\" or reference contains "
                        "\"3:23\") and translation equals \"NIV\"\n"
                        "categories contains \"Salvation\"\n"
                        "not chapter equals \"3\""),
                        QLineEdit::Normal, mSearchQuery, &ok, this);
    if (ok)
    {
        if (FilterQuery::patternValid(newQuery))
        {
            setSearchQuery(newQuery);
        }
        else
        {
            QMessageBox::warning(this, "Bad pattern", "The pattern you "
                    "selected is invalid.");
        }
    }
}

//void MainAct::setSearchQuery(FilterSearchQuery* newQuery)
//{
//    delete mSearchQuery;
//    mSearchQuery = newQuery;
//    mFilterComboBox->setCurrentItem(1);
//    mSearchFilter = mSearchQuery; //Doing here to prevent double deletion.
//    mFilterComboBox_activated(1);
//}

void MainAct::setSearchQuery(const QString& newQuery)
{
    mSearchQuery = newQuery;
    mFilterComboBox->setCurrentItem(1);
    mFilterComboBox_activated(1);
}

void MainAct::preferencesSettingsAction_activated()
{
    PreferencesAct pluginSelect;
    int result = pluginSelect.exec();
    if (result == QDialog::Accepted)
    {
        BiblePlugin* newPlugin = new BiblePlugin(pluginSelect.getMeta());
        for (int i = 0; i < mTabWidget->count(); i++)
        {
            QWidget *nextPage = mTabWidget->page(i);
            if (VerseAct* nextWidget = dynamic_cast<VerseAct*>(nextPage))
            {
                nextWidget->setPlugin(newPlugin);
            }
        }
        delete mPlugin;
        mPlugin = newPlugin;
    }
}

void MainAct::mVerseListBox_doubleClicked(BQListBoxItem* item)
{
    if (item != 0)
    {
        Verse* verse = mListItemMap[item];
        openVerse(verse, static_cast<VerseAct::PageType>
                (mOpenForComboBox->currentItem()));
    }
}

void MainAct::openVerse(Verse* verse, VerseAct::PageType pageType)
{
    if (verse != 0)
    {
        VerseAct* newAct = new VerseAct(verse, mPlugin,
                mVerses->getCategories(), pageType, *mTabWidget);
        mTabWidget->insertTab(newAct, verse->getReference());
        mTabWidget->showPage(newAct);
    }
    else
    {
        qWarning("Null verse.");
    }
}

void MainAct::editMenu_aboutToShow()
{
    disconnect(editCopyAction, SIGNAL(activated()), 0, 0);
    disconnect(editCutAction, SIGNAL(activated()), 0, 0);
    disconnect(editPasteAction, SIGNAL(activated()), 0, 0);
    disconnect(editUndoAction, SIGNAL(activated()), 0, 0);
    disconnect(editRedoAction, SIGNAL(activated()), 0, 0);
    //QWidget* currWidget = qApp->focusWidget();
    if ((currWidget != 0) && (currWidget->inherits(BQTextEdit_TypeName) ||
            currWidget->inherits("QLineEdit") ||
            (currWidget->inherits("QComboBox") &&
            static_cast<QComboBox*>(currWidget)->lineEdit() != NULL)))
    {
        QWidget* receiver = NULL;
        if (currWidget->inherits(BQTextEdit_TypeName))
        {
            BQTextEdit* currEdit = static_cast<BQTextEdit*>(currWidget);
            receiver = currEdit;
            editCopyAction->setEnabled(currEdit->hasSelectedText());
            editCutAction->setEnabled(currEdit->hasSelectedText() &&
                    !currEdit->isReadOnly());
            editPasteAction->setEnabled(!currEdit->isReadOnly());
            editUndoAction->setEnabled(currEdit->isUndoAvailable());
            editRedoAction->setEnabled(currEdit->isRedoAvailable());
        }
        else if (currWidget->inherits("QComboBox") ||
                currWidget->inherits("QLineEdit"))
        {
            QLineEdit* currEdit;
            if (currWidget->inherits("QComboBox"))
            {
                currEdit = static_cast<QComboBox*>(currWidget)->lineEdit();
            }
            else
            {
                currEdit = static_cast<QLineEdit*>(currWidget);
            }
            receiver = currEdit;
            editCopyAction->setEnabled(currEdit->hasSelectedText());
            editCutAction->setEnabled(currEdit->hasSelectedText() &&
                    !currEdit->isReadOnly());
            editPasteAction->setEnabled(!currEdit->isReadOnly());
            editUndoAction->setEnabled(currEdit->isUndoAvailable());
            editRedoAction->setEnabled(currEdit->isRedoAvailable());
        }
        editPasteAction->setEnabled(editPasteAction->isEnabled() && 
                !qApp->clipboard()->text(QClipboard::Clipboard).isEmpty());
        connect(editCopyAction, SIGNAL(activated()), receiver, SLOT(copy()));
        connect(editCutAction, SIGNAL(activated()), receiver, SLOT(cut()));
        connect(editPasteAction, SIGNAL(activated()), receiver, SLOT(paste()));
        connect(editUndoAction, SIGNAL(activated()), receiver, SLOT(undo()));
        connect(editRedoAction, SIGNAL(activated()), receiver, SLOT(redo()));
    }
    else
    {
        editCopyAction->setEnabled(false);
        editCutAction->setEnabled(false);
        editPasteAction->setEnabled(false);
        editUndoAction->setEnabled(false);
        editRedoAction->setEnabled(false);
    }
}

bool MainAct::eventFilter(QObject* watched, QEvent* e)
{
    if (e->type() == QEvent::FocusIn)
    {
        if (watched->inherits("QWidget") && !watched->inherits("QMenuBar"))
        {
            currWidget = static_cast<QWidget*>(watched); 
        }
    }
    return false;
}

}

