#include "FFReadHandler.h"
#include "FFReadStream.h"
#include "avm_output.h"

#include "avm_avformat.h"

#include <stdlib.h>
#include <string.h>
#include <unistd.h>

AVM_BEGIN_NAMESPACE;

class AVStreamPacket : public StreamPacket
{
    AVPacket packet;

    /// \internal disabled
    AVStreamPacket(const AVStreamPacket&) {}
    /// \internal disabled
    AVStreamPacket& operator=(const AVStreamPacket&) { return *this; }

public:
    AVStreamPacket() : StreamPacket() {}
    virtual ~AVStreamPacket() {
	if (memory)
	{
	    av_free_packet(&packet);
	    memory = 0;
	}
    }

    AVPacket& GetAVPacket() { return packet; }
    const AVPacket& GetAVPacket() const { return packet; }
    void UpdateMembers() {
	memory = packet.data;
	size = packet.size;
        read = 0;
    }
};


FFReadHandler::FFReadHandler()
    : m_pContext(0)
{
    static int g_iInitilized = 0;

    if (!g_iInitilized)
    {
	av_register_all();
	g_iInitilized++;
    }
}

FFReadHandler::~FFReadHandler()
{
    if (m_pContext)
    {
	flush();

	while (m_Streams.size() > 0)
	{
	    delete m_Streams.back();
	    m_Streams.pop_back();
	}
	av_close_input_file(m_pContext);
    }
}

int FFReadHandler::Init(const char* url)
{
    AVFormatParameters avfp;
    AVInputFormat* fmt = 0;

    // av_find_input_format(url);
    //printf("find input format  %p   %s\n", fmt, b);
    memset(&avfp, 0, sizeof(avfp));
    int r = av_open_input_file(&m_pContext, url,
			       fmt, 64000, &avfp);
    if (r < 0)
    {
	AVM_WRITE("FF reader", "Open Input failed %d\n", r);
	return -1;
    }

    if (av_find_stream_info(m_pContext) < 0)
	return -1;

    AVM_WRITE("FF reader", "Format:'%s' Stream(s):%d\n", m_pContext->iformat->long_name, m_pContext->nb_streams);

    m_Streams.resize(m_pContext->nb_streams);
    for (unsigned i = 0; i < m_pContext->nb_streams; ++i)
    {
	AVCodecContext* avc = m_pContext->streams[i]->codec;
	AVCodec* codec = avcodec_find_decoder(avc->codec_id);
	AVM_WRITE("FF reader", "S: %d %s CodecID:%x  bitrate:%d (%d) samplerate:%d  chn:%d  framerate:%d/%d  wxh %dx%d  %d/%d\n",
		  i, codec ? codec->name : "", avc->codec_id, avc->bit_rate, avc->bit_rate_tolerance,
		  avc->sample_rate, avc->channels, avc->time_base.num, avc->time_base.den,
		  avc->width, avc->height, avc->sample_aspect_ratio.num, avc->sample_aspect_ratio.den);
	m_Streams[i] = new FFReadStream(this, i, m_pContext->streams[i]);
    }

    //m_pContext->iformat->read_header(m_pContext, &avfp);
    //printf("samprate %d\nchn %d\nframerate %d\nwidth %d\nheigth %d\n",
    //	     avfp.sample_rate, avfp.channels, avfp.frame_rate, avfp.width, avfp.height);
    return 0;
}

size_t FFReadHandler::GetHeader(void* header, size_t size)
{
    if (header)//&& (size >= sizeof(AVIMainHeader)))
    {
	memset(header, 0, size);
	//memcpy(pheader, &m_MainHeader, sizeof(AVIMainHeader));
    }
    return 0;//sizeof(AVIMainHeader);
}

IMediaReadStream* FFReadHandler::GetStream(uint_t stream_id,
					   IStream::StreamType type)
{
    int t;
    uint_t cnt = 0;
    switch (type)
    {
    case IStream::Audio: t = CODEC_TYPE_AUDIO; break;
    case IStream::Video: t = CODEC_TYPE_VIDEO; break;
    default: return 0;
    }

    for (unsigned i = 0; i < m_pContext->nb_streams; ++i)
    {
	if (m_pContext->streams[i]->codec->codec_type == t)
	{
	    if (cnt == stream_id)
		return m_Streams[i];
	    cnt++;
	}
    }

    return 0;
}

size_t FFReadHandler::GetStreamCount(IStream::StreamType type)
{
    int t;
    size_t cnt = 0;

    switch (type)
    {
    case IStream::Audio: t = CODEC_TYPE_AUDIO; break;
    case IStream::Video: t = CODEC_TYPE_VIDEO; break;
    default: return 0;
    }

    for (unsigned i = 0; i < m_pContext->nb_streams; ++i)
	if (m_pContext->streams[i]->codec->codec_type == t)
	    cnt++;
    return cnt;
}

void FFReadHandler::flush()
{
    for (size_t i = 0; i < m_Streams.size(); ++i)
    {
	while (m_Streams[i]->m_Packets.size())
	{
	    m_Streams[i]->m_Packets.front()->Release();
	    m_Streams[i]->m_Packets.pop();
	}
	m_Streams[i]->m_uiPosition = 0;
    }
}

//int FFReadHandler::seek(framepos_t pos)
int FFReadHandler::seek(double timepos)
{
    Locker locker(m_Mutex);
    flush();
    //AVM_WRITE("FF reader", 1, "seek(%f)   %" PRId64 "\n", timepos, m_pContext->start_time);
    int64_t sp = ((m_pContext->start_time != (int64_t)AV_NOPTS_VALUE)
		  ? m_pContext->start_time : 0) + int64_t(timepos * AV_TIME_BASE);
#if 1
    av_seek_frame(m_pContext, -1, sp, 0);
#else
#ifndef INT64_MIN
#define INT64_MIN (-0x7fffffffffffffffLL-1)
#endif
#ifndef INT64_MAX
#define INT64_MAX INT64_C(9223372036854775807)
#endif
    avformat_seek_file(m_pContext, -1, INT64_MIN, sp, INT64_MAX, 0);
#endif
    return 0;
}

int FFReadHandler::readPacket()
{
    Locker locker(m_Mutex);
    AVStreamPacket* p = new AVStreamPacket();
    AVPacket& pkt = p->GetAVPacket();
    AVM_WRITE("FF reader", 5, "readPacket()\n");
    if (av_read_frame(m_pContext, &pkt) < 0)
    {
        p->Release();
	if (url_ferror(m_pContext->pb))
	    AVM_WRITE("FF reader", "error seen\n");

	//if (!url_feof(m_pContext->pb))
	//    AVM_WRITE("FF reader", "ffmpeg packet error and not eof??\n");

	return -1;
    }

    //printf("FFMPEG pktsize: %u %" PRId64 "   %d\n", pkt.size, pkt.pts, pkt.stream_index);fflush(stdout);

    p->UpdateMembers();

    if ((size_t)pkt.stream_index >= m_Streams.size()) {
	AVM_WRITE("FF reader", "new stream??  i:%d  n:%d  sz:%" PRIsz "\n",
		  pkt.stream_index, m_pContext->nb_streams, m_Streams.size());
        p->Release();
	return 0;
    }

    FFReadStream& s = *m_Streams[pkt.stream_index];
    AVStream& ast = *m_pContext->streams[pkt.stream_index];

    p->SetPos(s.m_uiPosition);
    p->SetTimestamp(((pkt.pts != (int64_t)AV_NOPTS_VALUE) ? pkt.pts : pkt.dts)
		    * AV_TIME_BASE * ast.time_base.num / ast.time_base.den - s.GetStartTime());
    if (pkt.flags & PKT_FLAG_KEY)
	p->SetFlags(p->GetFlags() | KEYFRAME);

    if (p->GetTimestamp() < 0)
        p->SetTimestamp(0);

    //AVM_WRITE("FF reader", "st:%d n/d:%d/%d  ts:%" PRId64 " d:%" PRId64 " dur:%d  p:%" PRId64 "\n",
    //          pkt.stream_index, ast.time_base.num, ast.time_base.den,
    //          p->timestamp, pkt.dts, pkt.duration, pkt.pts);
    //if (ast->codec.codec_type == CODEC_TYPE_VIDEO) printf("FRATE %d pts:%lld %d  %d  t:%lld\n", p->position, pkt.pts,ast->codec.frame_rate_base, ast->codec.frame_rate, p->timestamp);
    //else printf("Bitrate  %d\n", ast->codec.bit_rate);
    //printf("TIMESTAMP %" PRId64 "   pts: %" PRId64  "  dts: %" PRId64 "\n", p->timestamp, pkt.pts,  pkt.dts);

    switch (ast.codec->codec_type)
    {
    case CODEC_TYPE_AUDIO:
	s.m_uiPosition += pkt.size;
	break;
    case CODEC_TYPE_VIDEO:
    default:
	s.m_uiPosition++;
	break;
    }
#if 0
    printf("---ReadPacket i:%d	 sz:%d	 pts:%" PRId64 "  size:%d   (%" PRId64 ",  %d)\n",
	   pkt.stream_index, s.m_Packets.size(), pkt.pts, pkt.size, p->timestamp, pkt.flags);
#endif

    if (s.m_Packets.size() >= s.m_Packets.capacity() - 1)
    {
	if (1)
	    AVM_WRITE("FF reader", "got too many packets in the queue %d??? (%" PRIsz ", %" PRIsz ")\n",
		      pkt.stream_index, s.m_Packets.size(), s.m_Packets.capacity());
	s.m_Packets.front()->Release();
	s.m_Packets.pop();
    }

    s.m_Packets.push(p);
    return 0;
}

IMediaReadHandler* CreateFFReadHandler(const char *pszFile)
{
    FFReadHandler* h = new FFReadHandler();
    if (h && h->Init(pszFile) == 0)
	return h;

    delete h;
    return 0;
}

AVM_END_NAMESPACE;
