/*
 * Copyright (C) 2005, 2006, 2007 Junjiro Okajima
 *
 * This program, aufs is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/* $Id: wkq.h,v 1.16 2007/10/15 01:29:36 sfjro Exp $ */

#ifndef __AUFS_WKQ_H__
#define __AUFS_WKQ_H__

#ifdef __KERNEL__

#include <linux/sched.h>
#include <linux/version.h>
#include <linux/workqueue.h>

#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 18)
#define DECLARE_COMPLETION_ONSTACK(work) DECLARE_COMPLETION(work)
#endif

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 20)
#define AuInitWkq(wk, func)	INIT_WORK(wk, func)
#define AuWkqFunc(name, arg)	void name(struct work_struct *arg)
#else
#define AuInitWkq(wk, func)	INIT_WORK(wk, func, wk)
#define AuWkqFunc(name, arg)	void name(void *arg)
#endif

/* ---------------------------------------------------------------------- */

/* internal workqueue named AUFS_WKQ_NAME */
struct au_wkq {
	struct workqueue_struct *q;

	/* accounting */
	atomic_t busy;
	unsigned int max_busy;
};

/* wait for 'nowait' tasks in system-wide workqueue */
struct au_nowait_tasks {
#ifdef CONFIG_AUFS_HINOTIFY
//currently, the 'nowait' task which should be waited for is only hinotify.
	atomic_t		nw_len;
	struct completion	nw_comp;
#endif
};

/* ---------------------------------------------------------------------- */

extern struct au_wkq *au_wkq;
typedef void (*au_wkq_func_t)(void *args);
struct au_wkq_flags {
	unsigned int dlgt:1;
	unsigned int wait:1;
};
int au_wkq_run(au_wkq_func_t func, void *args, struct super_block *sb,
	       struct au_wkq_flags *flags);
int __init au_wkq_init(void);
void au_wkq_fin(void);

/* ---------------------------------------------------------------------- */

static inline int au_test_wkq(struct task_struct *tsk)
{
	return (!tsk->mm && !strcmp(current->comm, AUFS_WKQ_NAME));
#if 0 // multithread workqueue
	return (!tsk->mm
		&& !memcmp(current->comm, AUFS_WKQ_NAME "/",
			   sizeof(AUFS_WKQ_NAME)));
#endif
}

static inline int au_wkq_wait(au_wkq_func_t func, void *args, int dlgt)
{
	struct au_wkq_flags flags = {
		.dlgt	= !!dlgt,
		.wait	= 1
	};
	return au_wkq_run(func, args, /*sb*/NULL, &flags);
}

static inline int au_wkq_nowait(au_wkq_func_t func, void *args,
				struct super_block *sb, int dlgt)
{
	struct au_wkq_flags flags = {
		.dlgt	= !!dlgt,
		.wait	= 0
	};
	return au_wkq_run(func, args, sb, &flags);
}

#ifdef CONFIG_AUFS_HINOTIFY
static inline void au_nwt_init(struct au_nowait_tasks *nwt)
{
	atomic_set(&nwt->nw_len, 0);
	smp_mb(); /* atomic_set */
	init_completion(&nwt->nw_comp);
}

static inline int au_nwt_inc(struct au_nowait_tasks *nwt)
{
	return atomic_inc_return(&nwt->nw_len);
}

static inline int au_nwt_dec(struct au_nowait_tasks *nwt)
{
	int ret = atomic_dec_return(&nwt->nw_len);
	if (!ret)
		complete_all(&nwt->nw_comp);
	return ret;
}

static inline int au_nwt_flush(struct au_nowait_tasks *nwt)
{
	if (unlikely(atomic_read(&nwt->nw_len))) {
		//return wait_for_completion_interupptible(&nwt->nw_comp);
		wait_for_completion(&nwt->nw_comp);
	}
	return 0;
}
#else
static inline void au_nwt_init(struct au_nowait_tasks *nwt)
{
	/* nothing */
}

static inline int au_nwt_inc(struct au_nowait_tasks *nwt)
{
	return 0;
}

static inline int au_nwt_dec(struct au_nowait_tasks *nwt)
{
	return 0;
}

static inline int au_nwt_flush(struct au_nowait_tasks *nwt)
{
	return 0;
}
#endif /* CONFIG_AUFS_HINOTIFY */

#endif /* __KERNEL__ */
#endif /* __AUFS_WKQ_H__ */
