#pragma once
#ifndef	FLOATTEXT_DEFINE
#define	FLOATTEXT_DEFINE

/*
 * atanks - obliterate each other with oversize weapons
 * Copyright (C) 2003  Thomas Hudson
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * */


#include "virtobj.h"
#include "main.h"
#include "environment.h"

#define newmakecol(r,g,b) makecol((int)(r),(int)(g),(int)(b))

class FLOATTEXT: public VIRTUAL_OBJECT
  {
  private:
    // empty ctor, copy-ctor and assign operator are private, so the compiler won't create implicit ones!
    inline FLOATTEXT () { }
    inline FLOATTEXT (FLOATTEXT &sourceText _UNUSED)_UNUSED;
    inline const FLOATTEXT& operator= (const FLOATTEXT &sourceText) { return(sourceText); }

    char	_text[80];
    int	_color;
    int _halfColor; // for shadowd text!

  public:

    /* --- constructor --- */
    inline FLOATTEXT (GLOBALDATA *global, ENVIRONMENT *env, char *text, int xpos, int ypos, int color, alignType alignment):
        VIRTUAL_OBJECT(),_color(color),_halfColor(color)
    {
      initialise ();
      setEnvironment (env);
      x = (double)xpos;
      y = (double)ypos;
      _current.x = (int)x;
      _current.y = (int)y;
      _current.w = 0;
      _current.h = 0;
      set_pos (xpos, ypos);
      if (text)
        set_text (text);
      else
        set_text( (char *)"");
      set_color (color);
      _align = alignment;
      _global = global;

    }

    /* --- destructor --- */
    inline virtual ~FLOATTEXT ()
    {
      if (_env)
        {
          /* To be sure to really remove the full text (might fail due to font differences),
             we "enlarge" the text by 25%: */
          if (_current.w)
            {
              _current.x -= (int)((double)_current.w * 1.125);
              _current.w  = (int)((double)_current.w * 1.250);
            }
          if (_current.h)
            {
              _current.y -= (int)((double)_current.h * 1.125);
              _current.h  = (int)((double)_current.h * 1.250);
            }

          switch (_align)
            {
            case LEFT:
              _env->make_bgupdate (_current.x, _current.y, _current.w, _current.h);
              _env->make_bgupdate (_old.x, _old.y, _old.w, _old.h);
              break;
            case RIGHT:
              _env->make_bgupdate (_current.x - _current.w, _current.y - _current.h, _current.w, _current.h);
              _env->make_bgupdate (_old.x - _old.w, _old.y - _old.h, _old.w, _old.h);
              break;
            case CENTRE:
              _env->make_bgupdate (_current.x - (_current.w / 2), _current.y - (_current.h / 2), _current.w + 2, _current.h + 2);
              _env->make_bgupdate (_old.x - (_old.w / 2), _old.y - (_old.h / 2), _old.w + 2, _old.h + 2);
              break;
            default:
              _env->make_bgupdate (_current.x - _current.w, _current.y - _current.h, 2 * _current.w, 2 * _current.h);
              _env->make_bgupdate (_old.x - _old.w, _old.y - _old.h, 2 * _old.w, 2 * _old.h);
            }
          _env->removeObject (this);
        }
      _global = NULL;
      _env    = NULL;
    }

    /* --- non-inline methods --- */

    /* --- inline methods --- */
    inline void setEnvironment(ENVIRONMENT *env)
    {
      if (!_env || (_env != env))
        {
          _env = env;
          _index = _env->addObject (this);
        }
    }
    inline virtual void		initialise ()
    {
      VIRTUAL_OBJECT::initialise ();
    }

    inline virtual int		applyPhysics ()
    {
      VIRTUAL_OBJECT::applyPhysics ();

      set_pos ((int)x, (int)y);

      age++;
      if ((maxAge != -1) && (age > maxAge))
        destroy = TRUE;

      return (0);
    }

    inline virtual void		draw (BITMAP *dest)
    {
      if (_current.w)
        {
          double dFrontFade		=	1.0;
          double dShadFade		= 0.75;
          int iFrontCol				=	_color;
          int iShadCol				=	_halfColor;
          int iBackCol				=	_color;

          // get Background color:
          if (  (_env->dShadowedText > 0.0) || (_env->dFadingText > 0.0)  )
            switch (_align)
              {
              case LEFT:
                iBackCol = _env->getAvgBgColor(	_current.x,								_current.y,
                                                _current.x + _current.w,	_current.y + _current.h,
                                                xv,												yv);
                break;
              case RIGHT:
                iBackCol = _env->getAvgBgColor(	_current.x - _current.w,	_current.y - _current.h,
                                                _current.x,								_current.y,
                                                xv,												yv);
                break;
              case CENTRE:
                iBackCol = _env->getAvgBgColor(	_current.x - (_current.w / 2),		_current.y - (_current.h / 2),
                                                _current.x + (_current.w / 2) + 2,_current.y + (_current.h / 2) + 2,
                                                xv,																yv);
                break;
              }

          if ((maxAge > 0) && (age >= (maxAge / 2)) && (_env->dFadingText > 0.0))
            {
              double dCalcAge = (double)age - (double)(maxAge / 2);
              double dCalcMax = (double)(maxAge / 2);
              dFrontFade	= 1.0 - (dCalcAge / dCalcMax);
              dShadFade		=	0.75 - (0.75 * (dCalcAge / dCalcMax));
              if (dFrontFade	< 0.0) dFrontFade		= 0.0;
              if (dFrontFade	> 1.0) dFrontFade		= 1.0;
              if (dShadFade < 0.0) dShadFade	= 0.0;
              if (dShadFade >	1.0) dShadFade	= 1.0;
              iFrontCol = newmakecol((getr(iFrontCol) * dFrontFade) + (getr(iBackCol) * (1.0 - dFrontFade)),
                                  (getg(iFrontCol) * dFrontFade) + (getg(iBackCol) * (1.0 - dFrontFade)),
                                  (getb(iFrontCol) * dFrontFade) + (getb(iBackCol) * (1.0 - dFrontFade)));
            }
          if (_env->dShadowedText > 0.0)
            iShadCol = newmakecol(	(getr(iShadCol) * dShadFade) + (getr(iBackCol) * (1.0 - dShadFade)),
                                (getg(iShadCol) * dShadFade) + (getg(iBackCol) * (1.0 - dShadFade)),
                                (getb(iShadCol) * dShadFade) + (getb(iBackCol) * (1.0 - dShadFade)));
          if (_align == LEFT)
            {
              if (_env->dShadowedText > 0.0)
                textout_ex (dest, font, _text,
                            _current.x + 1, _current.y + 1, iShadCol, -1);
              textout_ex (dest, font, _text,
                          _current.x, _current.y, iFrontCol, -1);
            }
          else if (_align == RIGHT)
            {
              if (_env->dShadowedText > 0.0)
                textout_ex (dest, font, _text,
                            _current.x - _current.w + 1,
                            _current.y - _current.h + 1, iShadCol, -1);
              textout_ex (dest, font, _text,
                          _current.x - _current.w,
                          _current.y - _current.h, iFrontCol, -1);
            }
          else
            {
              if (_env->dShadowedText > 0.0)
                textout_centre_ex (dest, font, _text,
                                   _current.x + 1, _current.y + 1, iShadCol, -1);
              textout_centre_ex (dest, font, _text,
                                 _current.x, _current.y, iFrontCol, -1);
            }
        }
    }
    inline void		set_text (char * text)
    {
      strncpy (_text, text, 80);
      if (strlen (text) > 0)
        {
          _current.w = text_length (font, _text) + 3;
          _text[79] = 0;
        }
      else
        _current.w = 0;
      _current.h = text_height (font) + 5;
    }

    inline void		set_pos (int x, int y)
    {
      _current.x = x;
      _current.y = y;
    }

    inline void		set_color (int color)
    {
      int red		=	getr(color);
      int green	=	getg(color);
      int blue	=	getb(color);
      int iAverage = (red + green + blue) / 3;

      _color = color;

      if (!(red & 0xc0) && !(green & 0xc0) && !(blue & 0xc0))
        _halfColor = makecol(red | 0x80, green | 0x80, blue | 0x80); // Color is too dark, shadow should be bright
      else
        _halfColor = makecol(	red		> iAverage?(red		/ 4):(red		/ 6),
                              green	> iAverage?(green	/ 4):(green	/ 6),
                              blue	> iAverage?(blue	/ 4):(blue	/ 6));
    }

    inline virtual int		isSubClass (int classNum)
    {
      if (classNum != FLOATTEXT_CLASS)
        return (FALSE);
      //return (VIRTUAL_OBJECT::isSubClass (classNum));
      else
        return (TRUE);
    }

    inline virtual int		getClass ()
    {
      return (FLOATTEXT_CLASS);
    }

    inline void            newRound()
    {
      age = maxAge + 1;
    }
  };

#endif
