// This file is part of PUMA.
// Copyright (C) 1999-2003  The PUMA developer team.
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

#ifndef __ACTree_h__
#define __ACTree_h__

#include "Puma/CTree.h"

/** \file
 *  AspectC++ specific syntax tree classes. */

namespace Puma {


/** \class CT_AdviceDecl ACTree.h Puma/ACTree.h
 *  Tree node representing an advice declaration. 
 *  Example: 
 *  \code 
 * advice "% main(...)" : before() { 
 *   printf('init'); 
 * } 
 *  \endcode */
class CT_AdviceDecl : public CT_Decl {
  CTree *sons[4]; // advice, pointcut, colon, decl

public:
  /** Constructor.
   *  \param a The keyword 'advice'.
   *  \param p The pointcut expression.
   *  \param c The colon before the advice declaration.
   *  \param d The advice declaration. */
  CT_AdviceDecl (CTree *a, CTree *p, CTree *c, CTree *d) {
    AddSon (sons[0], a); AddSon (sons[1], p);
    AddSon (sons[2], c); AddSon (sons[3], d);
  }
  /** Get the identifier for this node type. Can be compared with NodeName(). */
  static const char *NodeId ();
  /** Get the name of the node. Can be compared with NodeId(). */
  const char *NodeName () const { return NodeId (); }
  /** Get the number of sons. */
  int Sons () const { return 4; }
  /** Get the n-th son.
   *  \param n The index of the son.
   *  \return The n-th son or NULL. */
  CTree *Son (int n) const { return CTree::Son (sons, 4, n); }
  /** Replace a son.
   *  \param old_son The son to replace.
   *  \param new_son The new son. */
  void ReplaceSon (CTree *old_son, CTree *new_son) { 
    CTree::ReplaceSon (sons, 4, old_son, new_son);
  }
  /** Get the pointcut expression. */
  CTree *Pointcut () const { return sons[1]; }
  /** Get the advice declaration. */
  CTree *Decl () const { return sons[3]; }
};

/** \class CT_OrderList ACTree.h Puma/ACTree.h
 *  Tree node representing an order list. 
 *  Example: \code ( "pointcut1", "pointcut2" ) \endcode */
class CT_OrderList : public CT_List {
public:
  /** Constructor. */
  CT_OrderList () { AddProperties (SEPARATORS | OPEN_CLOSE); }
  /** Get the identifier for this node type. Can be compared with NodeName(). */
  static const char *NodeId ();
  /** Get the name of the node. Can be compared with NodeId(). */
  const char *NodeName () const { return NodeId (); }
};

/** \class CT_OrderDecl ACTree.h Puma/ACTree.h
 *  Tree node representing an order declaration. 
 *  Example: \code order("pointcut1","pointcut2") \endcode */
class CT_OrderDecl : public CTree {
  CTree *sons[3]; // order, order_list, semi_colon

public:
  /** Constructor.
   *  \param o The keyword 'order'.
   *  \param ol The list of pointcut expressions. 
   *  \param s The trailing semi-colon. */
  CT_OrderDecl (CTree *o, CTree *ol, CTree *s) {
    AddSon (sons[0], o); AddSon (sons[1], ol); AddSon (sons[2], s);
  }
  /** Get the identifier for this node type. Can be compared with NodeName(). */
  static const char *NodeId ();
  /** Get the name of the node. Can be compared with NodeId(). */
  const char *NodeName () const { return NodeId (); }
  /** Get the number of sons. */
  int Sons () const { return 3; }
  /** Get the n-th son.
   *  \param n The index of the son.
   *  \return The n-th son or NULL. */
  CTree *Son (int n) const { return CTree::Son (sons, 3, n); }
  /** Replace a son.
   *  \param old_son The son to replace.
   *  \param new_son The new son. */
  void ReplaceSon (CTree *old_son, CTree *new_son) { 
    CTree::ReplaceSon (sons, 3, old_son, new_son);
  }
  /** Get the list of pointcut expressions. */
  CT_OrderList *OrderList () const { return (CT_OrderList*)sons[1]; }
};

/** \class CT_PointcutDecl ACTree.h Puma/ACTree.h
 *  Tree node representing a pointcut declaration. 
 *  Example: \code pointcut main() = "% main(...)"; \endcode */
class CT_PointcutDecl : public CT_Decl {
  CTree *_pointcut;    // CT_Token
  CTree *_decl;

public:
  /** Constructor.
   *  \param p The keyword 'pointcut'.
   *  \param d The pointcut declaration. */
  CT_PointcutDecl (CTree *p, CTree *d) {
    AddSon (_pointcut, p); AddSon (_decl, d);
  }
  /** Get the identifier for this node type. Can be compared with NodeName(). */
  static const char *NodeId ();
  /** Get the name of the node. Can be compared with NodeId(). */
  const char *NodeName () const { return NodeId (); }
  /** Get the number of sons. */
  int Sons () const { return 2; }
  /** Get the n-th son.
   *  \param n The index of the son.
   *  \return The n-th son or NULL. */
  CTree *Son (int n) const { 
    switch (n) { 
      case 0: return _pointcut;
      case 1: return _decl;
      default: return (CTree*)0;
    }
  }
  /** Replace a son.
   *  \param old_son The son to replace.
   *  \param new_son The new son. */
  void ReplaceSon (CTree *old_son, CTree *new_son) { 
    if (old_son == _decl)
      CTree::ReplaceSon (_decl, old_son, new_son); 
    else if (old_son == _pointcut)
      CTree::ReplaceSon (_pointcut, old_son, new_son); 
  }
  /** Get the pointcut declaration. */
  CTree *Decl () const { return _decl; }
};

/** \class CT_Intro ACTree.h Puma/ACTree.h
 *  Tree node representing an introduction advice declaration.
 *  Example: \code around() \endcode */
class CT_Intro : public CT_List, public CSemScope {
  // indices of aspect or slice names in introduction
  Array<int> _name_indices;    // start index
  Array<int> _name_to_indices; // end index
  Array<bool> _name_qual;      // true if the token should be replaced by a qualified name

public:
  /** Get the identifier for this node type. Can be compared with NodeName(). */
  static const char *NodeId ();
  /** Get the name of the node. Can be compared with NodeId(). */
  const char *NodeName () const { return NodeId (); }
  /** Add a name index. 
   *  \param index The name index. */
  void AddNameIndex (int index) {
    _name_indices.append (index);
    _name_to_indices.append (index);
    _name_qual.append (false);
  }
  /** Add a name index.
   *  \param index_from Start index.
   *  \param index_to End index. */
  void AddNameIndex (int index_from, int index_to) {
    _name_indices.append (index_from);
    _name_to_indices.append (index_to);
    _name_qual.append (true);
  }
  /** Roll back the name index to the given position. 
   *  \param pos The position up to which to roll back. */
  void RollbackNameIndex (int pos) {
    for (int i = NameIndices () - 1; i >= 0; i--) {
      if (NameIndex (i) >= pos) {
        _name_indices.remove (i);
        _name_to_indices.remove (i);
        _name_qual.remove (i);
      }
      else
        break;
    }
  }
  /** Get the name indices. */
  int NameIndices () const { return _name_indices.length (); }
  /** Get the start index of the name with the given index.
   *  \param i The index. */
  int NameIndex (int i) const { return _name_indices.lookup (i); }
  /** Get the end index for the name with the given index.
   *  \param i The index. */
  int NameToIndex (int i) const { return _name_to_indices.lookup (i); }
  /** Check if the name at the given index should 
   *  be replaced by a qualified name.
   *  \param i The index. */
  bool NameQual (int i) const { return _name_qual.lookup (i); }
};

/** \class CT_ClassSliceDecl ACTree.h Puma/ACTree.h
 *  Tree node representing a slice declaration for a class. 
 *  Example: 
 *  \code 
 * slice class X : Y { 
 *   int x; 
 * }; 
 *  \endcode */
class CT_ClassSliceDecl : public CTree, public CSemObject {
  CTree *sons[6]; // SLICE? <key>? <name>? <baseclasses>? <members> ;
  
public:
  /** Constructor.
   *  \param sl The keyword 'slice'.
   *  \param k The keyword 'class' or 'struct'.
   *  \param n The name of the class.
   *  \param b The base class list.
   *  \param m The class member declarations list.
   *  \param se The trailing semi-colon. */
  CT_ClassSliceDecl (CTree *sl, CTree *k, CTree *n, CTree *b, CTree *m, CTree *se) {
    AddSon (sons[0], sl); AddSon (sons[1], k); AddSon (sons[2], n);
    AddSon (sons[3], b); AddSon (sons[4], m); AddSon (sons[5], se);
  }
  /** Get the identifier for this node type. Can be compared with NodeName(). */
  static const char *NodeId ();
  /** Get the name of the node. Can be compared with NodeId(). */
  const char *NodeName () const { return NodeId (); }
  /** Get the number of sons. */
  int Sons () const { return CTree::Sons (sons, 6); }
  /** Get the n-th son.
   *  \param n The index of the son.
   *  \return The n-th son or NULL. */
  CTree *Son (int n) const { return CTree::Son (sons, 6, n); }
  /** Replace a son.
   *  \param old_son The son to replace.
   *  \param new_son The new son. */
  void ReplaceSon (CTree *old_son, CTree *new_son) { 
    CTree::ReplaceSon (sons, 6, old_son, new_son);
  }
  /** Get the class keyword, i.e. 'class' or 'struct'. */
  CT_Token *key () const { return (CT_Token*)sons[1]; }
  /** Get the name of the class. */
  CT_SimpleName *name () const { return (CT_SimpleName*)sons[2]; }
  /** Get the base class list. */
  CT_Intro *base_clause () const { return (CT_Intro*)sons[3]; }
  /** Get the class member declarations list. */
  CT_Intro *members () const { return (CT_Intro*)sons[4]; }
};

/** \class CT_SliceRef ACTree.h Puma/ACTree.h
 *  Tree node representing a slice reference. 
 *  Example: \code slice X; \endcode */
class CT_SliceRef : public CTree {
  CTree *sons[3];
  
public:
  /** Constructor.
   *  \param sl The keyword 'slice'.
   *  \param n The name of the slice.
   *  \param se The trailing semi-colon. */
  CT_SliceRef (CTree *sl, CTree *n, CTree *se) {
    AddSon (sons[0], sl); AddSon (sons[1], n); AddSon (sons[2], se);
  }
  /** Get the identifier for this node type. Can be compared with NodeName(). */
  static const char *NodeId ();
  /** Get the name of the node. Can be compared with NodeId(). */
  const char *NodeName () const { return NodeId (); }
  /** Get the number of sons. */
  int Sons () const { return 3; }
  /** Get the n-th son.
   *  \param n The index of the son.
   *  \return The n-th son or NULL. */
  CTree *Son (int n) const { return CTree::Son (sons, 3, n); }
  /** Replace a son.
   *  \param old_son The son to replace.
   *  \param new_son The new son. */
  void ReplaceSon (CTree *old_son, CTree *new_son) { 
    CTree::ReplaceSon (sons, 3, old_son, new_son);
  }
  /** Get the name of the slice. */
  CT_SimpleName *name () const { return (CT_SimpleName*)sons[1]; }
};


} // namespace Puma

#endif /* __ACTree_h__ */
