// This file is part of PUMA.
// Copyright (C) 1999-2003  The PUMA developer team.
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

#include "Puma/InstantiationCandidate.h"
#include "Puma/CStructure.h"
#include "Puma/CClassInstance.h"
#include "Puma/CUnionInstance.h"
#include "Puma/CFctInstance.h"
#include "Puma/CNamespaceInfo.h"
#include "Puma/CTemplateInfo.h"
#include "Puma/CTemplateParamInfo.h"
#include "Puma/CTree.h"
#include "Puma/CCOverloading.h"
#include "Puma/CConstant.h"
#include "Puma/CCConversions.h"
#include "Puma/CUnit.h"
#include "Puma/TemplateInstanceUnit.h"
#include "Puma/CCParser.h"
#include "Puma/TokenStream.h"
#include "Puma/UnitManager.h"
#include "Puma/CSemDatabase.h"
#include "Puma/PreMacroManager.h"
#include "Puma/CTranslationUnit.h"
#include "Puma/PreprocessorParser.h"
#include "Puma/CFileInfo.h"
#include "Puma/CSourceInfo.h"
#include "Puma/CCConversions.h"
#include "Puma/CCSemExpr.h"
#include "Puma/CAttributeInfo.h"
#include <sstream>
#include <string.h>

namespace Puma {


#define SEM_ERROR(loc__,mesg__) \
  { \
    if (err) { \
      (*err) << sev_error << loc__->token ()->location () \
             << mesg__ << endMessage; \
    } \
  }


extern int TRACE_INSTANCE_CODE;
extern int TRACE_PARSE_INSTANCE;


InstantiationCandidate::InstantiationCandidate () {
  obj_info   = 0;
  tpl_info   = 0;
  poi        = 0;
  err        = 0;
  trans_unit = 0;
  inst_scope = 0;
  instance   = 0;
}


InstantiationCandidate::~InstantiationCandidate () {
  for (long i = 0; i < darguments.length (); i++)
    if (darguments[i])
      delete darguments[i];
  if (trans_unit) {
    delete trans_unit;
  }
}


void InstantiationCandidate::reset () {
  obj_info   = 0;
  tpl_info   = 0;
  poi        = 0;
  err        = 0;
  inst_scope = 0;

  if (trans_unit) {
    delete trans_unit;
    trans_unit = 0;
  }

  for (long i = 0; i < darguments.length (); i++)
    if (darguments[i])
      delete darguments[i];

  arguments.reset ();
  darguments.reset ();
  iarguments.reset ();
}


void InstantiationCandidate::initialize (ErrorSink *e) {
  err = e;
}


void InstantiationCandidate::initialize (CTree *p, CObjectInfo *o) {
  poi      = p;
  obj_info = o;
}


void InstantiationCandidate::initialize (CTree *p, CObjectInfo *o, CTemplateInfo *t, ErrorSink *e) {
  poi      = p;
  obj_info = o;
  tpl_info = t;
  err      = e ? e : err;
}


int InstantiationCandidate::getPosition (CTemplateParamInfo *param) const {
  for (unsigned i = 0; i < tpl_info->Parameters (); i++) 
    if (*param == *tpl_info->Parameter (i))
      return i;
  return -1;
}


CProject* InstantiationCandidate::getProject() {
  return obj_info->SemDB()->Project();
}


CTemplateParamInfo* InstantiationCandidate::getMatchingParameter (CTemplateParamInfo *param) {
  if (tpl_info && param && param->TemplateInfo () != tpl_info)
    param = tpl_info->Parameter (param->getPosition ());
  return param;
}


bool InstantiationCandidate::canBeInstantiated () {
  // template instantiation disabled?
  if (! getProject()->config().Option("--real-instances"))
    return false;
  // point of instantiation depends on template parameter?
  CTree* name = poi;
  if (name->NodeName () == CT_QualName::NodeId () ||
      name->NodeName () == CT_RootQualName::NodeId ())
    name = ((CT_QualName*)name)->Name ();
  if (name->NodeName () == CT_TemplateName::NodeId ())
    if (CCSemExpr::isDependent(((CT_TemplateName*)name)->Arguments ()))
      return false;
  // everything OK
  return true;
}


// §14.8.2 deduce template arguments (if necessary)
bool InstantiationCandidate::deduceArguments (bool real_inst) {
  // check if template really can be instantiated
  if (real_inst && ! canBeInstantiated())
    real_inst = false;

  CTree* name = poi;
  if (name->NodeName () == CT_QualName::NodeId () ||
      name->NodeName () == CT_RootQualName::NodeId ())
    name = ((CT_QualName*)name)->Name ();

  // template arguments
  CT_TemplateArgList *args;
  unsigned numargs;
  if (name->NodeName () == CT_TemplateName::NodeId ()) {
    args = ((CT_TemplateName*)name)->Arguments ();
    numargs = args->Entries ();
  } else {
    args = 0;
    numargs = 0;
  }

  // get template parameter list information
  if (obj_info->TemplateParamInfo ())
    tpl_info = obj_info->TemplateParamInfo ()->TemplateTemplate ();
  else
    tpl_info = obj_info->Template ();
  if (! tpl_info) {
    if (real_inst) {
      SEM_ERROR (poi, "fatal error: missing template parameter information");
      return false;
    }
    return true;
  }

  // check number of template arguments for function templates
  if (obj_info->FunctionInfo () && numargs > tpl_info->Parameters ()) {
    if (real_inst) {
      SEM_ERROR (poi, "wrong number of template arguments ("
        << numargs << ", should be " << tpl_info->Parameters () << ")");
      return false;
    }
    return true;
  }

  // calculate index of first parameter for which no direct template
  // argument or default template argument is given
  unsigned pos = 0;
  for (; pos < tpl_info->Parameters(); pos++) {
    CTemplateParamInfo *param = tpl_info->Parameter(pos);
    if (pos >= numargs) {
      // no direct template argument for the current parameter,
      // check default template arguments for non-function templates
      if (! obj_info->FunctionInfo()) {
        // not a function template, check default argument
        if (! param->DefaultArgument()) {
          // no default argument
          if (real_inst) {
            SEM_ERROR(poi, "missing default argument for parameter " << pos+1);
            return false;
          }
          return true;
        }
      } else {
        // function template parameters do not have default arguments,
        // ommited arguments must be deduced from the function call
        // argument list
        break;
      }
    }
  }

  // deduce the direct and default arguments
  if (! (real_inst ? parseDirectArguments(pos, numargs, args) :
                     matchDirectArguments(pos, numargs, args))) {
    return false;
  }

  if (pos < tpl_info->Parameters()) {
    // deduce remaining arguments from function call
    return deduceArgumentsFromFctCall(pos);
  }
  return true;
}


bool InstantiationCandidate::matchDirectArguments (unsigned pos, unsigned numargs, CT_TemplateArgList *args) {
  // match direct and default template arguments
  for (unsigned i = 0; i < pos; i++) {
    CTemplateParamInfo *param = tpl_info->Parameter (i);
    bool is_default_arg = (i >= numargs);
    CObjectInfo* oinfo = 0;
    CTree *arg;

    if (is_default_arg) {
      // default template argument
      arg = param->DefaultArgument ()->Entry (0);
    } else {
      // direct template argument
      arg = args->Entry (i);
    }

    // template template parameter: expect name of class base template
    if (param->isTemplate ()) {
      if (arg->SemObject () && (oinfo = arg->SemObject ()->Object ()) && oinfo->TypeInfo ())
        darguments.append (new DeducedArgument (param,
          oinfo->TypeInfo ()->VirtualType (), arg, is_default_arg, !is_default_arg));
    }
    // template type parameter: expect type-id
    else if (param->isTypeParam ()) {
      if (arg->SemObject () && (oinfo = arg->SemObject ()->Object ()) && oinfo->TypeInfo ())
        darguments.append (new DeducedArgument (param,
          oinfo->TypeInfo ()->VirtualType (), arg, is_default_arg, !is_default_arg));
    }
    // template non-type parameter: expect constant expression
    else {
      if (arg->Type () && arg->Type ()->TypeTemplateParam () &&
          arg->Type ()->TypeTemplateParam ()->isNonType ()) {
        // template non-type parameter as argument
        darguments.append (new DeducedArgument (param,
          arg->Type (), arg, is_default_arg, !is_default_arg));
      }
      else if (arg->Value () && arg->Value ()->Constant ()) {
        darguments.append (new DeducedArgument (param,
          arg->Value ()->Constant (), arg, is_default_arg, !is_default_arg));
      }
    }
  }
  return true;
}


// §14.8.2 deduce template arguments from function call
bool InstantiationCandidate::deduceArgumentsFromFctCall (unsigned skip, bool exact_match) {
  DeducedArgumentList dargs (tpl_info->Parameters ());
  CTypeInfo *atype, *ptype;
  CFunctionInfo *finfo;
  CTypeList *fargs;
  unsigned numargs;
  bool succeeded;
  CTree *arg;

  succeeded = true;
  finfo = obj_info->FunctionInfo ();
  fargs = finfo->TypeInfo ()->ArgTypes ();
  numargs = Arguments ();

  // compare number of function arguments and parameters
  if (fargs->Entries () > numargs) {
    numargs = fargs->Entries ();
  } else if (fargs->Entries () < numargs) {
    SEM_ERROR (poi, "too many arguments in call to function `"
      << finfo->Name () << "'");
    return false;
  }

  // prepare temporary deduced argument container
  for (unsigned i = 0; i < tpl_info->Parameters (); i++)
    dargs.append (0);

  // iterate and analyse arguments
  for (unsigned i = 0; i < numargs && succeeded; i++) {
    // consider default arguments for function parameters
    if (i >= Arguments ()) {
      arg = finfo->DefaultArgument (i);
      if (! arg) {
        SEM_ERROR (poi, "too few arguments in call to function `"
          << finfo->Name () << "'");
        return false;
      }
    } else
      arg = Argument (i);

    atype = arg->Type () ? arg->Type () : arg->SemObject ()->Object ()->TypeInfo ();
    ptype = fargs->Entry (i);

    if (ptype->isAddress ()) {
      while (ptype->isAddress ())
        ptype = ptype->BaseType ();
      while (atype->isAddress ())
        atype = atype->BaseType ();
    } else {
      // §14.3.2.5 implicit argument type conversions
      atype = atype->VirtualType ();
      ptype = ptype->VirtualType ();
      // array-to-pointer and function-to-pointer conversions
      if (atype->isArray () || atype->isFunction ()) {
        atype = atype->isArray () ? atype->VirtualType ()->BaseType () : atype;
        atype = new CTypePointer (atype->Duplicate ());
      }
    }

    if (deduceArguments (ptype, atype, dargs, exact_match) == 1) {
      succeeded = false; // ambiguous type for parameter
    }
  }

  for (unsigned i = skip; i < (unsigned)dargs.length () && succeeded; i++) {
    if (! dargs[i]) {
      SEM_ERROR (poi, "could not deduce argument for parameter " << i+1);
      succeeded = false;
    } else
      darguments.append (dargs[i]);
  }

  return succeeded;
}


// return 0 if all ok, 1 if ambiguous type for parameter, 2 if types don't match
int InstantiationCandidate::deduceArguments (CTypeInfo *ftype, CTypeInfo *atype, DeducedArgumentList &dargs, bool exact_match) {
  int ret;
  
  // template parameter
  if (ftype->TypeTemplateParam ()) {
    return deduceTemplateParam (ftype, atype, dargs, exact_match);
  }
  
  // deduce from qualified type
  if (ftype->TypeQualified ()) {
    return deduceFromQualifiedType (ftype, atype, dargs, exact_match);
  }
  
  // types must be equal to deduce template arguments
  if (ftype->Id () != atype->Id ()) { 
    return exact_match ? 2 : 0;
  }
  
  // nothing to deduce
  if (ftype->TypeEmpty () || ftype->TypePrimitive ()) {
    return 0;
  }
  
  // deduce from pointer to class member
  if (ftype->TypeMemberPointer ()) {
    return deduceFromMemberPointer (ftype, atype, dargs, exact_match);

  // deduce from class or union type
  } else if (ftype->TypeRecord ()) {
    return deduceFromRecord (ftype, atype, dargs, exact_match);

  // deduce from array type
  } else if (ftype->TypeArray ()) {
    if ((ret = deduceFromArray (ftype, atype, dargs, exact_match)) != 0) { 
      return ret;
    }

  // deduce from function type
  } else if (ftype->TypeFunction ()) {
    if ((ret = deduceFromFunction (ftype, atype, dargs, exact_match)) != 0) { 
      return ret;
    }
  }
  
  // deduce from the base types, types must match exactly
  return deduceArguments (ftype->BaseType (), atype->BaseType (), dargs, exact_match);
}


// return 0 if all ok, 1 if ambiguous type for parameter, 2 if types don't match
int InstantiationCandidate::deduceFromQualifiedType (CTypeInfo *ftype, CTypeInfo *atype, DeducedArgumentList &dargs, bool exact_match) {
  CTypeQualified* fqt = ftype->TypeQualified ();
  CTypeQualified* aqt = atype->TypeQualified ();

  // exact match required: qualifiers must be the same
  if (exact_match) {
    if (! aqt || fqt->isConst () != aqt->isConst () || fqt->isVolatile () != aqt->isVolatile ()) {
      return 2;
    }
    return deduceArguments (ftype->UnqualType (), atype->UnqualType (), dargs, exact_match);  
  }

  // ordinary parameter type 
  if (! ftype->UnqualType ()->TypeTemplateParam ()) {
    // only matches if equal or more qualified than the argument type
    if (! CCConversions::equalOrMoreQualified (ftype, atype)) {
      return 0;
    }
    
    // deduce from the unqualified versions of the types
    return deduceArguments (ftype->UnqualType (), atype->UnqualType (), dargs, exact_match);  
  }

  // qualified template parameter type, remove all equal qualifiers
  // from the argument type
  
  bool has_const = aqt && aqt->isConst () && ! fqt->isConst ();
  bool has_volatile = aqt && aqt->isVolatile () && ! fqt->isVolatile ();
  
  // remove all qualifiers from argument type
  if (! has_const && ! has_volatile) {
    return deduceArguments (ftype->UnqualType (), atype->UnqualType (), dargs, exact_match);
  } 

  // remove no qualifier from argument type
  else if (! (aqt->isConst () && aqt->isVolatile ())) {
    return deduceArguments (ftype->UnqualType (), atype, dargs, exact_match);
  }

  // remove one qualifier from argument type
  else {
    CTypeQualified tmp(atype->UnqualType (), has_const, has_volatile, aqt && aqt->isRestrict ());
    return deduceArguments (ftype->UnqualType (), &tmp, dargs, exact_match);  
  }
}


// return 0 if all ok, 1 if ambiguous type for parameter, 2 if types don't match
int InstantiationCandidate::deduceFromFunction (CTypeInfo *ftype, CTypeInfo *atype, DeducedArgumentList &dargs, bool exact_match) {
  CTypeFunction *fft = ftype->TypeFunction ();
  CTypeFunction *aft = atype->TypeFunction ();
  
  // number of function parameters must be equal
  if (fft->ArgTypes ()->Entries () == aft->ArgTypes ()->Entries ()) {
    CTypeList *ftl = fft->ArgTypes ();
    CTypeList *atl = aft->ArgTypes ();
    
    bool matching_args = true;
    bool ambiguous_type = false;
    DeducedArgumentList curr_dargs;
        
    // prepare temporary deduced argument container
    for (unsigned i = 0; i < tpl_info->Parameters (); i++) {
      curr_dargs.append (0);
    }
    
    // deduce from the function parameters
    for (unsigned i = 0; i < ftl->Entries () && matching_args && ! ambiguous_type; i++) {
      switch (deduceArguments (ftl->Entry (i), atl->Entry (i), dargs, true)) {
        case 2: matching_args = false; break;
        case 1: ambiguous_type = true; break;
        default: break;
      }
    }
    
    // join the deduced template arguments if possible
    if (! joinDeducedArguments (dargs, curr_dargs, matching_args, ambiguous_type)) {
      return matching_args && ambiguous_type ? 1 : 
             exact_match && ! matching_args ? 2 : 0;
    }

  // different number of function parameters
  } else if (exact_match) {
    return 2;
  }

  return 0; 
}


bool InstantiationCandidate::joinDeducedArguments (DeducedArgumentList &dargs, DeducedArgumentList &curr_dargs, bool &matching_args, bool &ambiguous_type) {
  // check if the current deduced template arguments are valid
  if (! matching_args || ambiguous_type) {
    // discard the deduced template arguments
    for (long i = 0; i < curr_dargs.length (); i++) {
      DeducedArgument *darg = curr_dargs.fetch (i);
      if (darg) delete darg;
    }
  } 
  // if not ambiguous, add the deduced arguments to dargs
  if (matching_args) {
    // join the deduced template arguments
    if (! ambiguous_type) {
      for (long i = 0; i < curr_dargs.length (); i++) {
        DeducedArgument *darg = curr_dargs.fetch (i);
        if (darg && ! setDeducedArgument (dargs, i, darg)) {
          ambiguous_type = true;
        }
      }
    }
  }
  return matching_args && ! ambiguous_type;
}


// return 0 if all ok, 1 if ambiguous type for parameter, 2 if types don't match
int InstantiationCandidate::deduceFromArray (CTypeInfo *ftype, CTypeInfo *atype, DeducedArgumentList &dargs, bool exact_match) {
  CTypeArray *array = ftype->TypeArray ();
  
  // array size depends on a template parameter
  if (array->DepDim ()) {
    CTemplateParamInfo *param = array->DepDim ()->TemplateParamInfo ();
    
    if (param) {
      int pos = getPosition (param);
      
      // non-type template parameter
      if (pos != -1 && ! param->isTypeParam ()) {
        array = atype->TypeArray ();
        
        // array has dimension
        if (array->Dimension ()) { 
          if (! setDeducedArgument (dargs, pos, new DeducedArgument (param, array->Dimension ()))) {
            return 1;
          }
        
        // match non-type template parameter for partial ordering of function templates
        } else if (array->DepDim ()) {     
          CTemplateParamInfo *aparam = array->DepDim ()->TemplateParamInfo ();
          if (aparam && 
              ! aparam->isTypeParam () && 
              ! setDeducedArgument (dargs, pos, new DeducedArgument (param, atype))) {
            return 1;
          }
        }
      }
    }
    
  // ordinary types must be an exact match
  } else if (exact_match && (*ftype != *atype)) {
    return 2;
  }
  
  return 0;
}


// return 0 if all ok, 1 if ambiguous type for parameter, 2 if types don't match
int InstantiationCandidate::deduceTemplateParam (CTypeInfo *ftype, CTypeInfo *atype, DeducedArgumentList &dargs, bool exact_match) {
  CTemplateParamInfo *param = ftype->TypeTemplateParam ()->TemplateParamInfo ();
  int ret, pos = getPosition (param);
  
  if (pos != -1) {
    // template template parameter
    if (param->isTemplate ()) {
      CObjectInfo *ainfo = atype->TypeRecord () ? (CObjectInfo*)atype->TypeRecord ()->Record () : 
                           atype->TypeTemplateParam () ? (CObjectInfo*)atype->TypeTemplateParam ()->TemplateParamInfo () : 0;

      // template parameter list must match
      if (! matchingTemplateParameters (param, ainfo)) {
        return exact_match ? 2 : 0;
      }
      
      // get the template instance info of the argument, if it is an template instance
      CTemplateInstance *ti = ainfo->Record () ? ainfo->TemplateInstance () : ainfo->TemplateParamInfo ()->TemplateInstance ();  
      CTemplateInfo *tpl = getTemplateInfo (ainfo);

      // match template instance
      if (param->TemplateInstance ()) {
        if (! ti) {
          // not a template instance
          return exact_match ? 2 : 0;
        }

        // deduce from the argument list
        ret = deduceArgumentsFromTemplateArgList (param->TemplateInstance (), ti, dargs);
        if (ret != 0) {
          return ret;
        }
        
        // matches the template itself
        if (! setDeducedArgument (dargs, pos, new DeducedArgument (param, tpl->ObjectInfo ()->TypeInfo ()))) {
          return 1;
        } 
      
      // template template parameter does not match template instances
      } else if (ti) {
        return exact_match ? 2 : 0;

      // match template itself (template name)
      } else {
        if (! setDeducedArgument (dargs, pos, new DeducedArgument (param, tpl->ObjectInfo ()->TypeInfo ()))) {
          return 1;
        }
      }

    // type template parameter, matches any type
    } else if (param->isTypeParam ()) {
      if (! setDeducedArgument (dargs, pos, new DeducedArgument (param, atype))) {
        return 1;
      }

    // non-type template parameter, matches any value or non-type template parameter,
    // for partial ordering of function templates
    } else if (! param->isTypeParam ()) {
      if (atype->TypeTemplateParam ()) {
        CTemplateParamInfo *aparam = atype->TypeTemplateParam ()->TemplateParamInfo ();
        if (aparam && 
            ! aparam->isTypeParam () && 
            ! setDeducedArgument (dargs, pos, new DeducedArgument (param, atype))) {
          return 1;
        }
      }
    }
  }
  
  return 0;
}


bool InstantiationCandidate::matchingTemplateParameters (CTemplateParamInfo *param, CObjectInfo *ainfo) {
  if (! ainfo)
    return false;

  CTemplateInfo *atpl = getTemplateInfo (ainfo);
  CTemplateInfo *ftpl = param->TemplateTemplate ();

  if (! ftpl && ! atpl)
    return true;
  if (! ftpl || ! atpl || ftpl->Parameters () != atpl->Parameters ())
    return false;

  for (unsigned i = 0; i < ftpl->Parameters (); i++) {
    CTemplateParamInfo *fparam = ftpl->Parameter (i);
    CTemplateParamInfo *aparam = atpl->Parameter (i);

    if (! fparam->match (*aparam))
      return false;
  }

  return true;
}


CTemplateInfo *InstantiationCandidate::getTemplateInfo (CObjectInfo *info) {
  CTemplateInfo *tpl = 0;
  if (info->TemplateInstance ())
    tpl = info->TemplateInstance ()->Template ();
  else if (info->Record ())
    tpl = info->Record ()->TemplateInfo ();
  else if (info->TemplateParamInfo ())
    tpl = info->TemplateParamInfo ()->TemplateTemplate ();
  return tpl;
}


// return 0 if all ok, 1 if ambiguous type for parameter, 2 if types don't match
int InstantiationCandidate::deduceFromMemberPointer (CTypeInfo *ftype, CTypeInfo *atype, DeducedArgumentList &dargs, bool exact_match) {
  CTemplateParamInfo *param = ftype->TypeMemberPointer ()->TemplateParam ();

  // member pointer of form T::member where T is a type template parameter
  if (param) {
    int pos = getPosition (param);
    if (pos != -1 && param->isTypeParam ()) {

      CRecord *record = atype->TypeMemberPointer ()->Record ();

      // argument type is class or union, can be deduced
      if (record) {
        if (! setDeducedArgument (dargs, pos, new DeducedArgument (param, record->TypeInfo ()))) {
          return 1;
        }

      // argument member pointer type contains template parameter, handled as unique type for 
      // partial ordering of function templates
      } else {
        CTemplateParamInfo *tp = atype->TypeMemberPointer ()->TemplateParam ();
        if (tp && tp->isTypeParam () && ! setDeducedArgument (dargs, pos, new DeducedArgument (param, atype))) {
          return 1;
        }
      }
    }
  
  // ordinary types must match, if not the top level types
  } else if (exact_match && (*ftype != *atype)) {
    return 2;
  }
  
  return 0; 
}


// return 0 if all ok, 1 if ambiguous type for parameter, 2 if types don't match
int InstantiationCandidate::deduceFromRecord (CTypeInfo *ftype, CTypeInfo *atype, DeducedArgumentList &dargs, bool exact_match) {
  CRecord *finfo = ftype->TypeRecord ()->Record ();
  CRecord *ainfo = atype->TypeRecord ()->Record ();
  
  if (finfo && ainfo) {
    // types must have same record type, i.e. either class or union 
    if ((finfo->ClassInfo () && ! ainfo->ClassInfo ()) ||
        (finfo->UnionInfo () && ! ainfo->UnionInfo ())) {
      return exact_match ? 2 : 0;
    }
  
    // parameter type is template instance
    if (finfo->isTemplateInstance ()) {
      // deduce template arguments from the type arguments of the template instances,
      // the template used to instantiate finfo must be equal to the template used
      // to instantiate ainfo or one of its base classes

      CTemplateInstance *fi = finfo->TemplateInstance (); 
      Array<CTemplateInstance*> instances;
      getTemplateInstances (ainfo, instances); // search the class hierarchy of ainfo

      bool at_least_one_match = false;
      for (long j = 0; j < instances.length (); j++) {
        CTemplateInstance *ai = instances.fetch (j);

        // same template and same number of template arguments
        CTemplateInfo *ft = fi->Template ()->BaseTemplate () ? fi->Template ()->BaseTemplate () : fi->Template ();
        CTemplateInfo *at = ai->Template ()->BaseTemplate () ? ai->Template ()->BaseTemplate () : ai->Template ();
        if (*ft != *at || fi->InstantiationArgs () != ai->InstantiationArgs ()) {
          continue;
        }

        // deduce from the template arguments
        int ret = deduceArgumentsFromTemplateArgList (fi, ai, dargs);
        if (ret == 1) {
          return 1;
        }
        at_least_one_match = at_least_one_match || (ret == 0);
      }

      // require exact match
      if (! at_least_one_match) {
        return 2;
      }

    // ordinary types must match, if not the top level types
    } else if (exact_match && (*finfo != *ainfo)) {
      return 2;
    }
  }

  return 0; 
}


// return 0 if all ok, 1 if ambiguous type for parameter, 2 if types don't match
int InstantiationCandidate::deduceArgumentsFromTemplateArgList (CTemplateInstance *fi, CTemplateInstance *ai, DeducedArgumentList &dargs) {
  bool matching_args = true;
  bool ambiguous_type = false;
  DeducedArgumentList curr_dargs;

  // number of template arguments must match
  if (fi->InstantiationArgs () != ai->InstantiationArgs ()) {
    return 2;
  }
  
  // prepare temporary deduced argument container
  for (unsigned i = 0; i < tpl_info->Parameters (); i++) {
    curr_dargs.append (0);
  }

  // now try to deduce from the template arguments, must match exactly 
  // for a successful deduction
  for (unsigned i = 0; i < fi->InstantiationArgs () && matching_args && ! ambiguous_type; i++) {
    CTypeInfo *fdtype = fi->InstantiationArg (i)->Type ();
    CTypeInfo *adtype = ai->InstantiationArg (i)->Type ();
    CConstant *fvalue = fi->InstantiationArg (i)->Value ();
    CConstant *avalue = ai->InstantiationArg (i)->Value ();

    if (fdtype) {
      // type template arguments
      if (adtype) {
        switch (deduceArguments (fdtype, adtype, curr_dargs, true)) {
          case 2: matching_args = false; break;
          case 1: ambiguous_type = true; break;
          default: break;
        }

      // non-type template arguments
      } else if (avalue && 
                 fdtype->TypeTemplateParam () && 
                 ! fdtype->TypeTemplateParam ()->isTypeParam ()) {
        CTemplateParamInfo *param = fdtype->TypeTemplateParam ()->TemplateParamInfo ();
        if (param) {
          int pos = getPosition (param);
          if (pos != -1 && ! setDeducedArgument (curr_dargs, pos, new DeducedArgument (param, avalue))) {
            ambiguous_type = true;
          }
        }

      } else {
        matching_args = false;
      }

    // non-type template arguments
    } else if (fvalue && (! avalue || (*fvalue != *avalue))) {
      matching_args = false;
    }
  }

  // check if deduced template arguments are valid
  if (! matching_args || ambiguous_type) {
    // discard the deduced template arguments
    for (long i = 0; i < curr_dargs.length (); i++) {
      DeducedArgument *darg = curr_dargs.fetch (i);
      if (darg) delete darg;
    }
  } 
  // if not ambiguous, add the deduced arguments to dargs
  if (matching_args) {
    // add the deduced template arguments
    if (! ambiguous_type) {
      for (long i = 0; i < curr_dargs.length (); i++) {
        DeducedArgument *darg = curr_dargs.fetch (i);
        if (darg && ! setDeducedArgument (dargs, i, darg)) {
          ambiguous_type = true;
        }
      }
    }
    if (ambiguous_type) {
      return 1;
    }
    return 0;
  }
  return 2;
}


void InstantiationCandidate::getTemplateInstances(CRecord* c, Array<CTemplateInstance*>& instances) {
  if (c->isTemplateInstance ()) {
    instances.append (c->TemplateInstance ());
  }
  CClassInfo *cl = c->ClassInfo ();
  if (cl) {
    for (unsigned i = 0; i < cl->BaseClasses (); i++) {
      getTemplateInstances (cl->BaseClass (i)->Class (), instances);
    }
  }
}


// §14.5.4.1.2 try to match a partial specialization 
// against a given template argument list
bool InstantiationCandidate::match () {
  CT_TemplateArgList *args;
  CTemplateInfo *base;
  CTree *arg, *parg, *name;
  unsigned numargs;
  
  // template argument list
  name = poi;
  if (name->NodeName () == CT_QualName::NodeId () ||
      name->NodeName () == CT_RootQualName::NodeId ())
    name = ((CT_QualName*)name)->Name ();
  if (name->NodeName () == CT_TemplateName::NodeId ()) {
    args = ((CT_TemplateName*)name)->Arguments ();
    numargs = args->Entries ();
  } else {
    args = 0;
    numargs = 0;
  }
  
  // prepare deduced argument container
  darguments.reset ();
  for (unsigned i = 0; i < tpl_info->Parameters (); i++)
    darguments.append (0);

  // compare argument lists
  base = tpl_info->BaseTemplate ();
  for (unsigned i = 0; i < Arguments (); i++) {
    // current partial specialization argument
    parg = Argument (i); 

    // more partial specialization arguments than arguments
    // given at point of instantiation => consider default arguments
    if (i >= numargs) {
      if (base && base->Parameters () > i && base->Parameter (i)->DefaultArgument ()) {
        arg = base->Parameter (i)->DefaultArgument ()->Entry (0);
      } else {
        // oops, no default argument? this is an error
        return false;
      }
    } else
      arg = args->Entry (i);
      
    // try to match the current argument, must be an exact match!
    if (! matchArgument (parg, arg))
      return false;
  }
  
  // check if all template parameters could be deduced while matching
  for (unsigned i = 0; i < DeducedArgs (); i++)
    if (! DeducedArg (i))
      return false;
  
  return true;
}


bool InstantiationCandidate::sameType (int pos, CTypeInfo *type) {
  DeducedArgument *darg = darguments[pos];
  if (darg && darg->Value ())
    return false; // not a type
  CTypeInfo *otype = darg ? darg->Type () : 0;
  if (! type)
    return false;
  if (otype) {
    if (otype->TypeTemplateParam () && type->TypeTemplateParam ()) {
      CTemplateParamInfo *p1 = otype->TypeTemplateParam ()->TemplateParamInfo ();
      CTemplateParamInfo *p2 = type->TypeTemplateParam ()->TemplateParamInfo ();
      if (! p1 || ! p2 || *p1 != *p2)
        return false;
    } else if (*otype != *type)
      return false;
  }
  return true;
}


bool InstantiationCandidate::sameValue (int pos, CConstant *value) {
  DeducedArgument *darg = darguments[pos];
  if (darg && darg->Type ())
    return false; // not a value
  CConstant *ovalue = darg ? darg->Value () : 0;
  if (! value || (ovalue && (*ovalue != *value)))
    return false;
  return true;
}


bool InstantiationCandidate::matchArgument (CTree *pexpr, CTree *aexpr) {
  CConstant *value;

  // match against value, must be equal
  if (pexpr->Value () && pexpr->Value ()->Constant ()) {
    if (aexpr->Value () && (value = aexpr->Value ()->Constant ()) &&
        (*value == *pexpr->Value ()->Constant ())) 
      return true;
    return false;
  }

  CTypeInfo *ptype = 0, *atype = 0;
  if (pexpr->NodeName () == CT_NamedType::NodeId () &&
      pexpr->SemObject () && pexpr->SemObject ()->Object ())
    ptype = pexpr->SemObject ()->Object ()->TypeInfo ();
  if (aexpr->NodeName () == CT_NamedType::NodeId () &&
      aexpr->SemObject () && aexpr->SemObject ()->Object ())
    atype = aexpr->SemObject ()->Object ()->TypeInfo ();
  if (! ptype)
    ptype = pexpr->Type ();
  if (! atype)
    atype = aexpr->Type ();
  if (! ptype || ! atype)
    return false;

  return matchArgument (ptype, atype, aexpr);
}


bool InstantiationCandidate::matchArgument (CTypeInfo *ptype, CTypeInfo *atype, CTree *aexpr) {
  CTemplateInstance *pti, *ati;
  CTemplateInfo *ptinfo, *atinfo;
  CObjectInfo *pinfo, *ainfo;
  CTemplateParamInfo *param, *param2;
  CTypeList *ptl, *atl;
  CConstant *value, *value2;
  CTypeInfo *type, *type2;
  CRecord *prec, *arec;
  long pdim, adim;
  int pos;

  // match against template parameter
  if (ptype->TypeTemplateParam ()) {
    param = ptype->TypeTemplateParam ()->TemplateParamInfo ();
    pos = getPosition (param);
    if (pos != -1) {
      // template template parameter
      if (param->isTemplate ()) {
        if (atype->TypeTemplateParam ()) {
          if (sameType (pos, atype)) {
            darguments[pos] = new DeducedArgument (param, atype);
            return true;
          }
        } else if (atype->TypeRecord ()) {
          ainfo = atype->TypeRecord ()->Record ();
          type = ainfo->TypeInfo ();
          if (ainfo->isTemplateInstance ()) {
            type = ainfo->TemplateInstance ()->Template ()->ObjectInfo ()->TypeInfo ();
          } 
          // template parameter must always deduce to the same type
          if (sameType (pos, type)) {
            darguments[pos] = new DeducedArgument (param, type);
            return true;
          }
        }
      // type template parameter
      } else if (param->isTypeParam ()) {
        if (sameType (pos, atype)) {
          darguments[pos] = new DeducedArgument (param, atype);
          return true;
        }
      // non-type template parameter
      } else if (atype->TypeTemplateParam ()) {
        if (atype->TypeTemplateParam ()->isNonTypeParam ()) {
          if (sameType (pos, atype)) {
            darguments[pos] = new DeducedArgument (param, atype);
            return true;
          }
        }
      // non-type template parameter
      } else if (aexpr) {
        if (aexpr->Value () && (value = aexpr->Value ()->Constant ()) &&
            sameValue (pos, value)) {
          darguments[pos] = new DeducedArgument (param, value);
          return true;
        }
      }
    } 
    return false;
  } 
  
  // non-template parameter arguments and non-value arguments 
  // must be equal in type
  if (ptype->Id () != atype->Id ())
    return false;

  // match against primitive type, no further analyses needed
  if (ptype->TypeEmpty () || ptype->TypePrimitive ())
    return true;
  
  // match against class or union
  if (ptype->TypeRecord ()) {
    pinfo = ptype->TypeRecord ()->Record ();
    ainfo = atype->TypeRecord ()->Record ();
    if (pinfo && ainfo) {
      if (pinfo->isTemplateInstance () && ainfo->isTemplateInstance ()) {
        pti = pinfo->TemplateInstance (); 
        ati = ainfo->TemplateInstance (); 
        if (pti && ati && pti->DeducedArgs () == ati->DeducedArgs ()) {
          ptinfo = pti->Template ();
          atinfo = ati->Template ();
          if (! ptinfo || ! atinfo)
            return false;
          ptinfo = ptinfo->BaseTemplate () ? ptinfo->BaseTemplate () : ptinfo;
          atinfo = atinfo->BaseTemplate () ? atinfo->BaseTemplate () : atinfo;
          if (*ptinfo != *atinfo)
            return false;
          for (unsigned i = 0; i < pti->DeducedArgs (); i++) {
            type = pti->DeducedArg (i)->Type ();
            type2 = ati->DeducedArg (i)->Type ();
            value = pti->DeducedArg (i)->Value ();
            value2 = ati->DeducedArg (i)->Value ();
            if (type) {
              if (type->TypeTemplateParam () &&
                  type->TypeTemplateParam ()->isNonTypeParam ()) {
                param = type->TypeTemplateParam ()->TemplateParamInfo ();
                if (! param || (pos = getPosition (param)) == -1)
                  return false;
                if (type2 && type2->TypeTemplateParam ()) {
                  if (! type2->TypeTemplateParam ()->isNonTypeParam () ||
                      ! sameType (pos, type2))
                    return false;
                  darguments[pos] = new DeducedArgument (param, type2);
                  continue;
                } else if (! value2 || ! sameValue (pos, value2)) 
                  return false;
                darguments[pos] = new DeducedArgument (param, value2);
              } else if (! type2 || ! matchArgument (type, type2, 0))
                return false;
            } else if (! value || ! value2 || *value != *value2)
              return false;
          }
          return true;
        }
      } else if (*pinfo == *ainfo)
        return true;
    } 
    return false;
  // match against pointer to class member
  } else if (ptype->TypeMemberPointer ()) {
    prec = ptype->TypeMemberPointer ()->Record ();
    arec = atype->TypeMemberPointer ()->Record ();
    param = ptype->TypeMemberPointer ()->TemplateParam ();
    if (param) {
      if (! arec)
        return false;
      type = arec->TypeInfo ();
      pos = getPosition (param);
      if (pos == -1 || ! param->isTypeParam () || ! sameType (pos, type)) 
        return false;
      darguments[pos] = new DeducedArgument (param, type);
      return true;
    }
    if ((bool)prec != (bool)arec || *prec != *arec)
      return false;
  // match against array
  } else if (ptype->TypeArray ()) {
    pdim = ptype->TypeArray ()->Dimension ();
    adim = atype->TypeArray ()->Dimension ();
    if (ptype->TypeArray ()->DepDim ()) {
      CConstant dim ((LONG_LONG)adim, &CTYPE_LONG);
      param = ptype->TypeArray ()->DepDim ()->TemplateParamInfo ();
      if (! param || param->isTypeParam () || 
          (pos = getPosition (param)) == -1)
        return false;
      if (atype->TypeArray ()->DepDim ()) {
        param2 = atype->TypeArray ()->DepDim ()->TemplateParamInfo ();
        if (! param2 || param2->isTypeParam () || 
            ! sameType (pos, param2->TypeInfo ()))
          return false;
        darguments[pos] = new DeducedArgument (param, param2->TypeInfo ());
      } else if (sameValue (pos, &dim)) {
        darguments[pos] = new DeducedArgument (param, adim);
      } else 
        return false;
    } else if (pdim != adim)
      return false;
  // match against function
  } else if (ptype->TypeFunction ()) {
    ptl = ptype->TypeFunction ()->ArgTypes ();
    atl = atype->TypeFunction ()->ArgTypes ();
    if (ptl->Entries () != atl->Entries ())
      return false;
    for (unsigned i = 0; i < ptl->Entries (); i++)
      if (! matchArgument (ptl->Entry (i), atl->Entry (i), 0)) 
        return false;
  }
  
  // match base types
  return matchArgument (ptype->BaseType (), atype->BaseType (), aexpr);
}


// §14.5.5.2 partial ordering rules
// return 0 if equal, 1 if more, and -1 if less specialized than other
int InstantiationCandidate::compare (InstantiationCandidate &other) {
  InstantiationCandidate cand1;
  cand1.initialize (other.TemplateInfo ()->SpecializationName (), ObjectInfo (), TemplateInfo ());
  for (unsigned i = 0; i < Arguments (); i++) 
    cand1.addArgument (Argument (i));
  
  InstantiationCandidate cand2;
  cand2.initialize (TemplateInfo ()->SpecializationName (), other.ObjectInfo (), other.TemplateInfo ());
  for (unsigned i = 0; i < other.Arguments (); i++) 
    cand2.addArgument (other.Argument (i));

  // match against the other
  bool matches1 = cand1.match (); 
  bool matches2 = cand2.match (); 
  
  // equal specialized
  if (matches1 && matches2) 
    return 0;
  // more specialized
  if (matches1) 
    return -1;
  // less specialized
  if (matches2) 
    return 1;
  // equal specialized
  return 0;
}


bool InstantiationCandidate::setDeducedArgument (DeducedArgumentList &args, int pos, DeducedArgument *arg) {
  bool res = true;
  DeducedArgument *old = args[pos];
  if (! old) {
    args[pos] = arg;
  } else {
    res = (*arg == *old);
    delete arg;
  }
  return res;
}


void InstantiationCandidate::printArgumentList (std::ostream &out, bool print_default_args) const {
  out << "<";
  for (unsigned i = 0; i < DeducedArgs (); i++) {
    DeducedArgument *arg = DeducedArg (i);

    // do not list default arguments
    if (! print_default_args && arg->isDefaultArg ())
      break;

    if (i) out << ",";
    out << *arg;
  }
  out << ">";
}


Token* InstantiationCandidate::getPointOfInstantiationToken(CTree* p) const {
  Token* token = 0;
  CTree* point = p ? p : poi;
  if (point) {
    do {
      token = point->token();
    } while (! token && (point = point->Parent()));
  }
  return token;
}


bool InstantiationCandidate::maxInstDepthReached () {
  // ISO says not more than 17
  unsigned max_depth = 17;
  // maximum set by user?
  const ConfOption *opt = getProject ()->config ().Option ("--template-depth");
  if (opt && opt->Arguments () > 0) {
    max_depth = strtol (opt->Argument (0), NULL, 10);
  }
  if (tpl_info->Depth () > max_depth) {
    *err << sev_error << getPointOfInstantiationToken()->location ()
         << "maximum instantiation depth ("
         << max_depth << ") reached" << endMessage;
    return true;
  }
  return false;
}


void InstantiationCandidate::printCodeLine (Token* token) const {
  int line = token->location().line();
  Unit* u = token->unit();
  for (token = u->first(); token; token = u->next(token)) {
    if (token->location().line() == line) {
      if (token->is_whitespace() && strchr(token->text(), '\n'))
        break;
      *err << token->text();
    }
  }
  *err << endMessage;
}


void InstantiationCandidate::printInstantiationErrorHeader (Token* token) const {
  if (token) {
    CObjectInfo* obj = obj_info;
    if (! obj && instance)
      obj = instance->Object();

    if (obj) {
      bool verbose = obj->SemDB()->Project()->config().Option("--verbose-errors");
      *err << token->location()
           << "In instantiation of `" << *obj->TypeInfo();
      if (!obj->TemplateInstance()) {
        std::ostringstream args;
        printArgumentList(args);
        *err << args.str().c_str();
      }
      *err << "'" << endMessage;
      if (verbose)
        printCodeLine(token);

      Unit* unit = token->unit();
      if (unit->isTemplateInstance()) {
        unit = ((TemplateInstanceUnit*)unit)->ContainingUnit();

        while (unit->isTemplateInstance()) {
          TemplateInstanceUnit* tiunit = (TemplateInstanceUnit*)unit;
          CTemplateInstance* inst = tiunit->TemplateInstance();
          CTree* poi = inst ? inst->PointOfInstantiation() : 0;
          token = poi ? poi->token() : 0;

          if (token && inst->Object()) {
            *err << token->location()
                 << "  instantiated from `" << *inst->Object()->TypeInfo()
                 << "'" << endMessage;
            if (verbose)
              printCodeLine(token);
          }

          unit = tiunit->ContainingUnit();
        }
      }
    }
  }
}


CStructure *InstantiationCandidate::createInstanceScope() {
  CStructure *scope = tpl_info->Parent()->newNamespace();
  scope->NamespaceInfo()->aroundInstantiation(true);
  scope->Name("%<anonymous>");
  scope->TypeInfo(&CTYPE_UNDEFINED);
  scope->SemDB(obj_info->SemDB());
  scope->SourceInfo()->FileInfo(obj_info->SourceInfo()->FileInfo());
  scope->SourceInfo()->StartToken(PointOfInstantiation()->token_node());
  return scope;
}


CTranslationUnit* InstantiationCandidate::createInstanceTranslationUnit (CStructure *scope) {
  // create a new token unit for the template arguments
  CUnit* unit = new TemplateInstanceUnit(*err, getPointOfInstantiationToken()->unit(), instance);
  unit->scanner().configure(getProject()->config());
  unit->name(scope->Name());

  // create a new translation unit
  return new CTranslationUnit(*unit, *getProject());
}


bool InstantiationCandidate::parseDirectArguments (unsigned pos, unsigned numargs, CT_TemplateArgList *args) {
  // check maximal instantiation depth
  if (maxInstDepthReached()) {
    return false;
  }

  // create new template instance scope
  inst_scope = createInstanceScope();

  // create a new translation unit
  trans_unit = createInstanceTranslationUnit(inst_scope);

  // create code for the template parameters
  if (! insertCodeForArguments(pos, numargs, args)) {
    return false;
  }

  // parse the arguments
  if (parseArguments()) {
    return deduceFromParsedArguments(pos, numargs, args);
  }
  // failed due to parse errors
  return false;
}


bool InstantiationCandidate::parseArguments () {
  // setup new parser
  CCParser p;
  p.configure(getProject()->config());
  if (TRACE_PARSE_INSTANCE) {
#ifdef __PUMA_TRACING__
    p.trace(std::cout);
#endif
  }

  // setup the preprocessor
  TokenStream stream;
  stream.push(trans_unit->unit());
  getProject()->unitManager().init();
  PreprocessorParser cpp(&getProject()->err(), &getProject()->unitManager(), &trans_unit->local_units());
  cpp.macroManager()->init(trans_unit->unit()->name());
  cpp.stream(&stream);
  cpp.configure(getProject()->config(), false); // do not process --include option

  // initialize semantic analyzer
  bool is_fct = obj_info->FunctionInfo();
  Unit* primary = obj_info->SourceInfo()->FileInfo()->Primary();
  p.semantic().init(*obj_info->SemDB(), *primary, inst_scope, is_fct, !is_fct, 0);
  ((ErrorCollector&)p.builder().err()).index(0);
  p.semantic().error_sink(p.builder().err());

  // increase instantiation depth
  tpl_info->increaseDepth();

  // parse the code
  TokenProvider provider(cpp);
  trans_unit->tree(p.syntax().run(provider));
  trans_unit->cpp_tree(cpp.syntaxTree());

  // decrease instantiation depth
  tpl_info->decreaseDepth ();

  // return true if no parse error occured
  return ((ErrorCollector&)p.builder().err()).severity() < sev_error;
}


bool InstantiationCandidate::deduceFromParsedArguments(unsigned pos, unsigned numargs, CT_TemplateArgList *args) {
  // get the argument declaration scope
  CStructure* arg_scope = inst_scope->Namespace("__puma_dargs");
  if (! arg_scope) {
    return false;
  }

  // deduce arguments from parsed direct and default arguments
  for (unsigned i = 0; i < pos; i++) {
    CTemplateParamInfo *param = tpl_info->Parameter(i);
    bool is_default_arg = (i >= numargs);
    CTree *arg;

    if (is_default_arg) {
      // default template argument
      arg = param->DefaultArgument()->Entry(0);
    } else {
      // direct template argument
      arg = args->Entry(i);
    }

    // get the name of the parameter
    const char* name = getParameterName(param);

    // get the corresponding object in the instance unit
    CObjectInfo* oinfo = arg_scope->Object(name);
    if (! oinfo || ! oinfo->TypeInfo()) {
      return false;
    }

    // template template parameter: class base template
    if (param->isTemplate()) {
      // template<...> struct [name] {
      //   typedef [arg]<...> __puma_redirect;
      // };
      oinfo = ((CStructure*)oinfo)->Object("__puma_redirect")->TypeInfo()->ClassInfo();
      if (! oinfo || ! oinfo->isTemplateInstance()) {
        return false;
      }
      oinfo = oinfo->TemplateInstance()->Template();
      darguments.append(new DeducedArgument(param,
        oinfo->TypeInfo(), arg, is_default_arg, !is_default_arg));
    }
    // template type parameter: expect type-id
    else if (param->isTypeParam()) {
      // typedef [arg] [name];
      darguments.append(new DeducedArgument(param,
        oinfo->TypeInfo()->VirtualType(), arg, is_default_arg, !is_default_arg));
    }
    // template non-type parameter: expect constant expression
    else {
      // static [type] [name] = [arg];
      CTree* init = ((CAttributeInfo*)oinfo)->Init();
      if (! init || ! init->Value() || ! init->Value()->Constant()) {
        return false;
      }
      darguments.append(new DeducedArgument(param,
        init->Value()->Constant(), arg, is_default_arg, !is_default_arg));
    }
  }
  return true;
}


bool InstantiationCandidate::insertCodeForArguments(unsigned pos, unsigned numargs, CT_TemplateArgList *args) {
  CUnit& unit = (CUnit&)*trans_unit->unit();

  // insert code for each direct or default argument
  unit << "namespace __puma_dargs {\n" << endu;
  for (unsigned i = 0; i < pos; i++) {
    CTemplateParamInfo *param = tpl_info->Parameter(i);
    bool is_default_arg = (i >= numargs);
    CTree *arg;

    if (is_default_arg) {
      // default template argument
      arg = param->DefaultArgument()->Entry(0);
    } else {
      // direct template argument
      arg = args->Entry(i);
    }

    // get the name of the parameter, generate one
    // if the parameter is anonymous
    const char* name = getParameterName(param);

    if (param->isTemplate()) {
      // template-template parameter
      CTypeInfo* type = 0;
      if (! is_default_arg) {
        // get the template type
        CObjectInfo* oinfo;
        CTree* sn = arg;
        if (sn->NodeName() == CT_NamedType::NodeId())
          sn = sn->Son(0)->Son(0);
        if (! (sn->NodeName() == CT_SimpleName::NodeId() ||
               sn->NodeName() == CT_RootQualName::NodeId() ||
               sn->NodeName() == CT_QualName::NodeId()) ||
            ! sn->SemObject() ||
            ! (oinfo = sn->SemObject()->Object()) ||
            ! (oinfo->Record() && oinfo->Record()->isTemplate())) {
          SEM_ERROR(arg, "expected a class template as argument " << i+1);
          return false;
        }
        type = oinfo->TypeInfo()->VirtualType();
      }
      if (! insertTemplateTemplateArgument(param, arg, type, name, unit))
        return false;
    }
    else if (param->isTypeParam()) {
      // template type parameter
      CTypeInfo* type = 0;
      if (! is_default_arg) {
        // get the type
        CObjectInfo* oinfo;
        if (! (arg->NodeName() == CT_NamedType::NodeId()) ||
            ! arg->SemObject() ||
            ! (oinfo = arg->SemObject()->Object())) {
          SEM_ERROR(arg, "expected a type as argument " << i+1);
          return false;
        }
        type = oinfo->TypeInfo()->VirtualType();
      }
      if (! insertTypeTemplateArgument(param, arg, type, name, unit))
        return false;
    }
    else {
      // template non-type parameter
      CConstant* value = 0;
      if (! is_default_arg) {
        // not a non-type argument
        if (arg->NodeName() == CT_NamedType::NodeId()) {
          SEM_ERROR(arg, "expected a constant as argument " << i+1);
          return false;
        }
        // need value of constant
        if (! arg->Value() || ! arg->Value()->Constant()) {
          SEM_ERROR(arg, "expected a constant as argument " << i+1);
          return false;
        }
        value = arg->Value()->Constant();
      }
      if (! insertNonTypeTemplateArgument(param, arg, value, name, unit))
        return false;
    }
  }

  // finish and scan the code
  unit << "}\n" << endu;

  // dump code to stdout if tracing enabled
  if (TRACE_PARSE_INSTANCE) {
    dumpInstanceCode(unit);
  }
  return true;
}


bool InstantiationCandidate::insertNonTypeTemplateArgument(CTemplateParamInfo* param, CTree* tree, CConstant* value, const char* name, CUnit& unit) {
  // static [type] [name] = [arg];
  unit << "static ";
  CT_NonTypeParamDecl* decl = (CT_NonTypeParamDecl*)param->Tree();
  copyTree(unit, decl->DeclSpecs(), 0, 0, name);
  unit << " ";
  copyTree(unit, decl->Declarator(), 0, findName(decl->Declarator()),
    name, param->ValueType() && ! param->ValueType()->isConst());
  unit << " = ";
  //CTypeInfo* type = tree ? tree->Type() : 0;
  //bool ref = CCSemExpr::isAddrExpr(tree) || (type && (type->isArray() || type->isFunction()));
  if (value/* && !(ref && tree)*/) {
    unit << *value;
  } else if (tree) {
    copyTree(unit, tree, tree->end_token());
  } else {
    return false;
  }
  unit << ";\n";
  return true;
}


bool InstantiationCandidate::insertTypeTemplateArgument(CTemplateParamInfo* param, CTree* tree, CTypeInfo* type, const char* name, CUnit& unit) {
  // typedef [arg] [name];
  unit << "typedef ";
  if (type) {
    type->TypeText(unit, name, true);
  } else if (tree) {
    copyTree(unit, tree, 0, findPrivateName(tree), name);
  } else {
    return false;
  }
  unit << ";\n";
  return true;
}


bool InstantiationCandidate::insertTemplateTemplateArgument(CTemplateParamInfo* param, CTree* tree, CTypeInfo* type, const char* name, CUnit& unit) {
  // template<...> struct [name] {
  //   typedef [arg]<...> __puma_redirect;
  // };
  CTemplateInfo* ti = param->TemplateTemplate();
  unit << "template< ";
  listParameters(unit, ti);
  unit << " > struct " << name << " {\n  typedef ";
  if (type) {
    type->TypeText(unit, 0, true);
  } else if (tree) {
    copyTree(unit, tree, tree->end_token());
  } else {
    return false;
  }
  unit << "<";
  listParameterNames(unit, ti);
  unit << "> __puma_redirect;\n};\n";
  return true;
}


bool InstantiationCandidate::insertCodeForInstance (CUnit &unit) {
  // add all non-direct non-default arguments
  for (unsigned i = 0; i < DeducedArgs (); i++) {
    DeducedArgument *arg = DeducedArg (i);
    CTemplateParamInfo *param = getMatchingParameter(arg->TemplateParam ());
    if (! param)
      continue;

    // get the name of the parameter, generate one
    // if the parameter is anonymous
    const char* name = getParameterName(param);

    // direct or default argument
    if ((arg->isDefaultArg() || arg->isDirectArg()) &&
        arg->TemplateParam()->Template() == tpl_info) {
      // code already inserted, only introduce the name
      // of the parameter into the instance scope
      unit << "using __puma_dargs::" << name << ";\n";
      continue;
    }

    if (param->isTemplate ()) {
      // deduced template-template argument
      if (! insertTemplateTemplateArgument(param, 0, arg->Type(), name, unit))
        return false;
    } else if (param->isTypeParam ()) {
      // deduced type argument
      if (! insertTypeTemplateArgument(param, 0, arg->Type(), name, unit))
        return false;
    } else {
      // deduced non-type argument
      if (! insertNonTypeTemplateArgument(param, 0, arg->Value(), name, unit))
        return false;
    }
  }

  // check for a template parameter in the class name -> desturbs
  Array<CTree*> desturbing;
  calculateDesturbing (desturbing);

  // copy the object declaration without the template header
  if (TemplateInfo ()->Tree ()) {
    CTree* decl = TemplateInfo ()->Tree ()->Declaration ();
    copyTree (unit, decl, decl->end_token(), 0, 0, false, &desturbing);
  }

  unit << "\n" << endu;

  if (TRACE_INSTANCE_CODE) {
    dumpInstanceCode(unit);
  }
  return true;
}


void InstantiationCandidate::calculateDesturbing (Array<CTree*>& desturbing) {
  // check for a template parameter in the class name -> desturbs
  if (ObjectInfo ()->Tree ()->NodeName () == CT_ClassDef::NodeId ()) {
    CT_ClassDef *clsdef = (CT_ClassDef*)ObjectInfo ()->Tree ();
    if (clsdef->Name ()->NodeName () == CT_TemplateName::NodeId ())
      desturbing.append (((CT_TemplateName*)clsdef->Name ())->Arguments ());
    CT_MembList *members = clsdef->Members ();
    if (members) { // normally true, but in case of parse errors (?) ...
      for (int i = 0; i < members->Entries (); i++) {
        if (members->Entry (i)->NodeName () == CT_FctDef::NodeId ()) {
          CT_FctDef *fctdef = (CT_FctDef*)members->Entry (i);
          CT_SimpleName *nm =((CT_Declarator*)fctdef->Declarator ())->Name ();
          if (nm->NodeName () == CT_TemplateName::NodeId ())
            desturbing.append (((CT_TemplateName*)nm)->Arguments ());
        }
      }
    }
  }
}


void InstantiationCandidate::dumpInstanceCode(CUnit& unit) {
  std::cout << endl;
  if (PointOfInstantiation()) {
    std::cout << getPointOfInstantiationToken()->location() << ": ";
  }
  std::cout << obj_info->Name();
  printArgumentList(std::cout);
  std::cout << ": instantiated here" << std::endl;
  if (tpl_info->Tree() && tpl_info->Tree()->token() && tpl_info->ObjectInfo()) {
    std::cout << tpl_info->Tree()->token()->location()
              << ": instantiation of template "
              << tpl_info->ObjectInfo()->QualName(true) << std::endl;
  }
  std::cout << "-------------------" << std::endl;
  unit.print(std::cout);
  std::cout << "-------------------\n" << std::endl;
}


void InstantiationCandidate::copyTree(CUnit &unit, CTree *tree, Token* end_token, CTree *sn, const char *name, bool make_const, Array<CTree*> *desturbing) {
  // first check if this subtree is disturbing and should
  // be omitted in the generated unit
  if (desturbing)
    for (int i = 0; i < desturbing->length (); i++)
      if (desturbing->lookup (i) == tree)
        return; // prune here

  if (tree == sn) {
    // replace name with name of the current template parameter
    unit << " ";
    if (make_const)
      unit << "const ";
    if (sn->NodeName() == CT_PrivateName::NodeId())
      unit << name << " ";
  }
  else if (tree->NodeName() == CT_Token::NodeId()) {
    // copy the rest
    Token* t = tree->token();
    if (t) {
      unit << t->text();
      if (t != end_token) {
        Token* next = t->unit()->next(t);
        if (next && (next->is_whitespace() || next->is_comment()))
          unit << (next->is_whitespace() ? next->text() : " ");
        else if (! next)
          unit << " ";
      }
    }
  }
  else if (tree->NodeName() == CT_TemplateName::NodeId() &&
           tree->Parent()->NodeName() == CT_ClassDef::NodeId() &&
           ((CT_ClassDef*)tree->Parent())->Name() == tree) {
    // name of a template specialization, cut the parameter list
    CT_SimpleName* tplname = ((CT_TemplateName*)tree)->TemplateName();
    copyTree(unit, tplname, end_token, sn, name, make_const);
    unit << " ";
    return;
  }

  CTree* body = 0;
  if (tree->NodeName() == CT_FctDef::NodeId()) {
    body = ((CT_FctDef*)tree)->Body();
  }

  for (unsigned i = 0; i < (unsigned)tree->Sons(); i++) {
    CTree* son = tree->Son(i);
    if (body && body == son) {
      if (body->NodeName() == CT_Error::NodeId()) {
        // if function body is error node => replace by ';'
        unit << ";\n";
        continue;
      } else if (! getProject ()->config ().Option ("--inst-fct-bodies")) {
        // don't instantiate function body, instantiated on demand
        unit << "{}\n";
        continue;
      }
    }
    copyTree(unit, tree->Son(i), end_token, sn, name, make_const);
  }

  if (tree->NodeName() == CT_TemplateName::NodeId()) {
    extendTemplateName(unit, (CT_TemplateName*)tree);
  }
  else if (tree->NodeName() == CT_SimpleName::NodeId()) {
    extendClassName(unit, (CT_SimpleName*)tree);
  }
}


void InstantiationCandidate::extendClassName(CUnit &unit, CT_SimpleName *name) {
  CObjectInfo *info = name->Object();
  if (! info)
    return;

  CTree* parent = name->Parent();
  if (info == ObjectInfo() && info->Record() && parent->NodeName() != CT_TemplateName::NodeId()) {
    // change X to X<T> if X is the instantiated class template
    // do not extend X in the class head
    if (parent->IsSimpleName())
      parent = parent->Parent();
    if (parent->NodeName() != CT_ClassDef::NodeId()) {
      unit << "<";
      if (TemplateInfo()->isSpecialization())
        listInstantiationArguments(unit, info);
      else
        listParameterNames(unit, TemplateInfo());
      unit << "> ";
    }
  }
}


void InstantiationCandidate::extendTemplateName(CUnit &unit, CT_TemplateName *name) {
  CObjectInfo *info = name->TemplateName()->Object();
  if (info) {
    // is template template parameter?
    CTemplateParamInfo *pinfo = info->TemplateParamInfo();
    if (pinfo && pinfo->isTemplate()) {
      // which template parameter is it?
      if (getPosition(pinfo) != -1) {
        // extend the template name
        unit << "::__puma_redirect ";
      }
    }
  }
}


void InstantiationCandidate::listParameters(CUnit &unit, CTemplateInfo *ti) {
  CTemplateParamInfo *tp;
  CT_TypeParamDecl *tpd;
  CT_NonTypeParamDecl *ntpd;
  const char* name;

  for (unsigned i = 0; i < ti->Parameters(); ++i) {
    if (i > 0) {
      unit << ",";
    }
    tp = ti->Parameter(i);
    name = getParameterName(tp);
    if (tp->isTypeParam()) {
      tpd = (CT_TypeParamDecl*)tp->Tree();
      copyTree(unit, tpd, 0, findPrivateName(tpd->Name()), name);
    } else {
      ntpd = (CT_NonTypeParamDecl*)tp->Tree();
      copyTree(unit, ntpd, 0, findPrivateName(ntpd->Declarator()), name);
    }
  }
}


void InstantiationCandidate::listParameterNames(CUnit &unit, CTemplateInfo *ti) {
  for (unsigned j = 0; j < ti->Parameters(); ++j) {
    if (j > 0) {
      unit << ",";
    }
    CTemplateParamInfo *tp = ti->Parameter(j);
    if (*tp->Name() != '%') {
      unit << tp->Name();
    } else {
      unit << "__puma_" << tp->Name()+1;
    }
  }
}


void InstantiationCandidate::listInstantiationArguments(CUnit &unit, CObjectInfo *info) {
  for (unsigned i = 0; i < InstantiationArgs(); i++) {
    DeducedArgument *arg = InstantiationArg(i);

    // do not list default arguments
    if (arg->isDefaultArg())
      break;

    if (i) unit << ",";
    unit << *arg;
  }
}


const char* InstantiationCandidate::getParameterName(CTemplateParamInfo* param) {
  const char* name = param->Name();
  if (*name == '%') {
    sprintf(anon_name_buf, "__puma_%s", name+1);
    name = anon_name_buf;
  }
  return name;
}


CT_SimpleName *InstantiationCandidate::findPrivateName (CTree *node) {
  const char *id = node->NodeName ();
  if (id == CT_PrivateName::NodeId ())
    return (CT_SimpleName*)node;
  else if (id == CT_NamedType::NodeId ())
    return findPrivateName (((CT_NamedType*)node)->Declarator ());
  else if (id == CT_FctDeclarator::NodeId ())
    return findPrivateName (((CT_FctDeclarator*)node)->Declarator ());
  else if (id == CT_ArrayDeclarator::NodeId ())
    return findPrivateName (((CT_ArrayDeclarator*)node)->Declarator ());
  else if (id == CT_PtrDeclarator::NodeId ())
    return findPrivateName (((CT_PtrDeclarator*)node)->Declarator ());
  else if (id == CT_MembPtrDeclarator::NodeId ())
    return findPrivateName (((CT_MembPtrDeclarator*)node)->Declarator ());
  else if (id == CT_BracedDeclarator::NodeId ())
    return findPrivateName (((CT_BracedDeclarator*)node)->Declarator ());
  else if (id == CT_BitFieldDeclarator::NodeId ())
    return findPrivateName (((CT_BitFieldDeclarator*)node)->Declarator ());
  else if (id == CT_RefDeclarator::NodeId ())
    return findPrivateName (((CT_RefDeclarator*)node)->Declarator ());
  else if (id == CT_InitDeclarator::NodeId ())
    return findPrivateName (((CT_InitDeclarator*)node)->Declarator ());
  return (CT_SimpleName*)0;
}


CT_SimpleName *InstantiationCandidate::findName (CTree *node) {
  const char *id = node->NodeName ();
  if (node->IsSimpleName ())
    return (CT_SimpleName*)node;
  else if (id == CT_NamedType::NodeId ())
    return findName (((CT_NamedType*)node)->Declarator ());
  else if (id == CT_FctDeclarator::NodeId ())
    return findName (((CT_FctDeclarator*)node)->Declarator ());
  else if (id == CT_ArrayDeclarator::NodeId ())
    return findName (((CT_ArrayDeclarator*)node)->Declarator ());
  else if (id == CT_PtrDeclarator::NodeId ())
    return findName (((CT_PtrDeclarator*)node)->Declarator ());
  else if (id == CT_MembPtrDeclarator::NodeId ())
    return findName (((CT_MembPtrDeclarator*)node)->Declarator ());
  else if (id == CT_BracedDeclarator::NodeId ())
    return findName (((CT_BracedDeclarator*)node)->Declarator ());
  else if (id == CT_BitFieldDeclarator::NodeId ())
    return findName (((CT_BitFieldDeclarator*)node)->Declarator ());
  else if (id == CT_RefDeclarator::NodeId ())
    return findName (((CT_RefDeclarator*)node)->Declarator ());
  else if (id == CT_InitDeclarator::NodeId ())
    return findName (((CT_InitDeclarator*)node)->Declarator ());
  return (CT_SimpleName*)0;
}


} // namespace Puma
