package Cache::Apt::Package;
use Class::Struct;
use Exporter;
use Data::Dumper;
use strict;
use warnings;
use vars qw(@ISA @EXPORT $verbose );
@ISA=qw(Exporter);
@EXPORT=qw( lookup_pkg output_pkg );

=pod

=head1 Name

Cache::Apt::Package - Package handler for apt-cross and related tools

=head2 Description

Turns the awkward libapt-pkg-perl API into an Object-oriented
set of structs that can be passed around between scripts.
Uses an extensively modified portion of NorthernCross
to provide the data contained in the structs. Together with
Cache and Config, Package will form a new Debian package:
libcache-apt-perl

The structures are designed to be discrete and do not contain
self-referencing variables as are found in apt-pkg-perl. All instances
can be safely used with Data::Dumper to inspect the contents - most
functions support a $verbose setting that can dump the contents of the
current package to the console using Data::Dumper.

=head1 Copyright and Licence

 Copyright (C) 2007  Neil Williams <codehelp@debian.org>

 This package is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 3 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program.  If not, see <http://www.gnu.org/licenses/>.

=head1 Bugs

Please report bugs via the Debian Bug Tracking System.

=head1 Support

All enquiries to the C<<debian-embedded@lists.debian.org>> mailing list.

=cut

=head1 AptCrossPackage

The struct that is used to describe a package.

Note that Depends is an array - it needs to be an array of
AptCrossDependency structs.

 struct (AptCrossPackage => {
	"Package" =>             '$',
	"Version" =>             '$',
	"Source" =>              '$',
	"Distribution" =>        '$',
	"Architecture" =>        '$',
	"Maintainer" =>          '$',
	"Size" =>                '$',
	"Installed" =>           '$',
	"Section" =>             '$',
	"Priority" =>            '$',
	"Build_Depends" =>       '$',
	"Build_Depends_Indep" => '$',
	"Depends" =>      '@',
	"Description" =>         '$',
	"Summary" =>             '$',
	"Recommends" =>          '$',
	"Suggests" =>            '$',
	"Filename" =>            '$',
	"FileMD5Hash" =>         '$',
 });

=cut

struct (AptCrossPackage => {
	"Package" =>             '$',
	"Version" =>             '$',
	"Source" =>              '$',
	"Distribution" =>        '$',
	"Architecture" =>        '$',
	"Cross_Architecture" =>  '$',
	"Maintainer" =>         '$',
	"Size" =>      '$',
	"Installed" => '$',
	"Section" =>             '$',
	"Priority" =>            '$',
	"Build_Depends" => '$',
	"Build_Depends_Indep" => '$',
	"Depends" =>   '@',
	"Description" =>         '$',
	"Summary" =>             '$',
	"Recommends" => '$',
	"Suggests" =>  '$',
	"Filename" =>            '$',
	"FileMD5Hash" =>         '$',
});

=head1 AptCrossDependency

AptCrossPackage->Depends is an array of
AptCrossDependency instances describing the
dependency.

VersionLimit is the version indicator for this dependency and the
version string itself - but not the parentheses.

 e.g.
 >= 1.2.3-1
 << 0.56-2
 >> 1.3~r345-1

 struct (AptCrossDependency => {
	"Package" => '$',
	"VersionLimit" => '$',
	"Installed" => '$',
	"Type" => '$',
 });

=cut

struct (AptCrossDependency => {
	"Package" => '$',
	"VersionLimit" => '$',
	"Installed" => '$',
	"Type" => '$',
});

sub populate_depends ($)
{
	my ($depends, $pkg) = @_;
	my @list=();
	foreach my $name (keys %$depends)
	{
		next if ($$depends{$name}[2] ne "Depends");
		my $dep = new AptCrossDependency;
		$dep->Package($name);
		$dep->VersionLimit($$depends{$name}[0]);
		$dep->Installed($$depends{$name}[1]);
		$dep->Type($$depends{$name}[2]);
		push @list, $dep;
	}
	$$pkg->Depends(\@list);
}

=head1 lookup_pkg

Populate an AptCrossPackage with data from the cache.

Use this function to remove some of the pain of working directly
with the apt-pkg-perl constructs or even AptCross::Cache functions.

Note that this lookup operates on the binary package name.

=cut

sub lookup_pkg($)
{
	my $pkg = shift;
	$verbose = Cache::Apt::Lookup::get_verbose();
	my $rec = Cache::Apt::Lookup::binlookup($pkg->Package);
	print Dumper ($rec) if ($verbose >= 3);
	$pkg->Version($rec->{VerStr});
	# all dpkg-cross packages are Architecture: all
	if ($pkg->Package =~ /-([^\-]+)-cross$/)
	{
		$pkg->Cross_Architecture($1);
		$pkg->Architecture("all");
	}
	(defined $rec->{SourcePkg}) ? $pkg->Source($rec->{SourcePkg}) : $pkg->Source($pkg->Package);
	#$pkg->Size();
	#$pkg->Installed();
	$pkg->Section($rec->{Section});
	#$pkg->Priority();
	$pkg->Maintainer($rec->{Maintainer});
	$pkg->Build_Depends();
	$pkg->Build_Depends_Indep();
	$pkg->Depends();
	$pkg->Description($rec->{LongDesc});
	$pkg->Summary($rec->{ShortDesc});
	(defined $rec->{Recommends}) ? $pkg->Recommends($rec->{Recommends}) : undef;
	(defined $rec->{Suggests}) ? $pkg->Suggests($rec->{Suggests}) : undef;
	(defined $rec->{FileName}) ? $pkg->Filename($rec->{FileName}) : undef;
	# reset Architecture: all packages
	if (defined($pkg->Filename) && ($pkg->Filename =~ /_all\.deb$/))
	{
		$pkg->Architecture("all");
	}
	$pkg->FileMD5Hash($rec->{MD5Hash});
	&get_depends_list ($pkg);
	return \$pkg;
}

=head1 output_pkg

Creates output on STDOUT to imitate dpkg-query -s or apt-cache show

Certain values are not available via the current perl bindings.

=cut

sub output_pkg($) {
	my $emp = shift;
	print "Package: " . $$emp->Package . "\n" if ($$emp->Package);
	print "Version: " . $$emp->Version . "\n" if ($$emp->Version);
	print "Architecture: " . $$emp->Architecture . "\n" if ($$emp->Architecture);
	print "Maintainer: " . $$emp->Maintainer . "\n" if ($$emp->Maintainer);
	print "Source: " . $$emp->Source . "\n" if ($$emp->Source);
	my $dep = $$emp->Depends;
	my $c = 0;
	my $deps;
	my $line = "";
	foreach my $d (@$dep)
	{
		$line .= ", " if ($c > 0);
		$deps = $$d->Type . ":";
		$line .= $$d->Package . " (" . $$d->VersionLimit . ")" if ($$d->VersionLimit);
		$c++;
	}
	print $deps . $line . "\n" if (defined($deps));
#	print "Suggests: \n";
#	print "Recommends: \n";
	print "Priority: " . $$emp->Priority . "\n" if ($$emp->Priority);
	print "Section: " . $$emp->Section . "\n" if ($$emp->Section);
	print "Filename: " . $$emp->Filename . "\n" if ($$emp->Filename);
#	print "Size: " . $emp->Size . "\n";
	print "MD5sum: " . $$emp->FileMD5Hash . "\n" if ($$emp->FileMD5Hash);
	print "Description: " . $$emp->Description . "\n" if ($$emp->Description);
}

# TODO: work out how to support Recommends and Provides etc.
sub get_depends_list ($)
{
	my %sorter = ();
	my @list=();
	my $emp = shift;
	my $dephash = Cache::Apt::Lookup::deplookup(\$emp);
	foreach my $href (@$dephash)
	{
		foreach my $hh (keys %$href)
		{
			# TODO: bring the subroutine inline.
			&populate_depends($$href{$hh}, \$emp);
			my $nested = $$href{$hh};
			foreach my $n (keys %$nested)
			{
				my $g = $$nested{$n};
				foreach my $name (@$g)
				{
					# skip repeat dependencies
					next if (defined ($sorter{$n}));
					# skip Suggests/Recommends etc.
					next if ($$g[2] ne "Depends");
					my $dep = new AptCrossDependency;
					$dep->Package($n);
					$dep->VersionLimit($$g[0]);
					$dep->Installed($$g[1]);
					$dep->Type($$g[2]);
					$sorter{$n} = \$dep;
				}
			}
		}
	}
	my @s = keys %sorter;
	@s = sort (@s);
	# add the unique dependencies, sorted by name.
	foreach my $uniq (@s)
	{
		push @list, $sorter{$uniq};
	}
	$emp->Depends(\@list);
}

1;
