
//#define LOCAL_DEBUG
#include "debug.h"

#include "expiration.h"
#include "lockable.h"
#include "acfg.h"
#include "meta.h"
#include "filereader.h"
#include "fileitem.h"
#include "dlcon.h"
#include "dirwalk.h"
#include "header.h"
#include "job.h"
#include "dljob.h"

#include "fileio.h"
#include <errno.h>
#include <unistd.h>
#include <dirent.h>

#include <map>
#include <string>
#include <iostream>
#include <algorithm>

using namespace MYSTD;

#define ENABLED

#ifndef ENABLED
#warning Unlinking parts defused
#endif

#define SZABSPATH(x) (CACHE_BASE+(x)).c_str()
#define SABSPATH(x) (CACHE_BASE+(x))
static cmstring diffIdxSfx(".diff/Index");
static cmstring sPatchBaseRel("_actmp/patch.base");
static cmstring sPatchResRel("_actmp/patch.result");

tCacheMan::tCacheMan(int fd) :
	tWuiBgTask(fd), m_nTimeNow(0),
	m_bErrAbort(false), m_bVerbose(false), m_bForceDownload(false),
	m_bScanInternals(false), m_bByPath(false), m_bByChecksum(false), m_bSkipHeaderChecks(false),
	m_bNeedsStrictPathsHere(false),
	m_nErrorCount(0),
	m_nProgIdx(0), m_nProgTell(1), m_pDlcon(NULL)
{
	m_szDecoFile="maint.html";
	m_nTimeNow=time(NULL);

}

tCacheMan::~tCacheMan()
{
	delete m_pDlcon;
	m_pDlcon=NULL;
}

bool tCacheMan::ProcessOthers(const string & sPath, const struct stat &)
{
	// NOOP
	return true;
}

bool tCacheMan::ProcessDirAfter(const string & sPath, const struct stat &)
{
	// NOOP
	return true;
}


void tCacheMan::AddIFileCandidate(const string &sPathRel)
{
	enumIndexType t;
	if ( (rechecks::FILE_INDEX == rechecks::GetFiletype(sPathRel)
	// SUSE stuff, not volatile but also contains file index data
	|| endsWithSzAr(sPathRel, ".xml.gz") )
	&& (t=GuessIndexType(sPathRel)) != EIDX_UNSUPPORTED)
	{
		tIfileAttribs & atts=m_indexFilesRel[sPathRel];
		atts.vfile_ondisk=true;
		atts.eIdxType=t;
    }
}

const tCacheMan::tIfileAttribs & tCacheMan::GetFlags(cmstring &sPathRel) const
{
	static const tIfileAttribs def; // good enough for single thread
	tS2IDX::const_iterator it=m_indexFilesRel.find(sPathRel);
	if(m_indexFilesRel.end()==it)
		return def;
	return it->second;
}

tCacheMan::tIfileAttribs &tCacheMan::SetFlags(cmstring &sPathRel)
{
	return m_indexFilesRel[sPathRel];
}


bool tCacheMan::RecDownload(tFileItemPtr pFi, const char *pURL, int nTTL)
{
	if(--nTTL<0)
	{
		SendChunk("Redirection count reached, aborting!<br>");
		return false;
	}
	if(!m_pDlcon || !pFi)
		return false;
	// get fileitem into a pristine state
	pFi->ResetCacheState();
	pFi->Setup(true);
	tHttpUrl url;
	if(!pURL || !*pURL || !url.SetHttpUrl(pURL))
		return false;
	m_pDlcon->AddJob(pFi, url);
	m_pDlcon->WorkLoop();
	switch (pFi->GetHeader().getStatus())
	{
	case 200:
		return true;
	case 301:
	case 302:
		return RecDownload(pFi, pFi->GetHeader().h[header::LOCATION]);
	default:
		return false;
	}
}

bool tCacheMan::RefetchFile(tFileItemPtr fi, cmstring &sFilePathRel, mstring &sErr)
{
	StartDlder();
	if(!m_pDlcon)
	{
		sErr="General error while creating remote connection";
		return false;
	}

	/*
	 * Three ways to find the source to download from:
	 * 1. Interpret the base directory as repository name
	 * 2. Interpret the whole subpath as URL (host/path)
	 * 3. Use X-Original-Source (which might be broken)
	 */

	// abusing url class to extract base directory

	tHttpUrl url;
	acfg::tHostiVec *pBackends(NULL);

	if (!url.SetHttpUrl(sFilePathRel))
	{
		// should never happen, though
		sErr=sFilePathRel+" does not seem to contain valid repository or host name.";
		return false;
	}

	pBackends = acfg::GetBackendVec(&url.sHost);

	if (pBackends)
		// HIT, control by backend scheme, strip leading slash from path
		 m_pDlcon->AddJob(fi, pBackends, url.sPath.substr(1));
	else
	{
		tHttpUrl urlOrig;
		string sOrig;
		{
			lockguard g(fi.get());
			const header & pHead=fi->GetHeaderUnlocked();
			if (pHead.h[header::XORIG])
				sOrig=pHead.h[header::XORIG];
		}
		if(startsWithSz(sOrig, "http://") && urlOrig.SetHttpUrl(sOrig))
		{
			// ok, looks valid, is it better than the one from the path?
			if(url != urlOrig)
			{
				if(urlOrig.sHost.find(".") != stmiss)
				{
					if(url.sHost.find(".") != stmiss)
					{
						// Both have dots, prefer directory as host
						//goto dl_from_url;
					}
					else
					{
						// dir has no dots, orig-url host has -> use orig url
						url=urlOrig;
						//goto dl_from_url;
					}
				}
				else // no dots in urlOrig host, most likely broken
				{
					/*
					 * if the directory has dots, use it and be quiet (unless verbosity is enabled).
					 * Otherwise, warn the user.
					 * */
					if (m_bVerbose || url.sHost.find(".") != stmiss)
					{
						SendFmt()<<"<font color=\"orange\">Code 520824! "
							"Read the manual about known bugs! Attempting to use "
								<< url.sHost << " as hostname</font>";
					}
				}
			}
		}
		m_pDlcon->AddJob(fi, url);
	}

	m_pDlcon->WorkLoop();

	return FIST_ERROR != fi->GetStatus(); // this is precise enough for this multi-purpose function, caller might need to recheck it against the own expectations
}

static unsigned int nKillLfd=1;

bool tCacheMan::DownloadIdxFile(const MYSTD::string & sFilePathRel, string &sErr)
{
	sErr.clear();
	bool bSuccess=false;
	FiStatus fistate=FIST_FRESH;
	tFileItemPtr fi;

	//bool nix=StrHas(sFilePathRel, "debrep/dists/experimental/main/source/Sources.diff/Index");

	const tIfileAttribs &flags=GetFlags(sFilePathRel);
	if(flags.uptodate)
	{
		SendFmt()<<"Checking "<<sFilePathRel<<"... (fresh)<br>\n";
		return true;
	}

	fi=fileitem::GetFileItem(sFilePathRel);

	if (fi)
	{
		fistate = fi->Setup(true);
		if(m_bForceDownload) // setup first and drop the contents then
		{
			fi->ResetCacheState();
			SendFmt()<<"Downloading "<<sFilePathRel<<"...\n";
		}
		else
		{
			SendFmt()<<"Checking/Updating "<<sFilePathRel<<"...\n";
		}
	}
	else
	{
		SendFmt()<<"Checking "<<sFilePathRel<<"...\n"; // just display the name

		sErr=" could not create file item handler.";
		goto rep_dlresult;
	}


#ifdef WIN32
#error rewrite for the backslashes or change reliably in the walker to slashes
#endif

	if (RefetchFile(fi, sFilePathRel, sErr))
	{
		int code=fi->GetHeader().getStatus();
		if(301==code || 302==code)
		{
			RecDownload(fi, fi->GetHeader().h[header::LOCATION]);
			code=fi->GetHeader().getStatus();
		}
		if(200==code)
		{
			bSuccess=true;
			SendFmt() << "<i>(" << fi->GetTransferCount() / 1024 << "KiB)</i>\n";
		}
		else
			bSuccess = flags.forgiveDlErrors;
	}
#ifdef DEBUG
	else
	{
		bSuccess=false;
	}
#endif

	rep_dlresult:

	if (bSuccess)
		SetFlags(sFilePathRel).uptodate=true;
	else if (fi)
		sErr += fi->GetHttpMsg();
	else if(sErr.empty())
		sErr += "Download error";

	UpdateFingerprint(sFilePathRel, -1, NULL, NULL);

	if (fi)
		fi->Unreg();

	if(m_bVerbose || !bSuccess)
	{
		m_bShowControls=true;
		SendFmt() << "<label><input type=\"checkbox\" name=\"kf"
		<< (nKillLfd++) << "\" value=\"" << sFilePathRel
		<< "\">Tag</label>";
	}

	SendChunk("<br>\n");
	return bSuccess;
}


bool tCacheMan::DownloadSimple(const MYSTD::string & sFilePathRel)
{
	bool bSuccess=false;
	FiStatus fistate=FIST_FRESH;
	tFileItemPtr fi=fileitem::GetFileItem(sFilePathRel);

	if (!fi)
		return false;

	fistate = fi->Setup(false);
	if(fistate == FIST_COMPLETE)
	{
		fi->Unreg();
		return true;
	}

	SendFmt()<<"Downloading "<<sFilePathRel<<"...\n";

	string sErr;
	if (RefetchFile(fi, sFilePathRel, sErr))
	{
		int code=fi->GetHeader().getStatus();
		if(301==code || 302==code)
		{
			RecDownload(fi, fi->GetHeader().h[header::LOCATION]);
			code=fi->GetHeader().getStatus();
		}
		if(200==code)
		{
			bSuccess=true;
			SendFmt() << "<i>(" << fi->GetTransferCount() / 1024 << "KiB)</i>\n";
		}
	}

	if (fi)
		fi->Unreg();

	SendChunk("<br>\n");
	return bSuccess;
}


static const string relKey("/Release");

#define ERRMSGABORT if(m_nErrorCount && m_bErrAbort) { SendChunk(sErr); return; }
#define ERRABORT if(m_nErrorCount && m_bErrAbort) { return; }

inline tStrPos FindComPos(const string &s)
{
	tStrPos compos(stmiss);
	for(const string *p=suxe; p<suxe+_countof(suxe) && stmiss==compos; p++)
		if(endsWith(s, *p))
			compos=s.size()-p->size();
	return compos;
}
static unsigned short FindCompIdx(cmstring &s)
{
	unsigned short i=0;
	for(;i<_countof(suxe); i++)
		if(endsWith(s, suxe[i]))
			break;
	return i;
}

tContId BuildEquivKey(const tFingerprint &fpr, const string &sDir, const string &sFile,
		tStrPos maxFnameLen)
{
	static const string dis("/binary-");
	tStrPos pos=sDir.rfind(dis);
	//pos=(stmiss==pos) ? sDir.size() : pos+dis.size();
	return make_pair(fpr,
			(stmiss==pos ? sEmptyString : sDir.substr(pos)) + sFile.substr(0, maxFnameLen));
}

static void DelTree(const string &what)
{
	class killa : public IFileHandler
	{
		virtual bool ProcessRegular(const mstring &sPath, const struct stat &data)
		{
			::unlink(sPath.c_str()); // XXX log some warning?
			return true;
		}
		bool ProcessOthers(const mstring &sPath, const struct stat &x)
		{
			return ProcessRegular(sPath, x);
		}
		bool ProcessDirAfter(const mstring &sPath, const struct stat &x)
		{
			::rmdir(sPath.c_str()); // XXX log some warning?
			return true;
		}
	} hh;
	DirectoryWalk(what, &hh, false, false);
}

struct fctLessThanCompMtime
{
	string m_base;
	fctLessThanCompMtime(const string &base) :
		m_base(base)
	{
	}
	bool operator()(const string &s1, const string &s2) const
	{
		struct stat stbuf1, stbuf2;
		tStrPos cpos1(FindComPos(s1) ), cpos2(FindComPos(s2));
		if(cpos1!=cpos2)
			return cpos1 > cpos2; // sfx found -> less than npos (=w/o sfx) -> be smaller
		// s1 is lesser when its newer
		if (::stat((m_base + s1).c_str(), &stbuf1))
			stbuf1.st_mtime = 0;
		if (::stat((m_base + s2).c_str(), &stbuf2))
			stbuf2.st_mtime = 0;
		return stbuf1.st_mtime > stbuf2.st_mtime;
	}
};

struct tCompByState : public tFingerprint
{
	tCompByState(const tFingerprint &re) : tFingerprint(re) {}
	bool operator()(const tPatchEntry &other) const { return other.fprState == *this; }
};

/*
struct tCompByEnd : private mstring
{
	tCompByEnd(const mstring &x) : mstring(x) {}
	bool operator()(const mstring &haystack) const { return endsWith(haystack, *this);}
};
*/

tFingerprint * BuildPatchList(string sFilePathAbs, deque<tPatchEntry> &retList)
{
	retList.clear();
	string sLine;
	static tFingerprint ret;
	ret.csType=CSTYPE_INVALID;

	filereader reader;
	if(!reader.OpenFile(sFilePathAbs))
		return NULL;

	enum { eCurLine, eHistory, ePatches} eSection;
	eSection=eCurLine;

	UINT peAnz(0);

	// This code should be tolerant to minor changes in the format

	tStrVec tmp;
	while(reader.GetOneLine(sLine))
	{
		int nTokens=Tokenize(sLine, SPACECHARS, tmp);
		if(3==nTokens)
		{
			if(tmp[0] == "SHA1-Current:")
				ret.Set(tmp[1], CSTYPE_SHA1, atoofft(tmp[2].c_str()));
			else
			{
				tFingerprint fpr;
				fpr.Set(tmp[0], CSTYPE_SHA1, atoofft(tmp[1].c_str()));

				if(peAnz && retList[peAnz%retList.size()].patchName == tmp[2])
					// oh great, this is also our target
				{
					if (eHistory == eSection)
						retList[peAnz%retList.size()].fprState = fpr;
					else
						retList[peAnz%retList.size()].fprPatch = fpr;
				}
				else
				{
					retList.resize(retList.size()+1);
					retList.back().patchName=tmp[2];
					if (eHistory == eSection)
						retList.back().fprState = fpr;
					else
						retList.back().fprPatch = fpr;
				}

				peAnz++;
			}
		}
		else if(1==nTokens)
		{
			if(tmp[0] == "SHA1-History:")
				eSection=eHistory;
			else if(tmp[0] == "SHA1-Patches:")
				eSection=ePatches;
			else
				return NULL;
		}
		else if(nTokens) // not null but weird count
			return NULL; // error
	}

	return ret.csType != CSTYPE_INVALID ? &ret : NULL;
}

bool tCacheMan::PatchFile(const string &srcRel,
		const string &diffIdxPathRel, tPListConstIt pit, tPListConstIt itEnd,
		const tFingerprint *verifData)
{
	if(m_bVerbose)
		SendFmt()<< "Patching from " << srcRel << " via " << diffIdxPathRel << "...<br>\n";

	string sFinalPatch(CACHE_BASE+"_actmp/combined.diff");
	if(diffIdxPathRel.length()<=diffIdxSfx.length())
		return false; // just be sure about that

	FILE_RAII pf;
	for(;pit!=itEnd; pit++)
	{
		string pfile(diffIdxPathRel.substr(0, diffIdxPathRel.size()-diffIdxSfx.size()+6)
				+pit->patchName+".gz");
		if(!DownloadSimple(pfile))
		{
			m_indexFilesRel.erase(pfile); // remove the mess for sure
			SendFmt() << "Failed to download patch file " << pfile << " , stop patching...<br>";
			return false;
		}

		if(CheckAbortCondition())
			return false;

		SetFlags(pfile).parseignore=true; // not an ifile, never parse this
		::mkbasedir(sFinalPatch);
		if(!pf.p && ! (pf.p=fopen(sFinalPatch.c_str(), "w")))
		{
			SendChunk("Failed to create intermediate patch file, stop patching...<br>");
			return false;
		}
		tFingerprint probe;
		if(!probe.ScanFile(CACHE_BASE+pfile, CSTYPE_SHA1, true, pf.p))
		{

			if(CheckAbortCondition())
				return false;

			if(m_bVerbose)
				SendFmt() << "Failure on checking of intermediate patch data in " << pfile << ", stop patching...<br>";
			return false;
		}
		if ( ! (probe == pit->fprPatch))
		{
			SendFmt()<< "Bad patch data in " << pfile <<" , stop patching...<br>";
			return false;
		}
	}
	if(pf.p)
	{
		::fprintf(pf.p, "w patch.result\n");
		::fflush(pf.p); // still a slight risk of file closing error but it's good enough for now
		if(::ferror(pf.p))
		{
			SendChunk("Patch merging error<br>");
			return false;
		}
		checkForceFclose(pf.p);
	}

	if(m_bVerbose)
		SendChunk("Patching...<br>");

	tSS cmd;
	cmd << "cd '" << CACHE_BASE << "_actmp' && red patch.base < combined.diff";
	if (::system(cmd.c_str()))
	{
#ifdef DEBUG
		SendFmt() << "Command failed: " << cmd << "<br>\n";
#endif
		return false;
	}

	tFingerprint probe;
	string respathAbs = CACHE_BASE + sPatchResRel;
	if (!probe.ScanFile(respathAbs, CSTYPE_SHA1, false))
	{
#ifdef DEBUG
		SendFmt() << "Scan of " << respathAbs << " failed<br>\n";
#endif
		return false;
	}

	if(verifData && probe != *verifData)
	{
#ifdef DEBUG
		SendFmt() << "Verification against: " << respathAbs << " failed<br>\n";
#endif
		return false;
	}

	return true;

}

bool tCacheMan::GetAndCheckHead(cmstring & sDataFileRel, cmstring &sReferencePathRel,
		off_t nWantedSize)
{
	time_t timeStarted=time(0);
	class tHeadOnlyStorage : public fileitem
	{
	public:
		tHeadOnlyStorage(cmstring &path) : fileitem(path)
		{
			m_bAllowStoreData=false;
			m_bHeadOnly=true;
		}
		int AddHeadData(cmstring &path)
		{
			return m_head.LoadFromFile(path+".head");
		}
		void ForceStoreHead()
		{
			m_head.StoreToFile(m_sPath+".head");
		}
	};
	tHeadOnlyStorage *p=new tHeadOnlyStorage(sDataFileRel);
	tFileItemPtr pItem(static_cast<fileitem*>(p));
	string sErr;
	if(!pItem
			|| p->AddHeadData(SABSPATH(sReferencePathRel))<=0
			|| !RefetchFile(pItem, sReferencePathRel, sErr))
		return false;
	p->ForceStoreHead();

	const char *plen=pItem->GetHeaderUnlocked().h[header::CONTENT_LENGTH];
	cmstring sHeadPath=SABSPATH(sDataFileRel)+".head";
	struct stat stbuf;
	if(plen && nWantedSize == atoofft(plen)
			&& 0==stat(sHeadPath.c_str(), &stbuf)
			)
	{
		return timeStarted<=stbuf.st_mtime; // looks sane, fetched right now -> thrust the data
	}
	return false;
}

bool tCacheMan::Inject(cmstring &from, cmstring &to)
{
	// XXX should it really filter it here?
	if(GetFlags(to).uptodate)
		return true;

#ifdef DEBUG
	SendFmt()<<"Replacing "<<to<<" with " << from <<  "<br>\n";
#endif

	header h;
	filereader data;

	if(h.LoadFromFile(SABSPATH(from+".head")) <= 0 || !data.OpenFile(SABSPATH(from), true))
	{
#ifdef DEBUG
	SendFmt()<<"Cannot read "<<from<<".head<br>\n";
#endif
		return false;
	}

	tFileItemPtr fi=fileitem::GetFileItem(to);
	if (!fi)
		return false;

	bool bRes=(fi->Setup(true) != FIST_ERROR
			&& fi->DownloadStartedStoreHeader(h, NULL)
			&& fi->StoreFileData(data.GetBuffer(), data.GetSize())
			&& fi->StoreFileData(NULL, 0)
			&& fi->GetStatus() == FIST_COMPLETE);

	UpdateFingerprint(to, -1, NULL, NULL);

	fi->Unreg();

#ifdef DEBUG
	if(!bRes) SendChunk("<font color=red>Inject failed</font><br>");
#endif
	tIfileAttribs &atts = SetFlags(to);
	atts.uptodate=atts.vfile_ondisk=bRes;
	return bRes;
}


bool tCacheMan::Propagate(const string &donor, tContId2eqClass::iterator eqClassIter)
{
#ifdef DEBUG
	SendFmt()<< "Installing " << donor << "<br>\n";
	bool nix=StrHas(donor, "debrep/dists/experimental/main/binary-amd64/Packages");
#endif

	const tStrDeq &tgts = eqClassIter->second.paths;

	// we know it's uptodate, make sure that attempts to modify it in background
	// it also needs to be reliable disarmed before releasing, use RAII for that
	struct keeper
	{
		tFileItemPtr pItem;
		keeper(const string &path, bool y) { if(y) pItem=fileitem::GetFileItem(path); }
		~keeper() { if(pItem) pItem->Unreg(); }
	} src(donor, GetFlags(donor).uptodate);

	//tStrPos cpos=FindComPos(donor);

	// make sure that the source file has the .head file and has the final size
	if(src.pItem)
	{
		src.pItem->Setup(false);
		// something changed it in meantime?!
		if(src.pItem->GetStatus() != FIST_COMPLETE)
			return false;

		/* Disable the double-check, doesn't work well when compressed files serves as content id
		if(stmiss == cpos)
		{
			off_t nCheckSize(0); // just to be sure
			src.pItem->GetStatusUnlocked(nCheckSize);
			if(eqClassIter->first.first.size != nCheckSize) // hm?
				return false;
		}
		*/
	}

	int nInjCount=0;
	for (tStrDeq::const_iterator it = tgts.begin(); it != tgts.end(); it++)
	{
		const string &tgtCand=*it;
		if(donor == tgtCand)
			continue;
		const tIfileAttribs &flags=GetFlags(tgtCand);
		if(!flags.vfile_ondisk)
			continue;

		if(FindCompIdx(donor) == FindCompIdx(tgtCand)) // same compression type -> replace it?
		{
			// counts fresh file as injected, no need to recheck them in Inject()
			if (flags.uptodate || Inject(donor, tgtCand))
				nInjCount++;
#ifdef DEBUG
			else
				SendFmt() << "Inject failed<br>\n";
#endif
		}
	}

	// defuse some stuff located in the same directory, like .gz variants of .bz2 files
	for (tStrDeq::const_iterator it = tgts.begin(); it != tgts.end(); it++)
	{
		const tIfileAttribs &myState = GetFlags(*it);
		if(!myState.vfile_ondisk || !myState.uptodate || myState.parseignore)
			continue;

		tStrPos cpos=FindComPos(*it);
		string sBasename=it->substr(0, cpos);
		string sux=cpos==stmiss ? "" : it->substr(FindComPos(*it));
		for(cmstring *ps=suxeWempty; ps<suxeWempty+_countof(suxeWempty); ps++)
		{
			if(sux==*ps) continue; // touch me not
			tS2IDX::iterator kv=m_indexFilesRel.find(sBasename+*ps);
			if(kv!=m_indexFilesRel.end() && kv->second.vfile_ondisk)
			{
				kv->second.parseignore=true; // gotcha
#ifdef DEBUG
				SendFmt() << "Defused bro: " << sBasename<<*ps<<"<br>\n";
#endif
			}
		}
	}

	if(!nInjCount && endsWith(donor, sPatchResRel))
	{
		/*
		 * Now that's a special case, the file was patched and
		 * we need to store the latest state somewhere. But there
		 * was no good candidate to keep that copy. Looking closer for
		 * some appropriate location.
		 * */
		string sLastRessort;
		for (tStrDeq::const_iterator it = tgts.begin(); it != tgts.end(); it++)
		{
			if(stmiss!=FindComPos(*it))
				continue;
			// ultimate ratio... and then use the shortest path
			if(sLastRessort.empty() || sLastRessort.size()>it->size())
				sLastRessort=*it;
		}
		Inject(donor, sLastRessort);
	}

	return true;
}

void tCacheMan::StartDlder()
{
	if(!m_pDlcon)
		m_pDlcon=new dlcon(true);
}

void tCacheMan::UpdateIndexFiles()
{
	LOGSTART("expiration::UpdateIndexFiles()");

	SendChunk("<b>Bringing index files up to date...</b><br>\n");

	string sErr; // for download error output
	const string sPatchBaseAbs=CACHE_BASE+sPatchBaseRel;
	mkbasedir(sPatchBaseAbs);

	// just reget them as-is and we are done
	if (m_bForceDownload)
	{
		for (tS2IDX::const_iterator it = m_indexFilesRel.begin(); it != m_indexFilesRel.end(); it++)
		{
			// nope... tell the fileitem to ignore file data instead ::truncate(SZABSPATH(it->first), 0);
			if (!DownloadIdxFile(it->first, sErr))
				m_nErrorCount+=!m_indexFilesRel[it->first].forgiveDlErrors;
		}
		ERRMSGABORT;
		return;
	}

	typedef map<string, tContId> tFile2Cid;

#ifdef DEBUG
	SendChunk("<br><br><b>STARTING ULTIMATE INTELLIGENCE</b><br><br>");
#endif

	/*
	 * Update all Release files
	 *
	 */
	class releaseStuffReceiver : public ifileprocessor
	{
	public:
		tFile2Cid m_file2cid;
		virtual void HandlePkgEntry(const tRemoteFileInfo &entry, bool bUncompressForChecksum)
		{
			if(bUncompressForChecksum) // dunno, ignore
				return;

			tStrPos compos=FindComPos(entry.sFileName);

			// skip some obvious junk and its gzip version
			if(0==entry.fpr.size || (entry.fpr.size<33 && stmiss!=compos))
				return;

			m_file2cid[entry.sDirectory+entry.sFileName] = BuildEquivKey(entry.fpr,
					entry.sDirectory, entry.sFileName, compos);
		}
	};

	for(tS2IDX::const_iterator it=m_indexFilesRel.begin(); it!=m_indexFilesRel.end(); it++)
	{
		const string &sPathRel=it->first;

		if(!endsWith(sPathRel, relKey))
			continue;

		if(!DownloadIdxFile(sPathRel, sErr))
		{
			m_nErrorCount+=(!m_indexFilesRel[sPathRel].hideDlErrors);
			if(!m_indexFilesRel[sPathRel].hideDlErrors)
				SendFmt() << "<font color=\"red\">" << sErr << "</font><br>\n";

			if(CheckAbortCondition())
				return;

			continue;
		}
		m_indexFilesRel[sPathRel].uptodate=true;

		releaseStuffReceiver recvr;
		ParseAndProcessIndexFile(recvr, sPathRel, EIDX_RELEASE);

		if(recvr.m_file2cid.empty())
			continue;

		for(tFile2Cid::iterator it=recvr.m_file2cid.begin(); it!=recvr.m_file2cid.end(); it++)
		{
			string sNativeName=it->first.substr(0, FindComPos(it->first));
			tContId sCandId=it->second;
			// find a better one which serves as the flag content id for the whole group
			for(cmstring *ps=suxeByLhood; ps<suxeByLhood+_countof(suxeByLhood); ps++)
			{
				tFile2Cid::iterator it2=recvr.m_file2cid.find(sNativeName+*ps);
				if(it2 != recvr.m_file2cid.end())
					sCandId=it2->second;
			}
			tClassDesc &tgt=m_eqClasses[sCandId];
			tgt.paths.push_back(it->first);

			// pick up the id for bz2 verification later
			if(tgt.bz2VersContId.second.empty() && endsWithSzAr(it->first, ".bz2"))
				tgt.bz2VersContId=it->second;

			// also the index file id
			if(tgt.diffIdxId.second.empty()) // XXX if there is no index at all, avoid repeated lookups somehow?
			{
				tFile2Cid::iterator j = recvr.m_file2cid.find(sNativeName+diffIdxSfx);
				if(j!=recvr.m_file2cid.end())
					tgt.diffIdxId=j->second;
			}

			// and while we are at it, check the checksum of small files in order to reduce server requests
			if(it->second.first.size<10000 && ContHas(m_indexFilesRel, it->first))
			{
				if(it->second.first.CheckFile(CACHE_BASE+it->first))
					m_indexFilesRel[it->first].uptodate=true;
			}
		}
	}

	/*
	for(tContId2eqClass::iterator it=m_eqClasses.begin(); it!=m_eqClasses.end();it++)
	{
		SendFmt()<<"__TID: " << it->first.first<<it->first.second<<"<br>"
				<< "bz2TID:" << it->second.bz2VersContId.first<< it->second.bz2VersContId.second<<"<br>"
				<< "idxTID:"<<it->second.diffIdxId.first << it->second.diffIdxId.second <<"<br>"
				<< "Paths:<br>";
		for(tStrDeq::const_iterator its=it->second.paths.begin();
				its!=it->second.paths.end(); its++)
		{
			SendFmt()<<"&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;" << *its<<"<br>";
		}
	}
	*/

	if(CheckAbortCondition())
		return;

	/*
	 *
	 * OK, the equiv-classes map is built, now post-process the knowledge
	 *
	 * First, strip the list down to those which are at least partially present in the cache
	 */

	for(tContId2eqClass::iterator it=m_eqClasses.begin(); it!=m_eqClasses.end();)
	{
		bool bFound=false;
		for(tStrDeq::const_iterator its=it->second.paths.begin();
				its!= it->second.paths.end(); its++)
		{
			if(GetFlags(*its).vfile_ondisk)
			{
				bFound=true;
				break;
			}
		}
		if(bFound)
			++it;
		else
			m_eqClasses.erase(it++);
	}
	ERRMSGABORT;

	// Let the most recent files be in the front of the list, but the uncompressed ones have priority
	for(tContId2eqClass::iterator it=m_eqClasses.begin(); it!=m_eqClasses.end();it++)
	{
		sort(it->second.paths.begin(), it->second.paths.end(), fctLessThanCompMtime(CACHE_BASE));
		// and while we are at it, give them pointers back to the eq-classes
		for(tStrDeq::const_iterator its=it->second.paths.begin();
						its!= it->second.paths.end(); its++)
		{
			SetFlags(*its).bros=&(it->second.paths);
		}
	}

#ifdef DEBUG
	for(tContId2eqClass::iterator it=m_eqClasses.begin(); it!=m_eqClasses.end();it++)
	{
		SendFmt()<<"TID: " << it->first.first<<it->first.second<<"<br>"
				<< "bz2TID:" << it->second.bz2VersContId.first<< it->second.bz2VersContId.second<<"<br>"
				<< "idxTID:"<<it->second.diffIdxId.first << it->second.diffIdxId.second <<"<br>"
				<< "Paths:<br>";
		for(tStrDeq::const_iterator its=it->second.paths.begin();
				its!=it->second.paths.end(); its++)
		{
			SendFmt()<<"&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;" << *its<<"<br>";
		}
	}
	SendChunk("<br><br><b>FOLLOWING VOID WHICH BINDS</b><br><br>");
#endif

	DelTree(SABSPATH("_actmp")); // do one to test the permissions
	/* wrong check but ignore for now
	if(::access(SZABSPATH("_actmp"), F_OK))
		SendFmt()
		<< "<font color=\"orange\">Warning, failed to purge temporary directory "
		<< CACHE_BASE << "_actmp/, this could impair some additional functionality"
				"</font><br>";
*/

	// Iterate over classes and do patch-update where possible
	for(tContId2eqClass::iterator cid2eqcl=m_eqClasses.begin(); cid2eqcl!=m_eqClasses.end();cid2eqcl++)
	{
		tContId2eqClass::iterator itDiffIdx; // iterator pointing to the patch index descriptor
		int nProbeCnt(3);
		string patchidxFileToUse;
		deque<tPatchEntry> patchList;
		tFingerprint *pEndSum;
		tPListConstIt itPatchStart;

		if(CheckAbortCondition())
			return;

		DelTree(SABSPATH("_actmp"));

		if (cid2eqcl->second.diffIdxId.second.empty() || m_eqClasses.end() == (itDiffIdx
				= m_eqClasses.find(cid2eqcl->second.diffIdxId)) || itDiffIdx->second.paths.empty())
			goto NOT_PATCHABLE; // no patches available

		// iterate over patch paths and fine a present one which is most likely the most recent one
		for (tStrDeq::const_iterator ppit = itDiffIdx->second.paths.begin(); ppit
				!= itDiffIdx->second.paths.end(); ppit++)
		{
			if (m_indexFilesRel[*ppit].vfile_ondisk)
			{
				patchidxFileToUse = *ppit;
				break;
			}
		}
		if (patchidxFileToUse.empty()) // huh, not found? Then just take the first one
			patchidxFileToUse = itDiffIdx->second.paths.front();

		if (!DownloadIdxFile(patchidxFileToUse, sErr))
			continue;

		if(CheckAbortCondition())
			return;

		pEndSum=BuildPatchList(CACHE_BASE+patchidxFileToUse, patchList);

		if(!pEndSum)
			goto NOT_PATCHABLE;

		/*
		for(deque<tPatchEntry>::const_iterator itPinfo = patchList.begin();
				pEndSum && itPinfo!=patchList.end(); ++itPinfo)
		{
			SendFmt() << itPinfo->patchName<< " -- " << itPinfo->fprState
					<<" / " << itPinfo->fprPatch<<  " <br>";
		}
		*/

		/* ok, patches should be available, what to patch? Probe up to three of the most recent ones */
		// XXX now ideally, it should unpack on each test into an extra file and then get the one which matched best. But it's too cumbersome, and if the code works correctly, the first hit should always the best version
		for(tStrDeq::const_iterator its=cid2eqcl->second.paths.begin();
				nProbeCnt-->0 && its!= cid2eqcl->second.paths.end(); its++)
		{
			FILE_RAII df;
			tFingerprint probe;
			::mkbasedir(sPatchBaseAbs);
			df.p = fopen(sPatchBaseAbs.c_str(), "w");
			if(!df.p)
			{
				SendFmt() << "Cannot write temporary patch data to " << sPatchBaseAbs << "<br>";
				break;
			}
			if (GetFlags(*its).vfile_ondisk)
			{
				header h;
				if(h.LoadFromFile(SABSPATH(*its)+ ".head")<=0
						|| ! h.h[header::CONTENT_LENGTH]
						|| GetFileSize(SABSPATH(*its), -2) != atoofft(h.h[header::CONTENT_LENGTH]))
				{
#ifdef DEBUG
					SendChunk("########### Header looks suspicious<br>");
#endif
					continue;
				}
#ifdef DEBUG
				SendFmt() << "########### Testing file: " << *its << " as patch base candidate<br>";
#endif
				if (probe.ScanFile(CACHE_BASE + *its, CSTYPE_SHA1, true, df.p))
				{

					if(CheckAbortCondition())
						return;

					// Hit the current state, no patching needed for it?
					if(probe == *pEndSum)
					{
						// since we know the stuff is fresh, no need to refetch it later
						m_indexFilesRel[*its].uptodate=true;
						if(m_bVerbose)
							SendFmt() << "Found fresh version in " << *its << "<br>";

						Propagate(*its, cid2eqcl);

						if(CheckAbortCondition())
							return;

						goto CONTINUE_NEXT_GROUP;
					}
					// or found at some previous state, try to patch it?
					else if (patchList.end() != (itPatchStart = find_if(patchList.begin(),
							patchList.end(), tCompByState(probe))))
					{
						df.close(); // write the whole file to disk!
						// XXX for now, construct a replacement header based on some assumptions
						// tried hard and cannot imagine any case where this would be harmful
						if (h.h[header::XORIG])
						{
							string s(h.h[header::XORIG]);
							h.set(header::XORIG, s.substr(0, FindComPos(s)));
						}

						if(CheckAbortCondition())
							return;

						if (m_bVerbose)
							SendFmt() << "Found patching base candidate, unpacked to " << sPatchBaseAbs << "<br>";

						if (PatchFile(sPatchBaseAbs, patchidxFileToUse, itPatchStart, patchList.end(),
								pEndSum))
						{

							if(CheckAbortCondition())
								return;

							h.set(header::LAST_MODIFIED, "Sat, 26 Apr 1986 01:23:39 GMT+3");
							h.set(header::CONTENT_LENGTH, pEndSum->size);
							if (h.StoreToFile(SABSPATH(sPatchResRel) + ".head") <= 0)
							{
	#ifdef DEBUG
								SendFmt() << "############ Failed to store target header as "
										<< SABSPATH(sPatchResRel) << ".head<br>";
	#endif
								continue;
							}

							SendChunk("Patching result: succeeded<br>");
							Propagate(sPatchResRel, cid2eqcl);

							if(CheckAbortCondition())
								return;

							InstallBz2edPatchResult(cid2eqcl);

							if(CheckAbortCondition())
								return;

							break;
						}
						else
						{
							SendChunk("Patching result: failed<br>");
							// don't break, maybe the next one can be patched
						}

						if(CheckAbortCondition())
							return;
					}
				}
			}
		}

		// ok, now try to get a good version of that file and install this into needed locations
		NOT_PATCHABLE:
		/*
		if(m_bVerbose)
			SendFmt() << "Cannot update " << it->first << " by patching, what next?"<<"<br>";
*/
		// prefer to download them in that order, no uncompressed versions because
		// mirrors usually don't have them
		static const string preComp[] = { ".lzma", ".xz", ".bz2", ".gz"};
		for (const string *ps = preComp; ps < preComp + _countof(preComp); ps++)
		{
			for (tStrDeq::const_iterator its = cid2eqcl->second.paths.begin(); its
					!= cid2eqcl->second.paths.end(); its++)
			{
				cmstring &cand=*its;
				if(!endsWith(cand, *ps))
					continue;
				if(DownloadIdxFile(cand, sErr))
				{
					if(CheckAbortCondition())
						return;
					if(Propagate(cand, cid2eqcl)) // all locations are covered?
						goto CONTINUE_NEXT_GROUP;
				}
				else if(! GetFlags(cand).hideDlErrors)
					SendFmt() << "<font color=\"red\">" << sErr << "</font><br>\n";

				if(CheckAbortCondition())
					return;
			}
		}

		CONTINUE_NEXT_GROUP:

		if(CheckAbortCondition())
			return;
	}

#ifdef DEBUG
	SendChunk("<br><br><b>NOW GET THE REST</b><br><br>");
#endif

	// fetch all remaining stuff
	for(tS2IDX::citer it=m_indexFilesRel.begin(); it!=m_indexFilesRel.end(); it++)
	{
		if(it->second.uptodate || it->second.parseignore || !it->second.vfile_ondisk)
			continue;
		string sErr;
		if(DownloadIdxFile(it->first, sErr))
			continue;
		m_nErrorCount+=(!it->second.forgiveDlErrors);
		if(!it->second.hideDlErrors)
			SendFmt() << "<font color=\"red\">" << sErr << "</font><br>\n";
	}

}

void tCacheMan::InstallBz2edPatchResult(tContId2eqClass::iterator eqClassIter)
{
	if(!acfg::recompbz2)
		return;

	string sFreshBz2Rel;
	tFingerprint &bz2fpr=eqClassIter->second.bz2VersContId.first;
	string sRefBz2Rel;

	for (tStrDeq::const_iterator it = eqClassIter->second.paths.begin(); it
			!= eqClassIter->second.paths.end(); it++)
	{
		if (endsWithSzAr(*it, ".bz2"))
		{
			const tIfileAttribs &fl = GetFlags(*it);
			if (fl.vfile_ondisk)
			{
				// needs a reference location to get the HTTP headers for, pickup something
				if(sRefBz2Rel.empty())
					sRefBz2Rel=*it;

				if (fl.uptodate)
				{
					sFreshBz2Rel = *it;
					goto inject_bz2s;
				}
				else
				{
					if(sFreshBz2Rel.empty())
						sFreshBz2Rel = sPatchResRel+".bz2";
					// continue searching, there might be a working version
				}
			}
		}
	}

	// not skipped this code... needs recompression then?
	if (sFreshBz2Rel.empty())
		return;

	// ok, it points to the temp file then, create it

	if (Bz2compressFile(SZABSPATH(sPatchResRel), SZABSPATH(sFreshBz2Rel))
			&& bz2fpr.CheckFile(SABSPATH(sFreshBz2Rel))
	// fileitem implementation may nuke the data on errors... doesn't matter here
			&& GetAndCheckHead(sFreshBz2Rel, sRefBz2Rel, bz2fpr.size))
	{
		if (m_bVerbose)
			SendFmt() << "Compressed into " << sFreshBz2Rel << "<br>\n";
	}
	else
		return;

	inject_bz2s:
	// use a recursive call to distribute bz2 versions

	if(CheckAbortCondition())
		return;

	if (!sFreshBz2Rel.empty())
	{
#ifdef DEBUG
		SendFmt() << "Recursive call to install the bz2 version from " << sFreshBz2Rel << "<br>";
#endif
		Propagate(sFreshBz2Rel, eqClassIter);
	}
}

tCacheMan::enumIndexType tCacheMan::GuessIndexType(const mstring &sPath)
{
	tStrPos pos = sPath.rfind(SZPATHSEP);
	string sPureIfileName = (stmiss == pos) ? sPath : sPath.substr(pos + 1);
	stripSuffix(sPureIfileName, ".gz");
	stripSuffix(sPureIfileName, ".bz2");
	stripSuffix(sPureIfileName, ".xz");
	stripSuffix(sPureIfileName, ".lzma");
	if (sPureIfileName=="Packages") // Debian's Packages file
	return EIDX_PACKAGES;
	if (endsWithSzAr(sPureIfileName, ".db.tar"))
		return EIDX_ARCHLXDB;
	if (sPureIfileName == "setup")
		return EIDX_CYGSETUP;

	if (sPureIfileName == "repomd.xml")
		return EIDX_SUSEREPO;

	if (sPureIfileName.length() > 50 && endsWithSzAr(sPureIfileName, ".xml") && sPureIfileName[40]
			== '-')
		return EIDX_SUSEOTHER;

	if (sPureIfileName == "Sources")
		return EIDX_SOURCES;

	if (sPureIfileName == "Release")
		return EIDX_RELEASE;

	if (sPureIfileName == "Index")
		return endsWithSzAr(sPath, "i18n/Index") ? EIDX_TRANSIDX : EIDX_DIFFIDX;


	return EIDX_UNSUPPORTED;
}

bool tCacheMan::ParseAndProcessIndexFile(ifileprocessor &ret, const MYSTD::string &sPath,
		enumIndexType idxType)
{

#ifdef DEBUG
	bool bNix=StrHas(sPath, "/i18n/");
#endif


	// pre calc relative base folders for later
	string sDirname(SZPATHSEP);
	string sDebBaseDir = sDirname; // may differ from sDirname if the path looks like a Debian mirror path
	tStrPos pos = sPath.rfind(CPATHSEP);
	if(stmiss!=pos)
	{
		sDirname.assign(sPath, 0, pos+1);
		pos=sDirname.rfind("/dists/");
		if(stmiss!=pos)
			sDebBaseDir.assign(sDirname, 0, pos+1);
		else
			sDebBaseDir=sDirname;
	}
	else
	{
		m_nErrorCount++;
		SendFmt() << "Unexpected index file without subdir found: " << sPath;
		return false;
	}


	LOGSTART("expiration::_ParseAndProcessIndexFile");
	filereader reader;

	if (!reader.OpenFile(CACHE_BASE+sPath))
	{
		aclog::errnoFmter err;
		SendFmt()<<"<font color=orange>WARNING: unable to open "<<sPath<<"("
				<< err << ")</font><br>\n";
		return false;
	}

	// some common variables
	string sLine, key, val;
	tRemoteFileInfo info;
	info.SetInvalid();
	tStrVec vsMetrics;
	string sStartMark;
	bool bUse(false);

	enumIndexType origIdxType=idxType;


	REDO_AS_TYPE:
	switch(idxType)
	{
	case EIDX_PACKAGES:
		LOG("filetype: Packages file");
		static const string sMD5sum("MD5sum"), sFilename("Filename"), sSize("Size");

		for (bool bNoEof=true; bNoEof;)
		{
			bNoEof=reader.GetOneLine(sLine);
			trimBack(sLine);
			//cout << "file: " << *it << " line: "  << sLine<<endl;
			if (sLine.empty() || !bNoEof)
			{
				if(info.IsUsable())
					ret.HandlePkgEntry(info, false);
				info.SetInvalid();

				if(CheckAbortCondition())
					return true; // XXX: should be rechecked by the caller ASAP!

				continue;
			}
			else if (_ParseLine(sLine, key, val))
			{
				// not looking for data we already have
				if(key==sMD5sum)
					info.fpr.Set(val, CSTYPE_MD5, info.fpr.size);
				else if(key==sSize)
					info.fpr.size=atoofft(val.c_str());
				else if(key==sFilename)
				{
					info.sDirectory=sDebBaseDir;
					tStrPos pos=val.rfind(SZPATHSEPUNIX);
					if(pos==stmiss)
						info.sFileName=val;
					else
					{
						info.sFileName=val.substr(pos+1);
						info.sDirectory.append(val, 0, pos+1);
					}
				}
			}
		}
		break;
	case EIDX_ARCHLXDB:
		LOG("assuming Arch Linux package db");
		{
			UINT nStep = 0;
			enum tExpData
			{
				_fname, _csum, _csize, _nthng
			} typehint(_nthng);

			while (reader.GetOneLine(sLine)) // last line doesn't matter, contains tar's padding
			{
				trimLine(sLine);

				if (nStep >= 2)
				{
					if (info.IsUsable())
						ret.HandlePkgEntry(info, false);
					info.SetInvalid();
					nStep = 0;

					if (CheckAbortCondition())
						return true;

					continue;
				}
				else if (endsWithSzAr(sLine, "%FILENAME%"))
					typehint = _fname;
				else if (endsWithSzAr(sLine, "%CSIZE%"))
					typehint = _csize;
				else if (endsWithSzAr(sLine, "%MD5SUM%"))
					typehint = _csum;
				else
				{
					switch (typehint)
					{
					case _fname:
						info.sDirectory = sDirname;
						info.sFileName = sLine;
						nStep = 0;
						break;
					case _csum:
						info.fpr.Set(sLine, CSTYPE_MD5, info.fpr.size);
						nStep++;
						break;
					case _csize:
						info.fpr.size = atoofft(sLine.c_str());
						nStep++;
						break;
					default:
						continue;
					}
					// next line is void for now
					typehint = _nthng;
				}
			}
		}
		break;
	case EIDX_CYGSETUP:
		LOG("assuming Cygwin package setup listing");

		for (bool bNoEof=true; bNoEof;)
		{
			if(CheckAbortCondition())
				return true;

			static const string cygkeys[]={"install: ", "source: "};

			bNoEof=reader.GetOneLine(sLine);
			trimBack(sLine);
			for(UINT i=0;i<_countof(cygkeys);i++)
			{
				if(!startsWith(sLine, cygkeys[i]))
					continue;
				//LOG("interesting line: " << sLine);
				if (3 == Tokenize(sLine, "\t ", vsMetrics, false, cygkeys[i].length())
						&& info.fpr.Set(vsMetrics[2], CSTYPE_MD5, atoofft(
								vsMetrics[1].c_str())))
				{
					tStrPos pos = vsMetrics[0].rfind(SZPATHSEPUNIX);
					if (pos == stmiss)
					{
						info.sFileName = vsMetrics[0];
						info.sDirectory = sDirname;
					}
					else
					{
						info.sFileName = vsMetrics[0].substr(pos + 1);
						info.sDirectory = sDirname + vsMetrics[0].substr(0, pos + 1);
					}
					ret.HandlePkgEntry(info, false);
					info.SetInvalid();
				}
			}
		}
		break;
	case EIDX_SUSEREPO:
		LOG("SUSE index file, entry level");
		while(reader.GetOneLine(sLine))
		{
			Tokenize(sLine, "\"/", vsMetrics);
			for(UINT i=0;i<vsMetrics.size();i++)
			{
				LOG("testing filename: " << vsMetrics[i]);
				if(!endsWithSzAr(vsMetrics[i], ".xml.gz"))
					continue;
				LOG("index basename: " << vsMetrics[i]);
				info.sFileName = vsMetrics[i];
				info.sDirectory = sDirname;
				ret.HandlePkgEntry(info, false);
				info.SetInvalid();
			}
		}
		break;
	case EIDX_SUSEOTHER:
		LOG("SUSE list file, pickup any valid filename ending in .rpm");
		while(reader.GetOneLine(sLine))
		{
			Tokenize(sLine, "\"'><=/", vsMetrics);
			for(UINT i=0;i<vsMetrics.size();i++)
			{
				LOG("testing token: " << vsMetrics[i]);
				if(!endsWithSzAr(vsMetrics[i], ".rpm"))
					continue;
				LOG("RPM basename: " << vsMetrics[i]);
				info.sFileName = vsMetrics[i];
				info.sDirectory = sDirname;
				ret.HandlePkgEntry(info, false);
				info.SetInvalid();
			}
		}
		break;
	/* not used for now, just covered by wfilepat
	else if( (sPureIfileName == "MD5SUMS" ||
			sPureIfileName == "SHA1SUMS" ||
			sPureIfileName == "SHA256SUMS") && it->find("/installer-") != stmiss)
	{

	}
	*/
	case EIDX_DIFFIDX:
		info.fpr.csType = CSTYPE_SHA1;
		info.sDirectory = sDirname; // same for all
		sStartMark = "SHA1-Patches:";
		idxType = EIDX_DEBSRCLIKE;
		goto REDO_AS_TYPE;

	case EIDX_TRANSIDX:
		//	quite similar but not the same
		info.fpr.csType = CSTYPE_SHA1;
		info.sDirectory = sDirname; // same for all
		sStartMark = "SHA1:";
		idxType = EIDX_DEBSRCLIKE;
		goto REDO_AS_TYPE;

		case EIDX_SOURCES:
			info.fpr.csType = CSTYPE_MD5;
			sStartMark="Files:";
			idxType = EIDX_DEBSRCLIKE;
			goto REDO_AS_TYPE;
	case EIDX_RELEASE:
			info.fpr.csType = CSTYPE_MD5;
			sStartMark="MD5Sum:";
			// fall-through, parser follows
	case EIDX_DEBSRCLIKE:
		for (bool bNoEof=true; bNoEof;)
		{
			bNoEof=reader.GetOneLine(sLine);
			trimBack(sLine);
			//if(sLine.find("unp_")!=stmiss)
			//	int nWtf=1;
			//cout << "file: " << *it << " line: "  << sLine<<endl;
			if (sLine.empty() || !bNoEof)
			{
				// optional, but better be sure
				info.sDirectory.clear();
				continue;
			}
			else if(startsWith(sLine, sStartMark))
				bUse=true;
			else if(startsWithSz(sLine, "Directory:"))
			{
				trimBack(sLine);
				tStrPos pos=sLine.find_first_not_of(SPACECHARS, 10);
				if(pos!=stmiss)
					info.sDirectory=sDebBaseDir+sLine.substr(pos)+SZPATHSEP;
			}
			else if(!startsWithSz(sLine, " ")) // unimportant directive or data block?
				bUse = false;
			else if (bUse && 3 == Tokenize(sLine, SPACECHARS, vsMetrics)
					&& info.fpr.Set(vsMetrics[0], info.fpr.csType, atoofft(
							vsMetrics[1].c_str())))
			{
				switch(origIdxType)
				{
				case EIDX_SOURCES:
					info.sFileName=vsMetrics[2];
					ret.HandlePkgEntry(info, false);
					break;
				case EIDX_DIFFIDX:
					info.sFileName=vsMetrics[2];
					ret.HandlePkgEntry(info, false);
					info.sFileName+=".gz";
					ret.HandlePkgEntry(info, true);
					break;
				case EIDX_TRANSIDX: // csum refers to the files as-is
					info.sFileName=vsMetrics[2];
					ret.HandlePkgEntry(info, false);
					break;
				case EIDX_RELEASE:
					// usually has subfolders
					pos=vsMetrics[2].rfind(SZPATHSEPUNIX);
					if (stmiss!=pos)
					{
						info.sFileName=vsMetrics[2].substr(pos+1);
						info.sDirectory=sDirname
								+vsMetrics[2].substr(0, pos+1);
					}
					else // something new in main folder? unlikely...
					{
						info.sFileName=vsMetrics[2];
						info.sDirectory=sDirname;
					}
					ret.HandlePkgEntry(info, false);
					break;
				default:
					ASSERT(!"Originally determined type cannot lead into this case!");
					break;
				}
			}

			if(CheckAbortCondition())
				return true;

		}
		break;

	default:
		SendChunk("<font color=orange>WARNING: unable to read this file (unsupported format)</font><br>\n");
		return false;
	}
	return reader.CheckGoodState(false);
}

void tCacheMan::_ProcessSeenIndexFiles()
{
	LOGSTART("expiration::_ParseVolatileFilesAndHandleEntries");
	for(tS2IDX::const_iterator it=m_indexFilesRel.begin(); it!=m_indexFilesRel.end(); it++)
	{
		if(CheckAbortCondition())
			return;

		const tIfileAttribs &att=it->second;
		enumIndexType itype = att.eIdxType;
		if(!itype)
			itype=GuessIndexType(it->first);
		if(!itype) // still unknown. Where does it come from? Just ignore.
			continue;
		if(att.parseignore || (!att.vfile_ondisk && !att.uptodate))
			continue;

		m_bNeedsStrictPathsHere=(m_bByPath ||
				m_bByChecksum || (it->first.find("/installer-") != stmiss));

		/*
		 * Actually, all that information is available earlier when analyzing index classes.
		 * Could be implemented there as well and without using .bros pointer etc...
		 *
		 * BUT: what happens if some IO error occurs?
		 * Not taking this risk-> only skipping when file was processed correctly.
		 *
		 */

		if(!m_bNeedsStrictPathsHere && att.alreadyparsed)
		{
			SendChunk(string("Skipping in ")+it->first+" (equivalent checks done before)<br>\n");
			continue;
		}

		SendChunk(string("Parsing metadata in ")+it->first+"<br>\n");

		if( ! ParseAndProcessIndexFile(*this, it->first, itype))
		{
			SendChunk("<font color=red>An error occured while reading this file, some contents may have been ignored.</font><br>\n");
			m_nErrorCount+=!m_indexFilesRel[it->first].forgiveDlErrors;
			continue;
		}
		else if(!m_bNeedsStrictPathsHere && att.bros)
		{
			for(tStrDeq::const_iterator broIt=att.bros->begin(); broIt!=att.bros->end(); broIt++)
			{
#ifdef DEBUG
				SendFmt() << "Marking " << *broIt << " as processed<br>";
#endif
				SetFlags(*broIt).alreadyparsed=true;
			}
		}

		//		cout << "found package files: "<< m_trashCandidates.size()<<endl;
	}
}

void expiration::HandlePkgEntry(const tRemoteFileInfo &entry, bool bUnpackForCsumming)
{
	LOGSTART2("expiration::_HandlePkgEntry:",
			"\ndir:" << entry.sDirectory <<
			"\nname: " << entry.sFileName <<
			"\nsize: " << entry.fpr.size <<
			"\ncsum: " << entry.fpr.GetCsAsString());

	// debian-installer files also need to do path checks
	// checksum mode also needs to be sure about path in order to not display false positives to the user

	tFileNdir startHook(entry.sFileName, m_bNeedsStrictPathsHere ? entry.sDirectory : "");
	for(tS2DAT::iterator it=m_trashCandSet.lower_bound(startHook);
			it!=m_trashCandSet.end();
			/* erases inside -> step forward there */ )
	{
		const tFileNdir &k = it->first; // shortcut

		// where are we, still at the right filename?
		if(entry.sFileName != k.file)
			return;

		tFingerprint & fprHave=it->second.fpr;
		const tFingerprint & fprNeed=entry.fpr;

		string sPathRel(k.dirRel+k.file);
		string sPathAbs(CACHE_BASE+sPathRel);
		header h;
		tS2IDX::const_iterator j;

		// needs to match the exact file location
		if(m_bNeedsStrictPathsHere)
		{
			string sEntrPath=entry.sDirectory+entry.sFileName;
			LOG("Checking exact path: " << sEntrPath << " vs. " << sPathAbs);
			pathTidy(sEntrPath);
			if(sPathRel != sEntrPath)
				goto keep_in_trash;
		}

		// Basic header checks. Skip if the file was forcibly updated/reconstructed before.
		if (m_bSkipHeaderChecks || it->second.bHeaderTestDone
				/*
				|| (j = m_indexFilesRel.find(sPathRel)) == m_indexFilesRel.end()
				|| j->second.uptodate
				*/
				)
		{
			LOG("Skipped header check for " << sPathRel);
		}
		else
		{
			LOG("Doing basic header checks");
			it->second.bHeaderTestDone = true;

			if (0<h.LoadFromFile(sPathAbs+".head"))
			{
				if (h.h[header::CONTENT_LENGTH])
				{
					off_t len=atoofft(h.h[header::CONTENT_LENGTH]);
					struct stat stinfo;
					off_t lenInfo=0;


#warning XXX RLLY? Review and maybe enable later, should work without it ATM
					/*
					// avoid duplicate stat call if the data from the includer is still there
					if(fprFile.bUnpack == false
							&& fprFile.csType == CSTYPE_INVALID
							&& fprFile.size>0)
					{
						//SendChunk("data from fingerprint");
						lenInfo=fprFile.size;
					}

					else
					*/if (0==stat(sPathAbs.c_str(), &stinfo))
					{
						//SendChunk("data from stat info");
						lenInfo = stinfo.st_size;
					}
					else
					{
						SendFmt() << ": error reading attributes, ignoring" << sPathRel;
						goto keep_in_trash;
					}

					if(len<lenInfo)
					{
						SendFmt() << "<font color=red>WARNING, header file of "
								<< sPathRel
								<< " reported too small file size ("
								<< len << " vs. " << lenInfo
								<< "), invalidating</font><br>\n";
						goto keep_in_trash;
					}
				}
				else
				{
					SendFmt() << "<font color=red>WARNING, header file of "
							<< sPathRel << " does not contain content length";
					goto keep_in_trash;
				}
			}
			else
			{
				if(!GetFlags(sPathRel).parseignore)
					SendFmt()<<"<font color=\"orange\">WARNING, header file missing for "
						<<sPathRel<<"</font><br>\n";
			}
		}

		if(m_bByChecksum)
		{
			bool bSkipDataCheck=false;

			// scan file if not done before or the checksum type differs
			if(fprNeed.csType != fprHave.csType)
			{
				if(!fprHave.ScanFile(sPathAbs, fprNeed.csType, bUnpackForCsumming))
				{
					// IO error? better keep it for now
					aclog::err(string("An error occured while checksumming ")
							+sPathAbs+", not touching it.");
					bSkipDataCheck=true;
				}
			}

			if ( !bSkipDataCheck)
			{
				if ( ! (fprHave == entry.fpr))
				{
					SendFmt()<<"<font color=red>BAD: "<<sPathAbs<<"</font><br>\n";
					goto keep_in_trash;
				}
			}
		}

		// ok, package matched, contents ok if checked, drop it from the removal list
		if (m_bVerbose)
			SendFmt() << "<font color=green>OK: " << sPathRel << "</font><br>\n";

		{ // dark side of goto use...
			tFileNdir headKey=it->first;
			headKey.file+=".head";
			m_trashCandHeadSet.erase(headKey);

			// delete&increment using a backup copy
			m_trashCandSet.erase(it++);
			continue;
		}

		continue;

		keep_in_trash:
		it++;
	}
}

inline void expiration::_RemoveAndStoreStatus(bool bPurgeNow)
{
	LOGSTART("expiration::_RemoveAndStoreStatus");
	FILE *f(NULL);
    if(!bPurgeNow)
    {
        string sDbFileAbs=CACHE_BASE+"_expending_dat";

        f = fopen(sDbFileAbs.c_str(), "w");
        if(!f)
        {
            SendChunk("Unable to open _expending.dat for writing, trying to recreate... ");
            ::unlink(sDbFileAbs.c_str());
            f=::fopen(sDbFileAbs.c_str(), "w");
            if(f)
                SendChunk("OK<br>\n");
            else
            {
                SendChunk("<font color=red>FAILED. ABORTING. Check filesystem and file permissions.");
                return;
            }
        }
    }

	int n(0);
	off_t tagSpace(0);
	for (tS2DAT::iterator it = m_trashCandSet.begin(); it != m_trashCandSet.end(); it++)
	{
		const tFileNdir &k = it->first; // shortcut
		string sPathRel = k.dirRel+k.file;

		if (rechecks::MatchWhitelist(k.file) || rechecks::MatchWhitelist(sPathRel))
		{
			LOG("File not to be removed, ignoring");
			continue;
		}

		if(!it->second.nLostAt) // no shit, it should be assigned at least once
			continue;

		//cout << "Unreferenced: " << it->second.sDirname << it->first <<endl;

		string sPathAbs=CACHE_BASE+sPathRel;

		// file will be removed (with its header) or tagged ASAP,
		// don't consider its header for anything afterwards
	    m_trashCandHeadSet.erase(tFileNdir(k.file+".head", k.dirRel));

	    //cout << "Took " << sWhatHead << " from the list" <<endl;

		if(bPurgeNow || (it->second.nLostAt < (m_nTimeNow-acfg::extreshhold*86400)))
		{
			SendFmt() << "Removing " << sPathRel << "<br>\n";

#ifdef ENABLED

			::unlink(sPathAbs.c_str());
			::unlink((sPathAbs+".head").c_str());
			::rmdir((CACHE_BASE + k.dirRel).c_str());
#endif
        }
		else if(f)
		{
			SendFmt() << "Tagging " << sPathRel << "<br>\n";

			n++;
			tagSpace+=it->second.fpr.size;
			fprintf(f, "%lu\t%s\t%s\n",  it->second.nLostAt,
					k.dirRel.c_str(), k.file.c_str());
		}
	}
    if(f)
        fclose(f);

    // now just kill dangling header files
	for(set<tFileNdir>::iterator it=m_trashCandHeadSet.begin();
            it != m_trashCandHeadSet.end(); it++)
	{
		string sPathRel(it->dirRel + it->file);
		if (rechecks::MatchWhitelist(sPathRel) || (endsWithSzAr(sPathRel, ".head")
				&& rechecks::MatchWhitelist(sPathRel.substr(0, sPathRel.size() - 5))))
		{
			continue;
		}

		if(m_bVerbose)
			SendFmt() << "Removing orphaned head file: " << sPathRel;

#ifdef ENABLED
        string sPathAbs=CACHE_BASE+sPathRel;
        ::unlink(sPathAbs.c_str());
        string::size_type pos=sPathAbs.find_last_of(SZPATHSEPUNIX SZPATHSEPWIN);
        if(pos!=stmiss)
        	::rmdir(sPathAbs.substr(0, pos).c_str());
#endif
    }
    if(n>0)
    	TellCount(n, tagSpace);
}

void tCacheMan::TellCount(uint nCount, off_t nSize)
{
	SendFmt() << "<br>\n" << nCount <<" package file(s) marked "
			"for removal in few days. Estimated disk space to be released: "
			<< offttosH(nSize) << ".<br>\n<br>\n";
}

void tCacheMan::SetCommonUserFlags(cmstring &cmd)
{
	m_bErrAbort=(cmd.find("abortOnErrors=aOe")!=stmiss);
	m_bByPath=(cmd.find("byPath")!=stmiss);
	m_bByChecksum=(cmd.find("byChecksum")!=stmiss);
	m_bVerbose=(cmd.find("beVerbose")!=stmiss);
	m_bForceDownload=(cmd.find("forceRedownload")!=stmiss);
	m_bSkipHeaderChecks=(cmd.find("skipHeadChecks")!=stmiss);
}

void expiration::Action(const string & cmd)
{
	if (cmd.find("justRemove")!=stmiss)
	{
		_LoadTrashMapFromFile(true);
		_RemoveAndStoreStatus(true);
		return;
	}
	if (cmd.find("justShow")!=stmiss)
	{
		_LoadTrashMapFromFile(true);
		off_t nSpace(0);
		uint cnt(0);
		for (tS2DAT::iterator it=m_trashCandSet.begin(); it
				!=m_trashCandSet.end(); it++)
		{
			const tFileNdir &k=it->first;
			string rel=k.dirRel+k.file;
			off_t sz=GetFileSize(CACHE_BASE+rel, -2);
			if(sz<0)
				continue;

			cnt++;
			SendChunk(rel+"<br>\n");
			nSpace+=sz;

			sz = GetFileSize(CACHE_BASE + rel+".head", -2);
			if (sz >= 0)
			{
				nSpace += sz;
				SendChunk(rel + ".head<br>\n");
			}
		}
		TellCount(cnt, nSpace);
		return;
	}

	SetCommonUserFlags(cmd);

	SendChunk("<b>Locating potentially expired files in the cache...</b><br>\n");

	DirectoryWalk(acfg::cachedir, this);
	if(CheckAbortCondition())
		return;
	SendFmt()<<"Found "<<m_nProgIdx<<" files.<br />\n";

	//cout << "found package files: " << m_trashCandidates.size()<<endl;
	//for(tS2DAT::iterator it=m_trashCandSet.begin(); it!=m_trashCandSet.end(); it++)
	//	SendChunk(tSS()<<it->second.sDirname << "~~~" << it->first << " : " << it->second.fpr.size<<"<br>");

	_LoadIgnoreList();
	UpdateIndexFiles();

	if(CheckAbortCondition())
		return;

	if (m_nErrorCount > 0 && m_bErrAbort)
	{
		SendChunk("<font color=\"red\">Found errors during processing, "
				"aborting as requested.</font><!-- TELL:THE:ADMIN -->");
		return;
	}


	SendChunk("<b>Validating cache contents...</b><br>\n");
	_ProcessSeenIndexFiles();

	if (m_bErrAbort && m_nErrorCount>0)
	{
		SendChunk("<font color=\"red\">Found errors during processing, aborting as requested.</font>"
			"<!-- TELL:THE:ADMIN -->");
		return;
	}

	if(CheckAbortCondition())
			return;

	// update timestamps of pending removals
	_LoadTrashMapFromFile(false);

	_RemoveAndStoreStatus(cmd.find("purgeNow")!=stmiss);
	_PurgeMaintLogs();

	DelTree(CACHE_BASE+"_actmp");

	SendChunk("Done.");

}

void expiration::_PurgeMaintLogs()
{
	tStrDeq logs = ExpandFilePattern(acfg::logdir + SZPATHSEP"maint_*.log");
	if (logs.size() > 2)
		SendChunk(
				"Found required cleanup tasks: purging maintanence logs...<br>\n");
	for (tStrDeq::const_iterator it = logs.begin(); it != logs.end(); it++)
	{
		time_t id = atoofft(it->c_str() + acfg::logdir.size() + 7);
		//cerr << "id ist: "<< id<<endl;
		if (id == GetTaskId())
			continue;
		//cerr << "Remove: "<<globbuf.gl_pathv[i]<<endl;
#ifdef ENABLED
		::unlink(it->c_str());
#endif
	}
}

void expiration::UpdateFingerprint(const MYSTD::string &sPathRel,
		off_t nOverrideSize, uint8_t *pOverrideSha1, uint8_t *pOverrideMd5)
{
	// navigating to the data set...
	tStrPos nCutPos = sPathRel.rfind(CPATHSEP);
	tFileNdir key((stmiss == nCutPos) ? sPathRel : sPathRel.substr(nCutPos+1),
			(stmiss == nCutPos) ? "" : sPathRel.substr(0, nCutPos+1));
	tDiskFileInfo &finfo = m_trashCandSet[key];

	if (!finfo.nLostAt) // just has been created? smells like a bug
		return;

	if(pOverrideMd5)
		finfo.fpr.Set(pOverrideMd5, CSTYPE_MD5, nOverrideSize);
	else if(pOverrideSha1)
		finfo.fpr.Set(pOverrideSha1, CSTYPE_SHA1, nOverrideSize);
	else if(nOverrideSize != off_t(-1))
		finfo.fpr.Set(NULL, CSTYPE_INVALID, nOverrideSize);
	/*
	else if (!pDataSrc)
			finfo.fpr.ScanFile(sPath, finfo.fpr.csType, false);
	else if (CSTYPE_MD5 == finfo.fpr.csType) // or use the file object which is already mmap'ed
		pDataSrc->GetMd5Sum(sPath, finfo.fpr.csum, false, finfo.fpr.size);
	else if (CSTYPE_SHA1 == finfo.fpr.csType)
		pDataSrc->GetSha1Sum(sPath, finfo.fpr.csum, false, finfo.fpr.size);
		*/
	else // just update the file size
	{
		//SendChunk(tSS() << "cs from fresh stat info<br>");
		struct stat stbuf;
		finfo.fpr.csType=CSTYPE_INVALID;
		if (::stat((CACHE_BASE+sPathRel).c_str(), &stbuf))
			aclog::err(sPathRel + " << FAILED TO READ");
		else
			finfo.fpr.Set(NULL, CSTYPE_INVALID, stbuf.st_size);
	}
}


bool expiration::ProcessRegular(const string & sPath, const struct stat &stinfo)
{

	if(CheckAbortCondition())
		return false;

	if(sPath.size()<=CACHE_BASE_LEN) // heh?
		return false;

	if (++m_nProgIdx == m_nProgTell)
	{
		SendFmt()<<"Scanning, found "<<m_nProgIdx<<" file"
				<< (m_nProgIdx>1?"s":"") << "...<br />\n";
		m_nProgTell*=2;
	}

	if(0==sPath.compare(CACHE_BASE_LEN, 1, "_") && !m_bScanInternals)
		return true; // not for us

	tStrPos nSlashPos=sPath.rfind(CPATHSEP);
	string sBasename(sPath, nSlashPos+1);
	string sDirRel(sPath, CACHE_BASE_LEN, nSlashPos-CACHE_BASE_LEN+1);

	// handle the head files separately
    if (endsWithSzAr(sPath, ".head"))
		m_trashCandHeadSet.insert(tFileNdir(sBasename, sDirRel));
	else
	{
		AddIFileCandidate(sDirRel+sBasename);
		//SendChunk(string("<br>hm?<br>")+sDirnameAndSlash + " -- " + sBasename);
		tDiskFileInfo &finfo = m_trashCandSet[tFileNdir(sBasename, sDirRel)];
		finfo.nLostAt = m_nTimeNow;
		finfo.fpr.size = stinfo.st_size;
	}
    return true;
}


expiration::expiration(int fd) : tCacheMan(fd)
{
	m_sTypeName="Expiration";
}

expiration::~expiration()
{
}

void expiration::_LoadIgnoreList()
{
	filereader reader;
	if(!reader.OpenFile(acfg::confdir+SZPATHSEP+"ignore_list"))
		return;
	string sTmp;
	while (reader.GetOneLine(sTmp))
	{
		trimLine(sTmp);
		if (startsWithSz(sTmp, "#"))
			continue;
		if(startsWith(sTmp, CACHE_BASE))
			sTmp.erase(CACHE_BASE_LEN);
		SetFlags(sTmp).forgiveDlErrors=true;
	}
}
void expiration::_LoadTrashMapFromFile(bool bForceInsert)
{

	filereader reader;
	reader.OpenFile(CACHE_BASE+"_expending_dat");

	string sLine;

#if 0
	// stuff for user info
	time_t now=time(NULL);
	time_t oldest(now), newest(0);
#endif

	while(reader.GetOneLine(sLine))
	{
		char *eptr(NULL);
		const char *s = sLine.c_str();
		time_t timestamp = strtoull(s, &eptr, 10);
		if (!eptr || *eptr != '\t' || !timestamp || timestamp > m_nTimeNow) // where is the DeLorean?
			continue;
		const char *sep = strchr(++eptr, (unsigned) '\t');
		if (!sep)
			continue;
		string dir(eptr, sep - eptr);
		if (!dir.empty() && '/' != *(sep - 1))
			dir += "/";
		const char *term = strchr(++sep, (unsigned) '\t'); // just to be sure
		if (term)
			continue;
		tFileNdir key(sep, dir);

		if (bForceInsert)
		{
			// add with timestamp from the last century (implies removal later)
			m_trashCandSet[key].nLostAt = 1;
			continue;
		}

		tS2DAT::iterator it = m_trashCandSet.find(key);

		// file is in trash candidates now and was back then, set the old date
		if (it != m_trashCandSet.end())
			it->second.nLostAt = timestamp;

#if 0
		if(timestamp < oldest)
		oldest=timestamp;
		if(timestamp > newest)
		newest=timestamp;
#endif

	}

#if 0
	/*
	cout << "Unreferenced: ";
	for(trashmap_t::iterator it=m_trashCandidates.begin(); it!=m_trashCandidates.end(); it++)
		fprintf(stdout, "%lu\t%s\t%s\n",  it->second.first, it->second.second.c_str(), it->first.c_str());
	*/

	if(m_trashCandidates.size()>0)
	{
		// pretty printing for impatient users
		char buf[200];

		// map to to wait
		int nMax=acfg::extreshhold-((now-oldest)/86400);
		int nMin=acfg::extreshhold-((now-newest)/86400);

		snprintf(buf, _countof(buf), "Previously detected: %lu rotten package file(s), "
				"to be deleted in about %d-%d day(s)<br>\n",
				(unsigned long) m_trashCandidates.size(),
				nMin, nMax==nMin?nMax+1:nMax); // cheat a bit for the sake of code simplicity
		SendChunk(buf);
	}
#endif

}


