
#define LOCAL_DEBUG
#include "debug.h"

#include "config.h"

#include "fileitem.h"
#include "header.h"
#include "acfg.h"
#include "acbuf.h"

#include <iostream>
#include <sstream>
#include <utime.h>
#include <errno.h>

using namespace MYSTD;

static map<string, tFileItemPtr> mapItems;
static pthread_mutex_t mapLck = PTHREAD_MUTEX_INITIALIZER;

void fileitem::GetHeader(header &h)
{
   //ldbg("fileitem::GetHeader(...)");
	setLockGuard;
	h=m_RetHead;
	/*
	 * 
	 * UNRELIABLE because of date formats :-(
	 * 
	// weird, a file without modification date. Maybe imported by user, let's be optimistic for static files
	if(h.get("Last-Modified").empty() && !m_bIsDynType)
	{
		struct stat stinfo;
		
		if (0==stat(m_sPath.c_str(), &stinfo))
		{
			char buf[26];
			struct tm tmp;
			const time_t cur=stinfo.st_mtime;
			gmtime_r(&cur, &tmp);
			asctime_r(&tmp, buf);
			buf[24]=0x0;
			h.set("Last-Modified", string(buf));
		}
	}
	*/
   //ldbg("RETURN: fileitem::GetHeader(...)");
}

FiStatus fileitem::TrackStatusChanges(long const nKnownSize, long & nRetCachedLen, MYSTD::string & sErrorOut) {
	setLockGuard;
	ldbg("fileitem::TrackStatusChanges("<< nKnownSize<<	 ", ret:X, ret: Y)");
	/*
		
	while( ( !m_bIsUptodate || nKnownSize>=m_nCachedLen ) && m_sError.empty() )
	{
		ldbg("fileitem::TrackStatusChanges~wait: m_bIsUptodate: "<<m_bIsUptodate<<", nKnownSize: " << nKnownSize
				<< ", m_nCachedLen: "<<m_nCachedLen << ", Error: " << m_sError);
		wait();
	}
	FiStatus st = _GetStatus(nNewSizeOut, sErrorOut);
	 
	if( st&FIST_COMPLETE && nKnownSize==m_nContLen)
		st|=FIST_EOF;
	
	ldbg("RETURN: fileitem::TrackStatusChanges(..., " << nNewSizeOut <<", " << sErrorOut<<") -> " << st);
	return st;
	
	*/
	while(true)
	{
		FiStatus st = _GetStatus(nRetCachedLen, sErrorOut);
		ldbg("status: " << st << " cachedlen: "<< nRetCachedLen << " error: " << sErrorOut);
		if( ! m_bIsUptodate && sErrorOut.empty())
		{
			wait();
			continue;
		}
		
		if( !sErrorOut.empty() || (st&FIST_COMPLETE) || nKnownSize<m_nCachedLen)
		{
			// maybe complete, maybe not, but downloader exited no matter how
			if( nKnownSize==m_nCachedLen && m_bFilesLocked )
				st|=FIST_EOF;
			
			// finished, on known-size-downloads that will be okay, 
			// at open-end-downloads m_nContlen was -1 untill finished -> also ok
			if( m_bIsUptodate && m_nContLen==nKnownSize )
				st|=FIST_EOF;
			
			if(m_bNonFatal)
				st|=FIST_NONFATAL;
			
			return st;
		}
		
		wait();
	}
}

FiStatus fileitem::_GetStatus(long & nCachedLen, MYSTD::string & sErrorOut) 
{
   FiStatus ret(0);
	nCachedLen=m_nCachedLen;
	if( ! m_sError.empty())
	{
		ldbg("setze sErrorOut to "<<m_sError);
		sErrorOut=m_sError;
		ret|=FIST_ERROR;
	}
	if(m_nCachedLen==m_nContLen && m_nContLen>=0 && m_bIsUptodate)
		ret|=FIST_COMPLETE;
	if(m_bHaveDownloader)
		ret|=FIST_DLRUNNING;
	ldbg("RETURN: fileitem::_GetStatus(" << nCachedLen <<", "<<sErrorOut<<")");
	return ret;
}


FiStatus fileitem::GetStatus(long & nCachedLen, MYSTD::string & sErrorOut) 
{
   ldbg("fileitem::GetStatus(...)");
	setLockGuard;
	return _GetStatus(nCachedLen, sErrorOut);
}

void fileitem::GetStatusEx(long & nCachedLen, long &nContLen, bool & bIsDynType) 
{
//#error meldet ok obwohl .head nicht da ist und auch m_ncontlen m��ll enth��lt
	setLockGuard;
	nCachedLen=m_nCachedLen;
	nContLen=m_nContLen;
	bIsDynType=m_bIsDynType;
	ldbg("fileitem::GetStatusEx("<<nCachedLen<<", "<<nContLen<<", "<<bIsDynType);
}

void fileitem::DelUser() {
   ldbg("fileitem::DelUser()");
	lockguard managementLock(mapLck); // order does matter, check GetFileItem
  dbgline;
	setLockGuard;
	m_nUserCount--;
  ldbg("fileitem::DelUser~Usercount: "<<m_nUserCount);
	if(m_nUserCount==0) 
	{
		m_bFilesLocked=true;
		if(m_bRemovalScheduled)
		{
			ldbg("Removing "<<m_sPath<<" and company");
			unlink(m_sPath.c_str());
			unlink((m_sPath+".head").c_str());
		}
		m_bRemovalScheduled=false;
		
		// become orphaned and be deleted RSN, when the caller of this method loses
		// the reference on us
		mapItems.erase(m_sKey);
	}
	
  ldbg("RETURN: fileitem::DelUser()");
}

tFileItemPtr fileitem::GetFileItem(MYSTD::string sPathKey, bool bIsDynType)
{
	// lock global mutex as long as required, then carefully switch to the local and release the global
	dbgline;
	lockguard lockGlobalMap(mapLck);
	map<string, tFileItemPtr>::iterator it = mapItems.find(sPathKey);
	if (it != mapItems.end())
		return it->second;
	
	tFileItemPtr t(new fileitem(sPathKey, bIsDynType));
	lockguard lockItemItself(& t->__mutex);
	mapItems[sPathKey]=t;
	lockGlobalMap.unLock(); // don't keep global stuff blocked while doing IO stuff, internal lock here is enough

	t->_LoadCached();
	if (bIsDynType)
	{
		// cannot rely on anything there, StoreNewHead/StoreData will update this info
		t->m_bIsUptodate=false;
		t->m_nContLen=-1;
	}
	// now can see whether a downloader would start
	if (t->m_nCachedLen>=0&& t->m_nCachedLen==t->m_nContLen)
		t->m_bFilesLocked=true; // cannot be continued

	t->m_nUserCount++;
	
	return t;

}

fileitem::fileitem(MYSTD::string sPath, bool bDynType) :
	condition(),
	m_sPath(acfg::cachedir+sPathSep+sPath),
	m_sKey(sPath),
	m_bHaveDownloader(false),
	m_bIsUptodate(false),
	m_bIsDynType(bDynType),
	m_bFilesLocked(false),
	m_bRemovalScheduled(false),
	m_bNonFatal(false),
	m_nIncommingCount(0),
	m_nCachedLen(-1),
	m_nContLen(-1),
	m_nCachedWritePos(0),
	m_nUserCount(0),
	m_filefd(-1)
{
   ldbg("item created, m_sPath: "<< m_sPath);
}

uint64_t fileitem::GetTransferCount()
{
	setLockGuard;
	uint64_t ret=m_nIncommingCount;
	m_nIncommingCount=0;
	return ret;
}

void fileitem::AddTransferCount(UINT l)
{
	setLockGuard;
	m_nIncommingCount+=l;
}

int fileitem::GetFileFd() {
	setLockGuard;
	if(!m_bIsUptodate)
		return -1; // stay away from rotten data
	int fd=open(m_sPath.c_str(), O_RDONLY);
	if(fd<0 || m_nCachedLen<0)
		return fd;
	
#ifdef HAS_ADVISE
	// optional, experimental
	posix_fadvise(fd, 0, m_nCachedLen, POSIX_FADV_SEQUENTIAL);
#endif
	
	return fd;
}

// PREREQUISITE: runs locked.
void fileitem::_LoadCached() {
	
	// header file MUST be there
	
	if(m_RetHead.LoadFromFile(m_sPath+".head") >0 )
	{
		struct stat statbuf;
		if (0==stat(m_sPath.c_str(), &statbuf))
		{
			m_nCachedLen=statbuf.st_size;
			ldbg("on disk size: "<< m_nCachedLen);
		}
		
		m_RetHead.fix();

		if(200 != m_RetHead.getStatus())
		{
			unlink((m_sPath+".head").c_str());
			return; // unuseable, will be redownloaded
		}

		string f=m_RetHead.get("Content-Length");
		if( ! f.empty())
		{
			ldbg("hohoho, cont-len: "<< f);
			m_nContLen=atoi(f.c_str());
			m_bIsUptodate=true; // the constructors/wrappers might still change that
		}
		
		// detect some obvious crap
		if(m_nContLen>0 && m_nCachedLen>m_nContLen)
		{
			aclog::err("Warning, static file larger than it could be (Content-Length)");
			
			m_bRemovalScheduled=true;
			m_bFilesLocked=true;
			m_sError="500 Cache inconsistency";
		}
	}
}

void fileitem::SetFailureMode(string const & sFailMsg, bool bNonFatal)
{
	setLockGuard;
	return _SetFailureMode(sFailMsg, bNonFatal);
}

void fileitem::_SetFailureMode(string const & sFailMsg, bool bNonFatal)
{
	ldbg("Setting failure mode: " << sFailMsg <<", fatal? " << !bNonFatal);
	m_sError=sFailMsg;
	m_bNonFatal=bNonFatal;
	
	 // if failmsg is empty, it was a mere notification from 304 and downloads can start now
	m_bIsUptodate=true;
	m_bFilesLocked=true;
	
	ldbg("Got error: "<<sFailMsg);
	notifyAll();
}

bool fileitem::StoreNewHead(const header & h) 
{
	setLockGuard;
	return _StoreNewHead(h);
}

/*!
 * 
 * Saves a new header on the harddisk. 
 * 
 * \returns False on storage errors.
 * */
bool fileitem::_StoreNewHead(const header & h) 
{
	if(m_bFilesLocked)
		return true;
	if (m_bIsUptodate)
		return true;

	#if 0 // for what exactly...
	
	if (!h)
	{ // should pass a header now
		m_sError="503 Internal failure, code 1";
		return false;
	}
	string modate=h->get("Last-Modified");
	if (modate.empty() || modate!=m_RetHead.get("Last-Modified"))
	{ // outdated or unknown, purge it
		m_nCachedLen=0;
		unlink(m_sPath.c_str());
		unlink((m_sPath+".head").c_str());
		// TODO: just delete now, no checking yet, the files may be just missing
		/*
		if (0!=unlink(m_sPath.c_str()) || 0!=unlink((m_sPath+".head").c_str()))
		{ 
			
			m_sError="503 Internal failure, code 2";
			return false;
			
		}
		*/
	}
#endif
	
	m_RetHead=h;
	
	// pickup the real content length if needed and where possible
	if(m_nContLen<0) 
	{
		string sCl=h.get("Content-Length");
		if(!sCl.empty()) {
			m_nContLen=atol(sCl.c_str());
			notifyAll();
		}
	}
	
	// better later, avoid inconsistencies with head-only action
	if(m_bIsDynType)
		return true;
	
	return _StoreLocalHead();
}

bool fileitem::_StoreLocalHead() {

	string path=m_sPath+".head";
	mkbasedir(path);
	ldbg("Opening "+path);
	int count=m_RetHead.StoreToFile(path);
	
	// unlink and retry
	if(count<0)
		unlink(path.c_str());
	count=m_RetHead.StoreToFile(path);

	if(count<0)
	{
		const char *msg;
		if(-count!=ENOSPC)
		{
			msg="503 Cache storage error";
			aclog::err(string("Error writting header: ")+m_sPath+".head");
		}
		else msg="503 OUT OF DISK SPACE";
		_SetFailureMode(msg, false);
		return false;
	}

	return true;
}

// DO NOT THROW HERE
void fileitem::Finalize(long nNewContLen) {
	setLockGuard;

	// Anyway if the download was ok, truncate it to the proper size now
	if(m_filefd>=0 && !m_bFilesLocked)
	{
		ftruncate(m_filefd, nNewContLen);
	}
  /*
	if(m_nContLen>=0 && m_nContLen!=nNewContLen)
  {
  Whatever happened, if the file is oversized now, this will be reported in
  the future and then the file will be nuked, so don't care...
	}
  */
	if(m_RetHead.type != header::INVALID &&
			(m_RetHead.get("Content-Length").empty() || m_nContLen<0) )
	{
		m_nContLen=nNewContLen;
		m_RetHead.set("Content-Length", nNewContLen);
		_StoreLocalHead();
	}
	notifyAll();
	m_bHaveDownloader=false;
	m_bFilesLocked=true;
	
	// TODO: more fixes of that kind?
	
}

//! Stores the minimum of nRest or data size in buf at the position offset. All arguments are updated to reflect the new state.
long fileitem::StoreData(const char *data, const unsigned int size, off_t nOffset) {
	setLockGuard;ldbg("fileitem::StoreData("<<  size << " bytes at " << nOffset<<")");
	
	if(size==0)
		return 0;
	
	if (m_bFilesLocked)
	{
		// what now? Usually the user is gone, now either simulate storage to the caller, or stop and let 
		// the caller reconnect...
		// TODO: fixme, guessed number, reflects "costs" of a reconnect
		if(size>4000)
			return -1; 
		// pretend having stored it
		return size;
	}

	ldbg("shall write file, open it?");
	if (m_filefd<0|| !m_bIsUptodate)
	{
		// deffered hd dump of the header data for dyn.files
		if(m_bIsDynType)
		{
			if(!_StoreLocalHead())
				return -1;
		}
		
		m_filefd=open(m_sPath.c_str(), O_WRONLY | O_CREAT, 00644);
		ldbg("file opened?! returned: " << m_filefd);
		
		// self-recovery from cache poisoned with files with wrong permissions
		if (m_filefd<0)
		{
			unlink(m_sPath.c_str());
			m_filefd=open(m_sPath.c_str(), O_WRONLY | O_CREAT, 00644);
			ldbg("file opened?! returned: " << m_filefd);
		}
		
		if (m_filefd<0)
		{
			_SetFailureMode(errno==ENOSPC ? "503 OUT OF DISK SPACE"
			: "503 Cache storage error, opening data file.", false);
			return -1;
		}
	}
	
	if (nOffset!=m_nCachedWritePos)
	{
		if (nOffset!=lseek(m_filefd, nOffset, SEEK_SET))
		{
			_SetFailureMode("503 Seek failure", false);
			return -1;
		}
	}
	
	int r=write(m_filefd, data, size);
	if(r<0)
	{
		_SetFailureMode(errno==ENOSPC ? "503 OUT OF DISK SPACE" : "503 Cache storage error", false);
	}
	else
	{
		// now it's safe to read from it
		m_bIsUptodate=true;
		m_nCachedLen=m_nCachedWritePos=nOffset+r;
		ldbg("new m_nCachedLen: " << m_nCachedLen);
	}
	notifyAll();
	return r;
}

fileitem::~fileitem() {

	ldbg("Destroying fitem, " << m_sKey);
	setLockGuard;
	
	if(m_filefd>=0)
	{
		close(m_filefd);
	}
	/* done in DelUser
	if(m_bRemovalScheduled)
	{
		ldbg("Removing files");
		unlink(m_sPath.c_str());
		unlink((m_sPath+".head").c_str());
	}
	*/
	ldbg("Fitem destroyed, " << m_sKey);
}

// downloader instruments
bool fileitem::AssignDownloaderUnlessConflicting()
{
	setLockGuard;
	ldbg("Assigning downloader for " << m_sPath);
	if(m_bHaveDownloader || m_bFilesLocked)
		return false;
	m_bHaveDownloader=true;
	return true;
}

bool fileitem::ReleaseDownloaderUnlessBusy() {
	setLockGuard;
	ldbg("fileitem::ReleaseDownloaderUnlessBusy on " << m_sPath);
	if(m_nUserCount>0)
		return false;
	
	m_bHaveDownloader=false;
	m_bFilesLocked=true;
	
	return true;
}
