#include "conserver.h"
#include "meta.h"
#include "lockable.h"
#include "con.h"
#include "acfg.h"

#include <sys/socket.h>
#include <sys/un.h>
#include <signal.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netinet/tcp.h>
#include <errno.h>

#include <netdb.h>

#include <list>
#include <map>
#include <iostream>
using namespace MYSTD;

namespace conserver
{

int sockip(-1), sockunix(-1);

condition cond;

list<con*> qForHelper, qForWorker;
map<con*,int> mConStatus;
int nStandbyHelper(0), nStandbyWorker(0);
bool bTerminationMode(false);
int nAllConThreadCount(0);

#define TYPE_WORKER ((void*)0)
#define TYPE_HELPER ((void*)1)
#define STANDBY_LIMIT acfg::tpoolsize
#define MAX_BACKLOG 1

void * ThreadAction(void *type)
{

	list<con*> & myq = (type==TYPE_WORKER) ? qForWorker : qForHelper;
	int &nSpareThreads=(type==TYPE_WORKER) ? nStandbyWorker : nStandbyHelper;

	lockguard g(cond);
	nAllConThreadCount++;

	while (true)
	{
		con *c(NULL);
		if (bTerminationMode)
			break;

		nSpareThreads++;
		while (myq.empty())
			cond.wait();
		nSpareThreads--;
		c=myq.front();
		myq.pop_front();
	
		g.unLock();
		if (type==TYPE_WORKER)
		{
			c->WorkLoop();
		}
		else
		{
			c->HelpLoop();
		}
		c->SignalStop(); // just to be sure
		g.reLock();

		// don't care about deletion
		if(bTerminationMode)
			break;
		
		// increate return count, clean up when both threads are terminated
		map<con*,int>::iterator it=mConStatus.find(c);
		it->second++;
		if (2==it->second)

		{
			c->ShutDown();
			delete c;
			mConStatus.erase(it);
		}
		if (nSpareThreads>=STANDBY_LIMIT)
			return NULL;
	}
	
	nAllConThreadCount--;
	return NULL;
}

bool SpawnThread(void * type)
{
	pthread_t thr;
	pthread_attr_t attr; // detached, to be initialized in Setup
	pthread_attr_init(&attr);
	pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_DETACHED);

	return (0==pthread_create(&thr, NULL, ThreadAction, type));
}

void SetupConAndGo(int fd, const char *szClientName="")
{
	// protect from interference of con shutdown with OS on file descriptor 
	// management and also thread pool control
	lockguard g(cond);

	con *c(NULL);
	try
	{
		c=new con(fd, szClientName);
	}
	catch(bad_alloc)
	{
		goto failure_mode;
	}
	// better be sure...
	if (!c)
	{
		goto failure_mode;
	}

	// a thread will take it soon, wait till that happens

	if (qForWorker.size() > MAX_BACKLOG)
	{
		goto failure_mode;
	}

	if (0==nStandbyHelper && !SpawnThread(TYPE_HELPER))
			goto failure_mode;

	if (0==nStandbyWorker && !SpawnThread(TYPE_WORKER))
			goto failure_mode;
	
	mConStatus[c]=0;
	qForHelper.push_back(c);
	qForWorker.push_back(c);
	cond.notifyAll();

	return;

	failure_mode: if (c)
		delete c;
	shutdown(fd, SHUT_RDWR);
	close(fd);
}

void CreateUnixSocket() {
	string & sPath=acfg::fifopath;
	struct sockaddr_un addr_unx;
	memset(&addr_unx, 0, sizeof(addr_unx));
	
	int jo=1;
	size_t size = sPath.length()+1+offsetof(struct sockaddr_un, sun_path);
	
	if(sPath.length()>sizeof(addr_unx.sun_path))
	{
		errno=ENAMETOOLONG;
		goto unx_err;
	}
	
	addr_unx.sun_family = AF_UNIX;
	strncpy(addr_unx.sun_path, sPath.c_str(), sPath.length());
	
	mkbasedir(sPath);
	unlink(sPath.c_str());
	
	sockunix = socket(PF_UNIX, SOCK_STREAM, 0);
	setsockopt(sockunix, SOL_SOCKET, SO_REUSEADDR, &jo, sizeof(jo));
				
	if (sockunix<0)
		goto unx_err;
	
	if (bind(sockunix, (struct sockaddr *)&addr_unx, size) < 0)
		goto unx_err;
	
	if (0==listen(sockunix, SO_MAXCONN))
		return;
	
	unx_err:
	cerr << "Error creating Unix Domain Socket, ";
	cerr.flush();
	perror(acfg::fifopath.c_str());
	exit(EXIT_FAILURE);

}

void Setup()
{
	sockip=sockunix=-1;
	int yes(1);

	if (acfg::port>0)
	{
#ifdef HAS_IPV6SOCKETS
		sockip = socket(PF_INET6, SOCK_STREAM, IPPROTO_TCP);
		if (sockip>=0)
		{

			setsockopt(sockip, SOL_SOCKET, SO_REUSEADDR, &yes,
					sizeof(yes));
			
			struct sockaddr_in6 addr;
			addr.sin6_family=PF_INET6;
			addr.sin6_flowinfo=0;
			addr.sin6_port=htons(acfg::port);
			addr.sin6_addr = in6addr_any;

			if (0==bind(sockip, (struct sockaddr *)&addr, sizeof(addr))
					&& 0==listen(sockip, SO_MAXCONN))
			{
				//ldbg("Using IPv6");
			}
			else
			{
				close(sockip);
				sockip=-1;
			}
		}
#endif
		if(sockip <0)
		{
			sockip = socket(PF_INET, SOCK_STREAM, IPPROTO_TCP);
			if (sockip<0)
			{
				aclog::err("Could not create IPv4 socket");
			}
			else
			{

				setsockopt(sockip, SOL_SOCKET, SO_REUSEADDR, &yes,
						sizeof(yes));
				
				struct sockaddr_in addr;
				addr.sin_family = PF_INET;
				addr.sin_port = htons(acfg::port);
				addr.sin_addr.s_addr = INADDR_ANY;
				memset(&(addr.sin_zero), '\0', 8);

				if (bind(sockip, (struct sockaddr *)&addr,
								sizeof(struct sockaddr)) <0)
				{
					perror("bind");
					exit(1);
				}
				if (listen(sockip, SO_MAXCONN) == -1)
				{
					perror("listen");
					exit(1);
				}
			}
		}
		if (sockip<0)
		{
			cerr
			<< "Unable to create an IP socket, see error log for details."
			<<endl;
			exit(EXIT_FAILURE);
		}

		setsockopt(sockip, IPPROTO_TCP, TCP_NODELAY, &yes,
								sizeof(yes));
		
	}
	else
		aclog::err("Not creating TCP listening socket, no valid port specified!");

	if ( !acfg::fifopath.empty() )
	{
		CreateUnixSocket();
	}
	else
		aclog::err("Not creating Unix Domain Socket, fifo_path not specified");

	if (acfg::port<0 && acfg::fifopath.empty())
	{
		cerr << "Neither TCP nor UNIX interface configured, cannot proceed.\n";
		exit(1);
	}
}

int Run()
{
	fd_set rfds, wfds;
	int maxfd=1+std::max(sockunix, sockip);

	while (1)
	{ // main accept() loop

		FD_ZERO(&rfds);
		FD_ZERO(&wfds);
		if(sockunix>=0)
			FD_SET(sockunix, &rfds);
		if(sockip>=0)
			FD_SET(sockip, &rfds);
		
		//cerr << "Polling..." <<endl;
		int nReady=select(maxfd, &rfds, &wfds, NULL, NULL);
		if (nReady<0)
		{
			aclog::err("select", "failure");
			perror("Select died");
			exit(1);
		}
		
		if(sockunix>=0 && FD_ISSET(sockunix, &rfds))
		{
			int fd = accept(sockunix, NULL, NULL);
			if (fd>=0)
			{
				set_nb(fd);
				SetupConAndGo(fd);
			}
		}
		
		if(sockip>=0 && FD_ISSET(sockip, &rfds))
		{
			struct sockaddr_storage addr;
			socklen_t addrlen = sizeof(addr);
			int fd=accept(sockip,(struct sockaddr *)&addr, &addrlen);

			if (fd>=0)
			{
				set_nb(fd);
				char hbuf[NI_MAXHOST];
				
				if (getnameinfo((struct sockaddr*) &addr, addrlen, hbuf, sizeof(hbuf), 
						NULL, 0, NI_NUMERICHOST))
				{
					printf("ERROR: could not resolve hostname\n");
					return 1;
				}
				SetupConAndGo(fd, hbuf);
			}
		}
	}
	return 0;
}

void Shutdown()
{
	lockguard g(cond);
	
	if(bTerminationMode)
		return; // double SIGWHATEVER? Prevent it.
	
	for (map<con*,int>::iterator it=mConStatus.begin(); it !=mConStatus.end(); it++)
		it->first->SignalStop();
	printf("Signaled stop to all cons\n");
	bTerminationMode=true;
	printf("Notifying waiting threads\n");
	cond.notifyAll();
	
	printf("Closing sockets\n");
	if(sockip>=0)
	{
		shutdown(sockip, SHUT_RDWR);
		close(sockip);
		sockip=-1;
	}

	if(sockunix>=0)
	{
		shutdown(sockunix, SHUT_RDWR);
		close(sockunix);
		sockunix=-1;
	}
}

}
