//==============================================
//  copyright            : (C) 2003-2005 by Will Stokes
//==============================================
//  This program is free software; you can redistribute it
//  and/or modify it under the terms of the GNU General
//  Public License as published by the Free Software
//  Foundation; either version 2 of the License, or
//  (at your option) any later version.
//==============================================

//Systemwide includes
#include <qwidget.h>
#include <qlayout.h>
#include <qpixmap.h>
#include <qapplication.h>
#include <qtooltip.h>
#include <qpushbutton.h>

//Projectwide includes
#include "configuration.h"
#include "configurationWidget.h"
#include "groupsWidget.h"
#include "groupIcon.h"
#include "layoutSettingsWidget.h"
#include "loadingSavingWidget.h"
#include "alertsWidget.h"
#include "../config.h"

//==============================================
ConfigurationWidget::ConfigurationWidget(Configuration* config,
                                                                    QWidget *parent,
                                                                    const char* name ) :
                                                                    QDialog(parent,name)
{
  //store config pointer
  this->config = config;
  //--
  //set window title
  setCaption( tr("Settings"));
  //--
  //no icon currently hovered over
  currentPseudoSelection = NULL;
  //----------------------------------------------
  //create settings widget
  layoutWidget = new LayoutSettingsWidget(config, this);
  loadingSavingWidget = new LoadingSavingWidget(config, this);
  alertsWidget = new AlertsWidget( config, this );
  //----------------------------------------------
  //create iconview and icons for groups
  groups = new GroupsWidget( this );
  groups->setItemTextPos( QIconView::Right );
//  groups->setMaxItemWidth(20);
  //----
  //construct group labels
  QString labels[3];
  labels[0] = tr("Appearance");
  labels[1] = tr("Load/Save");
  labels[2] = tr("Alerts");
  //----
  //find max text width
  int maxLabelWidth = 0;
  int i;
  QFontMetrics fm( qApp->font() ); 
  for(i=0; i<3; i++)
  {
    if( fm.width( labels[i] ) > maxLabelWidth )
      maxLabelWidth = fm.width( labels[i] );
  }
  groups->setTextWidth( maxLabelWidth );
  //----
  //construct actual group icons
  layoutIcon = new GroupIcon(  groups, QPixmap(QString(IMAGE_PATH)+"settingsIcons/layout.png" ), 
                               labels[0], layoutWidget );
  layoutIcon->setDragEnabled(false);
  //----
  loadingSavingIcon = new GroupIcon( groups, QPixmap(QString(IMAGE_PATH)+"settingsIcons/loadsave.png" ),
                                     labels[1], loadingSavingWidget );
  loadingSavingIcon->setDragEnabled(false);
  //----
  alertsIcon = new GroupIcon( groups, QPixmap(QString(IMAGE_PATH)+"settingsIcons/alerts.png" ),
                              labels[2], alertsWidget );
  alertsIcon->setDragEnabled(false);
  //-------------------------
  //set default selection
  currentSettingsWidget = layoutWidget;
  layoutIcon->setSelected(true);
  loadingSavingWidget->hide();
  alertsWidget->hide();
  //-------------------------
  //connect selectionChanged signal to update which settings dialog is displayed
  connect( groups, SIGNAL(selectionChanged(QIconViewItem*)),
           SLOT(updateDialogue(QIconViewItem*)) );

  //connect mouse over events to paint pseudo selection in ligher blue
  connect( groups, SIGNAL(onItem(QIconViewItem*)),
           SLOT(repaintGroup(QIconViewItem*)) );

  //clear any pseudo selection when mouse moves off icons
  connect( groups, SIGNAL(onViewport()),
           SLOT(clearPseudoSelection()) );


  //create buttons frame and widgets
  buttonsFrame = new QFrame( this );
  okButton = new QPushButton( tr("Apply"), buttonsFrame );
  okButton->setSizePolicy( QSizePolicy::Fixed, QSizePolicy::Fixed );
  okButton->setDefault(true);
  connect( okButton, SIGNAL(clicked()), SLOT(saveSettings()) );
  cancelButton = new QPushButton( tr("Cancel"), buttonsFrame );
  cancelButton->setSizePolicy( QSizePolicy::Fixed, QSizePolicy::Fixed );
  connect( cancelButton, SIGNAL(clicked()), SLOT(reject()) );
  buttonsGrid = new QGridLayout( buttonsFrame, 1, 5, 0 );
  buttonsGrid->setColStretch( 0, 1 );
  buttonsGrid->addWidget( okButton, 0, 1 );
  buttonsGrid->addColSpacing( 2, 10 );
  buttonsGrid->addWidget( cancelButton, 0, 3 );
  buttonsGrid->setColStretch( 4, 1 );
  //----------------------------------------------
  //place all widgets in a grid
  grid = new QGridLayout( this, 5, 5, 0 );

  grid->setRowSpacing(0,8);

  grid->addWidget( groups, 1, 1);
  grid->addWidget( alertsWidget, 1, 3);
  grid->addWidget( layoutWidget, 1, 3);
  grid->addWidget( loadingSavingWidget, 1, 3);
  grid->setRowStretch( 1, 1 );
  grid->setColStretch( 3, 1 );

  grid->setRowSpacing(2,8);

  grid->addMultiCellWidget( buttonsFrame, 3, 3, 0, 4);

  grid->setRowSpacing(4,8);

  grid->setColSpacing(0,8);
  grid->setColSpacing(2,8);
  grid->setColSpacing(4,8);

  groups->setGridX(1);
  groups->arrangeItemsInGrid();

  int maxWidth = 0;
  int maxHeight = 0;

  layoutWidget->constPolish();
  loadingSavingWidget->constPolish();
  alertsWidget->constPolish();
  groups->constPolish();

  QSize s = layoutWidget->minimumSizeHint();
  if(maxWidth < s.width()) maxWidth = s.width();
  if(maxHeight < s.height()) maxHeight = s.height();

  s = loadingSavingWidget->minimumSizeHint();
  if(maxWidth < s.width()) maxWidth = s.width();
  if(maxHeight < s.height()) maxHeight = s.height();

  s = alertsWidget->minimumSizeHint();
  if(maxWidth < s.width()) maxWidth = s.width();
  if(maxHeight < s.height()) maxHeight = s.height();

  s = groups->minimumSizeHint();
  if(maxHeight < s.height()) maxHeight = s.height();

  maxWidth = maxWidth + s.width();
  maxHeight += okButton->minimumSizeHint().height();
  //add padding
  maxWidth += 3*8;
  maxHeight += 3*8;

  //add a little extra for when text entries need more space
  maxWidth += 100;

  resize( maxWidth, maxHeight );

  show();
  setFixedSize(size());
  //----------------------------------------------
  //load setting values
  layoutWidget->loadSettings();
  loadingSavingWidget->loadSettings();
  alertsWidget->loadSettings();
  //----------------------------------------------
}
//==============================================
void ConfigurationWidget::updateDialogue( QIconViewItem* selection)
{
  //hide current selection
  currentSettingsWidget->hide();

  //set current and show
  currentSettingsWidget = ((GroupIcon*)selection)->getSettingsWidget();
  currentSettingsWidget->show();
}
//==============================================
void ConfigurationWidget::repaintGroup( QIconViewItem* pseudoSelection)
{
  //if old pseudo selection unselect it
  clearPseudoSelection();

  //paint new selection
  currentPseudoSelection = (GroupIcon*)pseudoSelection;
  currentPseudoSelection->setMousedOver(true);
  groups->repaintItem(currentPseudoSelection);
}
//==============================================
void ConfigurationWidget::clearPseudoSelection()
{
  //if old pseudo selection unselect it
  if(currentPseudoSelection != NULL)
  {
    currentPseudoSelection->setMousedOver(false);
    groups->repaintItem(currentPseudoSelection);
    currentPseudoSelection = NULL;
  }
}
//==============================================
void ConfigurationWidget::saveSettings()
{
  layoutWidget->saveSettings();
  loadingSavingWidget->saveSettings();
  alertsWidget->saveSettings();
  close();
}
//==============================================
void ConfigurationWidget::closeEvent( QCloseEvent* e)
{
  QWidget::closeEvent( e );
  emit closed();
}
//==============================================
void ConfigurationWidget::reject()
{
  QDialog::reject();
  emit closed();
}
//==============================================
