# C preprocessor macros.
#
# Author::    Yutaka Yanoh <mailto:yanoh@users.sourceforge.net>
# Copyright:: Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
# License::   GPLv3+: GNU General Public License version 3 or later
#
# Owner::     Yutaka Yanoh <mailto:yanoh@users.sourceforge.net>

#--
#     ___    ____  __    ___   _________
#    /   |  / _  |/ /   / / | / /__  __/           Source Code Static Analyzer
#   / /| | / / / / /   / /  |/ /  / /                   AdLint - Advanced Lint
#  / __  |/ /_/ / /___/ / /|  /  / /
# /_/  |_|_____/_____/_/_/ |_/  /_/   Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
#
# This file is part of AdLint.
#
# AdLint is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# AdLint is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# AdLint.  If not, see <http://www.gnu.org/licenses/>.
#
#++

require "adlint/token"
require "adlint/report"
require "adlint/util"
require "adlint/cpp/syntax"

module AdLint #:nodoc:
module Cpp #:nodoc:

  class Macro
    def initialize(define_line)
      @define_line = define_line
    end

    attr_reader :define_line

    def name
      @define_line.identifier
    end

    def replacement_list
      @define_line.replacement_list
    end

    def location
      @define_line.location
    end

    def expand(tokens, macro_table, context)
      @define_line.mark_as_referred_by(tokens.first)
    end

    def function_like?
      subclass_responsibility
    end
  end

  class ObjectLikeMacro < Macro
    def replaceable_size(tokens)
      if tokens.first.value == "NULL" then
        0
      else
        name.value == tokens.first.value ? 1 : 0
      end
    end

    def expand(tokens, macro_table, context)
      super
      macro_table.notify_object_like_macro_replacement(self, tokens)

      if replacement_list
        location = tokens.first.location
        replacement_list.tokens.map { |t|
          ReplacedToken.new(t.type, t.value, location, t.type_hint, false)
        }
      else
        []
      end
    end

    def function_like?; false end
  end

  class FunctionLikeMacro < Macro
    def initialize(define_line)
      super
      if params = define_line.identifier_list
        @parameter_names = params.identifiers.map { |t| t.value }
      else
        @parameter_names = []
      end
    end

    attr_reader :parameter_names

    def replaceable_size(tokens)
      return 0 unless name.value == tokens.first.value
      arg_array, index = parse_arguments(tokens, 1)
      arg_array && @parameter_names.size >= arg_array.size ? index + 1 : 0
    end

    def expand(tokens, macro_table, context)
      super
      arg_array = parse_arguments(tokens, 1).first
      macro_table.notify_function_like_macro_replacement(self, tokens)

      args = {}
      @parameter_names.each_with_index do |param, i|
        arg = arg_array[i] and args[param] = arg
      end

      location = tokens.first.location
      expand_replacement_list(args, location, macro_table, context)
    end

    def function_like?; true end

    private
    def parse_arguments(tokens, index)
      while token = tokens[index]
        case
        when token.type == :NEW_LINE
          index += 1
        when token.value == "("
          index += 1
          break
        else
          return nil, index
        end
      end
      return nil, index unless token

      args = []
      while token = tokens[index]
        if token.value == ","
          args.push(nil)
          index += 1
        else
          arg, index = parse_one_argument(tokens, index)
          break unless arg
          args.push(arg)
        end
      end
      return args, index
    end

    def parse_one_argument(tokens, index)
      arg = []
      paren_depth = 0
      while token = tokens[index]
        case token.value
        when "("
          arg << token
          paren_depth += 1
        when ")"
          paren_depth -= 1
          if paren_depth >= 0
            arg << token
          else
            break
          end
        when ","
          if paren_depth > 0
            arg << token
          else
            index += 1
            break
          end
        when "\n"
          ;
        else
          arg << token
        end
        index += 1
      end
      return (arg.empty? ? nil : arg), index
    end

    def expand_replacement_list(args, expansion_location, macro_table, context)
      return [] unless replacement_list

      result = []
      index = 0
      while curr_token = replacement_list.tokens[index]
        next_token = replacement_list.tokens[index + 1]

        case
        when arg = args[curr_token.value]
          substitute_argument(curr_token, next_token, arg, result,
                              macro_table, context)
        when curr_token.value == "#"
          if next_token and arg = args[next_token.value]
            result.push(stringize_argument(arg, expansion_location))
            index += 1
          end
        when curr_token.value == "##" && next_token.value == "#"
          if next_next_token = replacement_list.tokens[index + 2] and
              arg = args[next_next_token.value]
            stringized = stringize_argument(arg, expansion_location)
            concat_with_last_token([stringized], expansion_location, result)
            index += 2
          end
        when curr_token.value == "##"
          if next_token and arg = args[next_token.value]
            concat_with_last_token(arg, expansion_location, result)
          else
            concat_with_last_token([next_token], expansion_location, result)
          end
          index += 1
        else
          result.push(ReplacedToken.new(curr_token.type, curr_token.value,
                                        expansion_location,
                                        curr_token.type_hint, false))
        end
        index += 1
      end
      result
    end

    def substitute_argument(param_token, next_token, arg, result,
                            macro_table, context)
      # NOTE: The ISO C99 standard saids;
      #
      # 6.10.3.1 Argument substitution
      #
      # 1 After the arguments for the invocation of a function-like macro have
      #   been identified, argument substitution take place.  A parameter in
      #   the replacement list, unless proceeded by a # or ## preprocessing
      #   token or followed by a ## preprocessing token, is replaced by the
      #   corresponding argument after all macros contained therein have been
      #   expanded.  Before being substituted, each argument's preprocessing
      #   tokens are completely macro replaced as if they formed the rest of
      #   the preprocessing file; no other preprocessing tokens are available.

      if next_token && next_token.value == "##"
        result.concat(arg.map { |t|
          ReplacedToken.new(t.type, t.value, t.location, t.type_hint, false)
        })
      else
        macro_table.replace(arg, context)
        result.concat(arg.map { |t|
          ReplacedToken.new(t.type, t.value, t.location, t.type_hint, true)
        })
      end
    end

    def stringize_argument(arg, expansion_location)
      # NOTE: The ISO C99 standard saids;
      #
      # 6.10.3.2 The # operator
      #
      # Constraints
      #
      # 1 Each # preprocessing token in the replacement list for a
      #   function-like macro shall be followed by a parameter as the next
      #   preprocessing token in the replacement list.
      #
      # Semantics
      #
      # 2 If, in the replacement list, a parameter is immediately proceeded by
      #   a # preprocessing token, both are replaced by a single character
      #   string literal preprocessing token that contains the spelling of the
      #   preprocessing token sequence for the corresponding argument.  Each
      #   occurrence of white space between the argument's preprocessing tokens
      #   becomes a single space character in the character string literal.
      #   White space before the first preprocessing token and after the last
      #   preprocessing token composing the argument is deleted.  Otherwise,
      #   the original spelling of each preprocessing token in the argument is
      #   retained in the character string literal, except for special handling
      #   for producing the spelling of string literals and character
      #   constants: a \ character is inserted before each " and \ character of
      #   a character constant or string literal (including the delimiting "
      #   characters), except that it is implementation-defined whether a \
      #   character is inserted before the \ character beginning of universal
      #   character name.  If the replacement that results is not a valid
      #   character string literal, the behavior is undefined.  The character
      #   string literal corresponding to an empty argument is "".  The order
      #   of evaluation of # and ## operators is unspecified.
      #
      # NOTE: This code does not concern about contents of the string literal.
      #       But, it is enough for analysis.

      ReplacedToken.new(:PP_TOKEN,
                        "\"#{arg.map { |token| token.value }.join(' ')}\"",
                        expansion_location, :STRING_LITERAL, false)
    end

    def concat_with_last_token(tokens, expansion_location, result)
      # NOTE: The ISO C99 standard saids;
      #
      # 6.10.3.3 The ## operator
      #
      # Constraints
      #
      # 1 A ## preprocessing token shall not occur at the beginning or at the
      #   end of a replacement list for either form of macro definition.
      #
      # Semantics
      #
      # 2 If, in the replacement list of a function-form macro, a parameter is
      #   immediately preceded or followed by a ## preprocessing token, the
      #   parameter is replaced by the corresponding argument's preprocessing
      #   token sequence; however, if an argument consists of no preprocessing
      #   tokens, the parameter is replaced by a placemarker preprocessing
      #   token instead.
      #
      # 3 For both object-like and function-like macro invocations, before the
      #   replacement list is reexamined for more macro names to replace, each
      #   instance of a ## preprocessing token in the replacement list (not
      #   from an argument) is deleted and the preceding preprocessing token is
      #   concatenated with the following preprocessing token.  Placemarker
      #   preprocessing tokens are handled specially: concatenation of two
      #   placemarkers results in a single placemarker preprocessing token, and
      #   concatenation of a placemarker with a non-placemarker preprocessing
      #   token results in the non-placemarker preprocessing token.  If the
      #   result is not a valid preprocessing token, the behavior is undefined.
      #   The resulting token is available for further macro replacement.  The
      #   order of evaluation of ## operators is unspecified.

      if last_token = result.pop
        result.push(
          ReplacedToken.new(:PP_TOKEN, last_token.value + tokens.first.value,
                            expansion_location, nil, false))

        result.concat(tokens[1..-1].map { |token|
          ReplacedToken.new(token.type, token.value, expansion_location,
                            token.type_hint, false)
        })
      end
    end
  end

  class SpecialMacro < ObjectLikeMacro
    def initialize(name_str)
      super(PseudoObjectLikeDefineLine.new(name_str))
      @replacement_list = nil
    end

    attr_reader :replacement_list

    def expand(tokens, macro_table, context)
      @replacement_list = generate_replacement_list(tokens.first)
      super
    end

    private
    def generate_replacement_list(token)
      subclass_responsibility
    end
  end

  class DateMacro < SpecialMacro
    def initialize
      super("__DATE__")
    end

    private
    def generate_replacement_list(token)
      date = Time.now.strftime("%h %d %Y")
      PPTokens.new.push(Token.new(:PP_TOKEN, "\"#{date}\"", token.location,
                                  :STRING_LITERAL))
    end
  end

  class FileMacro < SpecialMacro
    def initialize
      super("__FILE__")
    end

    private
    def generate_replacement_list(token)
      PPTokens.new.push(
        Token.new(:PP_TOKEN, "\"#{token.location.fpath}\"", token.location,
                  :STRING_LITERAL))
    end
  end

  class LineMacro < SpecialMacro
    def initialize
      super("__LINE__")
    end

    private
    def generate_replacement_list(token)
      PPTokens.new.push(
        Token.new(:PP_TOKEN, "#{token.location.line_no}", token.location,
                  :STRING_LITERAL))
    end
  end

  class StdcMacro < SpecialMacro
    def initialize
      super("__STDC__")
    end

    private
    def generate_replacement_list(token)
      PPTokens.new.push(Token.new(:PP_TOKEN, "1", token.location, :CONSTANT))
    end
  end

  class StdcHostedMacro < SpecialMacro
    def initialize
      super("__STDC_HOSTED__")
    end

    private
    def generate_replacement_list(token)
      PPTokens.new.push(Token.new(:PP_TOKEN, "1", token.location, :CONSTANT))
    end
  end

  class StdcMbMightNeqWcMacro < SpecialMacro
    def initialize
      super("__STDC_MB_MIGHT_NEQ_WC__")
    end

    private
    def generate_replacement_list(token)
      PPTokens.new.push(Token.new(:PP_TOKEN, "1", token.location, :CONSTANT))
    end
  end

  class StdcVersionMacro < SpecialMacro
    def initialize
      super("__STDC_VERSION__")
    end

    private
    def generate_replacement_list(token)
      PPTokens.new.push(Token.new(:PP_TOKEN, "199901L", token.location,
                                  :CONSTANT))
    end
  end

  class TimeMacro < SpecialMacro
    def initialize
      super("__TIME__")
    end

    private
    def generate_replacement_list(token)
      time = Time.now.strftime("%H:%M:%S")
      PPTokens.new.push(Token.new(:PP_TOKEN, "\"#{time}\"", token.location,
                                  :STRING_LITERAL))
    end
  end

  class StdcIec559Macro < SpecialMacro
    def initialize
      super("__STDC_IEC_559__")
    end

    private
    def generate_replacement_list(token)
      PPTokens.new.push(Token.new(:PP_TOKEN, "0", token.location, :CONSTANT))
    end
  end

  class StdcIec559ComplexMacro < SpecialMacro
    def initialize
      super("__STDC_IEC_559_COMPLEX__")
    end

    private
    def generate_replacement_list(token)
      PPTokens.new.push(Token.new(:PP_TOKEN, "0", token.location, :CONSTANT))
    end
  end

  class StdcIso10646Macro < SpecialMacro
    def initialize
      super("__STDC_ISO_10646__")
    end

    private
    def generate_replacement_list(token)
      PPTokens.new.push(Token.new(:PP_TOKEN, "199712L", token.location,
                                  :CONSTANT))
    end
  end

  class PragmaOperator < FunctionLikeMacro
    def initialize
      super(PseudoFunctionLikeDefineLine.new("_Pragma", ["str"]))
    end

    def expand(tokens, macro_table, context)
      # TODO: Should implement pragma handling feature.
      []
    end
  end

  class MacroReplacementContext
    def initialize
      @hide_sets = Hash.new { |h, k| h[k] = Set.new }
    end

    def add_to_hide_set(orig_token, new_tokens, macro_name)
      new_tokens.each do |new_token|
        @hide_sets[new_token].merge(@hide_sets[orig_token])
        @hide_sets[new_token].add(macro_name)
      end
    end

    def hidden?(token, macro_name)
      @hide_sets[token].include?(macro_name)
    end
  end

  class MacroTable
    def initialize
      @macros = {}
      predefine_special_macros
    end

    extend Pluggable

    def_plugin :on_object_like_macro_replacement
    def_plugin :on_function_like_macro_replacement

    def define(macro)
      @macros[macro.name.value] = macro
      self
    end

    def undef(name_str)
      @macros.delete(name_str)
      self
    end

    def lookup(name_str)
      @macros[name_str]
    end

    def replace(tokens, context = nil)
      replaced = false
      index = 0

      while token = tokens[index]
        case token.value
        when "defined"
          in_defined = true
        when "(", ")"
          ;
        else
          if in_defined
            in_defined = false
          else
            if new_index = do_replace(tokens, index, context)
              index = new_index
              replaced = true
            end
          end
        end
        index += 1
      end

      replaced
    end

    def notify_object_like_macro_replacement(macro, tokens)
      on_object_like_macro_replacement.invoke(macro, tokens)
    end

    def notify_function_like_macro_replacement(macro, tokens)
      on_function_like_macro_replacement.invoke(macro, tokens)
    end

    private
    def do_replace(tokens, index, context)
      context ||= MacroReplacementContext.new

      return nil unless token = tokens[index] and macro = lookup(token.value)
      return nil if context.hidden?(token, macro.name.value)

      size = macro.replaceable_size(tokens[index..-1])

      if tokens[index, size].all? { |t| t.need_no_further_replacement? }
        return nil
      end

      expanded = macro.expand(tokens[index, size], self, context)
      context.add_to_hide_set(tokens[index], expanded, macro.name.value)

      # NOTE: The ISO C99 standard saids;
      #
      # 6.10.3.4 Rescanning and further replacement
      #
      # 1 After all parameters in the replacement list have been substituted
      #   and # and ## processing has take place, all placemarker preprocessing
      #   tokens are removed.  Then, the resulting preprocessing token sequence
      #   is rescanned, along with all subsequent preprocessing tokens of the
      #   source file, for more macro names to replace.
      #
      # 2 If the name of the macro being replaced is found during this scan of
      #   the replacement list (not including the rest of the source file's
      #   preprocessing tokens), it is not replaced.  Furthermore, if any
      #   nested replacements encounter the name of the macro being replaced,
      #   it is not replaced.  These nonreplaced macro name preprocessing
      #   tokens are no longer available for further replacement even if they
      #   are later (re)examined in contexts in which that macro name
      #   preprocessing token whould otherwise have been replaced.
      while replace(expanded, context); end
      tokens[index, size] = expanded

      index + expanded.size - 1
    end

    def predefine_special_macros
      define(DateMacro.new)
      define(FileMacro.new)
      define(LineMacro.new)
      define(StdcMacro.new)
      define(StdcHostedMacro.new)
      define(StdcMbMightNeqWcMacro.new)
      define(StdcVersionMacro.new)
      define(TimeMacro.new)
      define(StdcIec559Macro.new)
      define(StdcIec559ComplexMacro.new)
      define(StdcIso10646Macro.new)
      define(PragmaOperator.new)
    end
  end

end
end
