/* **********************************************************
 * Copyright 1998 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

/*
 * main.c --
 *
 * Common part of Linux kernel module implementation of driver for the
 * VMware Host/Guest filesystem.
 */

/* Must come before any kernel header file */
#include "driver-config.h"

#include <linux/linkage.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/fs.h>
#include <linux/sched.h>
#include <linux/slab.h>
#include <linux/file.h>
#include <asm/uaccess.h>
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 9)
#include <linux/moduleparam.h>
#endif
#include "main.h"

#ifdef VMX86_DEVEL
/*
 * Logging is available only in devel build.
 */

int LOGLEVEL_THRESHOLD = 4;

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 9)
module_param(LOGLEVEL_THRESHOLD, int, 0444);
#else
MODULE_PARM(LOGLEVEL_THRESHOLD, "i");
#endif

MODULE_PARM_DESC(LOGLEVEL_THRESHOLD, "Set verbosity (0 means no log, 10 means very verbose, 4 is default)");
#endif

MODULE_AUTHOR("VMware, Inc.");
MODULE_DESCRIPTION("VMware Host/Guest File System");

spinlock_t hgfsBigLock = SPIN_LOCK_UNLOCKED;

HgfsReq requestPool[HGFS_MAX_OUTSTANDING_REQS];   /* pool of request objects */
struct semaphore requestSem;                      /* limits the max number of outstanding requests  */
struct list_head reqFreeList;                     /* free list of available (but uninitialized) requests */


/*
 * Gotta have Panic. Stolen from vmnix.
 */

void
Panic(const char *fmt, ...) // IN
{
   va_list args;
   char buffer[1024];
   volatile char *p = NULL;

   va_start(args, fmt);
   vsprintf(buffer, fmt, args);
   va_end(args);

   printk(KERN_EMERG "%s", buffer);

   /* force segfault */
   buffer[0] = *p;
   while (1) ; // avoid compiler warning
}


/*
 *----------------------------------------------------------------------
 *
 * vmstrdup --
 *
 *    Strdup using kmalloc
 *
 * Results:
 *    Returns a pointer to a new string identical to the input
 *    on success, or NULL if memory could not be allocated.
 *
 *    As with strdup, the returned string must be freed after use.
 *
 * Side effects:
 *    None
 *
 *----------------------------------------------------------------------
 */

char *
vmstrdup(char const *string) // IN: String to duplicate
{
   unsigned int size;
   char *new;

   ASSERT(string);

   size = strlen(string) + 1;
   new = kmalloc(size, GFP_KERNEL);
   if (!new) {
      return NULL;
   }

   memcpy(new, string, size);
   return new;
}


/*
 *----------------------------------------------------------------------
 *
 * HgfsFreeRequest --
 *
 *    Frees request object by setting its state to inactive
 *    and returning it to the free list. Should be called with
 *    the big lock held.
 *
 * Results:
 *    None
 *
 * Side effects:
 *    None
 *
 *----------------------------------------------------------------------
 */

void
HgfsFreeRequest(HgfsReq *req) // IN/OUT: The request to free
{
   ASSERT(req);

   req->stateFile = HGFS_REQ_STATE_INACTIVE;

   /*
    * Put request back on the free list, which is LIFO, and release
    * the request semaphore.
    */
   list_add(&req->list, &reqFreeList);
   up(&requestSem);
}


/*
 *----------------------------------------------------------------------
 *
 * init_module --
 *
 *    linux module entry point. Called by /sbin/insmod command.
 *    Registers the hgfs filesystem with the kernel.
 *
 * Results:
 *    Returns 0 on success, an error on failure.
 *
 * Side effects:
 *    None
 *
 *----------------------------------------------------------------------
 */

int
init_module(void)
{
   int error;

   error = register_filesystem(&hgfsType);
   if (error) {
      printk(KERN_WARNING "VMware hgfs: failed to register filesystem\n");
      return error;
   }

  /*
   * Setup our entry in /proc
   */
   HgfsSetupProcDevice();

   LOG(4, (KERN_DEBUG "VMware hgfs: Module Loaded\n"));

   return 0;
}


/*
 *----------------------------------------------------------------------
 *
 * cleanup_module --
 *
 *    Called by /sbin/rmmod. Unregisters filesystem with kernel,
 *    unloads module.
 *
 *    Note: for true kernel 2.4 compliance, this should be
 *    "module_exit".
 *
 * Results:
 *    None
 *
 * Side effects:
 *    None
 *
 *----------------------------------------------------------------------
 */

void
cleanup_module(void)
{
/* FIXME: Check actual kernel version when RR's modules went in */
#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 5, 45)
   if (MOD_IN_USE) {
      printk(KERN_WARNING "VMware hgfs: filesystem in use, removal failed\n");
   }
#endif

   if (unregister_filesystem(&hgfsType)) {
      printk(KERN_WARNING "VMware hgfs: failed to unregister filesystem\n");
   }

   /*
    * Remove our entry in /proc
    */
   HgfsCleanupProcDevice();

   LOG(4, (KERN_DEBUG "VMware hgfs: Module Unloaded\n"));
}
