/*
Copyright (C) 1997-2001 Id Software, Inc.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/

// gs_weapons.c	-	game shared weapons definitions 

#include "../game/wsw_shared.h"

#include "../game/q_shared.h"
#include "gs_public.h"

#ifndef WSW_RELEASE
#define ALLOW_WEAPONDEFS_READED_FROM_DISK
#endif

//============================================================================

//=================
// GS_Weaponstate_Run
//=================
int GS_Weaponstate_Run( weapon_state_t *state, unsigned int msec, firedef_t *firedef, qboolean pressing_fire )
{
	int actions = 0;

	// we go below zero for one call, but reset back to after that
	// the purpose of this, is to make it not matter in what intervals this function is called
#ifdef PREDICTSHOOTING
	if( state->nexttime > 0 )
		state->nexttime -= msec;
	else
		state->nexttime = 0;
#else
	state->nexttime -= msec;
	if( state->nexttime < 0 )
		state->nexttime = 0;
#endif

	// remove reload wait
	if( state->status == WEAPON_RELOADING && state->nexttime <= 0 )
	{
		state->poweredtime = 0; // remove power from last round
		if( firedef->cooldown_time ) {
			state->status = WEAPON_COOLDOWN;
			state->nexttime += firedef->cooldown_time;
			actions |= WEAPON_ACTION_STATE_CHANGE;
		} else {
			state->status = WEAPON_READY;
			actions |= WEAPON_ACTION_STATE_CHANGE;
		}
	}

	// remove cooldown wait
	if( state->status == WEAPON_COOLDOWN && state->nexttime <= 0 )
	{
		state->status = WEAPON_READY;
		actions |= WEAPON_ACTION_STATE_CHANGE;
	}

	// change weapon init
	if( state->changing && (state->status == WEAPON_READY || state->status == WEAPON_ACTIVATING)
		&& state->nexttime <= 0)
	{
		state->status = WEAPON_DROPPING;
		state->nexttime += firedef->weapondown_time;
		actions |= WEAPON_ACTION_STATE_CHANGE;
	}

	//change weapon do
	if( state->status == WEAPON_DROPPING && state->nexttime <= 0 )
	{
		actions |= WEAPON_ACTION_WEAPON_CHANGE;
		state->changing = qfalse;
		state->status = WEAPON_ACTIVATING;
		state->nexttime += firedef->weaponup_time;
		actions |= WEAPON_ACTION_STATE_CHANGE;
		return actions; //FIXME: this one MUST return to refresh the fire function pointers after weapon change
	}

	// change weapon changed
	if( state->status == WEAPON_ACTIVATING && state->nexttime <= 0 )
	{
		state->status = WEAPON_READY;
		actions |= WEAPON_ACTION_STATE_CHANGE;
	}

	//check for firing
	if( (state->status == WEAPON_READY || state->status == WEAPON_POWERING) && state->nexttime <= 0 )
	{
		if( pressing_fire )
		{
			if( state->status != WEAPON_POWERING ) {
				state->poweredtime = 0;
				actions |= WEAPON_ACTION_STATE_CHANGE;
			}
			state->status = WEAPON_POWERING;
			state->poweredtime += msec;

			// maximum time reached
			if( state->poweredtime >= firedef->powering_time ) {
				state->status = WEAPON_FIRING;
				actions |= WEAPON_ACTION_STATE_CHANGE;
			}
		}
		else if( state->status == WEAPON_POWERING ) // let go of fire button, launch it
		{
			state->status = WEAPON_FIRING;
			actions |= WEAPON_ACTION_STATE_CHANGE;
		}
	}

	// must fire a projectile? (never leave from here with WEAPON_FIRING state, or having power)
	// note: power is removed after calling the fire function, so it's value can be used in it
	if( state->status == WEAPON_FIRING )
	{
		actions |= WEAPON_ACTION_FIRE;
		state->status = WEAPON_RELOADING;
		actions |= WEAPON_ACTION_STATE_CHANGE;
		state->nexttime += firedef->reload_time;
		// it's the caller's responsibility to clear state->poweredtime after WEAPON_ACTION_FIRE action
	}

	return actions;
}

//======================================================
//
//	WEAPON DEFS
//
//======================================================

weapon_info_t gs_weaponInfos[WEAP_TOTAL];

firedef_t noweaponFireDef =
{
	WEAP_NONE,							// weapon name
	0,									// fire mode
	AMMO_NONE,							// ammo tag
	0,									// ammo usage per shot
	0,									// projectiles fired each shot

	//timings (in msecs)
	100,								// weapon up frametime
	100,								// weapon down frametime
	0,									// reload frametime
	0,									// cooldown frametime
	0,									// max powering up time
	0,									// projectile timeout

	//damages
	0,									// damage
	0,									// knockback
	0,									// splash radius
	0,									// splash minimum damage

	//projectile def
	0,									// speed
	0,									// h spread (I added a small spread while using the replacement gun)
	0,									// v spread

	//ammo
	0,									// pickup amount
	0									// max amount
};


#ifdef ALLOW_WEAPONDEFS_READED_FROM_DISK

static qbyte *GS_LoadWeaponDefFile( char *defname )
{
	int		length, filenum;
	qbyte	*data;
	char	filename[MAX_QPATH];

	Q_snprintfz( filename, sizeof(filename), "weapondefs/%s.def", defname );
	length = GS_FS_FOpenFile( filename, &filenum, FS_READ );

	if (length == -1) {
		GS_Printf( "Couldn't find script: %s.\n", filename );
		return NULL;
	}

	if( !length ) {
		GS_Printf( "Found empty script: %s.\n", filename );
		GS_FS_FCloseFile( filenum );
		return NULL;
	}

	//load the script data into memory
	data = GS_Malloc( length + 1 );
	GS_FS_Read( data, length, filenum );
	GS_FS_FCloseFile( filenum );

	if( !data[0] ) {
		GS_Printf( "Found empty script: %s.\n", filename );
		return NULL;
	}

	return data;
}

#define WEAPONDEF_NUMPARMS 17
static qboolean GS_ParseFiredefFile( qbyte *buf, firedef_t *firedef )
{
	char		*ptr, *token;
	int			count = 0;
	int			parm[WEAPONDEF_NUMPARMS];

	// jal: this is quite ugly.
	ptr = ( char * )buf;
	while ( ptr )
	{
		token = COM_ParseExt ( &ptr, qtrue );
		if( !token )
			break;

		if( !token[0] )
			continue;

		//ignore spacing tokens
		if( !Q_stricmp( token, "," ) ||
			!Q_stricmp( token, "{" ) ||
			!Q_stricmp( token, "}" ) )
			continue;

		//some token sanity checks
		if( token[strlen(token)-1] == ',' )
			token[strlen(token)-1] = 0;
		if( token[strlen(token)-1] == '}' )
			token[strlen(token)-1] = 0;
		//(I don't fix these ones, but show the error)
		if( token[0] == ',' ) {
			GS_Printf( "ERROR in script. Comma must be followed by space or newline\n" );
			return qfalse;
		}
		if( token[0] == '{' || token[0] == '}' ) {
			GS_Printf( "ERROR in script. Scorches must be followed by space or newline\n" );
			return qfalse;
		}

		if( count > WEAPONDEF_NUMPARMS )
			return qfalse;

		if( !Q_stricmp( token, "instant" ) )
			parm[count] = 0;
		else
			parm[count] = atoi( token );

		if( parm[count] < 0 )
			return qfalse;

		count++;
	}

	// incomplete or wrong file
	if( count < WEAPONDEF_NUMPARMS )
		return qfalse;

	// validate

	// 2 and 3 are weapon switches. They MUST be game.frametime msecs at least
	/*if( parm[2] < game.frametime )
		parm[2] = game.frametime;

	if( parm[3] < game.frametime )
		parm[3] = game.frametime;*/

	count = 0;
	// put the data into the firedef
	firedef->usage_count = parm[count++];
	firedef->projectile_count = parm[count++];

	firedef->weaponup_time = parm[count++];
	firedef->weapondown_time = parm[count++];
	firedef->reload_time = parm[count++];
	firedef->cooldown_time = parm[count++];
	firedef->powering_time = parm[count++];
	firedef->timeout = parm[count++];

	firedef->damage = parm[count++];
	firedef->knockback = parm[count++];
	firedef->splash_radius = parm[count++];
	firedef->splash_min_damage = parm[count++];

	firedef->speed = parm[count++];
	firedef->h_spread = parm[count++];
	firedef->v_spread = parm[count++];

	firedef->ammo_pickup = parm[count++];
	firedef->ammo_max = parm[count++];

	return qtrue;
}

static qboolean GS_LoadFiredefFromFile( firedef_t *firedef )
{
	char	filename[MAX_QPATH];
	qbyte	*data;

	if( !firedef || !firedef->weapon_id )
		return qfalse;

	Q_snprintfz( filename, sizeof(filename), "%s %s", GS_FindItemByTag(firedef->weapon_id)->pickup_name,
		(firedef->fire_mode == FIRE_MODE_STRONG) ? "strong" : "weak" );

	data = GS_LoadWeaponDefFile( filename );
	if( !data )
		return qfalse;

	//parse the file updating the firedef
	if( !GS_ParseFiredefFile( data, firedef ) )
	{
		GS_Printf( "'InitWeapons': Error in definition file %s\n", filename );
		GS_Free( data );
		return qfalse;
	}

	GS_Free( data );
	return qtrue;
}
#endif //ALLOW_WEAPONDEFS_READED_FROM_DISK

//=================
//GS_FiredefForAmmo
//=================
firedef_t *GS_FiredefForAmmo( int tag )
{
	gitem_t *item = GS_FindItemByTag( tag );

	if( item->type == IT_AMMO)
		return (firedef_t *)item->info;
	else
		return NULL;
}

//=================
//GS_FiredefForWeapon
//=================
weapon_info_t *GS_FiredefForWeapon( int tag )
{
	gitem_t *item = GS_FindItemByTag( tag );

	if( item->type == IT_WEAPON )
		return (weapon_info_t *)item->info;
	else
		return NULL;
}

//=================
//GS_InitWeapons
//=================
void GS_InitWeapons( void )
{
	int i;
	gitem_t *item;
	firedef_t *firedef;

	GS_Printf( "Initializing weapons\n" );

	//set up no weapon with no fires (not totally correct. I should not use item->info)
	gs_weaponInfos[WEAP_NONE].firedef = &noweaponFireDef;
	gs_weaponInfos[WEAP_NONE].firedef_weak = &noweaponFireDef;

	//assign the actual weapons
	for( i = WEAP_GUNBLADE; i < WEAP_TOTAL; i++ )
	{
		item = GS_FindItemByTag(i);
		gs_weaponInfos[i].firedef = gs_weaponInfos[i].firedef_weak = NULL; //start clean

		//find predefined firedefs for this weapon
		for( firedef = &ammoFireDefs[0]; firedef->weapon_id; firedef++ )
		{
			if( firedef->weapon_id == i ) {
				if( firedef->ammo_id < AMMO_WEAK_GUNBLADE ) {
					assert( firedef->fire_mode == FIRE_MODE_STRONG );
					gs_weaponInfos[i].firedef = firedef;
				} else {
					assert( firedef->fire_mode == FIRE_MODE_WEAK );
					gs_weaponInfos[i].firedef_weak = firedef;
				}
			}
		}

		assert( gs_weaponInfos[i].firedef );
		assert( gs_weaponInfos[i].firedef_weak );

		//link here too, but use gs_weaponInfos better
		item->info = (weapon_info_t *)&gs_weaponInfos[i];
		if( GS_FindItemByTag(item->weakammo_tag) ) {
			GS_FindItemByTag(item->weakammo_tag)->info = gs_weaponInfos[i].firedef_weak;
			GS_FindItemByTag(item->weakammo_tag)->quantity = gs_weaponInfos[i].firedef_weak->ammo_pickup;
		}
		if( GS_FindItemByTag(item->ammo_tag) ) {
			GS_FindItemByTag(item->ammo_tag)->info = gs_weaponInfos[i].firedef;
			GS_FindItemByTag(item->ammo_tag)->quantity = gs_weaponInfos[i].firedef->ammo_pickup;
		}
	}

#ifdef ALLOW_WEAPONDEFS_READED_FROM_DISK
	// ok, we assigned the default ones, but meanwhile we are designing the
	// game we will load replacements from text files
	for( i = WEAP_GUNBLADE; i < WEAP_TOTAL; i++ )
	{
		GS_LoadFiredefFromFile( gs_weaponInfos[i].firedef );
		GS_LoadFiredefFromFile( gs_weaponInfos[i].firedef_weak );
	}
#endif //ALLOW_WEAPONDEFS_READED_FROM_DISK
}
