/*
Copyright (C) 2002-2003 Victor Luchits

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/

#include "client.h"
#include "../ui/ui_public.h"

// Structure containing functions exported from user interface DLL
ui_export_t	*uie;

mempool_t	*ui_mempool;
static void	*module_handle;

/*
===============
CL_UIModule_CM_LoadMapMessage

Just adds cl.cms as first parameter
===============
*/
static inline char *CL_UIModule_CM_LoadMapMessage( char *name, char *message, int size )
{
	return CM_LoadMapMessage( cl.cms, name, message, size );
}

/*
===============
CL_UIModule_Print
===============
*/
static void CL_UIModule_Print( char *msg ) {
	Com_Printf( "%s", msg );
}

/*
===============
CL_UIModule_Error
===============
*/
static void CL_UIModule_Error( char *msg ) {
	Com_Error( ERR_FATAL, "%s", msg );
}

/*
===============
CL_UIModule_GetConfigString
===============
*/
static void CL_UIModule_GetConfigString( int i, char *str, int size )
{
	if( i < 0 || i >= MAX_CONFIGSTRINGS )
		Com_Error( ERR_DROP, "CL_UIModule_GetConfigString: i > MAX_CONFIGSTRINGS" );
	if( !str || size <= 0 )
		Com_Error( ERR_DROP, "CL_UIModule_GetConfigString: NULL string" );
	Q_strncpyz( str, cl.configstrings[i], size );
}

#ifdef ZEROTHREEAPI
typedef struct
{
	// halts the application
	void		(*Error)( char *str );

	// console messages
	void		(*Print)( char *str );

	// dynvars
	dynvar_t	*(*Dynvar_Create)( const char *name, qboolean console, dynvar_getter_f getter, dynvar_setter_f setter );
	void		(*Dynvar_Destroy)( dynvar_t *dynvar );
	dynvar_t	*(*Dynvar_Lookup)( const char *name );
	const char	*(*Dynvar_GetName)( dynvar_t *dynvar );
	dynvar_get_status_t (*Dynvar_GetValue)( dynvar_t *dynvar,void **value );
	dynvar_set_status_t (*Dynvar_SetValue)( dynvar_t *dynvar,void *value );
	void		(*Dynvar_AddListener)( dynvar_t *dynvar, dynvar_listener_f listener );
	void		(*Dynvar_RemoveListener)( dynvar_t *dynvar, dynvar_listener_f listener );

	// console variable interaction
	cvar_t 		*(*Cvar_Get)( const char *name, const char *value, int flags );
	cvar_t 		*(*Cvar_Set)( const char *name, const char *value );
	cvar_t 		*(*Cvar_ForceSet)( const char *name, const char *value );	// will return 0 0 if not found
	void		(*Cvar_SetValue)( const char *name, float value );
	float		(*Cvar_VariableValue)( const char *name );
	char		*(*Cvar_VariableString)( const char *name );

	//newgametypes[start] Allow the ui to read console args
	int			(*Cmd_Argc)( void );
	char		*(*Cmd_Argv)( int arg );
	char		*(*Cmd_Args)( void );	// concatenation of all argv >= 1
	//newgametypes[end]

	void			(*Cmd_AddCommand)( char *name, void(*cmd)(void) );
	void			(*Cmd_RemoveCommand)( char *cmd_name );
	void			(*Cmd_ExecuteText)( int exec_when, const char *text );
	void			(*Cmd_Execute)( void );

	void			(*R_ClearScene)( void );
	void			(*R_AddEntityToScene)( entity_t *ent );
	void			(*R_AddLightToScene)( vec3_t org, float intensity, float r, float g, float b, struct shader_s *shader );
	void			(*R_AddPolyToScene)( poly_t *poly );
	void			(*R_RenderScene)( refdef_t *fd );
	void			(*R_EndFrame)( void );
	void			(*R_ModelBounds)( struct model_s *mod, vec3_t mins, vec3_t maxs );
	struct model_s 	*(*R_RegisterModel)( char *name );
	struct shader_s *(*R_RegisterSkin)( char *name );
	struct shader_s *(*R_RegisterPic)( char *name );
	struct skinfile_s *(*R_RegisterSkinFile)( char *name );
	qboolean 		(*R_LerpTag)( orientation_t *orient, struct model_s *mod, int oldframe, int frame, float lerpfrac, const char *name );
	void			(*R_DrawStretchPic)( int x, int y, int w, int h, float s1, float t1, float s2, float t2, vec4_t color, struct shader_s *shader );
	void			(*R_TransformVectorToScreen)( refdef_t *rd, vec3_t in, vec2_t out );
	int				(*R_SkeletalGetNumBones)( struct model_s *mod, int *numFrames );
	int				(*R_SkeletalGetBoneInfo)( struct model_s *mod, int bone, char *name, int size, int *flags );
	void			(*R_SkeletalGetBonePose)( struct model_s *mod, int bone, int frame, bonepose_t *bonepose );

	char			*(*CM_LoadMapMessage)( char *name, char *message, int size );

	struct sfx_s	*(*S_RegisterSound)( const char *name );	
	void			(*S_StartLocalSound)( const char *s );
	void			(*S_StartBackgroundTrack)( const char *intro, const char *loop );
	void			(*S_StopBackgroundTrack)( void );

	// fonts
	struct mufont_s *(*SCR_RegisterFont)( char *name );
	void			(*SCR_DrawString)( int x, int y, int align, const char *str, struct mufont_s *font, vec4_t color );
	int				(*SCR_DrawStringWidth)( int x, int y, int align, const char *str, int maxwidth, struct mufont_s *font, vec4_t color );
	size_t			(*SCR_strHeight)( struct mufont_s *font );
	size_t			(*SCR_strWidth)( const char *str, struct mufont_s *font, int maxlen );
	size_t			(*SCR_StrlenForWidth)( const char *str, struct mufont_s *font, size_t maxwidth );

	void		(*CL_Quit)( void );
	void		(*CL_SetKeyDest)( int key_dest );
	void		(*CL_ResetServerCount)( void );
	char		*(*CL_GetClipboardData)( qboolean primary );
	void		(*CL_FreeClipboardData)( char *data );

	char		*(*Key_GetBindingBuf)( int binding );
	void		(*Key_ClearStates)( void );
	char		*(*Key_KeynumToString)( int keynum );
	void		(*Key_SetBinding)( int keynum, char *binding );
	qboolean 	(*Key_IsDown)( int keynum );

	qboolean	(*VID_GetModeInfo)( int *width, int *height, qboolean *wideScreen, int mode );

	void		(*GetConfigString)( int i, char *str, int size );
	unsigned int	(*Milliseconds)( void );

	// files will be memory mapped read only
	// the returned buffer may be part of a larger pak file,
	// or a discrete file from anywhere in the quake search path
	// a -1 return means the file does not exist
	// NULL can be passed for buf to just determine existance
	int			(*FS_FOpenFile)( const char *filename, int *filenum, int mode );
	int			(*FS_Read)( void *buffer, size_t len, int file );
	int			(*FS_Write)( const void *buffer, size_t len, int file );
	int			(*FS_Tell)( int file );
	int			(*FS_Seek)( int file, int offset, int whence );
	int			(*FS_Eof)( int file );
	int			(*FS_Flush)( int file );
	void		(*FS_FCloseFile)( int file );
	int			(*FS_GetFileList)( const char *dir, const char *extension, char *buf, size_t bufsize );
	int			(*FS_GetGameDirectoryList)( char *buf, size_t bufsize );
	const char	*(*FS_FirstExtension)( const char *filename, char **extensions, int num_extensions );

	struct mempool_s *(*Mem_AllocPool)( const char *name, const char *filename, int fileline );

	void		*(*Mem_Alloc)( struct mempool_s *pool, size_t size, const char *filename, int fileline );
	void		(*Mem_Free)( void *data, const char *filename, int fileline );
	void		(*Mem_FreePool)( struct mempool_s **pool, const char *filename, int fileline );
	void		(*Mem_EmptyPool)( struct mempool_s *pool, const char *filename, int fileline );
} ui_import03_t;

int CL_UIModule_FS_GetFileList_03( const char *dir, const char *extension, char *buf, size_t bufsize ) {
	return FS_GetFileList( dir, extension, buf, bufsize, 0, 0 );
}
#endif

/*
===============
CL_UIModule_MemAlloc
===============
*/
static void *CL_UIModule_MemAlloc( mempool_t *pool, size_t size, const char *filename, int fileline ) {
	return _Mem_Alloc( pool, size, MEMPOOL_USERINTERFACE, 0, filename, fileline );
}

/*
===============
CL_UIModule_MemFree
===============
*/
static void CL_UIModule_MemFree( void *data, const char *filename, int fileline ) {
	_Mem_Free( data, MEMPOOL_USERINTERFACE, 0, filename, fileline );
}

/*
===============
CL_UIModule_MemAllocPool
===============
*/
static mempool_t *CL_UIModule_MemAllocPool( const char *name, const char *filename, int fileline ) {
	return _Mem_AllocPool( ui_mempool, name, MEMPOOL_USERINTERFACE, filename, fileline );
}

/*
===============
CL_UIModule_MemFreePool
===============
*/
static void CL_UIModule_MemFreePool( mempool_t **pool, const char *filename, int fileline ) {
	_Mem_FreePool( pool, MEMPOOL_USERINTERFACE, 0, filename, fileline );
}

/*
===============
CL_UIModule_MemEmptyPool
===============
*/
static void CL_UIModule_MemEmptyPool( mempool_t *pool, const char *filename, int fileline ) {
	_Mem_EmptyPool( pool, MEMPOOL_GAMEPROGS, 0, filename, fileline );
}

/*
==============
CL_UIModule_InitImportStruct
==============
*/
#define CL_UIModule_InitImportStruct(import) \
( \
	import.Error = CL_UIModule_Error, \
	import.Print = CL_UIModule_Print, \
/* dynvars */ \
	import.Dynvar_Create = Dynvar_Create, \
	import.Dynvar_Destroy = Dynvar_Destroy, \
	import.Dynvar_Lookup = Dynvar_Lookup, \
	import.Dynvar_GetName = Dynvar_GetName, \
	import.Dynvar_GetValue = Dynvar_GetValue, \
	import.Dynvar_SetValue = Dynvar_SetValue, \
	import.Dynvar_AddListener = Dynvar_AddListener, \
	import.Dynvar_RemoveListener = Dynvar_RemoveListener, \
\
	import.Cvar_Get = Cvar_Get, \
	import.Cvar_Set = Cvar_Set, \
	import.Cvar_SetValue = Cvar_SetValue, \
	import.Cvar_ForceSet = Cvar_ForceSet, \
	import.Cvar_VariableString = Cvar_VariableString, \
	import.Cvar_VariableValue = Cvar_VariableValue, \
\
/* newgametypes[start] Allow the ui to read console args */ \
	import.Cmd_Argc = Cmd_Argc, \
	import.Cmd_Argv = Cmd_Argv, \
	import.Cmd_Args = Cmd_Args, \
/* newgametypes[end] */ \
\
	import.Cmd_AddCommand = Cmd_AddCommand, \
	import.Cmd_RemoveCommand = Cmd_RemoveCommand, \
	import.Cmd_ExecuteText = Cbuf_ExecuteText, \
	import.Cmd_Execute = Cbuf_Execute, \
\
	import.FS_FOpenFile = FS_FOpenFile, \
	import.FS_Read = FS_Read, \
	import.FS_Write = FS_Write, \
	import.FS_Tell = FS_Tell, \
	import.FS_Seek = FS_Seek, \
	import.FS_Eof = FS_Eof, \
	import.FS_Flush = FS_Flush, \
	import.FS_FCloseFile = FS_FCloseFile, \
/*	import.FS_GetFileList = FS_GetFileList,*/ \
	import.FS_GetGameDirectoryList = FS_GetGameDirectoryList, \
	import.FS_FirstExtension = FS_FirstExtension, \
\
	import.CL_Quit = CL_Quit, \
	import.CL_SetKeyDest = CL_SetKeyDest, \
	import.CL_ResetServerCount = CL_ResetServerCount, \
	import.CL_GetClipboardData = CL_GetClipboardData, \
	import.CL_FreeClipboardData = CL_FreeClipboardData, \
\
	import.Key_ClearStates = Key_ClearStates, \
	import.Key_GetBindingBuf = Key_GetBindingBuf, \
	import.Key_KeynumToString = Key_KeynumToString, \
	import.Key_SetBinding = Key_SetBinding, \
	import.Key_IsDown = Key_IsDown, \
\
	import.R_ClearScene = R_ClearScene, \
	import.R_AddEntityToScene = R_AddEntityToScene, \
	import.R_AddLightToScene = R_AddLightToScene, \
	import.R_AddPolyToScene = R_AddPolyToScene, \
	import.R_RenderScene = R_RenderScene, \
	import.R_EndFrame = R_EndFrame, \
	import.R_ModelBounds = R_ModelBounds, \
	import.R_RegisterModel = R_RegisterModel, \
	import.R_RegisterPic = R_RegisterPic, \
	import.R_RegisterSkin = R_RegisterSkin, \
	import.R_RegisterSkinFile = R_RegisterSkinFile, \
	import.R_LerpTag = R_LerpTag, \
	import.R_DrawStretchPic = R_DrawStretchPic, \
	import.R_TransformVectorToScreen = R_TransformVectorToScreen, \
	import.R_SkeletalGetNumBones = R_SkeletalGetNumBones, \
	import.R_SkeletalGetBoneInfo = R_SkeletalGetBoneInfo, \
	import.R_SkeletalGetBonePose = R_SkeletalGetBonePose, \
\
	import.CM_LoadMapMessage = CL_UIModule_CM_LoadMapMessage, \
\
	import.S_RegisterSound = CL_SoundModule_RegisterSound, \
	import.S_StartLocalSound = CL_SoundModule_StartLocalSound, \
	import.S_StartBackgroundTrack = CL_SoundModule_StartBackgroundTrack, \
	import.S_StopBackgroundTrack = CL_SoundModule_StopBackgroundTrack, \
\
	import.SCR_RegisterFont = SCR_RegisterFont, \
	import.SCR_DrawString = SCR_DrawString, \
	import.SCR_DrawStringWidth = SCR_DrawStringWidth, \
	import.SCR_strHeight = SCR_strHeight, \
	import.SCR_strWidth = SCR_strWidth, \
	import.SCR_StrlenForWidth = SCR_StrlenForWidth, \
\
	import.GetConfigString = CL_UIModule_GetConfigString, \
\
	import.Milliseconds = Sys_Milliseconds, \
\
	import.VID_GetModeInfo = VID_GetModeInfo, \
\
	import.Mem_Alloc = CL_UIModule_MemAlloc, \
	import.Mem_Free = CL_UIModule_MemFree, \
	import.Mem_AllocPool = CL_UIModule_MemAllocPool, \
	import.Mem_FreePool = CL_UIModule_MemFreePool, \
	import.Mem_EmptyPool = CL_UIModule_MemEmptyPool \
)

/*
==============
CL_UIModule_Init
==============
*/
void CL_UIModule_Init( void )
{
	int apiversion;
	ui_import_t	import;

	CL_UIModule_Shutdown();

	ui_mempool = Mem_AllocPool( NULL, "User Iterface" );

	CL_UIModule_InitImportStruct( import );
	import.FS_GetFileList = FS_GetFileList;

#ifdef UI_HARD_LINKED
	{
		EXTERN_API_FUNC void *GetUIAPI( void * );
		uie = (ui_export_t *)GetUIAPI( &import );
	}
#else
	uie = (ui_export_t *)Com_LoadGameLibrary( "ui", "GetUIAPI", &module_handle, &import, cls.sv_pure );
#endif
	if( !uie )
		Com_Error( ERR_DROP, "Failed to load UI dll" );

	apiversion = uie->API ();
	if( apiversion != UI_API_VERSION ) {
#ifdef ZEROTHREEAPI
		if( apiversion == 17 ) {
			ui_import03_t import03;

			Com_UnloadGameLibrary( &module_handle );

			CL_UIModule_InitImportStruct( import03 );
			import03.FS_GetFileList = CL_UIModule_FS_GetFileList_03;

			uie = (ui_export_t *)Com_LoadGameLibrary( "ui", "GetUIAPI", &module_handle, &import03, cls.sv_pure );
			if( !uie )
				Com_Error( ERR_DROP, "Failed to load UI DLL" );
			goto load;
		}
#endif

#ifndef UI_HARD_LINKED
		Com_UnloadGameLibrary( &module_handle );
#endif
		Mem_FreePool( &ui_mempool );
		uie = NULL;

		Com_Error( ERR_FATAL, "UI version is %i, not %i", apiversion, UI_API_VERSION );
	}

#ifdef ZEROTHREEAPI
load:
	cls.uiAPIversion = apiversion;
	uie->Init( viddef.width, viddef.height );
#else
	uie->Init( viddef.width, viddef.height, PROTOCOL_VERSION );
#endif
}

/*
===============
CL_UIModule_Shutdown
===============
*/
void CL_UIModule_Shutdown( void )
{
	if( !uie )
		return;

	uie->Shutdown();
#ifndef UI_HARD_LINKED
	Com_UnloadGameLibrary( &module_handle );
#endif
	Mem_FreePool( &ui_mempool );
	uie = NULL;
}

/*
===============
CL_UIModule_Refresh
===============
*/
void CL_UIModule_Refresh( qboolean backGround )
{
	if( uie )
		uie->Refresh( cls.realtime, Com_ClientState(), Com_ServerState(), backGround );
}

/*
===============
CL_UIModule_DrawConnectScreen
===============
*/
void CL_UIModule_DrawConnectScreen( qboolean backGround )
{
	if( uie ) {
		uie->DrawConnectScreen( cls.servername, cls.rejected ? cls.rejectmessage : NULL,
				( cls.download.filenum || cls.download.web ) ?
				va( "%s %3.1f%c", cls.download.name, cls.download.percent * 100.0, '%' ) : NULL,
				cls.connect_count, backGround );
	}
}

/*
===============
CL_UIModule_Keydown
===============
*/
void CL_UIModule_Keydown( int key )
{
	if( uie )
		uie->Keydown( key );
}

/*
===============
CL_UIModule_CharEvent
===============
*/
void CL_UIModule_CharEvent( int key )
{
	if( uie )
		uie->CharEvent( key );
}

/*
===============
CL_UIModule_ForceMenuOff
===============
*/
void CL_UIModule_ForceMenuOff( void )
{
	if( uie )
		uie->ForceMenuOff();
}

/*
===============
CL_UIModule_AddToServerList
===============
*/
void CL_UIModule_AddToServerList( char *adr, char *info )
{
	if( uie )
		uie->AddToServerList( adr, info );
}

/*
===============
CL_UIModule_MouseMove
===============
*/
void CL_UIModule_MouseMove( int dx, int dy )
{
	if( uie )
		uie->MouseMove( dx, dy );
}
