#ifndef __FREND_H__
# define __FREND_H__

# include <cstdio>
# include <ctype.h>
# include <iostream>
# include <string>
# include <list>
# include <set>
# include <map>
# include <vector>
# include <algorithm>

# include "config.h"

# ifdef FR_GNOME
#  ifndef GNOME_H
struct GnomeUIInfo;
#  endif
# endif

# ifndef __GTK_H__
struct GtkWidget;
struct GtkObject;
struct GtkAdjustment;
struct GtkTextTag;
enum GtkAttachOptions {};
# endif

# ifndef __GDK_H__
struct GdkPixbuf;
struct GdkPixmap;
struct GdkBitmap;
struct GdkEvent;
# endif

# ifndef __G_LIB_H__
struct GSList;
struct GList;
# endif

# define FR_PATH_MAX PATH_MAX

class fr_Element;
class fr_Style;
class fr_Label;
class fr_Button;
class fr_Box;
class fr_ToggleInGroup;
class fr_GroupHolder;
class fr_RadioButton;
class fr_RadioGroup;
class fr_MenuItem;
class fr_PulldownMenu;
class fr_File;
class fr_Notebook;
class fr_Notepage;
class fr_Window;
class fr_Event;
class fr_Listener;
class fr_DataTable;

enum fr_Direction
  {
    fr_Horizontal, fr_Vertical
  };

enum fr_CaseSensitivity
  {
    fr_CaseInsensitive, fr_CaseSensitive
  };

enum fr_EventType
  {
    fr_Changed, fr_Click, fr_DoubleClick, fr_Select, fr_Unselect,
    fr_MenuClick, fr_Destroy
  };

enum fr_StockItem
{
  FR_STOCK_OK         = 0x00001,
  FR_STOCK_CANCEL     = 0x00002,
  FR_STOCK_APPLY      = 0x00004,
  FR_STOCK_CLOSE      = 0x00008,
  FR_STOCK_OPEN       = 0x00010,
  FR_STOCK_SAVE       = 0x00020,
  FR_STOCK_SAVE_AS    = 0x00040,
  FR_STOCK_NEW        = 0x00080,
  FR_STOCK_ADD        = 0x00100,
  FR_STOCK_REMOVE     = 0x00200,
  FR_STOCK_YES        = 0x00400,
  FR_STOCK_NO         = 0x00800,
  FR_STOCK_REFRESH    = 0x01000,
  FR_STOCK_CONFIG     = 0x02000,
  FR_STOCK_EDIT       = 0x04000,
  FR_STOCK_ABOUT      = 0x08000,
  FR_STOCK_RUN        = 0x10000,
  FR_STOCK_CLEAR      = 0x20000,
  FR_STOCK_DELETE     = 0x40000,
  FR_STOCK_QUIT       = 0x80000
};
typedef unsigned int fr_StockItems;

/// case sensitive string compare
struct stringeqcs : public std::binary_function<const char, const char, bool>
{
	bool operator()(const char a, const char b)
		{ return a==b; }
};

/// case insensitive string compare
struct stringeqci : public std::binary_function<const char, const char, bool>
{
	bool operator()(const char a, const char b)
		{ return tolower(a)==tolower(b); }
};

typedef std::map<std::string,std::string> strmap;

class fr_ArgList
{
 private:
  char			**argv;
  bool			*marked;
  fr_CaseSensitivity	casesensitivity, *sensitive;
  int			size, countargs, ignore;
  void			ArgAlloc(int);
  void			Grow();
 public:
  fr_ArgList(int n=3);
  fr_ArgList(const char*args);
  fr_ArgList(int argc, char*argv[]);
  fr_ArgList(const fr_ArgList& a);
  virtual		~fr_ArgList();
  virtual void          clear(int n=3);
  virtual void          ignoreNext(int n=1) { ignore = n; }
  virtual fr_ArgList&	operator<< (const fr_CaseSensitivity cs);
  virtual fr_ArgList&	operator<< (const char*arg);
  virtual fr_ArgList&	operator<< (const std::string& arg) { return *this << arg.c_str(); }
  virtual fr_ArgList&	operator<< (int i);
  virtual void          loadUrlEncodedArgs(const char*uargs);
  int			GetSize() const { return size; }
  int			CountArgs() const { return countargs; }
  const char*		operator[] (int n) const;
  char**		GetArgs() const { return argv; }
  const std::string	GetString() const;
  
  int			MatchArg(const char*, fr_CaseSensitivity);
  int			MatchArgs(fr_ArgList&);
  void			ClearMarks();
  void			Mark(int);
  bool			IsMarked(int) const;
  const char*		GetPreferredArg() const;
  friend std::istream&  operator>> (std::istream& i, fr_ArgList& a);
  friend std::ostream&  operator<< (std::ostream& o, const fr_ArgList& a);
};

template <class T>
class fr_dimension
{
 private:
  T _x, _y;
 public:
  fr_dimension(T x, T y) { _x = x; _y = y; }
  fr_dimension(const fr_dimension& p) { _x = p._x; _y = p._y; }
  T x() const { return _x; }
  T y() const { return _y; }
  fr_dimension abs() const { return fr_dimension(_x<0?-_x:_x, _y<0?-_y:_y); }
  fr_dimension operator+(const fr_dimension& p) const { return fr_dimension(_x+p._x, _y+p._y); }
  fr_dimension operator-(const fr_dimension& p) const { return fr_dimension(_x-p._x, _y-p._y); }
  fr_dimension& operator= (const fr_dimension& p) { _x =p._x; _y =p._y; return *this; }
  fr_dimension& operator+=(const fr_dimension& p) { _x+=p._x; _y+=p._y; return *this; }
  fr_dimension& operator-=(const fr_dimension& p) { _x-=p._x; _y-=p._y; return *this; }
  void print(std::ostream& o) const;
  void input(std::istream& i);
};
typedef fr_dimension<int> fr_Point;
typedef fr_dimension<size_t> fr_Size;

class fr_Color
{
  private:
    int rgb;
  public:
    fr_Color(int OxRRGGBB=0x000000): rgb(OxRRGGBB) {}
    fr_Color(int red, int green, int blue): rgb(red*0xffff+green*0xff+blue) {}
    int r() const { return (rgb / 0xffff) & 0xff; }
    int g() const { return (rgb / 0xff) & 0xff; }
    int b() const { return rgb & 0xff; }
    int toInt() const { return rgb; }
    bool isValid() const { return (rgb >= 0) && (rgb <= 0xffffff); }
    std::string toString() const;
    fr_Color operator-() const { return ~rgb & 0xffffff; }
    fr_Color operator!() const { return isValid(); }
};

class fr_Colors: public std::vector<fr_Color>
{
  public:
    fr_Colors () {}
    fr_Colors (const fr_Color& c) { push_back(c); }
    fr_Colors (const fr_Color& c1, const fr_Color& c2) { push_back(c1); push_back(c2); }
    fr_Colors& operator+=(const fr_Color& c) { push_back(c); return *this; }
};

class fr_Image
{
 private:
  GdkPixbuf *pixbuf;
 public:
  fr_Image(fr_StockItem si);
  fr_Image(FILE*fptr, size_t filesize=0);
  virtual ~fr_Image();
  GtkWidget*		MakeGtkWidget() const;
  int getWidth() const;
  int getHeight() const;
 friend class fr_Element;
 friend class fr_CustomButton;
 friend class fr_DataTable;
 friend class fr_Window;
};

class fr_Element
{
  friend class fr_Image;
  friend class fr_Box;
  friend class fr_File;
  friend class fr_Window;
  friend class fr_Event;

 protected:
  fr_Element		*Parent;
  const char		*Name;
  GtkWidget		*Element, *GUI, *Window;
  std::list<fr_Listener*>*Listeners;
  void			CreateTooltip(const std::string& tip);
  fr_Element();
 public:
  fr_Element(fr_Element*parent, const char*name="");
  fr_Element(fr_Element*parent, GtkWidget*custom);
  fr_Element(fr_Element*parent, fr_Image& I);
  virtual		~fr_Element() {}
  GtkWidget*		getGUI();
  virtual void		SetName(const char*name);
  virtual const char*	GetName() { return Name; }
  virtual void		SetTooltip(const std::string& tip);
  virtual void		SetVisibility(bool s);
  virtual void		SetEditable(bool s);
  virtual void		SetSize(int x, int y);
  virtual fr_Size       getSize() const;
  virtual void		GrabFocus();
  virtual void		Contain(fr_Element*E);
  virtual void		AddLabel(const char*label);
  virtual void		AddLabel(fr_Label*label);
  virtual void		AddListener(fr_Listener*L);
  virtual void		RemoveListener(fr_Listener*L);
  void			EventTrapped(fr_Event*e);
};

class fr_Listener
{
 public:
  virtual void		EventOccurred(fr_Event*e) = 0;
  static void		Click(GtkObject*, fr_Element*E);
  static void		Changed(GtkObject*, fr_Element*E);
  static void		MenuClick(GtkObject*, fr_Element*E);
};

class fr_Event
{
 public:
  fr_Element		*Element;
  int			Type;
  int			intArg;
  void			*Arg;
  fr_Event(fr_Element*E, int t, int ia=0, void*arg=0);
  bool			Is(fr_Element* E, int t=-1);
  bool			Is(fr_Element& E, int t=-1) { return Is(&E, t); }
};

class fr_Label: public fr_Element
{
 protected:
  fr_Label() {};
 public:
  fr_Label(fr_Element*parent, const char*label);
  virtual void	SetLabel(const char*label);
};

class fr_Button: public fr_Element
{
 protected:
  GtkWidget		*img;
  fr_Button() {img = 0;};
 public:
  fr_Button(fr_Element*parent, const char*label, fr_Image*I=0);
  fr_Button(fr_Element*parent, fr_Label*label);
  fr_Button(fr_Element*parent, fr_StockItem b, const std::string& label="");
  virtual void		AddListener(fr_Listener*L);
};

class fr_CustomButton: public fr_Element
{
 protected:
  GdkPixbuf             *bgpixbuf;
  int                   bg_x_offset, bg_y_offset;
  GtkWidget		*Up, *Dn, *Box;
  bool			IsPressed, MouseInside;
  static void		ButtonPressed(GtkObject*, void*, fr_CustomButton*B);
  static void		ButtonReleased(GtkObject*, void*, fr_CustomButton*B);
  static void		MouseOver(GtkObject*, void*, fr_CustomButton*B);
  static void		MouseOff(GtkObject*, void*, fr_CustomButton*B);
  void			SetState(bool Pressed);
 public:
  fr_CustomButton(fr_Element*parent, const char*name);
  virtual void          setBgImg(const fr_Image*bg=0, int bg_x=0, int bg_y=0);
  virtual void		setUpImg(const fr_Image* I);
  virtual void		setDnImg(const fr_Image* I);
  virtual void		AddListener(fr_Listener*L);
};

class fr_Separator: public fr_Element
{
 protected:
  fr_Separator();
 public:
  fr_Separator(fr_Element*parent, fr_Direction Dir);
};

class fr_Box: public fr_Element
{
  friend class fr_Element;
  friend class fr_Notebook;
 public:
  enum fr_StretchMode { Normal, Shrink, Fill, Grow };
  typedef std::set<fr_Element*> ContentList;
 protected:
  int			SizeX, SizeY, PosX, PosY, PadX, PadY;
  GtkAttachOptions	ax, ay;
  fr_StretchMode	StretchX, StretchY;
  ContentList		Contents;
  bool			Place(GtkWidget*w, int x, int y);
 public:
  fr_Box(fr_Element*parent, const char*name="");
  virtual void		SetSpaceSize(int w, int h);
  virtual void		SetGridSize(int w, int h, bool SameSize=false);
  virtual void		Frame();
  virtual void		AddBorderBevelOut();
  virtual void		AddBorderBevelIn();
  virtual void		SetPadding(int x, int y);
  virtual void		SetStretch(fr_StretchMode x, fr_StretchMode y);
  virtual void		Pack(fr_Element& E, int x1, int y1, int x2 , int y2);
  virtual void		Pack(fr_Element& E, int dx=1, int dy=1);
  virtual void		Place(fr_Element& E, int x, int y);
  virtual void		Remove(fr_Element& E);
  virtual void		RemoveAll();
};

class fr_ButtonBox: public fr_Element
{
 public:
  fr_ButtonBox(fr_Element*parent,
	       fr_Direction dir=fr_Horizontal);
  virtual void		AddButton(fr_Button& btn, bool dflt=false);
};

class fr_Option: public fr_Element
{
 protected:
  const char		*KeyStroke;
  bool			Enabled;
 public:
  fr_Option(fr_Element*parent, const char*name="");
  fr_ArgList		Args;
  virtual void		SetKeyStroke(const char*K);
  virtual void		SetTooltip(const std::string& tip);
  virtual void		SetToDefault() = 0;
  virtual void		SiftArgs(fr_ArgList& L) = 0;
  virtual void		CompileArgs(fr_ArgList& L) = 0;
  virtual void		SetEditable(bool s);
  virtual void		EnableIf(bool cond);
  virtual bool          IsEnabled() { return Enabled; }
};

class fr_Toggle: public fr_Option
{
 protected:
  bool			DefaultState;
  void			SetDefaultState(bool s);
 public:
  fr_Toggle(fr_Element*parent, const char*name="");
  ~fr_Toggle();
  fr_ArgList		NotArgs;
  virtual void		SetTooltip(const std::string& tip);
  virtual void		SiftArgs(fr_ArgList& L);
  virtual void		SetState(bool s);
  virtual bool		GetState();
  virtual void		SetToDefault();
  virtual bool		IsDefault();
  virtual const char*	GetActiveArg();
  virtual void		CompileArgs(fr_ArgList& L);
  virtual void		AddListener(fr_Listener*L);
};

class fr_Checkbox: public fr_Toggle
{
 public:
  fr_Checkbox(fr_Element*parent, const char*label, bool s);
};

class fr_ToggleInGroup: public fr_Toggle
{
  friend class fr_GroupHolder;
  friend class fr_RadioGroup;
  friend class fr_PulldownMenu;
 protected:
  fr_ToggleInGroup	*Next;
 public:
  fr_ToggleInGroup(fr_GroupHolder*parent, const char*name="");
  virtual		~fr_ToggleInGroup();
  virtual void          SetToDefault();
};

class fr_GroupHolder: public fr_Option
{
 protected:
  int			DefaultIndex, GroupSize;
  fr_ToggleInGroup	*First, *Last;
  GSList          	*Group;
 public:
  fr_GroupHolder(fr_Element*parent, const char*name="");
  virtual		~fr_GroupHolder();
  virtual int		GetGroupSize() { return GroupSize; }
  virtual void		SelectIndex(int i);
  virtual fr_ToggleInGroup*GetSelected();
  virtual int		GetIndex();
  virtual bool		IsDefault();
  virtual void		SetToDefault();
  virtual void		SiftArgs(fr_ArgList& L);
  virtual void		CompileArgs(fr_ArgList& L);
};

class fr_RadioButton: public fr_ToggleInGroup
{
  friend class		fr_RadioGroup;
 public:
  fr_RadioButton(fr_RadioGroup*parent, const char*label);
};

class fr_RadioGroup: public fr_GroupHolder
{
  friend class			fr_RadioButton;
 protected:
  void			AddItem(fr_RadioButton*NewRadio);
 public:
  fr_RadioGroup(fr_Element*parent, const char*label,
		bool box=true);
};

class fr_MenuItem: public fr_ToggleInGroup
{
  friend class fr_PulldownMenu;
 public:
  fr_MenuItem(fr_PulldownMenu*parent, const char*label);
  virtual bool		GetState();
  virtual void    	SiftArgs(fr_ArgList& L);
};

class fr_PulldownMenu : public fr_GroupHolder
{
  friend class fr_MenuItem;
 protected:
  GtkWidget		*Menu;
  void			AddItem(fr_MenuItem*NewItem);
 public:
  fr_PulldownMenu(fr_Element*parent, const char*label);
  virtual void		SelectIndex(int i);
  virtual void        	SelectItem(fr_ToggleInGroup*I);
  virtual void        	SetToDefault();
  virtual void		AddListener(fr_Listener*L);
};

class fr_Adjustment: public fr_Option
{
 protected:
  GtkObject		*Adj;
  float		DefaultValue, Step, Minimum, Maximum;
  int			DecimalPlaces;
  char			Format[32];
  char			Printed[32];
  fr_Adjustment(fr_Element*parent, const char*name,
		float DV, float min, float max, int DP);
 public:
  void			SetValue(float v);
  float		GetValue();
  const char*		GetFormattedValue();
  void			ResetDefault(float DV);
  void			SetToDefault();
  bool			IsDefault();
  virtual void		SetTooltip(const std::string& tip);
  virtual void		SiftArgs(fr_ArgList& L);
  virtual void		CompileArgs(fr_ArgList& L);
  virtual void		AddListener(fr_Listener*L);
};

class fr_Slider: public fr_Adjustment
{
 public:
  fr_Slider(fr_Element*parent, const char*label, fr_Direction,
	    float DV, float min, float max, int DP,
	    bool ShowDigits);
};

class fr_NumEntry: public fr_Adjustment
{
 public:
  fr_NumEntry(fr_Element*parent, const char*label,
	      float DV, float min, float max, int DP);
};

class fr_Text: public fr_Option
{
 protected:
  std::string		DefaultText;
  bool			CaseSensitive;
 public:
  fr_Text(fr_Element*parent, const char*label,
	  const std::string& DefaultStr, bool Case=false,
	  int MaxLen=-1);
  void			SetText(const std::string& t);
  const std::string		GetText();
  void			SelectText();
  void			SetToDefault();
  bool			IsDefault();
  void			SiftArgs(fr_ArgList& L);
  void			CompileArgs(fr_ArgList& L);
  void			AddListener(fr_Listener*L);
};

class fr_DataTable: public fr_Element
{
 protected:
  bool			Single;
  int			Rows, Columns, SortColumn;
  typedef std::pair<void*,void*> ImageData;
  typedef std::map<const fr_Image*,ImageData*> ImageCollection;
  ImageCollection       imgcache;
  static void		EventRowSelected(GtkWidget*W, int row, int col,
					 void*event, fr_DataTable*D);
  static void		EventRowUnselected(GtkWidget*W, int row, int col,
					   void*, fr_DataTable*D);
  static void		EventColSelected(GtkWidget*W, int col, fr_DataTable*D);
 public:
  fr_DataTable(fr_Element*parent, int columns, bool sngl=true);
  virtual		~fr_DataTable();
  enum Alignment	{ Left, Right, Center };
  void			SetHeaders(const char**Headers);
  void			SetSortColumn(int col);
  void			SetSortOnInsert(bool s=true, bool ascend=true);
  int			AddRow(const char**RowData, int row=-1);
  void			RemoveRow(int row);
  void			AssociateData(int row, void*data);
  void*			AssociatedData(int row);
  void			Deselect(int row);
  void			DeselectAll();
  void			Select(int row);
  const std::string	GetCell(int row, int col);
  const fr_Image*	GetCellPic(int row, int col);
  void			SetCell(int row, int col, const std::string& txt,
				const fr_Image*I=0, bool showboth=false);
  void			AllowReorder(bool s=true);
  void			SetRowHeight(int h);
  void			SetColumnWidth(int col, int w=-1);
  void			SetColumnAlign(int col, Alignment a);
  void			Freeze(bool f=true);
  int			CountRows() { return Rows; }
  int			CountSelectedRows();
  int			GetSelected(int nth=0);
  bool			IsRowSelected(int row);
  bool			IsMultipleMode() { return !Single; }
  void			ShowRow(int row);
  void			RemoveAll();
  void			ReplaceRow(const char**RowData, int row);
  void			SetSingle(bool s);
  void			Sort(bool ascending=true);
  void			AddListener(fr_Listener*L);
};

class fr_TextArea: public fr_Box
{
 protected:
  GtkWidget *text;
  GtkTextTag *fgtag, *bgtag;
  typedef std::map<std::string,GtkTextTag*> TextTagCache;
  TextTagCache tagcache;
  int char_width, char_height;
 public:
  fr_TextArea(fr_Element*parent, const char*name="");
  virtual void SetEditable(bool e);
  virtual void setSizeForText(int cols, int rows);
  virtual void clear();
  virtual void setDefaultColors(const fr_Colors& c);
  virtual void print(const std::string& t);
  virtual void print(int i);
  virtual void println(const std::string& t);
  virtual void setColor(const fr_Colors& c);
  fr_TextArea& operator<<(const std::string& t) { print(t); return *this; }
  fr_TextArea& operator<<(const fr_Colors& c) { setColor(c); return *this; }
  fr_TextArea& operator<<(int i) { print(i); return *this; }
};

class fr_File: public fr_Option, protected fr_Listener
{
 protected:
  std::string		BrowseStr, DefaultPath;
  char			*DefaultEnv;
  fr_Button		BrowseButton;
  fr_Button		OK, Cancel;
  GtkWidget		*FileSelect;
  int			DirOnly;
  virtual void		EventOccurred(fr_Event*e);
  virtual void		FileOK();
  virtual void		FileCanceled();
 public:
  fr_File(fr_Element*parent, const char*title,
	  const char*ENVvar, const std::string& Path,
	  bool dironly=false);
  virtual		~fr_File();
  virtual bool        	IsDefault() const;
  virtual void		SetToDefault();
  virtual void		SetFileName(std::string filename);
  virtual std::string	GetFileName() const;
  virtual std::string	GetDefaultPath() const;
  virtual void          SetTooltip(const std::string& tip);
  virtual void		FilePopup();
  virtual void		SiftArgs(fr_ArgList& L);
  virtual void		CompileArgs(fr_ArgList& L);
};

class fr_Notebook: public fr_Box
{
 private:
  fr_Notepage           *current_page;
  static void           notebook_page_changed(GtkWidget*gnb,GtkWidget*,
                                              unsigned int pn, fr_Notebook*b);
 public:
  fr_Notebook(fr_Element*parent, const char*name="");
  virtual void		AddPage(fr_Notepage& P);
  bool                  isCurrent(const fr_Notepage*p) const { return p==current_page; }
};

class fr_Notepage: public fr_Box
{
  friend class fr_Notebook;
 protected:
  bool                  icon_visible, label_visible;
  GtkWidget		*TabContainer, *tab_icon, *tab_label;
  virtual void          pageChanged(bool to_me);
  bool                  hasIcon()  const { return tab_icon  != NULL; }
  bool                  hasLabel() const { return tab_label != NULL; }
 public:
  fr_Notepage(fr_Notebook*parent, const char*TabLabel=0);
  virtual		~fr_Notepage();
  virtual void		setIcon(fr_Image*I);
  virtual void          setLabelVisibility(bool v);
  virtual void          setIconVisibility(bool v);
  bool                  isCurrentPage() const { return ((fr_Notebook*)Parent)->isCurrent(this); }
};

typedef struct fr_MenuBarItem
{
  const char		*Label;
  fr_Element		*Element;
  fr_Image		*Icon;
} fr_MenuBarItem;

class fr_Window: public fr_Box
{
 protected:
  int			pos_x, pos_y;
  GtkWidget		*Vbox, *MenuBar;
  GdkPixmap             *backdrop;
  void			WindowInit();
  static void		EventDestroy(GtkObject*, fr_Window*W);
  static bool		EventDelete(GtkObject*, GdkEvent*, fr_Window*W);
  virtual void		MenuEvent(fr_Element*) {}
 public:
  fr_Window(fr_Element*parent, const char*Title);
  virtual    		~fr_Window();
  virtual void		SetGridSize(int w, int h, bool SameSize=false);
  virtual void		SetSpaceSize(int w, int h);
  virtual void          setResizable(bool r);
  virtual void		SetName(const char*name);
  virtual void		setIcon(const fr_Image*I=0);
  virtual void          setBackdrop(const fr_Image*I=0);
  virtual void		AddMenu(const char*label, fr_MenuBarItem Items[]);
  virtual bool		isVisible() const;
  virtual void		SetVisibility(bool s);
  virtual void		SaveWindowPos();
  virtual void		RestoreWindowPos();
  virtual void		AddListener(fr_Listener*L);
  virtual void		Mesg(const std::string& msg, bool err=false);
};

class fr_MainProgram: public fr_Window
{
 protected:
  std::string		executable;
  fr_ArgList            commandlineargs;
  int                   exitcode;
  fr_MainProgram(const char*name, const fr_ArgList& args);
  virtual void		setExecutable(const std::string& e) { executable=e; }
 public:
  virtual std::string	getExecutable() { return executable; }
  virtual void		SetToDefaults() {};
  virtual void		SiftArgs(fr_ArgList&) = 0;
  virtual void		CompileArgs(fr_ArgList&) = 0;
  virtual int		run();
  virtual void		exit(int exitcode=0);
};

/* ############################# Functions ################################ */
void			fr_Mesg(const std::string& Message, bool err=false);
bool	                fr_DirExists(const std::string& Dir);
bool	                fr_FileExists(const std::string& Filename);
bool	                fr_Exists(const std::string& Path);
std::string		fr_HomeDir();
std::string		fr_RealPath(const std::string& p);
std::string		fr_Dirname(const std::string& p);
std::string		fr_Basename(const std::string& p);
std::string             fr_FileExt(const std::string& p, bool all=false);
std::string		fr_EscapeArg(const std::string& a);
std::string		fr_UrlEncode(const std::string& s);
std::string		fr_UrlEncode(const strmap& sm);
void                    fr_UrlDecode(const std::string& s, strmap& sm);
void			fr_Flush();
std::string             fr_syserr();

void fr_init(const char*AppName, int argc, char*argv[], fr_ArgList&args);
int fr_exec(fr_ArgList& ExecArgs);
#endif
