/*
 * robvis.cpp
 * 
 * Copyright (c) 2000-2005 by Florian Fischer (florianfischer@gmx.de)
 * and Martin Trautmann (martintrautmann@gmx.de) 
 * 
 * This file may be distributed and/or modified under the terms of the 
 * GNU General Public License version 2 as published by the Free Software 
 * Foundation. 
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
 * WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */

// Implements the visualization SimSupervisors and tournament handlers

#include "robvis.h"
#include "robwxvis.h"
#include "robwxfe.h"
#include "robwxstruct.h"
#include "robsound.h"

#include "robuntrans.h"
#include "robtrans.h"

using namespace lrt;

namespace rt {

// mini helpie function: checks if we have a frame to do something for
bool HaveVisFrame()
{
	return !RobVisFrame::guiInfo.isDestroyed; 
}

// helpey function: waits until the event was really processes
// ALWAYS use this one!
void PostAndWait(wxEvent& evt, int maxWait)
{
	if(!HaveVisFrame())
		return; 

	static int numProcessors = wxThread::GetCPUCount();

	int numEventsSent = (++RobVisFrame::guiInfo.numEventsSent);

	// continue processing while handling non-critical events 
	// this is much more efficient
	// and we don't have to fear GUI unreactiveness any more
	// because we only handle RoboTourEvents in OnIdle. 
	RoboTourEvent* evtAsRT = wxDynamicCast(&evt, RoboTourEvent);
	bool uncritical = (evtAsRT && (evtAsRT->retData == 0));
	
	// has lots of real time elapsed since we sent the first event in the 
	// evtHandler's queue? 
	bool realTimeElapsed = false; 
	// ensure that the visFrame doesn't close just now... 
	RobVisFrame::visLocker.Enter(); 
	if(HaveVisFrame()){
		// check how old the oldest event in the queue is
		RobEventHandler* evtHandler = RobVisFrame::theVisFrame->GetRobEventHandler(); 
		lrt::Time oldestTime = evtHandler->GetLastTime(); 
		lrt::Time currentTime = lrt::Time::getCurrentTime(); 
		if((currentTime - oldestTime).toInt() > 200 /* 200 msec is a lot of real time */) {
			/*
			lrt::String str = lrt::String("[TimeElapsed!: lastTime=") + oldestTime.toInt()
				+ " currentTime=" + currentTime.toInt() + "]"; 
			wxLogDebug("%s\n", str.cStr()); 
			*/
			realTimeElapsed = true; 
		}
		
		evtHandler->AddPendingEvent(evt); 
	}
	RobVisFrame::visLocker.Leave(); 

	// we may be able to continue processing
	if(uncritical && (maxWait >= 0) && !realTimeElapsed) 
		numEventsSent -= 100; 


	int msWait = 0; 
	while(HaveVisFrame() && (RobVisFrame::guiInfo.restart != gtourExit)  
	  && (RobVisFrame::guiInfo.numEvents < numEventsSent) && ((maxWait < 0) || (msWait++ < maxWait)))
		wxThread::Sleep(1); // not yet finished
}


FpInterpretePos VisPlugin::getInterpretePos()
{
	return fpInterPosBefore; 
}

String VisPlugin::getCodeName()
{
	return "VisPlugin";
}

String VisPlugin::getName()
{
	return getTranslation("Robot Visualization Engine");
}

String VisPlugin::getHelpText()
{
	return getTranslation("  -vis       Enable built-in graphical visualization during simulation");
}
bool VisPlugin::interpreteParams(const Array<String>& params, Array<bool>& used)
{
	if(dynamic_cast<WxFrontend*>(parent))
		active = true; 

	for(int i = 0; i < params.length(); i++)
		if(params[i] == "-vis") 
		{
			used[i] = true; 
			active = true; 
		}
	return true; 
}

void VisPlugin::fillOptions(SimOptions& options)
{
	if(active) {
		VisDebugSupervisor* debugsup = new VisDebugSupervisor(parent);
		VisSupervisor* vsup = new VisSupervisor(parent, debugsup); 
		options.supervisors += vsup;
		options.supervisors += debugsup;
		options.tourDisps += new VisTourDisplayer(vsup);

		RoboTourEvent evt(EVT_RT_SHOW_TYPE);
		PostAndWait(evt);
	}
}

VisPlugin::VisPlugin(Frontend* parent) : FrontendPlugin(parent)
{
}

VisPlugin::~VisPlugin()
{
}


VisSupervisor::VisSupervisor(Frontend* parent, VisDebugSupervisor* debugger) 
	: parent(parent), initInfo(0,0), debugger(debugger), watch(new wxStopWatch)
{
}

VisSupervisor::~VisSupervisor()
{
	delete watch;
}

unsigned long VisSupervisor::getStepCycles()
{
	// observe the special case with 'Exit': if abortPressed and the vis frame is closed already 
	// we still have to process it
	if(!HaveVisFrame() && !RobVisFrame::guiInfo.abortPressed) 
		return 100000; // Frame closed => nothing to do

	if(RobVisFrame::guiInfo.slowMode)
		return 1;
	else
		return RobVisFrame::guiInfo.speed;
}

void VisSupervisor::initSim(Simulation* const curSim)
{
	curSimNum++;

	if(!HaveVisFrame())
		return;

	while(HaveVisFrame() && !RobVisFrame::guiInfo.startLocked && !RobVisFrame::guiInfo.startPressed && !RobVisFrame::guiInfo.abortPressed)
		processGuiRequests(curSim);
	RobVisFrame::guiInfo.startPressed = false; 

	RoboTourEvent evt(EVT_RT_INITSIM_TYPE);
	evt.LoadSimData(curSim, curSimNum);
	evt.LoadTourData(curSim, tourStatus, initInfo);
	PostAndWait(evt);
}

SimSupervisor::GameState VisSupervisor::exec(Simulation* const curSim)
{
	GameState state; 
	state.affectedProgram = 0;
	state.event = gameNothing; 

	if(RobVisFrame::guiInfo.abortPressed)
	{
		RobVisFrame::guiInfo.abortPressed = false;
		state.event = gameTie;
		return state;
	}

	if(!HaveVisFrame())
		return state;

	bool delayFinished = true;
	do {
		processGuiRequests(curSim);
		
		// additional delay may be required because we're in slow or pause mode
		if(RobVisFrame::guiInfo.pausePressed)
			delayFinished = false;
		else if(RobVisFrame::guiInfo.slowMode && (watch->Time() < RobVisFrame::guiInfo.speed))
			delayFinished = false;
		else
			delayFinished = true;

	} while(HaveVisFrame() && !delayFinished && !RobVisFrame::guiInfo.abortPressed);
	watch->Start();

	RoboTourEvent evt(EVT_RT_STEPSIM_TYPE);
	evt.LoadSimData(curSim, curSimNum);
	evt.LoadBotData(curSim);
	evt.LoadTourData(curSim, tourStatus, initInfo);
	PostAndWait(evt);

	return state; 
}

void VisSupervisor::exitSim(Simulation* const curSim, const GameState& simResult)
{
	if(!HaveVisFrame())
		return;
	RoboTourEvent evt(EVT_RT_EXITSIM_TYPE);
	evt.LoadSimData(curSim, curSimNum);
	if(simResult.event == gameTie || simResult.event == gameAbort)
		evt.simData.winner = -2;
	else if(simResult.event == gameFinished) {
		evt.simData.winner = simResult.affectedProgram;
		evt.simData.winnerName = curSim->getProgram(simResult.affectedProgram)->headers["name"].value;
	}
	PostAndWait(evt);
}

void VisSupervisor::updateTour(const Array<TourResult>& tourStatus)
{
	this->tourStatus.clear();
	this->tourStatus += tourStatus; 
}

void VisSupervisor::processGuiRequests(Simulation* sim)
{
	if(RobVisFrame::guiInfo.reqPlayerInfo >= 0)
	{
		// create player info 
		RoboTourEvent evt(EVT_RT_PLAYER_TYPE);
		evt.LoadPlayerData(RobVisFrame::guiInfo.reqPlayerInfo, sim, tourStatus);
		PostAndWait(evt);
		RobVisFrame::guiInfo.reqPlayerInfo = -1; 
	}
	// other GuiInfo requests here...
	else // no request pending
	{
		wxThread::Sleep(1); // so that time passes by
	}

	// GUI requests
	Request* req = 0; 
	while(req = RobVisFrame::guiInfo.reqDebug.removeFirst())
	{
		if(dynamic_cast<DebugRequest*>(req) && debugger)
			debugger->processGuiRequest(sim, (DebugRequest*) req); 

		if(dynamic_cast<SoundRequest*>(req)) {
			processSoundRequest((SoundRequest*) req); 
		}

		// request has been processed (hopefully) 
		delete req; 
	}
}

void VisSupervisor::processSoundRequest(SoundRequest* req) 
{
	SoundPlugin* plug = dynamic_cast<SoundPlugin*>(parent->findPlugin("SoundPlugin")); 
	if(plug) {
		if(!plug->canPlay()) 
			return; 

		switch(req->type) {
			case SoundRequest::TurnOn:
				plug->setActive(true); 
				break; 
			case SoundRequest::TurnOff:
				plug->setActive(false); 
				break; 
		
			case SoundRequest::None:
			default:
				return;
		}
	}
}

//////////////////// VisTourDisplayer ///////////
VisTourDisplayer::VisTourDisplayer(VisSupervisor* super)
	: super(super)
{
}

TourStatusType VisTourDisplayer::update(const Array<TourResult>& status)
{
	super->updateTour(status);

	if(RobVisFrame::guiInfo.abortTourPressed)
	{
		RobVisFrame::guiInfo.abortPressed = false;
		RobVisFrame::guiInfo.abortTourPressed = false;
		return tourAbort; 
	}

	if(!HaveVisFrame())
		return tourContinue;

	RoboTourEvent evt(EVT_RT_STEPTOUR_TYPE);
	evt.LoadTourData(0 /* no simulation */, status, super->initInfo);
	PostAndWait(evt);

	return tourContinue; 
}



////////////////////////////// VisDebugSupervisor ////////

/*class VisDebugSupervisor : public SimSupervisor, public BotErrorHandler 
{
public:
	VisDebugSupervisor(Frontend* parent);
	~VisDebugSupervisor();
	virtual unsigned long getStepCycles();
	virtual void initSim(Simulation* const sim);
	virtual GameState exec(Simulation* const curSim);
	virtual void exitSim(Simulation* const curSim, const GameState& simResult);

private: 
	void processGuiRequests(Simulation* sim); // my own (debug) requests

	Map<int, DebugInfo> debugInfo;
	WxFrontend* parent; // can be 0 (if another frontend is being used)
};*/

VisDebugSupervisor::VisDebugSupervisor(Frontend* fparent) : nextDebugID(1)
{
	parent = dynamic_cast<WxFrontend*>(fparent);
	if(parent)
		parent->addBotErrorHandler(this);
}

VisDebugSupervisor::~VisDebugSupervisor()
{
	if(parent)
		parent->removeBotErrorHandler(this);
}

unsigned long VisDebugSupervisor::getStepCycles()
{
	if(!HaveVisFrame())
		return 100000;
	else
		return 1;
}

void VisDebugSupervisor::initSim(Simulation* const sim)
{
}

SimSupervisor::GameState VisDebugSupervisor::exec(Simulation* const curSim)
{
	int oldID = nextDebugID;
	//processGuiRequests(curSim);
	for(Map<int,DebugInfo>::Iterator iter = debugInfo.begin(); iter.hasElement();)
	{
		DebugInfo& di = iter.get().getValue();
		bool keepDI = true; // keep the DI in the map after this iteration?

		if(di.debugID < oldID) // is an old request, and was not yet sent
		{
			keepDI = false; // need to find it!
			if(!di.alive) lrt::System::exit(201, getTranslation("dead bot found in debugInfo"));
			// Simulation.find(bot);
			for(List<Bot*>::Node* n = curSim->bots.first(); n; n = n->getNext())
			{
				if(di.botID == n->accessElement()) // here it is
				{
					if(di.updateFrom(n->accessElement(), true)) // data has changed
					{
						DebugEvent evt(EVT_DEBUG_UPDATE_TYPE);
						evt.data = di;
						PostAndWait(evt);
					}
					keepDI = true;
					break;
				}
			}
		}
		if(!keepDI)
			iter.remove();
		else
			++iter;
	}

	SimSupervisor::GameState result;
	result.affectedProgram = 0; result.event = gameNothing;
	return result;
}

void VisDebugSupervisor::exitSim(Simulation* const curSim, const GameState& simResult)
{
	// debug requests from the old simulation are no longer relevant in the next one
	RobVisFrame::guiInfo.reqDebug.clearOfType(typeid(DebugRequest)); 

	for(Map<int,DebugInfo>::Iterator iter = debugInfo.begin(); iter.hasElement(); iter.remove())
	{
		DebugInfo& di = iter.get().getValue();
		di.alive = false;
		DebugEvent evt(EVT_DEBUG_UPDATE_TYPE);
		evt.data = di;
		PostAndWait(evt);
	}
}

void VisDebugSupervisor::processGuiRequest(Simulation* sim, DebugRequest* req) // my own (debug) requests
{
	if(req->debugX >= 0 && req->debugY >= 0 && req->debugX < sim->fieldWidth && req->debugY < sim->fieldHeight)
	{
		// find bot at field
		Bot* bot = sim->field[req->debugX + req->debugY * sim->fieldWidth];
		// start debugging it
		if(bot) {
			DebugInfo& di = debugInfo[nextDebugID]; // insert into map
			di.updateFrom(bot, true);
			di.debugID = nextDebugID;
			DebugEvent evt(EVT_DEBUG_UPDATE_TYPE);
			evt.data = di;
			PostAndWait(evt);
			nextDebugID++;
		}
	}

	if(req->debugCloseID >= 0)
	{
		if(debugInfo.isSet(req->debugCloseID))
		{
			debugInfo.remove(req->debugCloseID);
		}
	}
}

void VisDebugSupervisor::handleBotError(const Bot* affectedBot, ExecReturnType type)
{
	// debugInfo.find(affectedBot);
	for(Map<int,DebugInfo>::Iterator iter = debugInfo.begin(); iter.hasElement();)
	{
		if(iter.get().getValue().botID == (const void*) affectedBot)
		{
			DebugInfo& di = iter.get().getValue();
			di.alive = false;
			DebugEvent evt(EVT_DEBUG_UPDATE_TYPE);
			evt.data = di;
			PostAndWait(evt);
			iter.remove();
		}
		else
			++iter;
	}

}



} // namespace
