/*
 * robtopmgr.cpp
 * 
 * Copyright (c) 2000-2005 by Florian Fischer (florianfischer@gmx.de)
 * and Martin Trautmann (martintrautmann@gmx.de) 
 * 
 * This file may be distributed and/or modified under the terms of the 
 * GNU General Public License version 2 as published by the Free Software 
 * Foundation. 
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
 * WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */
// Implements the top mode (useful for running competitions)

#include "robtopmgr.h"

#include "robhtml.h"
#include "robutils.h"
#include "robtrans.h"

#include <rtsystem.h>
#include <rtini.h>
#include <rtfile.h>

using namespace lrt;

namespace rt {

TopManager::TopManager(bool importOld, int size, const String& competitionFolder) : 
	importOld(importOld), size(size), newFiles(0), chartsFileName("charts.xxx"),
	competitionFolder(competitionFolder)
{
	// prepend competition folder before the charts file name
	chartsFileName = File(competitionFolder, chartsFileName).getName();
}

TopManager::~TopManager() {}

// reimplement to add the old programs too and load values for them, 
// but remember how many new programs we've got
// (we'll only play new ones against old ones)
void TopManager::setup(SimOptions* const options)
{
	if(options == 0) options->errHandler->handleSystemError(36, 
		_("options may not be NULL in TopManager::setup!"));
	this->options = options;

	newFiles = options->files;

	for(int n = 0; n < newFiles.length(); n++)
		newFiles[n] = File(newFiles[n]).getLocalName(competitionFolder);

	// now we can ask about tour information
	TourInitInfo initInfo = getTourInitInfo();
	// ...and send it to the displayers...
	for(int d = 0; d < options->tourDisps.length(); d++)
	  options->tourDisps[d]->init(initInfo);

	createSupervisors();
}


void TopManager::run()
{
	for(int i = 0; i < newFiles.length(); i++)
	{
		doFights(newFiles[i]);
	}

	// make charts html
}

TourInitInfo TopManager::getTourInitInfo()
{
	// all approximations only!
	int nPlayers = size + newFiles.length();
	int nSims = newFiles.length() * size * options->numRepeats;
	return TourInitInfo(nPlayers, nSims);
}

String TopManager::getResultFileName(const String& botname)
{
	String ret;
	if(botname.length() > 0) {
		ret = botname;
		ret[ret.length() - 1] = 'x';
	}
	else
		ret = "x.x";
	return File(competitionFolder, ret).getName();
}

String TopManager::getInfoFileName(const String& botname)
{
	String ret;
	if(botname.length() > 0) {
		ret = botname;
		ret[ret.length() - 1] = 'y';
	}
	else
		ret = "y.y";
	return File(competitionFolder, ret).getName();
}

String TopManager::getId(const String& botname)
{
	String infoFile = getInfoFileName(botname);
	TopInfo info(infoFile);
	info.read();
	return info.id;
}

bool TopManager::setIfInTop(const String& botname, const String& top)
{
	NoBreakSection nbs; // writing the info file is a critical task

	String infoFile = getInfoFileName(botname);
	TopInfo info(infoFile);
	if(!info.read())
		return false;
	info.top = top;
	info.write();

	return true;
}

bool TopManager::saveRes(const String& bot1, const String& bot2, 
						 int wins, int looses, int ties)
{
	NoBreakSection nbs; // writing the results is a critical task

	String resF1 = getResultFileName(bot1); // get filenames of the result files
	String resF2 = getResultFileName(bot2);

	IniFile res1, res2; // File format is like INI files with sections and options
                        //   section : filename of the opponent
                        //   options : result of the fights and info of the opp. 

	res1.read(resF1);
	res2.read(resF2);

	IniSection& sec1 = res1.getSection(bot2); // update result file 1; bot2's section
	sec1["wins"]   = String(wins);
	sec1["looses"] = String(looses);
	sec1["ties"]   = String(ties);
	sec1["id"]     = getId(bot2);

	IniSection& sec2 = res2.getSection(bot1); // update result file 2; bot1's section
	sec2["wins"]   = String(looses); // wins = other bot's looses
	sec2["looses"] = String(wins);
	sec2["ties"]   = String(ties);
	sec2["id"]     = getId(bot1);

	// save the changes to disk
	if(!res1.write(resF1))
	{
		options->errHandler->handleWarning(_("couldn't write result file ") + resF1);
		return false;
	}
	if(!res2.write(resF2))
	{
		options->errHandler->handleWarning(_("couldn't write result file ") + resF2);
		return false;
	}

	return true;
}

TopInfo* TopManager::updateBotInfo(const String& botname)
{
	NoBreakSection nbs; // writing the info file is a critical task

	Program* prog = Program::create(File(competitionFolder, botname).getName(), options->glob, 0);
	
	if(prog == 0)
		return 0;

	String infoFile = getInfoFileName(botname); // info file name
	TopInfo* info = new TopInfo(infoFile);

	info->read();		// read the info file if it exists
	if(!importOld)		// if it is a new bot
		info->incID();	// increase the id number
	info->fetchDate();	// get the current date as Upload date

	info->headers = prog->headers;
	info->top         = "yes"; // at first the bot is in the charts

	info->write();		// save the changed data to the file

	if(!importOld)		// ignore old results
	{
		// make the result file empty
		String resName = getResultFileName(botname);
		// a dummy ini which is empty
		IniFile dummy;
		dummy.write(resName);
	}

	delete prog;
	return info; 
}

bool TopManager::doFights(const String& botname)
{
//	NoBreakSection nbs; // just for test

    // get and save some information about the bot
	TopInfo* info = updateBotInfo(botname);

	if(info == 0)
	{
		options->errHandler->handleWarning(String::format(_("Cannot open the robot file %s."), botname.cStr()));
		return false;
	}

	String resF = getResultFileName(botname); // get filenames of the result files
	IniFile resultF; // File format is like INI files with sections and options
                     //   section : filename of the opponent
                     //   options : result of the fights and info of the opp.
	resultF.read(resF);

	NoBreakSection nbsThrowOut; // protect the charts result file
		IniFile charts; // ini file which stores the charts
		charts.read(chartsFileName);
		// remove the bot's entry from the charts if it is being updated
		charts.removeSection(botname);
		charts.write(chartsFileName);
	nbsThrowOut.leave();

	IniFile::Iterator iter = charts.begin();
	while(iter.hasElement())
	{
		IniSection& curSection = iter.get().getValue();
		String otherBotName = iter.get().getKey();

		if(importOld)
		{
			// check if this fight was done previously
			if(resultF.hasSection(otherBotName))
			{
				// yes, so go to the next bot.
				++iter;
				continue;
			}

		}

		System::println(String::format(_("*** %dx %s against %s ***"), (int) options->numRepeats,
			botname.cStr(), otherBotName.cStr()));

		deleteAll(programs);
		results.clear();
		options->files.clear();
		options->files += File(competitionFolder, botname).getName();
		options->files += File(competitionFolder, otherBotName).getName();

		if(!addAndCheck(botname)) // can I load this program?
		{
			setIfInTop(botname, "no");
			return false;
		}
		if(!addAndCheck(otherBotName)) // can I load the other program?
		{ // no, so go to the next program
			setIfInTop(otherBotName, "no");
			++iter;
			continue;
		}
		//assert(programs.length() == 2);

		// run the simulations between the programs 
		Array<int> programNums(2);
		programNums[0] = 0; programNums[1] = 1;
		doAllStdSims(programNums);
		// save the simulation results
		saveRes(botname, otherBotName, results[0].wins, results[0].looses, results[0].ties);

		++iter;
	}
	// ignore current charts contents;
	charts = IniFile();

	// reread charts to get changes by other programs (e.g. kickbot)
	charts.read(chartsFileName);

	IniSection& sec = charts.getSection(botname);
	sec["programname"] = info->headers["name"].value;
	sec["author"]      = info->headers["author"].value;
	sec["country"]     = info->headers["country"].value;
	sec["date"]        = info->date.toString();

	NoBreakSection nbsChartsWrite; // protect charts ini file
		Vector<TopTourResult> store(0);
		calcCharts(charts, store);
		checkChartsLength(charts, store);
		updateCharts(charts, store);
		charts.write(chartsFileName);
	nbsChartsWrite.leave();

	delete info;

	// create charts html
	HtmlCreator html(_("Any"), competitionFolder, competitionFolder, true, true);
	html.createChartsHtml(false, chartsFileName);

	return true;
}

void TopManager::readCharts(const IniFile& charts, Vector<TopTourResult>& store)
{
	store.clear();

	IniFile::Iterator iter = charts.begin();
	while(iter.hasElement())
	{
		const IniSection& sec = iter.get().getValue();

		StringMap<Program::HeaderField> headers; 
		headers["name"] = Program::HeaderField("Name", sec["programname"], headerPublished);
		headers["author"] = Program::HeaderField("Author", sec["author"], headerPublished);
		headers["country"] = Program::HeaderField("Country", sec["country"], headerPublished);

		TopTourResult res(headers, iter.get().getKey());
		res.wins   = sec["wins"].intValue(0);
		res.looses = sec["looses"].intValue(0);
		res.ties   = sec["ties"].intValue(0);
		res.points = (float)(sec["points"].intValue(0));
		res.date = sec["date"];
		res.id = sec["id"].intValue(0);
		store += res;
		++iter;
	}

}

void TopManager::calcCharts(const IniFile& charts, Vector<TopTourResult>& store)
{
	store.clear();

	// try all bot matches in the charts
	IniFile::Iterator iter = charts.begin();
	while(iter.hasElement())
	{
		// accumulated values for this bot
		int wins = 0, looses = 0, ties = 0; 

		String botname = iter.get().getKey();
		String resname = getResultFileName(botname);
		IniFile res;
		res.read(resname);

		IniFile::Iterator opponentIter = charts.begin();
		while(opponentIter.hasElement())
		{
			// are we both at the same bot?
			if(iter.get().getKey() == opponentIter.get().getKey())
			{ // yes, so proceed to the next one
				++opponentIter;
				continue;
			}

			// different bots
			String oppName = opponentIter.get().getKey();
			if(res.hasSection(oppName))
			{
				const IniSection& s = res.getSection(oppName);
				wins   += s["wins"].intValue(0);
				looses += s["looses"].intValue(0);
				ties   += s["ties"].intValue(0);
			}
			else
				options->errHandler->handleWarning(String::format(_("%s didn't fight against %s"), 
					botname.cStr(), oppName.cStr()));
			++opponentIter;
		}

		const IniSection& botSec = iter.get().getValue();

		StringMap<Program::HeaderField> headers; 
		headers["name"] = Program::HeaderField("Name", botSec["programname"], headerPublished);
		headers["author"] = Program::HeaderField("Author", botSec["author"], headerPublished);
		headers["country"] = Program::HeaderField("Country", botSec["country"], headerPublished);

		TopTourResult result(headers, botname); // botname is the bot's filename!

		result.wins = wins;
		result.looses = looses;
		result.ties = ties;
		result.points = (float)(wins * 3 + ties);

		store += result;

		++iter;
	}

	sortCharts(store);
}

void TopManager::sortCharts(Vector<TopTourResult>& store)
{
	store.sort(TopTourResult::compareByPoints);
}

void TopManager::checkChartsLength(IniFile& charts, Vector<TopTourResult>& store)
{
	int removeCount = store.length() - size;
	if(removeCount <= 0) // nothing to do
		return;

	// remove the not-good-enough bots from the charts
	for(int i = size; i < store.length(); i++)
	{
		setIfInTop(store[i].fileName, "no"); // mark the bot as a "looser"
		charts.removeSection(store[i].fileName);
	}
	// remove them from the store
	store.remove(size, removeCount);
}

bool TopManager::updateCharts(IniFile& charts, const Array<TopTourResult>& store)
{
	for(int i = 0; i < store.length(); i++)
	{
		const TopTourResult& res = store[i];
		IniSection& sec = charts.getSection(res.fileName);
		sec["programname"] = res.headers["name"].value;
		sec["wins"]        = String(res.wins);
		sec["looses"]      = String(res.looses);
		sec["ties"]        = String(res.ties);
		sec["points"]      = String(res.points);
	}
	return true;
}



bool TopManager::addAndCheck(const String& botname)
{
	Program* newProg = Program::create(File(competitionFolder, botname).getName(), options->glob, programs.length());
	// can I load this program?
	if(!newProg || (newProg->getBanks().length() == 0)) 
	{
		options->errHandler->handleWarning(String::format(_("cannot load %s."), botname.cStr()));
		return false;
	}
	else
	{
		programs += newProg;
		results += TourResult(newProg->headers, botname);
		return true;
	}
}

///////////////////////// TopInfo /////////////////////////////

TopInfo::TopInfo(const String& fileName) : fileName(fileName),
  id("0"), date(0,0), top("yes")
{
}

TopInfo::~TopInfo()
{}

void TopInfo::fetchDate()
{
	date = Time::getCurrentTime();
}

void TopInfo::incID()
{
	id = (id.intValue(0) + 1);
}

bool TopInfo::read()
{
	IniFile info;
	if(!info.read(fileName))
		return false;
	IniSection& sec = info.getSection("info");

	headers["name"] = Program::HeaderField("Name", sec["programname"], headerPublished);
	headers["author"] = Program::HeaderField("Author", sec["author"], headerPublished);
	headers["country"] = Program::HeaderField("Country", sec["country"], headerPublished);
	id          = sec["id"];
	date        = Time(sec["date"]);
	top         = sec["top"];

	return true;
}

bool TopInfo::write()
{
	NoBreakSection nbs; // writing the info file is a critical task

	IniFile info;
	IniSection& sec = info.getSection("info"); // the section will be created automagically

	sec["programname"] = headers["name"].value;
	sec["author"]      = headers["author"].value;
	sec["country"]     = headers["country"].value;
	sec["id"]          = id;
	sec["date"]        = date.toString();
	sec["top"]         = top;

	return info.write(fileName);
}

////////////////////// TopTourResult ////////////////////////

int TopTourResult::compareByPoints(const TopTourResult& t1, const TopTourResult& t2)
{
	return int(t2.points - t1.points);
}

int TopTourResult::compareByDate(const TopTourResult& t1, const TopTourResult& t2)
{
	// debug output
//	System::println("Created " + ti1.toString() + " from " + t1.date + " - correct?");
//	System::println("Created " + ti2.toString() + " from " + t2.date + " - correct?");

	return int(t2.date.sec - t1.date.sec);
}




} // namespace
