/*
 * robsound.cpp
 * 
 * Copyright (c) 2000-2005 by Florian Fischer (florianfischer@gmx.de)
 * and Martin Trautmann (martintrautmann@gmx.de) 
 * 
 * This file may be distributed and/or modified under the terms of the 
 * GNU General Public License version 2 as published by the Free Software 
 * Foundation. 
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
 * WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */

// robsound.cpp: Implementierung der Klasse SoundPlugin.
//
//////////////////////////////////////////////////////////////////////

#include "robsound.h"
#include "robtrans.h"

#include <rtsound.h>

using namespace lrt;

namespace rt {

FpInterpretePos SoundPlugin::getInterpretePos()
{
	return fpInterPosBefore;
}

String SoundPlugin::getCodeName()
{
	return "SoundPlugin";
}

String SoundPlugin::getName()
{
	return _("Sound Plugin");
}

String SoundPlugin::getHelpText()
{
	if(!Sound::isSupported())
		return "";
	
	return _("  -sound     Enable sound output.");
}

bool SoundPlugin::canPlay(bool checkSilent)
{
	if(!Sound::isSupported())
	{
		if(!checkSilent)
			parent->handleWarning(_("Cannot play any sounds."));
		return false;
	}

	if(!isSoundInstalled("tick", checkSilent)
	|| !isSoundInstalled("win", checkSilent)
	|| !isSoundInstalled("loose", checkSilent)
	|| !isSoundInstalled("tie", checkSilent)
	|| !isSoundInstalled("end", checkSilent))
		return false;

	return true;
}

void SoundPlugin::reset()
{
	// when the tournament is reset, the old 
	// sim supervisors and tour displayers will have been deleted. 
	// Thus, our pointers are now invalid. 
	tourDisp = 0; 
	simSup = 0; 
}

bool SoundPlugin::interpreteParams(const Array<String>& params, Array<bool>& used)
{
	for(int i = 0; i < params.length(); i++)
		if(params[i].lowerCase() == "-sound") // Yippee! We may play! ;-)
		{
			setActive(true); 
			used[i] = true;
		}
	return true;
}

void SoundPlugin::fillOptions(SimOptions& options)
{
	if(!active) return;

	if(!canPlay(false)) // really play?
	{
		setActive(false); 
		return;
	}

	tourDisp = new SoundTourDisplayer(); 
	tourDisp->setActive(active); 
	simSup = new SoundSimSupervisor(); 
	simSup->setActive(active); 

	options.tourDisps += tourDisp;
	options.supervisors += simSup;
}

void SoundPlugin::setActive(bool a) 
{
	FrontendPlugin::setActive(a); 
	if(tourDisp) 
		tourDisp->setActive(a); 
	if(simSup)
		simSup->setActive(a); 
}

bool SoundPlugin::isSoundInstalled(const String& sound, bool checkSilent)
{
	bool inst = Sound::isPresent(sound);
	if(!checkSilent && !inst)
		parent->handleWarning(_("Required sound not found: ") + sound + Sound::getExtension());
	return inst;
}


SoundPlugin::SoundPlugin(Frontend* parent) 
	: FrontendPlugin(parent), tourDisp(0), simSup(0)
{
}

//////////////////////// SoundTourDisplayer //////////////////////////////////

void SoundTourDisplayer::init(const TourInitInfo& info)
{}

TourStatusType SoundTourDisplayer::update(const Array<TourResult>& status)
{ return tourContinue; }

void SoundTourDisplayer::exit(TourStatusType exitType, const Array<TourResult>& result)
{
	if(active) 
		Sound::play("end", true);
}


///////////////////////// SoundSimSupervisor

const int SoundSimSupervisor::minMillis = 60000 / 320;	// 200 BPM
const int SoundSimSupervisor::maxMillis = 60000 / 80;		// 50 BPM


SoundSimSupervisor::SoundSimSupervisor() : active(true), cycles(1000), lastTime(Time::getCurrentTime())
{
	retState.affectedProgram = 0;
	retState.event = gameNothing;
}

SimSuperPosition SoundSimSupervisor::getPreferredPosition()
{
	return supPosDontCare;
}

unsigned long SoundSimSupervisor::getStepCycles()
{
	return cycles;
}

SimSupervisor::GameState SoundSimSupervisor::exec(Simulation* const curSim)
{
	Time curTime = Time::getCurrentTime();
	int diff = (curTime - lastTime).toInt();
	lastTime = curTime;

	if(curSim->curCycle <= 1) return retState; // do nothing at sim begin

	if((diff < minMillis) && (diff > 2))
		cycles *= 2;
	else if(diff > maxMillis)
		cycles /= 2;

	if(active) 
		Sound::play("tick", false, false);

	return retState;
}

void SoundSimSupervisor::exitSim(Simulation* const curSim, const GameState& simResult)
{
	if(!active)
		return; 


	if(simResult.event == gameFinished)
	{
		if(simResult.affectedProgram == 0)
			Sound::play("win", false);
		else
			Sound::play("loose", false);
	}
	else if(simResult.event == gameTie)
		Sound::play("tie", false);
}



} // namespace

