/*
 * robotour.cpp
 * 
 * Copyright (c) 2000-2005 by Florian Fischer (florianfischer@gmx.de)
 * and Martin Trautmann (martintrautmann@gmx.de) 
 * 
 * This file may be distributed and/or modified under the terms of the 
 * GNU General Public License version 2 as published by the Free Software 
 * Foundation. 
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
 * WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */

/** @mainpage RoboTour - The fast and portable RoboCom interpreter
  * @section desc Description  
  * RoboTour is a fast, portable, fully compatible and free RoboCom interpreter
  * and tournament engine. It can be used to run simple simulations, big tournaments
  * and even create competitions to which robots can be added later on. 
  * Graphical display of tournaments (even in 3D), robot debugging and simulation
  * history are all available. 
  */

/** @namespace rt
  * All classes in the RoboTour main program and the associated programs 
  * are inside the namespace rt. 
  */

/** @file robotour.cpp
  * Interpretes command line parameters, and holds RoboTour together.
  */

#include <rtstring.h>
#include <rtcollect.h>
#include <rtlist.h>
#include <rtnetwork.h> // for __NETWORK__ define
#include <rtfile.h>

#include "robvars.h"
#include "robglob.h"
#include "robfrontend.h"
#include "robconsole.h"
#include "robmanage.h"
#include "robloader.h"
#include "robbase.h"
#include "robpackage.h"

#ifdef __RBI__
#include "robbinary.h"
#endif

#ifdef __TELE__
#include "robtele.h"
#endif

#ifdef __WXWIN__
#include "robwxfe.h"
#endif

// Have Resources => Have Fun!
#include "resources.rh"

/** Since rtmain.h includes system includes, we may not import any namespace in
  * this file. Only in this file, really. */

enum WhichFrontend {
	UseTeleFrontend,
	UseWxFrontend,
	UseConsoleFrontend
};

WhichFrontend selectFrontend(const lrt::Array<lrt::String>& args)
{
#if defined(__NETWORK__) && defined(__TELE__)
	bool useTele = false;
	for(int at = 0; at < args.length(); at++)
		if(args[at].lowerCase() == "-tele")
			useTele = true;
	if(useTele)
		return UseTeleFrontend;
#endif // __NETWORK__ && __TELE__
#if defined(__WXWIN__)
	bool useWx = false;
	for(int aw = 0; aw < args.length(); aw++)
		if(args[aw].lowerCase() == "-vis")
			useWx = true;
	if(!args.length())
		useWx = true; 
	if(useWx)
		return UseWxFrontend;
#endif // __WXWIN__

	return UseConsoleFrontend;
}

rt::Frontend* createFrontend(WhichFrontend fe)
{
#if defined(__NETWORK__) && defined(__TELE__)
	if(fe == UseTeleFrontend)
		return new rt::TeleFrontend;
#endif
#if defined(__WXWIN__)
	if(fe == UseWxFrontend)
		return new rt::WxFrontend;
#endif
	return new rt::ConsoleFrontend;
}

int rtRun(const lrt::Array<lrt::String>& args)
{
	initAllResources();
	lrt::System::setAppName("RoboTour");
	rt::initPackage(); 

	rt::Frontend* frontend = createFrontend(selectFrontend(args));

	// create example loaders
	rt::Program::exampleLoaders += new rt::RobLoader(frontend);
#ifdef __RBI__
	rt::Program::exampleLoaders += new rt::RbiLoader(frontend);
#endif

	do{
		frontend->reset();
		if(!frontend->interpreteParams(args))
			goto rtMain_done;
		rt::SimOptions options(frontend);
		frontend->fillOptions(options);


		rt::SimManager* manager = frontend->getManager();
		manager->setup(&options);
		manager->run();
	
	}while(frontend->doAllAgain);

rtMain_done:
	delete frontend; lrt::deleteAll(rt::Program::exampleLoaders);
	return 0;
}

#ifndef __WXWIN__ // no wx-windows

// always include this file as the last one
#include <rtmain.h>

int rtMain(const lrt::Array<lrt::String>& args)
{
	return rtRun(args);
}

# else // with wx-windows

#include <wx/wx.h>
#include <wx/thread.h>
#include "robwxvis.h"
#include "robwxutil.h"
#include <rtmain.h>

const lrt::Array<lrt::String>* theArgs = 0; 

class RtWxApp : public wxApp {
public:
	int exitCode;
	virtual bool OnInit();
	virtual int OnExit();
};
DECLARE_APP(RtWxApp);

class SimThread : public wxThread {
public:
	virtual ExitCode Entry()
	{
		// Run simulations
		wxGetApp().exitCode = rtRun(*theArgs);
		// but if the frame *was* already closed, need to shut down ourselves...
		if(!RobVisFrame::theVisFrame)
		{ // we'll have to exit manually now
#ifdef __WXMSW__ // special workaround: if no windows present, ExitMainLoop won't help within MSW
			if(!wxTheApp->GetTopWindow()) 
				lrt::System::exit(wxGetApp().exitCode);
			else
#endif // MSW
			wxTheApp->ExitMainLoop(); // should quit the program
		}
		// tell vis frame that we're done
		RobVisFrame::guiInfo.haveSimThread = false; 
		// finished
		return 0; 
	}
};

wxLocale rtMyLocale; 

void rtInitTranslation()
{
	bool locOk = rtMyLocale.Init();

#ifdef EXTRA_DATA_DIR
	rtMyLocale.AddCatalogLookupPathPrefix(wxString(wxT(EXTRA_DATA_DIR)) + wxFILE_SEP_PATH + wxT("locale")); 
#endif
	// look near the executable file also
	rtMyLocale.AddCatalogLookupPathPrefix(Wx::Conv(lrt::File(lrt::File::getExecutableFile().getParent(), "locale").getName()));

	
	locOk &= rtMyLocale.AddCatalog(wxT("robotour"));
	/*if(!locOk)
		lrt::System::println("warning: Program translation for language "
		+ Wx::Conv(rtMyLocale.GetName()) + " could not be loaded. \nThe program will be in English.");
	*/
	// in some configurations, we bring out own version of the wxstd catalog
	rtMyLocale.AddCatalog(wxT("robotour-wxstd")); 
}

bool RtWxApp::OnInit()
{
	wxApp::OnInit(); 

	rtInitTranslation(); 

	exitCode = 0; 
    SetAppName(wxT("RoboTour"));
	SetVendorName(wxT("RoboComTeam"));
	SetExitOnFrameDelete(false);
	RobVisFrame::theVisFrame = new RobVisFrame();
	RobVisFrame::theVisFrame->Show();
	SetTopWindow(RobVisFrame::theVisFrame);

	wxThread* simThread = new SimThread();
	simThread->Create();
	simThread->SetPriority(0);
	simThread->Run();
    return true;
}
int RtWxApp::OnExit()
{
	return exitCode; 
}

IMPLEMENT_APP_NO_MAIN(RtWxApp)

int rtMain(const lrt::Array<lrt::String>& args)
{
	theArgs = &args; 

	if(selectFrontend(*theArgs) == UseWxFrontend)
		return Wx::Startup();
	else {
		wxChar* dummy_arg = wxT("dummy"); 
#ifdef __WXMSW__
		// There is a bug (at last I believe so) in wxMSW. 
		// It wants to free() all of the command line arguments 
		// that are passed to wxEntryStart, as well as the array itself. 
		// So we've got to copy them. 
		wxChar** dummy_argv = new wxChar*[2]; 
		dummy_argv[0] = wxStrdup(dummy_arg); dummy_argv[1] = NULL;
#else
		wxChar* dummy_argv[] = { dummy_arg, NULL };
#endif
		int dummy_argc = 1;

		bool initOK = wxEntryStart(dummy_argc, dummy_argv); 
		if(initOK) 
			rtInitTranslation(); 
		int retCode = rtRun(*theArgs);
		if(initOK)
			wxEntryCleanup(); 
		return retCode;
	}
}

#endif // WX


