/*
 * robhtml.h
 * 
 * Copyright (c) 2000-2005 by Florian Fischer (florianfischer@gmx.de)
 * and Martin Trautmann (martintrautmann@gmx.de) 
 * 
 * This file may be distributed and/or modified under the terms of the 
 * GNU General Public License version 2 as published by the Free Software 
 * Foundation. 
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
 * WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */

/** @file
  * Declares HTML generation related classes 
  */

#ifndef __RT_HTML__
#define __RT_HTML__

#include <rtstring.h>
#include <rtfile.h>

#include "robtopmgr.h"

namespace rt {

/** Creates HTML results files for top mode competitions. 
  * <b>Note:</b> Most methods return <tt>true</tt> on success, <tt>false</tt>
  * on failure (e.g. if some files were not found or could not be created). */
class HtmlCreator {

public:
	/** A suitable default for the charts ini file name (as used by TopManager).
	  * This file name is used relative to the <tt>competitionFolder</tt>. 
	  * Currently this value is "<tt>charts.xxx</tt>".
	  */
	static lrt::String defaultChartsIniName;
	/** A suitable default for the charts html file name.
	  * This file name is used relative to the <tt>outputFolder</tt>. 
	  * Currently this value is "<tt>charts.html</tt>".
	  */
	static lrt::String defaultChartsHtmlName;

	/// Escapes any critical characters in the given text to their HTML entities. 
	static lrt::String toHtml(const lrt::String& text);

	/** Creates a new HtmlCreator which reads and writes from/to the given paths,
	  * using the given options.
	  * @param competitionName The competition's name, as it should appear on the
	  *                        HTML pages, e.g. "Eternal".
	  * @param competitionFolder The folder in which the competition files (charts
	  *                          ini files, results ini files) are.
	  * @param outputFolder The folder in which the created HTML files should be
	  *                     placed. 
	  * @param createLinks Should links between the charts and the detailed pages
	  *                    be included into the pages?
	  * @param createDownloadLinks Should links to the robot files be created?
	  */
	HtmlCreator(const lrt::String& competitionName, const lrt::File& competitionFolder,
		const lrt::File& outputFolder, bool createLinks, bool createDownloadLinks);

	~HtmlCreator() {}

	/** Returns the file of the detailed html file for the given bot name. */
	lrt::File getDetailedHtml(const lrt::String& botFileName);
	
	/** Creates the main HTML page. 
	  * @param recalc If specified, rewrites the charts.cxx file at first from 
	  *               the details sources. 
	  */
	bool createChartsHtml(bool recalc = false, 
		const lrt::String& iniName = defaultChartsIniName,
		const lrt::String& htmlName = defaultChartsHtmlName);
	/** Creates a detailed HTML page for the given bot. */
	bool createDetailedHtml(const lrt::String& botFileName);


protected:
	lrt::String competitionName;
	// from last createChartsHtml() run
	lrt::String chartsHtmlName;
	lrt::File competitionFolder;
	lrt::File outputFolder;
	bool createLinks;
	bool createDownloadLinks;

	TopManager myManager;

	// Reads the Detailed Result lrt::File and all of the referenced info files for
	// the given bot into the store. The store is cleared first.
	bool readDetailed(const lrt::String& botFileName, lrt::Vector<TopTourResult>& store);
	
	static lrt::String cssName;
	bool cssExists();
	// Copies the CSS file (resource) to the output folder. 
	bool writeCss();
};



} // namespace

# endif

