/*
 * dialogs.cpp
 * 
 * Copyright (c) 2000-2005 by Florian Fischer (florianfischer@gmx.de)
 * and Martin Trautmann (martintrautmann@gmx.de) 
 * 
 * This file may be distributed and/or modified under the terms of the 
 * GNU General Public License version 2 as published by the Free Software 
 * Foundation. 
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
 * WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */

#include "dialogs.h"

#include <ctype.h>

#include <wx/config.h>
#include <wx/fileconf.h>

#include "wxfsel.h"
#include "robotop.h"

#include "bitmaps/open.xpm"

/// Notiez: wxFILE_SEP_PATH ist '\\' oder '/' je nach System
///   -"-   wxEndsWithPathSeparator() prft genau das

wxString GetMainTourFolder()
{
	wxConfigBase* globalCfg = wxConfigBase::Get();
	wxString ret = globalCfg->Read(wxT("TournamentFolder"), wxT("."));
	if(!wxEndsWithPathSeparator(ret))
		ret += wxFILE_SEP_PATH;
	return ret;
}

wxString GetProgramFileName(const wxString& progName)
{
	wxConfigBase* globalCfg = wxConfigBase::Get();
	wxString ret = globalCfg->Read(wxT("RoboTourExecutable"), wxT("robotour"));
	if(progName == wxT("robotour"))
		return ret;
	ret = ret.BeforeLast(wxFILE_SEP_PATH) + wxFILE_SEP_PATH + progName;
#ifdef __WXMSW__
	ret += wxT(".exe");
#endif
	return ret;
}

void wxDirCommand(wxArrayString& store, const wxChar* spec, int flags)
{
	store.Empty();
	wxString file = wxFindFirstFile(spec, flags);
	while ( !file.IsEmpty() )
	{
		store.Add(file);
		file = wxFindNextFile();
	}
}


BEGIN_EVENT_TABLE(GlobalPrefsDialog, wxDialog)
	EVT_BUTTON(wxID_OK, GlobalPrefsDialog::OnOK)
END_EVENT_TABLE()

GlobalPrefsDialog* GlobalPrefsDialog::instance = NULL;

GlobalPrefsDialog* GlobalPrefsDialog::Get()
{
	if(!instance)
		instance = new GlobalPrefsDialog(wxGetApp().GetTopWindow());
	return instance;
}


GlobalPrefsDialog::GlobalPrefsDialog(wxWindow* parent) : wxDialog(parent, -1, 
	_("RoboTop Preferences"), wxDefaultPosition, wxDefaultSize, 
	wxDIALOG_MODAL | wxDEFAULT_DIALOG_STYLE)
{
	wxBoxSizer* sizer = new wxBoxSizer( wxVERTICAL );
	wxConfigBase* conf = wxConfig::Get();

	wxString rtExe = conf->Read(wxT("RoboTourExecutable"), wxT("*.exe"));
	rtExeWin = new wxxFselWindow(this, -1, _("RoboTour executable:"), rtExe);
	sizer->Add(rtExeWin, 0, wxEXPAND | wxALL, 2);

	wxString tourFolder = conf->Read(wxT("TournamentFolder"), wxGetCwd());
	tourFolderWin = new wxxFselWindow(this, -1, _("Main competition folder:"), tourFolder, true);
	sizer->Add(tourFolderWin, 0, wxEXPAND | wxALL, 2);
	
#ifndef __WXMSW__
	wxString browser = conf->Read(wxT("BrowserCmd"), wxT("netscape"));
	browserWin = new wxxFselWindow(this, -1, _("Web browser command:"), browser);
	browserWin->SetTextReadOnly(false);
	sizer->Add(browserWin, 0, wxEXPAND | wxALL, 2);
#endif

	bool useSound; conf->Read(wxT("UseSound"), &useSound, false);
	soundWin = new wxCheckBox(this, -1, _("Play sound while simulating"));
	soundWin->SetValue(useSound);
	sizer->Add(soundWin, 0, wxEXPAND | wxALL, 2);

	wxGridSizer* bsizer = new wxGridSizer(3);

	wxButton* okB = new wxButton(this, wxID_OK, _("OK"), wxDefaultPosition);
	wxButton* cancelB = new wxButton(this, wxID_CANCEL, _("Cancel"), wxDefaultPosition);

	bsizer->Add(20, 20, 1, wxEXPAND | wxALL, 2);
	bsizer->Add(cancelB, 1, wxEXPAND | wxALL, 2);
	bsizer->Add(okB, 1, wxEXPAND | wxALL, 2);

	sizer->Add(bsizer, 0, wxEXPAND | wxALL, 2);

	SetSizer( sizer );      // actually set the sizer
	sizer->SetSizeHints( this );   // set size hints to honour mininum size

/******** these should be unnecessary *************/
/**/	SetAutoLayout( TRUE );     // tell dialog to use sizer
/**/	sizer->Fit( this );            // set size to minimum size as calculated by the sizer

	SetDefaultItem(okB);
}

GlobalPrefsDialog::~GlobalPrefsDialog()
{
}

bool GlobalPrefsDialog::Go()
{
	ShowModal();
	return true;
}

void GlobalPrefsDialog::OnOK(wxCommandEvent& event)
{
//	wxMessageBox(wxT("Clicked OK"));
	wxConfigBase* conf = wxConfig::Get();
	conf->Write(wxT("RoboTourExecutable"), rtExeWin->GetFile());
	conf->Write(wxT("TournamentFolder"), tourFolderWin->GetFile());
#ifndef __WXMSW__
	conf->Write(wxT("BrowserCmd"), browserWin->GetFile());
#endif
	conf->Write(wxT("UseSound"), soundWin->GetValue());
	conf->Flush();
	wxDialog::OnOK(event); // call super class after we're done
}

///////////////////////////////////////////
// Competition configuration data
TourData::TourData() : tourName(_("New Competition")), tourFile(_("NewCompetition.top")),
		tourFolder(_("NewCompetition")), optionSet(wxT("robocom.rco")), chartSize(20), numRepeats(10)
{
}

TourData::TourData(const wxString& filename) : tourName(), tourFile(filename), tourFolder(),
		optionSet(), chartSize(20), numRepeats(10)
{
	if(!Read())
		wxLogError(_("Couldn't read competition file in TourData::TourData(wxString&)!"));
}

TourData::~TourData() 
{
}

bool TourData::Read()
{
	wxString mainTourFolder = GetMainTourFolder();
	wxConfigBase* cfg = new wxFileConfig(wxT("RoboTop"), wxT("RoboComTeam"), mainTourFolder + tourFile);
	tourName = cfg->Read(wxT("TourName"), tourName);
	tourFolder = cfg->Read(wxT("TourFolder"), tourFolder);
	optionSet = cfg->Read(wxT("OptionSet"), optionSet);
	chartSize = cfg->Read(wxT("ChartSize"), chartSize);
	numRepeats = cfg->Read(wxT("NumRepeats"), numRepeats);
	delete cfg;
	return true;
}

bool TourData::Save()
{
	if(!tourFile)
	{
		wxLogError(_("Attempt to write to unset tourFile in TourData::Save()"));
		return false;
	}
	wxString mainTourFolder = GetMainTourFolder();
	wxConfigBase* cfg = new wxFileConfig(wxT("RoboTop"), wxT("RoboComTeam"), mainTourFolder + tourFile);
	cfg->Write(wxT("TourName"), tourName);
	cfg->Write(wxT("TourFolder"), tourFolder);
	cfg->Write(wxT("OptionSet"), optionSet);
	cfg->Write(wxT("ChartSize"), (long)chartSize);
	cfg->Write(wxT("NumRepeats"), (long)numRepeats);
	bool ret = cfg->Flush();
	delete cfg;
	return ret;
}

bool TourData::Create()
{
	bool ret = true;
	wxString mainTourFolder = GetMainTourFolder();
	wxString myTourFolder = mainTourFolder + tourFolder + wxFILE_SEP_PATH;
	ret &= wxMkdir(mainTourFolder + tourFolder);
	if(!ret) {
		wxLogError(_("Could not create my competition folder %s."), myTourFolder.c_str());
		return false;
	}
	ret &= wxMkdir(myTourFolder + wxT("newbots"));
	ret &= wxMkdir(myTourFolder + wxT("curbots"));
	ret &= wxMkdir(myTourFolder + wxT("results"));
	ret &= wxMkdir(myTourFolder + wxT("html"));
	if(!ret) {
		wxLogError(_("Could not create some of the subfolders."));
		return false;
	}
	return true;
}

/*	wxString tourName;
	wxString tourFile;
	wxString tourFolder;
	wxString optionSet;
	int chartSize;
*/


///////////////////// /////////////////// /////////////////////

BEGIN_EVENT_TABLE(TourPrefsDialog, wxDialog)
	EVT_BUTTON(wxID_OK, TourPrefsDialog::OnOK)
	EVT_BUTTON(tpTourFile, TourPrefsDialog::OnTourFile)
	EVT_BUTTON(tpOptionSet, TourPrefsDialog::OnOptionSet)
// it's also called in the constructor, which is bad
//	EVT_TEXT(tpTourName, TourPrefsDialog::OnTourName)
END_EVENT_TABLE()


TourPrefsDialog::TourPrefsDialog(wxWindow* parent, TourPrefsMode mode, TourData* data) :
    wxDialog(parent, -1, _("Competition Preferences"), wxDefaultPosition, wxDefaultSize, 
	wxDIALOG_MODAL | wxRESIZE_BORDER | wxDEFAULT_DIALOG_STYLE), data(data), mode(mode)
{
	wxBitmap* openBitmap = new wxBitmap( open_xpm );

		
	wxBoxSizer* sizer = new wxBoxSizer( wxVERTICAL );
	wxFlexGridSizer* tsizer = new wxFlexGridSizer(2, 2, 2);

	// Competition name
	tsizer->Add(new wxStaticText(this, -1, _("Competition name:"), wxDefaultPosition), 1, wxALIGN_CENTER_VERTICAL | wxALL, 2);
	tsizer->Add(tourNameWin = new wxTextCtrl(this, tpTourName, data->tourName, wxDefaultPosition, wxSize(250, 22)), 0, wxEXPAND | wxALL, 2);

	// Settings file name
	tsizer->Add(new wxStaticText(this, -1, _("Settings file name:"), wxDefaultPosition), 1, wxALIGN_CENTER_VERTICAL | wxALL, 2);
	if(mode == tourPrefsModeChange) {
		tsizer->Add(tourFileWin = new wxStaticText(this, -1, data->tourFile, wxDefaultPosition), 0, wxALIGN_CENTER_VERTICAL | wxALL, 2);
	}
	else if(mode == tourPrefsModeNew) {
		wxBoxSizer* tfilesizer = new wxBoxSizer(wxHORIZONTAL);
		tfilesizer->Add(tourFileWin = new wxStaticText(this, -1, data->tourFile, wxDefaultPosition), 1, wxALIGN_CENTER_VERTICAL, 0);
		tfilesizer->Add(new wxBitmapButton(this, tpTourFile, *openBitmap), 0, wxEXPAND, 0);
		tsizer->Add(tfilesizer, 0, wxEXPAND | wxALL, 2);
	}
	else
		wxLogError(_("There's no such TourPrefsMode."));

	// Competition folder
	tsizer->Add(new wxStaticText(this, -1, _("Competition folder:"), wxDefaultPosition), 1, wxALIGN_CENTER_VERTICAL | wxALL, 2);
	if(mode == tourPrefsModeChange) {
		wxBoxSizer* tfsizer = new wxBoxSizer(wxHORIZONTAL);
		tfsizer->Add(new wxStaticText(this, -1, GetMainTourFolder(), wxDefaultPosition), 0, wxALIGN_CENTER_VERTICAL, 0);
		tfsizer->Add(tourFolderWin = new wxStaticText(this, -1, data->tourFolder, wxDefaultPosition), 1, wxALIGN_CENTER_VERTICAL, 0);
		tsizer->Add(tfsizer, 0, wxEXPAND | wxALL, 2);
	}
	else if(mode == tourPrefsModeNew) {
		wxBoxSizer* tfoldersizer = new wxBoxSizer(wxHORIZONTAL);
		tfoldersizer->Add(new wxStaticText(this, -1, GetMainTourFolder(), wxDefaultPosition), 0, wxALIGN_CENTER_VERTICAL, 0);
		tfoldersizer->Add(tourFolderWin = new wxTextCtrl(this, -1, data->tourFolder, wxDefaultPosition, wxSize(150, 22)), 1, wxEXPAND, 0);
		tsizer->Add(tfoldersizer, 0, wxEXPAND | wxALL, 2);
	}
	else
		wxLogError(_("There's no such TourPrefsMode."));

	// Option set file
	tsizer->Add(new wxStaticText(this, -1, _("Option set file:"), wxDefaultPosition), 1, wxALIGN_BOTTOM | wxALL, 2);
	wxBoxSizer* ofilesizer = new wxBoxSizer(wxHORIZONTAL);
	ofilesizer->Add(optionSetWin = new wxStaticText(this, -1, data->optionSet, wxDefaultPosition), 1, wxALIGN_CENTER_VERTICAL, 0);
	ofilesizer->Add(new wxBitmapButton(this, tpOptionSet, *openBitmap), 0, wxEXPAND, 0);
	tsizer->Add(ofilesizer, 0, wxEXPAND | wxALL, 2);

	// Chart size
	tsizer->Add(new wxStaticText(this, -1, _("Chart size:"), wxDefaultPosition), 1, wxALIGN_BOTTOM | wxALL, 2);
	tsizer->Add(chartSizeWin = new wxSpinCtrl(this, -1, wxString::Format(wxT("%d"), data->chartSize), wxDefaultPosition,
		wxDefaultSize, wxSP_ARROW_KEYS, 5, 1000, data->chartSize), 0, wxALL, 2);

	// Repeat count
	tsizer->Add(new wxStaticText(this, -1, _("Repeat count:"), wxDefaultPosition), 1, wxALIGN_BOTTOM | wxALL, 2);
	tsizer->Add(numRepeatsWin = new wxSpinCtrl(this, -1, wxString::Format(wxT("%d"), data->numRepeats), wxDefaultPosition,
		wxDefaultSize, wxSP_ARROW_KEYS, 1, 100, data->numRepeats), 0, wxALL, 2);

	// Buttons
	wxGridSizer* bsizer = new wxGridSizer(3);

	wxButton* okB = new wxButton(this, wxID_OK, _("OK"), wxDefaultPosition);
	wxButton* cancelB = new wxButton(this, wxID_CANCEL, _("Cancel"), wxDefaultPosition);

	bsizer->Add(20, 20, 1, wxEXPAND | wxALL, 2);
	bsizer->Add(cancelB, 1, wxEXPAND | wxALL, 2);
	bsizer->Add(okB, 1, wxEXPAND | wxALL, 2);

	sizer->Add(tsizer, 0, wxEXPAND | wxALL, 2);
	sizer->Add(bsizer, 0, wxEXPAND | wxALL, 2);

	SetSizer( sizer );      // actually set the sizer
	sizer->SetSizeHints( this );   // set size hints to honour mininum size

/******** these should be unnecessary *************/
/**/	SetAutoLayout( TRUE );     // tell dialog to use sizer
/**/	sizer->Fit( this );            // set size to minimum size as calculated by the sizer

	SetDefaultItem(okB);

	Connect(tpTourName, wxEVT_COMMAND_TEXT_UPDATED, (wxObjectEventFunction) &TourPrefsDialog::OnTourName);
	
	delete openBitmap;
}

TourPrefsDialog::~TourPrefsDialog()
{
}

bool TourPrefsDialog::Go()
{
	return (ShowModal() == wxID_OK);
}

void TourPrefsDialog::OnOK(wxCommandEvent& event)
{
	data->tourName = tourNameWin->GetValue();
	if(mode == tourPrefsModeNew)
	{
 		data->tourFile = tourFileWin->GetLabel();
		data->tourFolder = ((wxTextCtrl*) tourFolderWin)->GetValue();
	}
	data->optionSet = optionSetWin->GetLabel();
	data->chartSize = chartSizeWin->GetValue();
	data->numRepeats = numRepeatsWin->GetValue();
//	wxMessageBox(wxString::Format(wxT("got: tourName [%s]\ntourFile [%s]\ntourFolder [%s]\noptionSet [%s]"), data->tourName.c_str(), data->tourFile.c_str(), data->tourFolder.c_str(), data->optionSet.c_str()));
	data->Save();
	wxDialog::OnOK(event); // call super class after we're done
}

void TourPrefsDialog::OnTourFile(wxCommandEvent& event)
{
	wxString nTourFile = wxFileSelector(_("Save competition file"), GetMainTourFolder(), 
		tourFileWin->GetLabel(), wxT(".top"), _("Competition files (*.top)|*.top"), wxSAVE, this);
	if(!nTourFile.IsEmpty()) { // not cancelled
		nTourFile = nTourFile.AfterLast(wxFILE_SEP_PATH);
	    data->tourFile = nTourFile;
	    tourFileWin->SetLabel(nTourFile);
	}
}

void TourPrefsDialog::OnOptionSet(wxCommandEvent& event)
{
	wxString path, name, ext;
	wxSplitPath(optionSetWin->GetLabel(), &path, &name, &ext);

	wxString nOptionSet = wxFileSelector(_("Select option set"), path, 
		name + ext, wxT(".rco"), _("Option sets (*.rco)|*.rco"), wxOPEN, this);
	if(!nOptionSet.IsEmpty()) { // not cancelled
		data->optionSet = nOptionSet;
		optionSetWin->SetLabel(nOptionSet);
	}
}

void TourPrefsDialog::OnTourName(wxCommandEvent& event)
{
	if(mode != tourPrefsModeNew)
		return;
	wxString newVal = tourNameWin->GetValue();
	if(newVal.IsEmpty())
		return;
	// remove spaces from newVal
	for(size_t pos = 0; pos < newVal.Length(); pos++)
	{
		wxChar c = newVal.GetChar(pos);
		if(isalpha(c) || isdigit(c) || (c == '_') || (c == '.'))
			continue;
		else {
			newVal.Remove(pos, 1);
			pos--;
		}
	}

	data->tourFolder = newVal;
	tourFolderWin->SetLabel(newVal);
    data->tourFile = newVal + wxT(".top");
	tourFileWin->SetLabel(newVal + wxT(".top"));	
}
