/*
 * rtnetwork.cpp
 * 
 * Copyright (c) 2000-2005 by Florian Fischer (florianfischer@gmx.de)
 * and Martin Trautmann (martintrautmann@gmx.de) 
 * 
 * This file may be distributed and/or modified under the terms of the 
 * GNU General Public License version 2 as published by the Free Software 
 * Foundation. 
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
 * WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */

#include "rtnetwork.h"
#ifdef __NETWORK__ // actually implement networking?

#include "rtsystem.h"
#include "rtiterator.h"
#include "rtmap.h"
#include "rtchar.h"

#include "sockfunc.h"

namespace lrt {

Socket* Network::create(const String& server, int port)
{
	SFSocket* sk = ::Sopen_client((char*) server.cStr(), port);
	if(!sk) return 0; // error
	return new Socket(sk);
}

ServerSocket* Network::createServer(int port)
{
	SFSocket* sk = ::Sopen_server(port);
	if(!sk) return 0; // error
	return new ServerSocket(sk);
}

const Time& Network::getTimeout()
{
	return timeout;
}

void Network::setTimeout(const Time& tn)
{
	timeout = tn;
}

void Network::splitURL(const String& url, String& proto, String& server, int& port, String& file)
{
	// split the url first
	// check for the protocol
	int index = url.indexOf("://");
	if(index >= 0) {
		proto = url.substring(0, index).lowerCase();
		index += 3;
	}
	else {
		proto = "http";
		index = 0;
	}
	// server <=> file separator
	int index2 = url.indexOf('/', index+3); 
	if(index2 < 0) // no file given
		index2 = url.length();
	server = url.substring(index, index2);
	// file
	file = url.substring(index2); // including the '/'! http wants that
	if(file.length() == 0) file = "/";
	
	/// extract port from server
	index = server.lastIndexOf(':');
	if(index > 0) {
		port = server.substring(index + 1).intValue(80);
		server = server.substring(0, index);
	}
	else
		port = 80;
}

// Translation method <=> name
char* methodTable[] = { "GET", "POST", "HEAD" };

/// Performs a HTTP request 
bool Network::httpRequest(const String& url, HttpMethod method, Page& page)
{
	String proto, server, file;
	int port;
	splitURL(url, proto, server, port, file);
	if(proto != "http") { page.code = 1089; page.message = "Unsupported protocol"; return false; }
	
	return httpRequest(server, port, method, file, page);
}

bool Network::httpRequest(const String& server, int port, HttpMethod method, const String& location, Page& page)
{
	if(method == POST) {
		if(!page.headers.isSet("Content-Type"))
			page.headers.put("Content-Type", "text/plain");
		page.headers.put("Content-Length", String(page.text.length()));
	}
	/* send things such as: 

     GET /some-path/index.html HTTP/1.0
	 User-Agent: libRT 0.1
	 Host: server:port
	 More Headers

    */
	// open connection
	Socket* sock = create(server, port);
	if(!sock) { page.code = 1099; page.message = "Cannot connect to host"; return false; }
	NetworkInputStream* in = sock->getInput();
	NetworkOutputStream* out = sock->getOutput();

	// build request
	String sMethod = methodTable[method];
	String endl = "\x0D\x0A";
	String sLocation = location;
	if(!sLocation.startsWith("/")) // there must always be a slash
		sLocation = "/" + sLocation;
	String request = sMethod + " " + sLocation + " HTTP/1.0" + endl;
	// now process headers 
	if(!page.headers.isSet("User-Agent"))
		page.headers["User-Agent"] = "libRT/0.1";
	if(!page.headers.isSet("Host"))
		page.headers["Host"] = server + ":" + port; 

	for(Iterator<Pair<String,String> > headers = page.headers.begin(); headers.hasElement(); ++headers)
	{
		Pair<String, String>& header = headers.get();
		request += header.getKey() + ": " + header.getValue() + endl;
	}
	// headers done
	request += endl;

	// for POST operation, add the content too
	if(method == POST) {
		request += page.text;
		page.text.clear(); // clear the text for the answer
	}

	// send it
	if(!out->write(request)) {
		page.code = 1098; 
		page.message = "Cannot send data to host";
		return false;
	}

	// process the answer; 
	ParseInputStream pin(in, " ");
	pin.detach();
	// read the answer first, should be 'HTTP/1.x CODE MESSAGE'
	String answer = pin.getLine(false);
	Array<String> aParts = answer.split(" ", "", 3);
	if(aParts.length() < 2 || !aParts[0].startsWith("HTTP/")) { // too few parts or HTTP/ missing
		page.code = 1001;
		page.message = "Bad Response (HTTP/x.x missing)";
	}
	else { // ok
		page.code = aParts[1].intValue(1002); // get return code
		if(page.code == 1002) // conversion to integer failed
			page.message = "Bad Response (return code missing)";
		else // ok, take the server's message
			page.message = aParts[2];
	}

	// read headers now
	page.headers.clear();
	while(true) {
		String line = pin.getLine(false);
		line = line.trim();
		if(line.length() == 0) break; // header finished
		// process header
		Array<String> parts = line.split(":", "", 2);
		if(parts.length() < 2) continue; // what's that?
		page.headers[parts[0]] = parts[1];
	}

	// read page text
	page.text.clear();
	Array<char> buf(512);
	int count = 0;
	while((count = in->read(buf, 0, 512)) > 0) // still some data
		for(int i = 0; i < count; i++)
			page.text += buf[i];

	delete sock;
	return true;
}

String Network::urlEncode(const StringMap<String>& data)
{
	String temp;
	for(Iterator< Pair<String, String> > it = data.begin(); it.hasElement(); ++it) {
		temp += urlEncode(it.get().getKey());
		temp += '=';
		temp += urlEncode(it.get().getValue());
		temp += '&';
	}
	// we have added one & too many
	temp.remove(temp.length() - 1);
	return temp;
}

String Network::urlEncode(const String& str)
{
	String ret;
	for(int i = 0; i < str.length(); i++) {
		char c = str[i];
		if((Char::isLetter(c) || Char::isDigit(c) || (c == '/') || (c == '.')) && (((unsigned char)c) < 128))
			ret += c;
		else {
			ret += '%'; // percent sign
			ret += String(((int)(unsigned char)c), 16, 2); // append hex number
			if(ret[1] == ' ') ret[1] = '0'; // fix up potential space character from hex conversion
		}
	}
	
	return ret;
}

Time Network::timeout(60, 0);

//////////////////////////////////////// Socket //////////////////////////////
Socket::~Socket()
{
	close();
	delete in;
	in = 0;
	delete out;
	out = 0;
}

NetworkInputStream* Socket::getInput()
{
	if(!sock)
		return 0;
	if(in)
		return in;
	else return (in = new NetworkInputStream(this));
}

NetworkOutputStream* Socket::getOutput()
{
	if(!sock)
		return 0;
	if(out)
		return out;
	else return (out = new NetworkOutputStream(this));
}

String Socket::getPeerName()
{
	char* peer = ::Speername((SFSocket*) sock);
	return String(peer);
}

void Socket::close()
{
	if(sock)
		::Sclose((SFSocket*) sock);
	sock = 0;
}

bool Socket::isClosed()
{
	return (sock == 0);
}

Socket::Socket(void* sock) : sock(sock), in(0), out(0)
{
}

/////////////////////////////////// ServerSocket /////////////////////////////
ServerSocket::~ServerSocket() {}

Socket* ServerSocket::accept()
{
	if(!sock)
		return 0;
	SFSocket* client = ::Saccept((SFSocket*) sock);
	return new lrt::Socket(client);
}


ServerSocket::ServerSocket(void* sock) : Socket(sock)
{
}

////////////////////////////// NetworkInputStream ///////////////////////////
NetworkInputStream::~NetworkInputStream()
{
	sock->in = 0;
}

int NetworkInputStream::read()
{
	SFSocket* sk = (SFSocket*) sock->sock;
	if(!sk) { _fail = true; return -1; } // socket is closed

	// process timeout
	const Time& timeout = Network::getTimeout();
	if(timeout.sec || timeout.msec) { // timeout enabled
		int ret = ::Stimeoutwait(sk, timeout.sec, timeout.msec*1000);
		if(ret <= 0) { // error, timeout or no more data
			_fail = true;
			return -1; 
		}
		// else continue
	}

	unsigned char c;
	int numRead = ::Sread(sk, &c, 1);
	if(numRead <= 0) { _fail = true; return -1; } // reading failed
	return c;
}

int NetworkInputStream::read(Array<char> &b, int off, int len)
{
	SFSocket* sk = (SFSocket*) sock->sock;
	if(!sk) { _fail = true; return -1; } // socket is closed

	// check array access
	if(b.length() < off + len) // doesn't seem to fit, can't do direct read
		return InputStream::read(b, off, len);
	char* buf = (char*) b.getData(); // create data pointer
	buf += off; // add offset

	// process timeout
	const Time& timeout = Network::getTimeout();
	if(timeout.sec || timeout.msec) { // timeout enabled
		int ret = ::Stimeoutwait(sk, timeout.sec, timeout.msec*1000);
		if(ret <= 0) { // error, timeout or no more data
			_fail = true; 
			return -1; 
		}
		// else continue
	}

	// actually read
	int numRead = ::Sread(sk, buf, len);
	if(numRead <= 0) { _fail = true; return -1; } // reading failed
	return numRead;
}

bool NetworkInputStream::eos()
{
	if(sock->isClosed())
		return true;
	int test = ::Stest((SFSocket*) sock->sock);
	return (test < 0 || test == EOF);
}
// does nothing
void NetworkInputStream::close() {}
NetworkInputStream::NetworkInputStream(Socket* sock) : sock(sock) {}

////////////////////////////// NetworkOutputStream ///////////////////////////
NetworkOutputStream::~NetworkOutputStream()
{
	sock->out = 0;
}

bool NetworkOutputStream::write(int b)
{
	SFSocket* sk = (SFSocket*) sock->sock;
	if(!sk) return false; // socket is closed

	unsigned char c = b;
	int ret = ::Swrite(sk, &c, 1);
	if(ret < 1) { _fail = true; return false; } // writing failed
	return true;
}

bool NetworkOutputStream::write(const Array<char> &b, int off, int len)
{
	SFSocket* sk = (SFSocket*) sock->sock;
	if(!sk) return false; // socket is closed

	const char* data = b.getData();
	data += off;
	int ret = ::Swrite(sk, (void*) data, len);
	if(ret < len) { _fail = true; return false; } // writing failed
	return true; 
}

bool NetworkOutputStream::write(const Array<char> &b) { return write(b, 0, b.length()); }
// does nothing
void NetworkOutputStream::close() {}
NetworkOutputStream::NetworkOutputStream(Socket* sock) : sock(sock) {}


} // namespace
#endif // __NETWORK__
