/*
 * rtini.h
 * 
 * Copyright (c) 2000-2005 by Florian Fischer (florianfischer@gmx.de)
 * and Martin Trautmann (martintrautmann@gmx.de) 
 * 
 * This file may be distributed and/or modified under the terms of the 
 * GNU General Public License version 2 as published by the Free Software 
 * Foundation. 
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
 * WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */

/** \file
  * Contains support for the INI file format, a text-mode configuration
  * file format.
  * @see IniFile
  * @see IniSection
  */

#ifndef __LRT_INI__
#define __LRT_INI__

#include "rtmap.h"

namespace lrt {

/** An IniSection is a subsection of an INI file. It contains 
  * several key-value string mappings. Currently, it is just implemented
  * by a linear map of Strings. 
  * @see StringMap 
  */
class IniSection : public StringMap<String> {
public:
	/** Creates an empty IniSection. Ensures that it is not case sensitive. */
	IniSection() : StringMap<String>(false) {}

	/** Copies all data from the given IniSection. */
	IniSection(const IniSection& is) : StringMap<String>(is) {}

	virtual ~IniSection() {}
};

/** IniFile implements an interface to a configuration file in INI format.
  * (This format is a well known text-mode configuration file format from 
  * the Windows world.) While the format of INI files varies
  * with different Windows programs, for our purposes, its structure is
  * as follows:
  * -	Each INI file consists of any number of sections. 
  * -	Every section is addressed by its name (a String), and itself contains
  * 	any number of INI entries.
  * -	Every INI entry is just a key-value pair of <tt>String</tt>s.
  */
class IniFile {
public:
	typedef StringMap<IniSection>::Iterator Iterator;

	/** Creates an empty IniFile. It will not contain any sections nor entries. */
	IniFile();
	/** Creates an IniFile and fills it with sections and entries from an INI 
	  * file on your disk. 
	  * @param filename The INI file to read. */
	IniFile(const String& filename);

	/** Checks if a section with the given name is present in this IniFile. */
	bool hasSection(const String& name) const;
	
	/** Gets an IniSection by its name from this IniFile.
	  * If the section does not yet exist, an empty section will be created and returned. 
	  * @param name The section's name.
	  * @return A reference to the section through which its contents can be modified.
	  */ 
	IniSection& getSection(const String& name);

	/** Gets an IniSection by its name from this IniFile. 
	  * If the section does not yet exist, some nonsense value is returned; use hasSection()
	  * beforewards to check. */
	const IniSection& getSectionConst(const String& name) const;
	
	/** Adds an IniSection to this IniFile.
	  * The section's contents will be copied over to the IniFile.
	  * If a section with the same name was already present in this IniFile,
	  * its contents will be discarded.
	  */
	void addSection(const String& name, const IniSection& section);

	/** Removes the given section (by its name), if it is present.
	  * @return <tt>True</tt> if there was actually a section to remove, <tt>false</tt> otherwise.
	  */
	bool removeSection(const String& name);

	/** Returns an iterator to the begin of the IniFile. You can use it to 
	  * traverse all IniSections that are present in it. 
	  */
	Iterator begin();

	/** Returns an unmodifyable iterator to the begin of the IniFile. */
	Iterator begin() const;

	/** Reads an INI file from your disk and inserts its contents into this IniFile.
	  * @return <tt>True</tt> if reading succeeded.
	  */
	bool read(const String& filename);

	/** Writes the contents of this IniFile to your disk.
	  * @param filename The file to write to.
	  * @return <tt>True</tt> if writing succeeded.
	  */
	bool write(const String& filename);

private:
	StringMap<IniSection> data;
};

} // namespace

#endif

