# This file is part of qVamps.
#
# qVamps is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# qVamps is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with qVamps; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA


use strict;
use warnings;


package MenuWriter;
use Qt;
use Qt::isa qw (Qt::Object);
use Qt::attributes qw (mtable_items menu_preview menu_xml fh
		       menu_bgnd menu_base menu_hilited menu_selected);
use Qt::constants qw (IO_WriteOnly);
use QVamps qw (read_setting replace error tr);
use File::Path;


# MenuWriter (mtable_items, menu_preview, base_path, parent, name)
sub NEW
{
  my $this         = shift;
  my $mtable_items = shift;
  my $menu_preview = shift;
  my $base_path    = shift;

  $this -> SUPER::NEW (@_);

  mkpath ("$base_path/menu");

  my $fh;
  my $menu_xml      = "$base_path/menu/spumux.xml";
  my $tv_standard   = read_setting ("/Menu/Preferences/tv_standard");
  my $mpv           = "$base_path/menu/video.mpv";
  my $menu_bgnd     = read_setting (sprintf "/Menu/Preferences/ppm2mpv_%s",
				    $tv_standard eq "NTSC" ? "ntsc" : "pal");
  $menu_bgnd        = replace ($menu_bgnd, { "o" => $mpv });
  my $menu_base     = "$base_path/menu/base.png";
  my $menu_hilited  = "$base_path/menu/hilited.png";
  my $menu_selected = "$base_path/menu/selected.png";

  unless (open $fh, ">", $menu_xml)
  {
    error (sprintf "%s: %s: %s",
	   &tr ("Failed creating menu XML file"), $menu_xml, $!);

    return;
  }

  mtable_items  = $mtable_items;
  menu_preview  = $menu_preview;
  menu_xml      = $menu_xml;
  menu_bgnd     = $menu_bgnd;
  menu_base     = $menu_base;
  menu_hilited  = $menu_hilited;
  menu_selected = $menu_selected;
  fh            = $fh;
}


sub DESTROY
{
  this -> close ();
  parent () -> removeChild (this);

  menu_selected = undef;
  menu_hilited  = undef;
  menu_base     = undef;
  menu_bgnd     = undef;
  menu_xml      = undef;
  menu_preview  = undef;
  mtable_items  = undef;

  SUPER -> DESTROY ();
}


sub close
{
  return -1 unless (fh);

  unless (CORE::close &fh)
  {
    error (sprintf "%s: %s: %s",
	   &tr ("Failed writing menu XML file"), menu_xml, $!);

    return -1;
  }

  fh = undef;

  return 0;
}


sub write_xml
{
  return -1 unless (fh);

  subpictures_begin_tag ();
  stream_begin_tag ();
  spu_begin_tag ();

  my $button_geos = menu_preview -> button_geometries ();

  for (my $i = 0; $i < @{$button_geos}; $i++)
  {
    my $rect = $button_geos -> [$i];

    button_tag (mtable_items -> label ($i),
		$rect -> left (), $rect -> top (),
		$rect -> right () + 1, $rect -> bottom () + 1);
  }

  spu_end_tag ();
  stream_end_tag ();
  subpictures_end_tag ();

  return 0;
}


sub subpictures_begin_tag
{
  print {&fh} "<subpictures>\n";
}


sub subpictures_end_tag
{
  print {&fh} "</subpictures>\n";
}


sub stream_begin_tag
{
  print {&fh} "  <stream>\n";
}


sub stream_end_tag
{
  print {&fh} "  </stream>\n";
}


sub spu_begin_tag
{
  print  {&fh} "    <spu start=\"00:00:00.00\" end=\"00:00:00.00\"\n";
  printf {&fh} "         image=\"%s\"\n", menu_base;
  printf {&fh} "         highlight=\"%s\"\n", menu_hilited;
  printf {&fh} "         select=\"%s\"\n", menu_selected;
  print  {&fh} "         force=\"yes\">\n";
#  print  {&fh} "         autoorder=\"columns\">\n";
}


sub spu_end_tag
{
  print {&fh} "    </spu>\n";
}


sub button_tag
{
  my $name = shift;
  my $x0   = shift;
  my $y0   = shift;
  my $x1   = shift;
  my $y1   = shift;

  printf {&fh} "      <button name=\"%s\" x0=\"%d\" y0=\"%d\" " .
    "x1=\"%d\" y1=\"%d\"/>\n", $name, $x0, $y0, $x1, $y1;
}


sub write_images
{
  return write_background () ||
         write_base ()       ||
	 write_hilited ()    ||
	 write_selected ();
}


sub write_background
{
  my $bg_filename = parent () -> background_filename ();

  # do not encode the video if an MPEG is supplied as background
  return 0 if ($bg_filename =~ /\.mpe?g$/);

  my $buf;
  my $qbuf       = Qt::Buffer ($buf);
  my $background = menu_preview -> background_image ();

  $qbuf -> open (IO_WriteOnly);
  $background -> save ($qbuf, "PPM");

  # this prevents us from being killed if encoding fails for some reason
  local $SIG {PIPE} = "IGNORE";

  my $cmd = menu_bgnd;
  my $rc  = open FH, "|-", $cmd;
  $rc   &&= print FH $buf;
  $rc   &&= CORE::close FH;

  unless ($rc)
  {
    error (sprintf "%s: %s: %s", &tr ("Failed encoding menu video"), $cmd, $!);

    return -1;
  }

  return 0;
}


sub write_base
{
  unless (menu_preview -> base_pixmap () -> save (menu_base, "PNG"))
  {
    error (sprintf "%s: %s: %s",
	   &tr ("Failed writing menu base image"), menu_base, $!);

    return -1;
  }

  return 0;
}


sub write_hilited
{
  unless (menu_preview -> hilited_pixmap () -> save (menu_hilited, "PNG"))
  {
    error (sprintf "%s: %s: %s",
	   &tr ("Failed writing menu highlighted image"), menu_hilited, $!);

    return -1;
  }

  return 0;
}


sub write_selected
{
  unless (menu_preview -> selected_pixmap () -> save (menu_selected, "PNG"))
  {
    error (sprintf "%s: %s: %s",
	   &tr ("Failed writing menu selected image"), menu_selected, $!);

    return -1;
  }

  return 0;
}


1;
