/*
**  prefs.c -- Handles our preferences
**
**	This file is subject to the terms and conditions of the GNU General Public
**	License.  See the file COPYING in the main directory of this archive
**	for more details.
**
*/

/* Note that there's not a lot of error checking done on the new prefs read/write
 * code. If the prefs don't get read or written because of an error, there's
 * probably a very good reason outside the scope of Penguin itself, and it's no
 * big loss if the prefs get mangled. - Tony Mantler <tonym@mac.linux-m68k.org> */

#include <string.h> // JDJ
#include <Folders.h>
#include <Resources.h>
#include <Finder.h>
#include <Script.h>
#include <Gestalt.h>
#include <Aliases.h>
#include <NumberFormatting.h>
#include <Files.h>

#include "penguin_prototypes.h"
#include "penguin_utils.h"
#include "prefs.h"
#include "prefs_prototypes.h"
#include "prefs_dialog.h"

/* this shouldn't be defined here, but oh well... */
#define RESID_ALERT				129

/* globals */
FSSpec		app_spec;


/*
 *	dupe_handle
 *
 *	Duplicate a fake handle
 */
OSErr dupe_handle (Handle *hand) {
	Handle dupe;
	Size size;
	
	if (!hand) return -1;
	dupe = NewHandle(size = GetHandleSize(*hand));
	if (dupe == NULL) return MemError(); // JDJ
	BlockMoveData(**hand, *dupe, size);
	*hand = dupe;
}


/*
 *	init_app_spec
 *
 *	obtain fsspec to current application
 *	used as starting point for alias derefs
 */
OSErr init_app_spec (void) {
	OSErr 				err;
#ifndef	AS_INIT
	ProcessSerialNumber	psn;
	ProcessInfoRec		procInfo;
	
	err = GetCurrentProcess(&psn);
	if (err) return err;
	procInfo.processInfoLength = sizeof procInfo;
	procInfo.processName = NULL;
	procInfo.processAppSpec = &app_spec;
	err = GetProcessInformation(&psn, &procInfo);
#else	// AS_INIT
	*app_spec.name = 0;
 	err = FindFolder( kOnSystemDisk, kExtensionFolderType, 
 						kCreateFolder, &app_spec.vRefNum, &app_spec.parID );
#endif	// AS_INIT
	return err;
}

/*
 *	blank_prefs
 *
 *	initialize "blank" prefs record -
 *	make sure kernel_alias defaults to ./vmlinux
 */
OSErr blank_prefs (penguin_config *prefs)
{
	FSSpec kernelFSS; // JDJ
	OSErr err;
	
	memset(prefs, 0, sizeof(penguin_config));
	err = FSMakeFSSpec(app_spec.vRefNum, app_spec.parID, "\pvmlinux", &kernelFSS);
	if (err == fnfErr) return noErr;
	if (err) return err;
	// if vmlinux exists in the app folder, this will find it.
	err = NewAlias(&app_spec, &kernelFSS, &(prefs->kernel_alias));
	
	return err;
}

/*
 *	write_prefs
 *
 *	write prefs to given FSSpec
 */
OSErr write_prefs (FSSpec *spec, penguin_config *prefs)
{
	FSSpec	tempFSS, destdirFSS;
	Handle	hand;
	short	rfile, dfile;
	OSErr err;
	short vol;
	long dir;
	
	// Always create new temp prefs file.
	err = FindFolder(spec->vRefNum, kTemporaryFolderType, kCreateFolder, &vol, &dir);
	if (err) return err;
	
	err = FSMakeFSSpec(vol, dir, "\pPenguin Prefs temp file", &tempFSS);
	if (err == noErr)
		err = FSpDelete(&tempFSS);
	
	FSpCreateResFile(&tempFSS, 'jim', 'jim', smSystemScript);
	
	rfile = FSpOpenResFile(&tempFSS, fsWrPerm);
	if (rfile == -1)
		return ResError();
	
	/* Write out the aliases to the Kernel, Ramdisk and Logfile */
	if (prefs->kernel_alias != NULL) {
		hand = (Handle)prefs->kernel_alias;
		HandToHand(&hand);
		if (hand == NULL) return MemError();
		AddResource(hand, rAliasType, 128, "\pKernel File Alias");
		if (ResError()) {
			DisposeHandle(hand);
			return ResError();
		}
	}
	if (prefs->ramdisk_alias != NULL) {
		hand = (Handle)prefs->ramdisk_alias;
		HandToHand(&hand);
		if (hand == NULL) return MemError();
		AddResource(hand, rAliasType, 129, "\pRamdisk File Alias");
		if (ResError()) {
			DisposeHandle(hand);
			return ResError();
		}
	}
	if (prefs->logfile_alias != NULL) {
		hand = (Handle)prefs->logfile_alias;
		HandToHand(&hand);
		if (hand == NULL) return MemError();
		AddResource(hand, rAliasType, 130, "\pLog File Alias");
		if (ResError()) {
			DisposeHandle(hand);
			return ResError();
		}
	}
	
	/* Do the old style prefs */
	hand = NewHandleClear(sizeof(penguin_prefs));
	if (hand == NULL)
		return MemError();
	
	/* Fill in the old prefs fields */
	((penguin_prefs *)(*hand))->version 				= 0x0100;
	BlockMoveData(prefs->command_line, ((penguin_prefs *)(*hand))->command_line, sizeof(cmd_ln));
	/* do the bitfields manually, since they move around now */
	((penguin_prefs *)(*hand))->do_ramdisk			= prefs->do_ramdisk;
	((penguin_prefs *)(*hand))->do_logfile			= prefs->do_logfile;
	((penguin_prefs *)(*hand))->logfile_append		= prefs->logfile_append;
	((penguin_prefs *)(*hand))->debug_boot_info		= prefs->debug_boot_info;
	((penguin_prefs *)(*hand))->debug_segment_info	= prefs->debug_segment_info;
	((penguin_prefs *)(*hand))->debug_mach_specs	= prefs->debug_mach_specs;
	((penguin_prefs *)(*hand))->debug_copy_and_go	= prefs->debug_copy_and_go;
	((penguin_prefs *)(*hand))->auto_boot			= prefs->auto_boot;
	((penguin_prefs *)(*hand))->disable_vbl			= 0;
	((penguin_prefs *)(*hand))->no_boot				= prefs->no_boot;
	((penguin_prefs *)(*hand))->reverse_video		= prefs->color_by_penguin;
	((penguin_prefs *)(*hand))->enable_ext_cache	= 0;
	((penguin_prefs *)(*hand))->use_old_mappings	= 0;
	((penguin_prefs *)(*hand))->noforce_020_bank_a	= 0;
	((penguin_prefs *)(*hand))->boot_into_linux		= prefs->boot_into_linux;
	
	/* Write the old style prefs. wheee */
	AddResource(hand, 'jim', 128, "\pSettings");
	WriteResource(hand);
	
	/* Do the new style prefs */
	err = FSpOpenDF(&tempFSS, fsWrPerm, &dfile);
	if (err != noErr) return err;
	
	/* Write the prefs header */
	{
		long tinybuf, size = sizeof(long);
		/* First the magic number */
		tinybuf = 'jim';
		FSWrite(dfile, &size, &tinybuf);
		if (size != sizeof(long))
			return -1; /* nothing got written. that's bad. */
		/* And the version, though this will probably never need to change */
		tinybuf = PREFS_FORMAT_VERSION;
		FSWrite(dfile, &size, &tinybuf);
	}
	
	/* Fill in the prefs resource. These are not order-sensitive */
	add_ramdisk_bits_v1(dfile, prefs);
	add_logfile_bits_v1(dfile, prefs);
	add_penguin_bits_v1(dfile, prefs);
	add_serial_bits_v1(dfile, prefs);
	add_serial_prefs_v1(dfile, prefs);
	add_boot_delay_prefs(dfile, prefs);
	add_command_line(dfile, prefs);
	
	FSClose(dfile);
	
	/* Close the file */
	CloseResFile(rfile);
	if (ResError())
		return ResError();
	
	// First try to exchange old and new.
	err = FSpExchangeFiles(&tempFSS, spec);
	if (err == fnfErr) {
		// Assume the 'existing' file doesn't.
		
		// Make a spec for the destination directory itself.
		err = FSMakeFSSpec(spec->vRefNum, spec->parID, "\p", &destdirFSS);
		
		err = FSpRename(&tempFSS, spec->name);
		if (err) return err;
		
		// The saved file now has the correct name.
		err = FSMakeFSSpec(tempFSS.vRefNum, tempFSS.parID, spec->name, &tempFSS);
		
		// Move the saved file to its normal place.
		err = FSpCatMove(&tempFSS, &destdirFSS);
		if (err) return err;
	} else if (err == noErr) {
		// Flush before delete to complete any pending I/O.
		err = FlushVol(NULL, spec->vRefNum);
		if (err) return err;
	
		err = FSpDelete(&tempFSS);
		if (err) return err; // Do we really care if this fails?  Why would it, anyway?
	} else
		return err;
	
	// ALWAYS flush the volume after saving a file.
	// Also, flush after delete.  Not so important generally, but in our case
	// it's quite likely that the system will crash or hang before flushing.
	err = FlushVol(NULL, spec->vRefNum);
	if (err) return err;
	
	return noErr;
}


/*
 *	read_prefs
 *
 *	read prefs from given FSSpec
 */
OSErr read_prefs (FSSpec *spec, penguin_config *prefs)
{
	OSErr			err;
	Handle			hand;
	short			rfile, dfile;
	long			prefsSize;
	unsigned char	fName[258];
	
	/* Open the prefs file */
	rfile = FSpOpenResFile(spec, fsRdPerm);
	if (rfile == -1)
		return ResError();
	
	/* Read in the Kernel, Ramdisk and Logfile aliases */
	prefs->kernel_alias = (AliasHandle)Get1Resource(rAliasType, 128);
	DetachResource((Handle)prefs->kernel_alias);
	prefs->ramdisk_alias = (AliasHandle)Get1Resource(rAliasType, 129);
	DetachResource((Handle)prefs->ramdisk_alias);
	prefs->logfile_alias = (AliasHandle)Get1Resource(rAliasType, 130);
	DetachResource((Handle)prefs->logfile_alias);
	
	/* Open up the old format prefs */
	hand = Get1Resource('jim', 128);
	if (hand == NULL)
		return (ResError() != noErr) ? ResError() : resNotFound;
	
	BlockMoveData( ((penguin_prefs *)(*hand))->command_line, prefs->command_line, sizeof(cmd_ln));
	
	/* Check the size */
	if (GetHandleSize(hand) < sizeof(penguin_prefs)) {
		/* Handle really old prefs that aren't long enough to cover all the bitfields... I think. - TM */
		prefs->boot_into_linux		= 0;
	} else {
		/* Otherwise, everything's groovy - TM */
		prefs->boot_into_linux		= ((penguin_prefs *)(*hand))->boot_into_linux;
	}
	
	prefs->color_by_penguin		= ((penguin_prefs *)(*hand))->reverse_video;
	prefs->no_boot				= ((penguin_prefs *)(*hand))->no_boot;
	prefs->auto_boot			= ((penguin_prefs *)(*hand))->auto_boot;
	prefs->debug_copy_and_go	= ((penguin_prefs *)(*hand))->debug_copy_and_go;
	prefs->debug_mach_specs		= ((penguin_prefs *)(*hand))->debug_mach_specs;
	prefs->debug_segment_info	= ((penguin_prefs *)(*hand))->debug_segment_info;
	prefs->debug_boot_info		= ((penguin_prefs *)(*hand))->debug_boot_info;
	prefs->logfile_append		= ((penguin_prefs *)(*hand))->logfile_append;
	prefs->do_logfile			= ((penguin_prefs *)(*hand))->do_logfile;
	prefs->do_ramdisk			= ((penguin_prefs *)(*hand))->do_ramdisk;
	
	/* Dump the resource now, we're done with it */
	ReleaseResource(hand);
	
	/* Close the prefs file */
	CloseResFile(rfile);
	if (ResError())
		return ResError();
	
	/* Open the data fork to see if there's some new-format prefs to read */
	err = FSpOpenDF(spec, fsRdPerm, &dfile);
	if (err != noErr)
		return err;
	
	{ /* Check the header to make sure there's something there to read */
		long tinybuf, size = sizeof(long);
		
		/* Read the magic number */
		FSRead(dfile, &size, &tinybuf);
		if ((size != sizeof(long)) || (tinybuf != 'jim'))
			goto nonewprefs; /* yes I used a goto. bite me. 8) - TM */
		
		/* Read the version */
		FSRead(dfile, &size, &tinybuf);
		if ((size != sizeof(long)) || (tinybuf != 1))
			goto nonewprefs; /* Ok, who changed the prefs format?? */
		
		SetFPos(dfile, fsFromStart, 0L);
	}
	
	get_ramdisk_bits_v1(dfile, prefs);
	get_logfile_bits_v1(dfile, prefs);
	get_penguin_bits_v1(dfile, prefs);
	get_serial_bits_v1(dfile, prefs);
	get_serial_prefs_v1(dfile, prefs);
	get_boot_delay_prefs(dfile, prefs);
	get_command_line(dfile, prefs);
	
nonewprefs:
	FSClose(dfile);
	
	if (path_from_FSSpec(spec, fName) != noErr) /* This looks remarkably ugly - TM */
		BlockMoveData(spec->name, fName, spec->name[0] + 1);
	fName[fName[0]+1] = 0;
	
	/* Hey look, we're using settings! */
	cprintf("Using '%s' settings.\n", &fName[1]);

	return noErr;
}

/*
 *	add_ramdisk_bits_v1 - called to tack on the ramdisk setting bits to the end of the file
 */
OSErr add_ramdisk_bits_v1 (short file, penguin_config *prefs) {
	OSErr			error = noErr;
	tag_header		header;
	ramdisk_bits_v1	data;
	long			size;
	
	memset(&data, 0, sizeof(data)); /* make sure to clear any unused bits */
	
	header.length	= sizeof(data);
	header.type		= RAMDISK_BITS_V1_TAG;
	size = sizeof(header);
	FSWrite(file, &size, &header);
	
	data.do_ramdisk = prefs->do_ramdisk;
	size = sizeof(data);
	FSWrite(file, &size, &data);
	
	return noErr;
}

/*
 *  get_ramdisk_bits_v1 - called to read in the ramdisk setting bits from the file
 */
OSErr get_ramdisk_bits_v1 (short file, penguin_config *prefs) {
	OSErr			error = noErr;
	tag_header		header;
	ramdisk_bits_v1	data;
	long			size;
	long			position;
	
	/* First, find our spot */
	error = find_prefs_tag(file, RAMDISK_BITS_V1_TAG, &position);
	if (error != noErr)
		return error;
	SetFPos(file, fsFromStart, position);
	
	/* Read in the header. Don't do anything with it at the moment, but oh well. */
	size = sizeof(header);
	FSRead(file, &size, &header);
	
	/* Read in the data. This is the good stuff, baby! :) */
	size = sizeof(data);
	FSRead(file, &size, &data);
	
	/* Fill out the appropriate fields */
	prefs->do_ramdisk = data.do_ramdisk;
	
	return noErr;
}

/*
 *	add_logfile_bits_v1 - called to tack on the logfile setting bits to the end of the file
 */
OSErr add_logfile_bits_v1 (short file, penguin_config *prefs) {
	OSErr			error = noErr;
	tag_header		header;
	logfile_bits_v1	data;
	long			size;
	
	memset(&data, 0, sizeof(data)); /* make sure to clear any unused bits */
	
	header.length	= sizeof(data);
	header.type		= LOGFILE_BITS_V1_TAG;
	size = sizeof(header);
	FSWrite(file, &size, &header);

	data.do_logfile			= prefs->do_logfile;
	data.logfile_append		= prefs->logfile_append;
	data.debug_boot_info	= prefs->debug_boot_info;
	data.debug_segment_info	= prefs->debug_segment_info;
	data.debug_mach_specs	= prefs->debug_mach_specs;
	data.debug_copy_and_go	= prefs->debug_copy_and_go;
	size = sizeof(data);
	FSWrite(file, &size, &data);
	
	return noErr;
}

/*
 *  get_logfile_bits_v1 - called to read in the logfile setting bits from the file
 */
OSErr get_logfile_bits_v1 (short file, penguin_config *prefs) {
	OSErr			error = noErr;
	tag_header		header;
	logfile_bits_v1	data;
	long			size;
	long			position;
	
	/* First, find our spot */
	error = find_prefs_tag(file, LOGFILE_BITS_V1_TAG, &position);
	if (error != noErr)
		return error;
	SetFPos(file, fsFromStart, position);
	
	/* Read in the header. Don't do anything with it at the moment, but oh well. */
	size = sizeof(header);
	FSRead(file, &size, &header);
	
	/* Read in the data. This is the good stuff, baby! :) */
	size = sizeof(data);
	FSRead(file, &size, &data);
	
	/* Fill out the appropriate fields */
	prefs->do_logfile			= data.do_logfile;
	prefs->logfile_append		= data.logfile_append;
	prefs->debug_boot_info		= data.debug_boot_info;
	prefs->debug_segment_info	= data.debug_segment_info;
	prefs->debug_mach_specs		= data.debug_mach_specs;
	prefs->debug_copy_and_go	= data.debug_copy_and_go;
	
	return noErr;
}

/*
 *	add_penguin_bits_v1 - called to tack on the penguin setting bits to the end of the file
 */
OSErr add_penguin_bits_v1 (short file, penguin_config *prefs) {
	OSErr			error = noErr;
	tag_header		header;
	penguin_bits_v1	data;
	long			size;
	
	memset(&data, 0, sizeof(data));
	
	header.length	= sizeof(data);
	header.type		= PENGUIN_BITS_V1_TAG;
	size = sizeof(header);
	FSWrite(file, &size, &header);
	
	data.auto_boot			= prefs->auto_boot;
	data.no_boot			= prefs->no_boot;
	data.color_by_penguin	= prefs->color_by_penguin;
	data.boot_into_linux	= prefs->boot_into_linux;
	size = sizeof(data);
	FSWrite(file, &size, &data);
	
	return noErr;
}

/*
 *  get_penguin_bits_v1 - called to read in the penguin setting bits from the file
 */
OSErr get_penguin_bits_v1 (short file, penguin_config *prefs) {
	OSErr			error = noErr;
	tag_header		header;
	penguin_bits_v1	data;
	long			size;
	long			position;
	
	/* First, find our spot */
	error = find_prefs_tag(file, PENGUIN_BITS_V1_TAG, &position);
	if (error != noErr)
		return error;
	SetFPos(file, fsFromStart, position);
	
	/* Read in the header. Don't do anything with it at the moment, but oh well. */
	size = sizeof(header);
	FSRead(file, &size, &header);
	
	/* Read in the data. This is the good stuff, baby! :) */
	size = sizeof(data);
	FSRead(file, &size, &data);
	
	/* Fill out the appropriate fields */
	prefs->auto_boot		= data.auto_boot;
	prefs->no_boot			= data.no_boot;
	prefs->color_by_penguin	= data.color_by_penguin;
	prefs->boot_into_linux	= 0;/*data.boot_into_linux;*/
	
	return noErr;
}

/*
 *	add_serial_bits_v1 - called to tack on the serial setting bits to the end of the file
 */
OSErr add_serial_bits_v1 (short file, penguin_config *prefs) {
	OSErr			error = noErr;
	tag_header		header;
	serial_bits_v1	data;
	long			size;
	
	memset(&data, 0, sizeof(data));
	
	header.length	= sizeof(data);
	header.type		= SERIAL_BITS_V1_TAG;
	size = sizeof(header);
	FSWrite(file, &size, &header);
	
	data.disable_appletalk		= prefs->disable_appletalk;
	data.do_config_modem_port	= prefs->modem_port.do_config_port;
	data.do_config_printer_port	= prefs->printer_port.do_config_port;
	size = sizeof(data);
	FSWrite(file, &size, &data);
	
	return noErr;
}

/*
 *  get_serial_bits_v1 - called to read in the serial setting bits from the file
 */
OSErr get_serial_bits_v1 (short file, penguin_config *prefs) {
	OSErr			error = noErr;
	tag_header		header;
	serial_bits_v1	data;
	long			size;
	long			position;
	
	/* First, find our spot */
	error = find_prefs_tag(file, SERIAL_BITS_V1_TAG, &position);
	if (error != noErr)
		return error;
	SetFPos(file, fsFromStart, position);
	
	/* Read in the header. Don't do anything with it at the moment, but oh well. */
	size = sizeof(header);
	FSRead(file, &size, &header);
	
	/* Read in the data. This is the good stuff, baby! :) */
	size = sizeof(data);
	FSRead(file, &size, &data);
	
	/* Fill out the appropriate fields */
	prefs->disable_appletalk			= data.disable_appletalk;
	prefs->modem_port.do_config_port	= data.do_config_modem_port;
	prefs->printer_port.do_config_port	= data.do_config_printer_port;
	
	return noErr;
}

/*
 *	add_serial_prefs_v1 - called to tack on the serial setting prefs to the end of the file
 */
OSErr add_serial_prefs_v1 (short file, penguin_config *prefs) {
	OSErr			error = noErr;
	tag_header		header;
	serial_prefs_v1	data;
	long			size;
	
	memset(&data, 0, sizeof(data)); /* make sure to clear any unused bits */
	
	header.length	= sizeof(data);
	header.type		= SERIAL_PREFS_V1_TAG;
	size = sizeof(header);
	FSWrite(file, &size, &header);
	
	data.modem_port_speed	= prefs->modem_port.port_speed;
	data.modem_data_size	= prefs->modem_port.data_size;
	data.modem_parity		= prefs->modem_port.parity;
	data.modem_stop_bits	= prefs->modem_port.stop_bits;
	
	data.printer_port_speed	= prefs->printer_port.port_speed;
	data.printer_data_size	= prefs->printer_port.data_size;
	data.printer_parity		= prefs->printer_port.parity;
	data.printer_stop_bits	= prefs->printer_port.stop_bits;
	
	size = sizeof(data);
	FSWrite(file, &size, &data);
	
	return noErr;
}

/*
 *  get_serial_prefs_v1 - called to read in the serial prefs from the file
 */
OSErr get_serial_prefs_v1 (short file, penguin_config *prefs) {
	OSErr			error = noErr;
	tag_header		header;
	serial_prefs_v1	data;
	long			size;
	long			position;
	
	/* First, find our spot */
	error = find_prefs_tag(file, SERIAL_PREFS_V1_TAG, &position);
	if (error != noErr)
		return error;
	SetFPos(file, fsFromStart, position);
	
	/* Read in the header. Don't do anything with it at the moment, but oh well. */
	size = sizeof(header);
	FSRead(file, &size, &header);
	
	/* Read in the data. This is the good stuff, baby! :) */
	size = sizeof(data);
	FSRead(file, &size, &data);
	
	/* Fill out the appropriate fields */
	prefs->modem_port.port_speed	= data.modem_port_speed;
	prefs->modem_port.data_size		= data.modem_data_size;
	prefs->modem_port.parity		= data.modem_parity;
	prefs->modem_port.stop_bits		= data.modem_stop_bits;
	
	prefs->printer_port.port_speed	= data.printer_port_speed;
	prefs->printer_port.data_size	= data.printer_data_size;
	prefs->printer_port.parity		= data.printer_parity;
	prefs->printer_port.stop_bits	= data.printer_stop_bits;
	
	return noErr;
}

/*
 *	add_boot_delay_prefs - called to tack on the serial setting prefs to the end of the file
 */
OSErr add_boot_delay_prefs (short file, penguin_config *prefs) {
	OSErr				error = noErr;
	tag_header			header;
	boot_delay_prefs	data;
	long				size;
	
	memset(&data, 0, sizeof(data)); /* make sure to clear any unused bits */
	
	header.length	= sizeof(data);
	header.type		= BOOT_DELAY_PREFS_TAG;
	size = sizeof(header);
	FSWrite(file, &size, &header);
	
	data.boot_delay_time	= prefs->delay_boot ? prefs->boot_delay_time : 0;
	
	size = sizeof(data);
	FSWrite(file, &size, &data);
	
	return noErr;
}

/*
 *  get_boot_delay_prefs - called to read in the serial prefs from the file
 */
OSErr get_boot_delay_prefs (short file, penguin_config *prefs) {
	OSErr				error = noErr;
	tag_header			header;
	boot_delay_prefs	data;
	long				size;
	long				position;
	
	/* First, find our spot */
	error = find_prefs_tag(file, BOOT_DELAY_PREFS_TAG, &position);
	if (error != noErr)
		return error;
	SetFPos(file, fsFromStart, position);
	
	/* Read in the header. Don't do anything with it at the moment, but oh well. */
	size = sizeof(header);
	FSRead(file, &size, &header);
	
	/* Read in the data. This is the good stuff, baby! :) */
	size = sizeof(data);
	FSRead(file, &size, &data);
	
	/* Fill out the appropriate fields */
	prefs->boot_delay_time	= data.boot_delay_time;
	prefs->delay_boot		= data.boot_delay_time ? 1 : 0;
	
	return noErr;
}

/*
 *	add_command_line - called to tack on the command line prefs to the end of the file
 */
OSErr add_command_line (short file, penguin_config *prefs) {
	OSErr		error = noErr;
	tag_header	header;
	long		size;
	
	header.length	= sizeof(cmd_ln);
	header.type		= COMMAND_LINE_TAG;
	size = sizeof(header);
	FSWrite(file, &size, &header);
	
	size = sizeof(cmd_ln);
	FSWrite(file, &size, prefs->command_line);
	
	return noErr;
}

/*
 *  get_command_line - called to read in the command line from the file
 */
OSErr get_command_line (short file, penguin_config *prefs) {
	OSErr			error = noErr;
	tag_header		header;
	long			size;
	long			position;
	
	/* First, find our spot */
	error = find_prefs_tag(file, COMMAND_LINE_TAG, &position);
	if (error != noErr)
		return error;
	SetFPos(file, fsFromStart, position);
	
	/* Read in the header. Don't do anything with it at the moment, but oh well. */
	size = sizeof(header);
	FSRead(file, &size, &header);
	
	/* Read in the data. This is the good stuff, baby! :) */
	size = sizeof(cmd_ln);
	FSRead(file, &size, prefs->command_line);
	
	return noErr;
}

/*
 * find_prefs_tag - called to search for the specified tag in a file
 */
OSErr find_prefs_tag (short file, long tag, long *position) {
	OSErr		error = noErr;
	tag_header	header = {0,0};
	long		size;
	
	SetFPos(file, fsFromStart, (sizeof(long) * 2)); /* skip the file header */
	
	size = sizeof(header);
	
	/* Read in the file 'till we find something or run out of file */
	do {
		error = SetFPos(file, fsFromMark, header.length);
		FSRead(file, &size, &header);
	} while ((size == sizeof(header)) && (header.type != tag) && (error == noErr));
	
	if (header.type == tag) {
		GetFPos(file, position);
		*position -= sizeof(header);
	} else {
		return -1; /* didn't find the tag. oh well */
	}
	return noErr;
}

/*
 *	init_prefs - called at start of program to read in default prefs
 */
OSErr init_prefs (penguin_config *prefs) {
	FSSpec prefsFSS; // JDJ
	OSErr	err;
	short	vol = 0;
	long	dir = 0;
	
	/* Find the prefs folder */
	err = FindFolder(kOnSystemDisk, kPreferencesFolderType, kDontCreateFolder, &vol, &dir);
	if (err) return err;
	
	/* Make an FSSpec pointing to the default prefs file */
	err = FSMakeFSSpec(vol, dir, "\pPenguin Prefs", &prefsFSS);
	if (err == fnfErr) {
		/* Not there? blank prefs. */
		err = blank_prefs(prefs);
	} else if (err == noErr) {
		/* Is there, read it in! */
		err = read_prefs(&prefsFSS, prefs);
	}
	
	return err;
}


/*
 *	save_prefs_as_default
 *
 *	save current preferences as the default ones
 */
OSErr save_prefs_as_default (penguin_config *prefs) {
	FSSpec	spec;
	OSErr	err;
	short	vol;
	long	dir;
	
	/* If we want to install the INIT, do it now */
#if 0 /* INIT not supported in Penguin-18 */
	if ( prefs->boot_into_linux )
	{
		err = install_init();
		if (err) return err;
	}	
#endif /* 0 */
	
	/* Find the prefs folder */
	err = FindFolder(kOnSystemDisk, kPreferencesFolderType, kCreateFolder, &vol, &dir);
	if (err) return err;
	
	/* Make and FSSpec pointing to the default prefs file */
	err = FSMakeFSSpec(vol, dir, "\pPenguin Prefs", &spec);
	if (err != noErr && err != fnfErr) return err;
	
	/* Write the prefs */
	err = write_prefs(&spec, prefs);
	return err;
}


/*
 *	open_prefs
 *
 *	open_prefs - prompt user to open prefs file
 */
OSErr open_prefs (penguin_config *prefs) {
	StandardFileReply	reply;
	SFTypeList			typelist = { 'jim', 0, 0, 0 };

	StandardGetFile(NULL, 1, (const unsigned long *)(&typelist), &reply);
	if (reply.sfGood) {
		if (prefs->kernel_alias) {
			DisposeHandle((Handle)prefs->kernel_alias);
			prefs->kernel_alias = nil;
		}
		if (prefs->ramdisk_alias) {
			DisposeHandle((Handle)prefs->ramdisk_alias);
			prefs->ramdisk_alias = nil;
		}
		if (prefs->logfile_alias) {
			DisposeHandle((Handle)prefs->logfile_alias);
			prefs->logfile_alias = nil;
		}
		return read_prefs(&(reply.sfFile), prefs);
	}
	return -1;
}


/*
 *	save_prefs
 *
 *	save_prefs - prompt user to save prefs file
 */
OSErr save_prefs (penguin_config *prefs) {
	StandardFileReply	reply;

	if ( prefs->boot_into_linux )
	{
		ParamText( "\pYou checked the 'Boot into Linux' option.\r",
					"\pThis will only work if you save your Preferences as default.\r",
					"\p", "\p" );
		NoteAlert( RESID_ALERT, NULL );
	}
	StandardPutFile("\pSave prefs file as:", "\pPenguin Prefs", &reply);
	if (reply.sfGood) {
		return write_prefs(&(reply.sfFile), prefs);
	}
	return -1;
}


/*
 *	do_prefs_dialog
 *
 */
int do_prefs_dialog (penguin_config *oldprefs) 
{
	DialogPtr			dialog;
	cmd_ln				command_line;
	OSErr				err;
	penguin_config		prefs;
	short				which_item, type;
	int					good, done = 0;
	short				current_tab;

	/* Copy old prefs */
	BlockMoveData(oldprefs, &prefs, sizeof(penguin_config));
	if (oldprefs->kernel_alias) {
		prefs.kernel_alias = oldprefs->kernel_alias;
		dupe_handle((Handle *)&prefs.kernel_alias);
	}
	if (oldprefs->ramdisk_alias) {
		prefs.ramdisk_alias = oldprefs->ramdisk_alias;
		dupe_handle((Handle *)&prefs.ramdisk_alias);
	}
	if (oldprefs->logfile_alias) {
		prefs.logfile_alias = oldprefs->logfile_alias;
		dupe_handle((Handle *)&prefs.logfile_alias);
	}
	
	/* Get prefs dialog */	
	dialog = GetNewDialog(PREFS_DLOG_ID, NULL, (WindowPtr) -1L);
	
	/* Notify MacOS of the OK and Cancel buttons */
	SetDialogDefaultItem(dialog, kPrefsOkButton);
	SetDialogCancelItem(dialog, kPrefsCancelButton);
	
	/* Default to the first tab, the Kernel/Ramdisk selection */
	current_tab = kPrefsOptionsTab; /* This enum is as good as any - TM */
	switch_to_optionstab(dialog, &prefs);
	
	/* Show the window to the world */
	MacShowWindow(dialog);
	
	/* Do the Modal Dialog thing */
	while (!done) {
		ModalDialog(NULL, &which_item);
		if (which_item > PREFS_NUM_ITEMS) {
			/* If it's one of the tab items, call the tab handler */
			switch (current_tab) {
				case kPrefsKernelTab:
					do_item_kerneltab(which_item, dialog, &prefs);
					break;
				case kPrefsOptionsTab:
					do_item_optionstab(which_item, dialog, &prefs);
					break;
				case kPrefsSerialTab:
					do_item_serialtab(which_item, dialog, &prefs);
					break;
			}
		} else { /* (which_item > PREFS_NUM_ITEMS) */
			/* One of the main prefs items was hit. Do something about it */
			switch (which_item) {
				/* Handle the OK and Cancel buttons */
				case kPrefsOkButton:
					good = done = 1;
					break;
				case kPrefsCancelButton:
					good = 0; done = 1;
					break;
				/* Otherwise, we're switching tabs */
				case kPrefsKernelTab:
					switch (current_tab) {
						case kPrefsKernelTab:
							/* Ignore it */
							break;
						case kPrefsOptionsTab:
							switch_from_optionstab(dialog, &prefs);
							switch_to_kerneltab(dialog, &prefs);
							break;
						case kPrefsSerialTab:
							switch_from_serialtab(dialog, &prefs);
							switch_to_kerneltab(dialog, &prefs);
							break;
					}
					current_tab = kPrefsKernelTab;
					break;
				case kPrefsOptionsTab:
					switch (current_tab) {
						case kPrefsKernelTab:
							switch_from_kerneltab(dialog, &prefs);
							switch_to_optionstab(dialog, &prefs);
							break;
						case kPrefsOptionsTab:
							/* Ignore it */
							break;
						case kPrefsSerialTab:
							switch_from_serialtab(dialog, &prefs);
							switch_to_optionstab(dialog, &prefs);
							break;
					}
					current_tab = kPrefsOptionsTab;
					break;
				case kPrefsSerialTab:
					switch (current_tab) {
						case kPrefsKernelTab:
							switch_from_kerneltab(dialog, &prefs);
							switch_to_serialtab(dialog, &prefs);
							break;
						case kPrefsOptionsTab:
							switch_from_optionstab(dialog, &prefs);
							switch_to_serialtab(dialog, &prefs);
							break;
						case kPrefsSerialTab:
							/* Ignore it */
							break;
					}
					current_tab = kPrefsSerialTab;
					break;
			}
		} /* if (which_item > PREFS_NUM_ITEMS) */
	} /* while (!done) */
	
	/* Clear out the current tab pane */
	switch (current_tab) {
		case kPrefsKernelTab:
			switch_from_kerneltab(dialog, &prefs);
			break;
		case kPrefsOptionsTab:
			switch_from_optionstab(dialog, &prefs);
			break;
		case kPrefsSerialTab:
			switch_from_serialtab(dialog, &prefs);
			break;
	}
	
	if (good) {
		/* either trash the old prefs in favor of the new ones... */
		DisposeHandle((Handle)oldprefs->kernel_alias);
		DisposeHandle((Handle)oldprefs->ramdisk_alias);
		DisposeHandle((Handle)oldprefs->logfile_alias);
		BlockMoveData(&prefs, oldprefs, sizeof(penguin_config));
	} else {
		/* ...or just destroy/forget about the new ones */
		DisposeHandle((Handle)prefs.kernel_alias);
		DisposeHandle((Handle)prefs.ramdisk_alias);
		DisposeHandle((Handle)prefs.logfile_alias);
	}
	
	/* All done! */
	DisposeDialog(dialog);
	return good;
}

/*
 *	switch_to_kerneltab
 *
 *  Overlay the Kernel pane on the dialog, and set up the fields.
 */
OSErr switch_to_kerneltab (DialogPtr dialog, penguin_config * prefs)
{
	Handle				ditl;
	ControlHandle		control;
	FSSpec				target_spec;
	DialogItemType		item_type;
	Rect				item_rect;
	Boolean				changed;
	
	/* Load the Kernel pane... */
	ditl = Get1Resource('DITL', KERNEL_DITL_ID);
	/* ... and overlay it on the dialog */
	AppendDITL(dialog, ditl, overlayDITL);
	/* That's all we need, get rid of the ditl now */
	//ReleaseResource(ditl);
	
	/* Set up the buttons to match the current prefs */
	if (prefs->kernel_alias) {
		ResolveAlias(&app_spec, prefs->kernel_alias, &target_spec, &changed);	
		GetDialogItem(dialog, kKernelKernelButton, &item_type, (Handle *)(&control), &item_rect);
		SetControlTitle(control, target_spec.name);
	}
	if (prefs->ramdisk_alias) {
		ResolveAlias(&app_spec, prefs->ramdisk_alias, &target_spec, &changed);	
		GetDialogItem(dialog, kKernelRamdiskButton, &item_type, (Handle *)(&control), &item_rect);
		SetControlTitle(control, target_spec.name);
	}
	
	/* Set up the ramdisk checkbox... */
	GetDialogItem(dialog, kKernelRamdiskCheckbox, &item_type, (Handle *)(&control), &item_rect);
	SetControlValue(control, prefs->do_ramdisk ? 1 : 0);
	
	/* ... and enable the ramdisk button accordingly */
	GetDialogItem(dialog, kKernelRamdiskButton, &item_type, (Handle *)(&control), &item_rect);
	HiliteControl(control, (prefs->do_ramdisk ? kControlNoPart : kControlInactivePart) );
	
	return noErr;
}


/*
 *	do_item_kerneltab
 *
 *  An item was clicked in the kernel pane. Do something about it.
 */
void do_item_kerneltab (short which_item, DialogPtr dialog, penguin_config * prefs)
{
	FSSpec				target_spec;
	StandardFileReply	reply;
	SFTypeList			typelist = { 'TEXT', 'BINA', '????', '****' };
	OSErr				err;
	ControlHandle		control;
	DialogItemType		item_type;
	Rect				item_rect;
	Boolean				changed;
	
	switch (which_item) {
		case kKernelKernelButton:
			if (prefs->kernel_alias) {
				/* Check the old target, and start from that directory */
				err = ResolveAlias(nil, prefs->kernel_alias, &target_spec, &changed);
				sfgetfile_dir(target_spec.vRefNum, target_spec.parID, -1, &typelist, &reply);
			} else {
				/* No old target, start from wherever */
				StandardGetFile(NULL, -1, (const unsigned long *)(&typelist), &reply);				
			}
			
			if (reply.sfGood) {
				/* We've selected a new Kernel. yay. */
				if (prefs->kernel_alias)
					DisposeHandle((Handle)prefs->kernel_alias);
				prefs->kernel_alias = nil;
				err = NewAlias(&app_spec, &reply.sfFile, &(prefs->kernel_alias));
				GetDialogItem(dialog, kKernelKernelButton, &item_type, (Handle *)(&control), &item_rect);
				SetControlTitle(control, reply.sfFile.name);
			};
			break;
		case kKernelRamdiskCheckbox:
			/* Twiddle the prefs */
			prefs->do_ramdisk = prefs->do_ramdisk ? 0 : 1;
			
			/* Change the dialog values */
			GetDialogItem(dialog, kKernelRamdiskCheckbox, &item_type, (Handle *)(&control), &item_rect);
			SetControlValue(control, prefs->do_ramdisk ? 1 : 0);
			GetDialogItem(dialog, kKernelRamdiskButton, &item_type, (Handle *)(&control), &item_rect);
			HiliteControl(control, prefs->do_ramdisk ? kControlNoPart : kControlInactivePart);
			
			/* If we have no ramdisk to point to, fall through and choose one */
			if (prefs->ramdisk_alias) break;
			
		case kKernelRamdiskButton:
			if (prefs->ramdisk_alias) {
				/* Check the old target, and start from that directory */
				err = ResolveAlias(nil, prefs->ramdisk_alias, &target_spec, &changed);
				sfgetfile_dir(target_spec.vRefNum, target_spec.parID, -1, &typelist, &reply);
			} else {
				/* No old target, start from wherever */
				StandardGetFile(NULL, -1, (const unsigned long *)(&typelist), &reply);
			}
			
			if (reply.sfGood) {
				/* We've selected a new Ramdisk. yay. */
				if (prefs->ramdisk_alias)
					DisposeHandle((Handle)prefs->ramdisk_alias);
				prefs->ramdisk_alias = nil;
				err = NewAlias(&app_spec, &reply.sfFile, &(prefs->ramdisk_alias));
				GetDialogItem(dialog, kKernelRamdiskButton, &item_type, (Handle *)(&control), &item_rect);
				SetControlTitle(control, reply.sfFile.name);
			} else if (!(prefs->ramdisk_alias)) {
				/* No file, reset everything */
				prefs->do_ramdisk = 0;
				GetDialogItem(dialog, kKernelRamdiskCheckbox, &item_type, (Handle *)(&control), &item_rect);
				SetControlValue(control, 0);
				GetDialogItem(dialog, kKernelRamdiskButton, &item_type, (Handle *)(&control), &item_rect);
				HiliteControl(control, kControlInactivePart);
			};
			break;
	}
}


/*
 *	switch_from_kerneltab
 *
 *  Store any prefs that haven't been stored already, and trim out the kernel pane.
 */
void switch_from_kerneltab (DialogPtr dialog, penguin_config * prefs)
{
	/* All the settings are already stored. Do nothing here */
	/* Trim out the kernel pane */
	ShortenDITL(dialog, KERNEL_NUM_ITEMS);	
}


/*
 *	switch_to_optionstab
 *
 *  Store any prefs that haven't been stored already, and trim out the kernel pane.
 */
OSErr switch_to_optionstab (DialogPtr dialog, penguin_config * prefs)
{
	Handle				ditl;
	ControlHandle		control;
	FSSpec				target_spec;
	long				processor;
	DialogItemType		item_type;
	Rect				item_rect;
	Boolean				changed;
	
	/* Load the Options pane... */
	ditl = Get1Resource('DITL', OPTIONS_DITL_ID);
	/* ... and overlay it on the dialog */
	AppendDITL(dialog, ditl, overlayDITL);
	/* That's all we need, get rid of the ditl now */
	//ReleaseResource(ditl);
	
	/* Set the Logfile button to match the current prefs */
	if (prefs->logfile_alias) {
		ResolveAlias(&app_spec, prefs->logfile_alias, &target_spec, &changed);
		GetDialogItem(dialog, kOptionsLogFileButton, &item_type, (Handle *)(&control), &item_rect);
		SetControlTitle(control, target_spec.name);
	}
	
	/* Copy the Commandline */
	GetDialogItem(dialog, kOptionsCommandline, &item_type, (Handle *)(&control), &item_rect);
	setdialogitemtext((Handle)control, prefs->command_line); /* C Glue */
	
	/* Set the Logfile checkbox, Append checkbox and Logfile button enable */
	GetDialogItem(dialog, kOptionsLogFileCheckbox, &item_type, (Handle *)(&control), &item_rect);
	SetControlValue(control, prefs->do_logfile ? 1 : 0);
	
	GetDialogItem(dialog, kOptionsLogFileButton, &item_type, (Handle *)(&control), &item_rect);
	HiliteControl(control, prefs->do_logfile ? kControlNoPart : kControlInactivePart);
	
	GetDialogItem(dialog, kOptionsAppendLogCheckbox, &item_type, (Handle *)(&control), &item_rect);
	SetControlValue(control, prefs->logfile_append ? 1 : 0);
	HiliteControl(control, prefs->do_logfile ? kControlNoPart : kControlInactivePart);
	
	/* Set the rest of the checkboxes */
	GetDialogItem(dialog, kOptionsAutoboot, &item_type, (Handle *)(&control), &item_rect);
	SetControlValue(control, prefs->auto_boot ? 1 : 0);
		
	GetDialogItem(dialog, kOptionsBootIntoLinux, &item_type, (Handle *)(&control), &item_rect);
	SetControlValue(control, prefs->boot_into_linux ? 1 : 0);
	/* No INIT support in Penguin-18 */
	HiliteControl(control, kControlInactivePart);
	
	GetDialogItem(dialog, kOptionsDontBoot, &item_type, (Handle *)(&control), &item_rect);
	SetControlValue(control, prefs->no_boot ? 1 : 0);
	
	GetDialogItem(dialog, kOptionsColorByPenguin, &item_type, (Handle *)(&control), &item_rect);
	SetControlValue(control, prefs->color_by_penguin ? 1 : 0);
	
	/* Set the delay stuff */
	GetDialogItem(dialog, kOptionsDelayBoot, &item_type, (Handle *)(&control), &item_rect);
	SetControlValue(control, prefs->delay_boot ? 1 : 0);
	
	{
		Str255 delay_string;
		
		GetDialogItem(dialog, kOptionsDelayAmount, &item_type, (Handle *)(&control), &item_rect);
		NumToString(prefs->boot_delay_time, delay_string);
		SetDialogItemText((Handle)control, delay_string);
	}
	
	/* Set the debugging options checkboxes */
	GetDialogItem(dialog, kOptionsDebugBootInfo, &item_type, (Handle *)(&control), &item_rect);
	SetControlValue(control, prefs->debug_boot_info ? 1 : 0);
	
	GetDialogItem(dialog, kOptionsDebugSegmentInfo, &item_type, (Handle *)(&control), &item_rect);
	SetControlValue(control, prefs->debug_segment_info ? 1 : 0);
	
	GetDialogItem(dialog, kOptionsDebugMachineSpecs, &item_type, (Handle *)(&control), &item_rect);
	SetControlValue(control, prefs->debug_mach_specs ? 1 : 0);
	
	GetDialogItem(dialog, kOptionsDebugBootstrap, &item_type, (Handle *)(&control), &item_rect);
	SetControlValue(control, prefs->debug_copy_and_go ? 1 : 0);
	
	return noErr;
}

/*
 *	do_item_optionstab
 *
 *  An item was clicked in the options pane. Do something about it.
 *  Note: the Commandline is handled automatically, and read out in switch_from_optionstab
 */
void do_item_optionstab (short which_item, DialogPtr dialog, penguin_config * prefs)
{
	FSSpec				target_spec;
	StandardFileReply	reply;
	SFTypeList			typelist = { 'TEXT', 'BINA', '????', '****' };
	ControlHandle		control;
	OSErr				err;
	DialogItemType		item_type;
	Rect				item_rect;
	Boolean				changed;
	
	switch (which_item) {
		case kOptionsLogFileCheckbox:
			/* Twiddle the prefs */
			prefs->do_logfile = prefs->do_logfile ? 0 : 1;
			
			/* Change the dialog values */
			GetDialogItem(dialog, kOptionsLogFileCheckbox, &item_type, (Handle *)(&control), &item_rect);
			SetControlValue(control, prefs->do_logfile ? 1 : 0);
			GetDialogItem(dialog, kOptionsLogFileButton, &item_type, (Handle *)(&control), &item_rect);
			HiliteControl(control, prefs->do_logfile ? kControlNoPart : kControlInactivePart);
			GetDialogItem(dialog, kOptionsAppendLogCheckbox, &item_type, (Handle *)(&control), &item_rect);
			HiliteControl(control, prefs->do_logfile ? kControlNoPart : kControlInactivePart);
			
			/* If we have no logfile to point to, fall through and choose one */
			if (prefs->logfile_alias || !prefs->do_logfile) break;
			
		case kOptionsLogFileButton:
			if (prefs->logfile_alias) {
				/* Check the old target, and start from that directory */
				err = ResolveAlias(nil, prefs->logfile_alias, &target_spec, &changed);
				sfputfile_dir(target_spec.vRefNum, target_spec.parID,
						"\pSave log file as:", target_spec.name, &reply);
			} else {
				/* No old target, start from wherever */
				StandardPutFile("\pSave log file as:", "\pPenguin.log", &reply);
			}
			
			if (reply.sfGood) {
				/* We've selected a new Logfile. yay. */
				if (prefs->logfile_alias)
					DisposeHandle((Handle)prefs->logfile_alias);
				prefs->logfile_alias = nil;
				err = NewAlias(&app_spec, &reply.sfFile, &(prefs->logfile_alias));
				if (err == fnfErr) {
					/* create temp file so we can make an alias to it */
					err = FSpCreate(&reply.sfFile, 'R*ch', 'TEXT', smSystemScript);
					err = NewAlias(&app_spec, &reply.sfFile, &(prefs->logfile_alias));
					err = FSpDelete(&reply.sfFile);
				}
				GetDialogItem(dialog, kOptionsLogFileButton, &item_type, (Handle *)(&control), &item_rect);
				SetControlTitle(control, reply.sfFile.name);
			} else if (!(prefs->logfile_alias)) {
				/* No file, reset everything */
				prefs->do_logfile = 0;
				GetDialogItem(dialog, kOptionsLogFileCheckbox, &item_type, (Handle *)(&control), &item_rect);
				SetControlValue(control, 0);
				GetDialogItem(dialog, kOptionsLogFileButton, &item_type, (Handle *)(&control), &item_rect);
				HiliteControl(control, kControlInactivePart);
			};
			break;
		
		case kOptionsAppendLogCheckbox:
			if (!prefs->do_logfile) break;
			prefs->logfile_append = prefs->logfile_append ? 0 : 1;
			GetDialogItem(dialog, kOptionsAppendLogCheckbox, &item_type, (Handle *)(&control), &item_rect);
			SetControlValue(control, prefs->logfile_append ? 1 : 0);
			break;
		
		/* Handle the checkboxes */
		case kOptionsAutoboot:
			prefs->auto_boot = prefs->auto_boot ? 0 : 1;
			GetDialogItem(dialog, kOptionsAutoboot, &item_type, (Handle *)(&control), &item_rect);
			SetControlValue(control, prefs->auto_boot ? 1 : 0);
			break;
		case kOptionsBootIntoLinux:
#if 0 /* not available in Penguin-18 */
			prefs->boot_into_linux = prefs->boot_into_linux ? 0 : 1;
			GetDialogItem(dialog, kOptionsBootIntoLinux, &item_type, (Handle *)(&control), &item_rect);
			SetControlValue(control, prefs->boot_into_linux ? 1 : 0);
#endif
			break;
		case kOptionsDontBoot:
			prefs->no_boot = prefs->no_boot ? 0 : 1;
			GetDialogItem(dialog, kOptionsDontBoot, &item_type, (Handle *)(&control), &item_rect);
			SetControlValue(control, prefs->no_boot ? 1 : 0);
			break;
		case kOptionsColorByPenguin:
			prefs->color_by_penguin = prefs->color_by_penguin ? 0 : 1;
			GetDialogItem(dialog, kOptionsColorByPenguin, &item_type, (Handle *)(&control), &item_rect);
			SetControlValue(control, prefs->color_by_penguin ? 1 : 0);
			break;
		
		/* Boot delay stuff */
		case kOptionsDelayBoot:
			prefs->delay_boot = prefs->delay_boot ? 0 : 1;
			GetDialogItem(dialog, kOptionsDelayBoot, &item_type, (Handle *)(&control), &item_rect);
			SetControlValue(control, prefs->delay_boot);
			break;
		
		/* The debugging checkboxes too */
		case kOptionsDebugBootInfo:
			prefs->debug_boot_info = prefs->debug_boot_info ? 0 : 1;
			GetDialogItem(dialog, kOptionsDebugBootInfo, &item_type, (Handle *)(&control), &item_rect);
			SetControlValue(control, prefs->debug_boot_info ? 1 : 0);
			break;
		case kOptionsDebugSegmentInfo:
			prefs->debug_segment_info = prefs->debug_segment_info ? 0 : 1;
			GetDialogItem(dialog, kOptionsDebugSegmentInfo, &item_type, (Handle *)(&control), &item_rect);
			SetControlValue(control, prefs->debug_segment_info ? 1 : 0);
			break;
		case kOptionsDebugMachineSpecs:
			prefs->debug_mach_specs = prefs->debug_mach_specs ? 0 : 1;
			GetDialogItem(dialog, kOptionsDebugMachineSpecs, &item_type, (Handle *)(&control), &item_rect);
			SetControlValue(control, prefs->debug_mach_specs ? 1 : 0);
			break;
		case kOptionsDebugBootstrap:
			prefs->debug_copy_and_go = prefs->debug_copy_and_go ? 0 : 1;
			GetDialogItem(dialog, kOptionsDebugBootstrap, &item_type, (Handle *)(&control), &item_rect);
			SetControlValue(control, prefs->debug_copy_and_go ? 1 : 0);
			break;
	}
}


/*
 *	switch_from_optionstab
 *
 *  Store any prefs that haven't been stored already, and trim out the options pane.
 */
void switch_from_optionstab (DialogPtr dialog, penguin_config * prefs)
{
	ControlHandle		control;
	DialogItemType		item_type;
	Rect				item_rect;
	Str255				delay_string;
	
	/* Save the command line */
	GetDialogItem(dialog, kOptionsCommandline, &item_type, (Handle *)(&control), &item_rect);
	getdialogitemtext((Handle)control, prefs->command_line); /* C Glue */
	
	/* Save the boot delay amount */
	GetDialogItem(dialog, kOptionsDelayAmount, &item_type, (Handle *)(&control), &item_rect);
	GetDialogItemText((Handle)control, delay_string);
	StringToNum(delay_string, &prefs->boot_delay_time);
	
	/* Trim out the options pane */
	ShortenDITL(dialog, OPTIONS_NUM_ITEMS);
}


/*
 *	switch_to_serialtab
 *
 *  Overlay the Serial pane on the dialog, and set up the fields.
 */
OSErr switch_to_serialtab (DialogPtr dialog, penguin_config * prefs)
{
	Handle				ditl;
	ControlHandle		control;
	long				serial_attr;
	DialogItemType		item_type;
	Rect				item_rect;
	
	/* Load the serial pane... */
	ditl = Get1Resource('DITL', SERIAL_DITL_ID);
	/* ... and overlay it on the dialog */
	AppendDITL(dialog, ditl, overlayDITL);
	/* That's all we need, get rid of the ditl now */
	//ReleaseResource(ditl);
	
	/* Check which ports are available */
	Gestalt(gestaltSerialAttr, &serial_attr);
	
	/* Set up the checkboxes */
	GetDialogItem(dialog, kSerialDisableAtalkCheckbox, &item_type, (Handle *)(&control), &item_rect);
	SetControlValue(control, prefs->disable_appletalk ? 1 : 0);
	
	GetDialogItem(dialog, kSerialModemPortCheckbox, &item_type, (Handle *)(&control), &item_rect);
	if (serial_attr & (1<<gestaltHidePortA)) {
		/* No port to configure */
		HiliteControl(control, kControlInactivePart);
	}
	SetControlValue(control, prefs->modem_port.do_config_port ? 1 : 0);
	
	GetDialogItem(dialog, kSerialPrinterPortCheckbox, &item_type, (Handle *)(&control), &item_rect);
	if (serial_attr & (1<<gestaltHidePortB)) {
		/* No port to configure */
		HiliteControl(control, kControlInactivePart);
	}
	SetControlValue(control, prefs->printer_port.do_config_port ? 1 : 0);
		
	/* Set up the menus */
	GetDialogItem(dialog, kSerialModemPortSpeedMenu, &item_type, (Handle *)(&control), &item_rect);
	if (prefs->modem_port.port_speed)
		SetControlValue(control, (prefs->modem_port.port_speed));
	HiliteControl(control, prefs->modem_port.do_config_port ? kControlNoPart : kControlInactivePart);
	
	GetDialogItem(dialog, kSerialModemDataSizeMenu, &item_type, (Handle *)(&control), &item_rect);
	if (prefs->modem_port.data_size)
		SetControlValue(control, (prefs->modem_port.data_size));
	HiliteControl(control, prefs->modem_port.do_config_port ? kControlNoPart : kControlInactivePart);
	
	GetDialogItem(dialog, kSerialModemParityMenu, &item_type, (Handle *)(&control), &item_rect);
	if (prefs->modem_port.parity)
		SetControlValue(control, (prefs->modem_port.parity));
	HiliteControl(control, prefs->modem_port.do_config_port ? kControlNoPart : kControlInactivePart);
	
	GetDialogItem(dialog, kSerialModemStopBitsMenu, &item_type, (Handle *)(&control), &item_rect);
	if (prefs->modem_port.stop_bits)
		SetControlValue(control, (prefs->modem_port.stop_bits));
	HiliteControl(control, prefs->modem_port.do_config_port ? kControlNoPart : kControlInactivePart);
	
	GetDialogItem(dialog, kSerialPrinterPortSpeedMenu, &item_type, (Handle *)(&control), &item_rect);
	if (prefs->printer_port.port_speed)
		SetControlValue(control, (prefs->printer_port.port_speed));
	HiliteControl(control, prefs->printer_port.do_config_port ? kControlNoPart : kControlInactivePart);
	
	GetDialogItem(dialog, kSerialPrinterDataSizeMenu, &item_type, (Handle *)(&control), &item_rect);
	if (prefs->printer_port.data_size)
		SetControlValue(control, (prefs->printer_port.data_size));
	HiliteControl(control, prefs->printer_port.do_config_port ? kControlNoPart : kControlInactivePart);
	
	GetDialogItem(dialog, kSerialPrinterParityMenu, &item_type, (Handle *)(&control), &item_rect);
	if (prefs->printer_port.parity)
		SetControlValue(control, (prefs->printer_port.parity));
	HiliteControl(control, prefs->printer_port.do_config_port ? kControlNoPart : kControlInactivePart);
	
	GetDialogItem(dialog, kSerialPrinterStopBitsMenu, &item_type, (Handle *)(&control), &item_rect);
	if (prefs->printer_port.stop_bits)
		SetControlValue(control, (prefs->printer_port.stop_bits));
	HiliteControl(control, prefs->printer_port.do_config_port ? kControlNoPart : kControlInactivePart);
	
	return noErr;
}


/*
 *	do_item_serialtab
 *
 *  An item was clicked in the serial pane. Do something about it.
 */
void do_item_serialtab (short which_item, DialogPtr dialog, penguin_config * prefs)
{
	long				serial_attr;	
	ControlHandle		control;
	DialogItemType		item_type;
	Rect				item_rect;
	
	switch (which_item) {
		case kSerialDisableAtalkCheckbox:
			/* Set the Prefs... */
			prefs->disable_appletalk = prefs->disable_appletalk ? 0 : 1;
			
			/* ...and control value */
			GetDialogItem(dialog, kSerialDisableAtalkCheckbox, &item_type, (Handle *)(&control), &item_rect);
			SetControlValue(control, prefs->disable_appletalk ? 1 : 0);
			break;
		case kSerialModemPortCheckbox:
			/* Set the Prefs... */
			prefs->modem_port.do_config_port = prefs->modem_port.do_config_port ? 0 : 1;
			
			/* ...and control values */
			GetDialogItem(dialog, kSerialModemPortCheckbox, &item_type, (Handle *)(&control), &item_rect);
			SetControlValue(control, prefs->modem_port.do_config_port ? 1 : 0);
			break;
		case kSerialPrinterPortCheckbox:
			/* Set the Prefs... */
			prefs->printer_port.do_config_port = prefs->printer_port.do_config_port ? 0 : 1;
			
			/* ...and control values */
			GetDialogItem(dialog, kSerialPrinterPortCheckbox, &item_type, (Handle *)(&control), &item_rect);
			SetControlValue(control, prefs->printer_port.do_config_port ? 1 : 0);
			break;
	}
	
	/* Enable and Disable stuff appropriatley */
	
	/* First the Appletalk Checkbox */
	GetDialogItem(dialog, kSerialDisableAtalkCheckbox, &item_type, (Handle *)(&control), &item_rect);
	HiliteControl(control, kControlNoPart);
	
	/* Check which ports are available */
	Gestalt(gestaltSerialAttr, &serial_attr);
	
	if (!(serial_attr & 1<<gestaltHidePortA)) {
		/* Modem port is good */
		GetDialogItem(dialog, kSerialModemPortCheckbox, &item_type, (Handle *)(&control), &item_rect);
		HiliteControl(control, kControlNoPart);
	}
	if (!(serial_attr & 1<<gestaltHidePortB)) {
		/* Printer port is good */
		GetDialogItem(dialog, kSerialPrinterPortCheckbox, &item_type, (Handle *)(&control), &item_rect);
		HiliteControl(control, kControlNoPart);
	}
	
	if (prefs->modem_port.do_config_port) {
		/* Enable the modem port menus */
		GetDialogItem(dialog, kSerialModemPortSpeedMenu, &item_type, (Handle *)(&control), &item_rect);
		HiliteControl(control, kControlNoPart);
		
		GetDialogItem(dialog, kSerialModemDataSizeMenu, &item_type, (Handle *)(&control), &item_rect);
		HiliteControl(control, kControlNoPart);
		
		GetDialogItem(dialog, kSerialModemParityMenu, &item_type, (Handle *)(&control), &item_rect);
		HiliteControl(control, kControlNoPart);
		
		GetDialogItem(dialog, kSerialModemStopBitsMenu, &item_type, (Handle *)(&control), &item_rect);
		HiliteControl(control, kControlNoPart);
	} else {
		/* Disable the modem port menus */
		GetDialogItem(dialog, kSerialModemPortSpeedMenu, &item_type, (Handle *)(&control), &item_rect);
		HiliteControl(control, kControlInactivePart);
		
		GetDialogItem(dialog, kSerialModemDataSizeMenu, &item_type, (Handle *)(&control), &item_rect);
		HiliteControl(control, kControlInactivePart);
		
		GetDialogItem(dialog, kSerialModemParityMenu, &item_type, (Handle *)(&control), &item_rect);
		HiliteControl(control, kControlInactivePart);
		
		GetDialogItem(dialog, kSerialModemStopBitsMenu, &item_type, (Handle *)(&control), &item_rect);
		HiliteControl(control, kControlInactivePart);
	}
	
	if (prefs->printer_port.do_config_port) {
		/* Enable the printer port menus */
		GetDialogItem(dialog, kSerialPrinterPortSpeedMenu, &item_type, (Handle *)(&control), &item_rect);
		HiliteControl(control, kControlNoPart);
		
		GetDialogItem(dialog, kSerialPrinterDataSizeMenu, &item_type, (Handle *)(&control), &item_rect);
		HiliteControl(control, kControlNoPart);
		
		GetDialogItem(dialog, kSerialPrinterParityMenu, &item_type, (Handle *)(&control), &item_rect);
		HiliteControl(control, kControlNoPart);
		
		GetDialogItem(dialog, kSerialPrinterStopBitsMenu, &item_type, (Handle *)(&control), &item_rect);
		HiliteControl(control, kControlNoPart);
	} else {
		/* Disable the printer port menus */
		GetDialogItem(dialog, kSerialPrinterPortSpeedMenu, &item_type, (Handle *)(&control), &item_rect);
		HiliteControl(control, kControlInactivePart);
		
		GetDialogItem(dialog, kSerialPrinterDataSizeMenu, &item_type, (Handle *)(&control), &item_rect);
		HiliteControl(control, kControlInactivePart);
		
		GetDialogItem(dialog, kSerialPrinterParityMenu, &item_type, (Handle *)(&control), &item_rect);
		HiliteControl(control, kControlInactivePart);
		
		GetDialogItem(dialog, kSerialPrinterStopBitsMenu, &item_type, (Handle *)(&control), &item_rect);
		HiliteControl(control, kControlInactivePart);
	}
}


/*
 *	switch_from_serialtab
 *
 *  Store any prefs that haven't been stored already, and trim out the serial pane.
 */
void switch_from_serialtab (DialogPtr dialog, penguin_config * prefs)
{
	ControlHandle		control;
	DialogItemType		item_type;
	Rect				item_rect;
	Handle				ditl;
	
	/* Save all the menu settings first */
	
	/* Modem Port */
	GetDialogItem(dialog, kSerialModemPortSpeedMenu, &item_type, (Handle *)(&control), &item_rect);
	prefs->modem_port.port_speed = GetControlValue(control);
	
	GetDialogItem(dialog, kSerialModemDataSizeMenu, &item_type, (Handle *)(&control), &item_rect);
	prefs->modem_port.data_size = GetControlValue(control);
	
	GetDialogItem(dialog, kSerialModemParityMenu, &item_type, (Handle *)(&control), &item_rect);
	prefs->modem_port.parity = GetControlValue(control);
	
	GetDialogItem(dialog, kSerialModemStopBitsMenu, &item_type, (Handle *)(&control), &item_rect);
	prefs->modem_port.stop_bits = GetControlValue(control);
	
	/* Printer Port */
	GetDialogItem(dialog, kSerialPrinterPortSpeedMenu, &item_type, (Handle *)(&control), &item_rect);
	prefs->printer_port.port_speed = GetControlValue(control);
	
	GetDialogItem(dialog, kSerialPrinterDataSizeMenu, &item_type, (Handle *)(&control), &item_rect);
	prefs->printer_port.data_size = GetControlValue(control);
	
	GetDialogItem(dialog, kSerialPrinterParityMenu, &item_type, (Handle *)(&control), &item_rect);
	prefs->printer_port.parity = GetControlValue(control);
	
	GetDialogItem(dialog, kSerialPrinterStopBitsMenu, &item_type, (Handle *)(&control), &item_rect);
	prefs->printer_port.stop_bits = GetControlValue(control);
	
	/* Trim out the serial pane */
	ShortenDITL(dialog, SERIAL_NUM_ITEMS);
}
