/*
**  boot_helper.c -- Load and launch the Linux kernel
**
**  based on bootstrap.c for Atari Linux booter, Copyright 1993 by Arjan Knor
**
**  This file is subject to the terms and conditions of the GNU General Public
**  License.  See the file COPYING in the main directory of this archive
**  for more details.
**
**  History:
**   5 Oct 1997  Functions factored out of bootstrap.c's boot.\
*/

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <stddef.h>
#include <string.h>
#include <ctype.h>
/*#include <unix.h>*/

/* linux specific include files */
#include "linux_a.out.h"
#include "linux_elf.h"
#include "asm_page.h"
#include "asm_setup.h"

#include <ShutDown.h>
#include <Gestalt.h> // JDJ
#include <LowMem.h>
//#include "penguin.h"
#include "bootstrap_prototypes.h"
#include "appkill_prototypes.h"
#include "penguin_prototypes.h"
#include "asm_prototypes.h"
#include "machine.h"
#include "videocard_prototypes.h"

#include "MMU.h"
#include "MMU_V2.h"

/* Protos */
static char					*add_v2_boot_record(char *dst, unsigned short tag,
								unsigned short in_data_size, void *in_data);
static unsigned short		get_v2_boot_record(char *start, unsigned short tag,
								void *out_data);

/*
 *	memory_init
 *
 */
void memory_init (void)
{
	gMemoryMappings.index = 0;
	gMemoryMappings.cacheEmitter = NULL;

	InitMMU_V2();
}

/*
 *	check_bootinfo_version
 *
 */
unsigned long check_bootinfo_version(char *memptr)
{
	struct bootversion *bv = (struct bootversion *)memptr;
	unsigned long		version = 0;
	unsigned long		i, boots_major, boots_minor;

	cprintf( "\n" );
	if (bv->magic == BOOTINFOV_MAGIC)
		for( i = 0; bv->machversions[i].machtype != 0; ++i ) {
			if (bv->machversions[i].machtype == MACH_MAC) {
				version = bv->machversions[i].version;
				break;
			}
		}
	if (!version)
		cprintf("Kernel has no bootinfo version info, assuming 0.0\n");

	v_kernel_major = BI_VERSION_MAJOR(version);
	v_kernel_minor = BI_VERSION_MINOR(version);
// XXX - change to MAC_BOOTI_VERSION later
	boots_major	 = BI_VERSION_MAJOR(MAC_BOOTI_VERSION);
	boots_minor	 = BI_VERSION_MINOR(MAC_BOOTI_VERSION);
	if (config.debug_boot_info) {
		cprintf("Bootstrap's bootinfo version: %d.%d\n",
			boots_major, boots_minor);
		cprintf("Kernel's bootinfo version	: %d.%d\n",
			v_kernel_major, v_kernel_minor);
	}
	
// must disable to test with stock Atari kernels.
//	if (kernel_major != boots_major) {
//		cprintf("\nThis bootstrap is too %s for this kernel!\n",
//			boots_major < kernel_major ? "old" : "new");
//		return 0;
//	}
	if (v_kernel_minor > boots_minor) {
		cprintf("Warning: boot_info versions of bootstrap and kernel differ!\n");
		cprintf("         Certain features may not work.\n");
	}

	return 1;
}

/*
 *	boot_init_bootinfo
 *
 *	bi is a global structur BootInfo
 *	
 */
unsigned char /* Boolean */ boot_init_bootinfo (void)
{
	long proc, fpu, mmu, mach, ram;
	unsigned long time;
	GDHandle theScreen;
	PixMapHandle theScreenMap;
	unsigned long rows, columns;
	unsigned long	gmt_bias;
	MachineLocation where;

	/*
	 *	Clear the structure first
	 */
	memset(&bi, 0, sizeof(bi));

	/* machine is Macintosh */
	bi.machtype = MACH_MAC;

	/*
	 *	Gather machine information
	 */
	Gestalt(gestaltProcessorType, &proc);
	/* Check if SoftwareFPU is present */
	if (Gestalt('FPUE', &fpu) == noErr) {
		fpu = 0;
	} else {
		Gestalt(gestaltFPUType, &fpu);
	}
	Gestalt(gestaltMMUType, &mmu);
	Gestalt(gestaltMachineType, &mach);
	Gestalt(gestaltPhysicalRAMSize, &ram);

	/*
	 *	Set the processor type
	 */
	switch (proc) {
		case gestalt68020:
			if (mmu != gestalt68851) {
				cprintf ("68020 has no MMU. Aborting boot...\n");
				return false;
			} else
				bi.cputype = CPU_68020;
			break;
		case gestalt68030:
			bi.cputype = CPU_68030; break;
		case gestalt68040:
			bi.cputype = CPU_68040; break;
		default:
			cprintf ("68000, PowerPC, or unknown processor.  Aborting...\n");
			return false;
	};

	/*
	 *	Set the FPU info
	 */
	switch (fpu) {
		case gestalt68881:
			bi.cputype |= FPU_68881; break;
		case gestalt68882:
			bi.cputype |= FPU_68882; break;
		case gestalt68040FPU:
			bi.cputype |= FPU_68040; break;
		case gestaltNoFPU:
		default:
			/* assume no FPU */
			break;
	};
	
	/*
	 *	Ram Disk is 0 until later
	 */
	bi.ramdisk_size = 0;

	/*
	 *	Copy the command line
	 */
	strncpy(bi.command_line, config.command_line, 256);
	
	/*
	 *	Gather video information
	 */
	theScreen = GetMainDevice();
	theScreenMap = (*theScreen)->gdPMap;
	bi.bi_mac.videoaddr = LogicalToPhysical ((unsigned long) (*theScreenMap)->baseAddr);
	bi.bi_mac.videological = (unsigned long) (*theScreenMap)->baseAddr;
	bi.bi_mac.videorow = ((*theScreenMap)->rowBytes & ~0xC000) & 0xFFFF; 			
	bi.bi_mac.videodepth = (*theScreenMap)->pixelSize;
	columns = (*theScreen)->gdRect.right - (*theScreen)->gdRect.left;
	rows = (*theScreen)->gdRect.bottom - (*theScreen)->gdRect.top;
	bi.bi_mac.dimensions = columns | (rows << 16);

	/*
	 *	Date and Time that we started the boot process
	 */
	GetDateTime(&time);
	bi.bi_mac.boottime = time - 2082844800;		/* time between Unix and MacOS epochs */ 
	ReadLocation(&where);
	gmt_bias = where.u.gmtDelta & 0x00FFFFFF;
	if (gmt_bias & 0x00800000)
		gmt_bias |= 0xFF000000;   /* sign-extend to 32 bits */
	gmt_bias = (long)gmt_bias / 60;  	/* convert to whole minutes, remember sign */
//	gmt_bias += 60 * where.u.dlsDelta;
	bi.bi_mac.gmtbias = gmt_bias;

	/*
	 *	General machine information
	 */
	bi.bi_mac.bootver = 108;					/* meaningless for Linux */
	bi.bi_mac.scc = LogicalToPhysical ((unsigned long) LMGetSCCRd());
	bi.bi_mac.id = mach;
	bi.bi_mac.memsize = ram / (1024L * 1024L);
	bi.bi_mac.serialmf = 0;						/* what? */
	bi.bi_mac.serialhsk = 9600;
	bi.bi_mac.serialgpi = 9600;
	bi.bi_mac.printf = 0;
	bi.bi_mac.printhsk = 9600;
	bi.bi_mac.printgpi = 9600;
	bi.bi_mac.cpuid = proc - gestalt68020;  	/* superfluous */
	bi.bi_mac.rombase = (unsigned long) LMGetROMBase();
	bi.bi_mac.timedbra = LMGetTimeDBRA();
	bi.bi_mac.adbdelay = *((unsigned short *)0x0cea);

	/*
	 *	
	 */

	return true;
}

/*
 *	boot_print_mach_specs
 *
 */
void boot_print_mach_specs (void)
{
	int			i;
	char		*model;
	Str255		strModel;
	long		sysVers;

	Gestalt(gestaltSystemVersion, &sysVers);
	cprintf("\nSystem: %d.%d.%d\n",
		(int)((sysVers>>8) & 255), (int)((sysVers>>4) & 15), (int)(sysVers & 15));
	model = GetModelName(bi.bi_mac.id, strModel);
	cprintf ("Gestalt ID: %d (%s)\n", bi.bi_mac.id, (model ? model : "Unknown"));
	cprintf ("CPU: %s\n", (bi.cputype & CPU_68040) ? "68040" : 
		((bi.cputype & CPU_68030) ? "68030" : "68020 w/68851 PMMU"));
	cprintf ("FPU: %s\n", (bi.cputype & FPU_68881) ? "68881" : 
		((bi.cputype & FPU_68882) ? "68882" :
		((bi.cputype & FPU_68040) ? "68040 built-in" : "none")));
	cprintf ("Physical RAM: %d MB\n", bi.bi_mac.memsize);
	
	for (i = 0; i < bi.num_memory; i++)
		cprintf("  block %d: at 0x%08lx, size 0x%lx\n",
			i, bi.memory[i].addr, bi.memory[i].size);
}


/*
 *	boot_dump_boot_info
 *
 */
void boot_dump_boot_info(char *v2_data,
	unsigned long major_version, unsigned long minor_version)
{
	int					i;
	unsigned long		l1;

	cprintf("\nDump of bootinfo, version %ld.%ld:\n", major_version, minor_version);
	if (major_version < 2) {
		cprintf("bi.machtype            = 0x%x\n", bi.machtype);
		cprintf("bi.cputype             = 0x%x\n", bi.cputype);
		for (i = 0; i < bi.num_memory; i++) {
			cprintf("bi.memory[%d].addr      = 0x%08x\n", i, bi.memory[i].addr);
			cprintf("bi.memory[%d].size      = 0x%08x\n", i, bi.memory[i].size);
		};
		cprintf("bi.num_memory          = 0x%x\n", bi.num_memory);
		cprintf("bi.ramdisk_size        = 0x%08x\n", bi.ramdisk_size);
		cprintf("bi.ramdisk_addr        = 0x%08x\n", bi.ramdisk_addr);
		cprintf("bi.command_line        = %s\n", &(bi.command_line));
		cprintf("bi.bi_mac.videoaddr    = 0x%08x\n", bi.bi_mac.videoaddr);
		cprintf("bi.bi_mac.videorow     = 0x%x\n", bi.bi_mac.videorow);
		cprintf("bi.bi_mac.videodepth   = 0x%x\n", bi.bi_mac.videodepth);
		cprintf("bi.bi_mac.dimensions   = 0x%08x\n", bi.bi_mac.dimensions);
		cprintf("bi.bi_mac.args         = 0x%x\n", bi.bi_mac.args);
		cprintf("bi.bi_mac.boottime     = 0x%x\n", bi.bi_mac.boottime);
		cprintf("bi.bi_mac.gmtbias      = 0x%x\n", bi.bi_mac.gmtbias);
		cprintf("bi.bi_mac.bootver      = 0x%x\n", bi.bi_mac.bootver);
		cprintf("bi.bi_mac.videological = 0x%08x\n", bi.bi_mac.videological);
		cprintf("bi.bi_mac.scc          = 0x%08x\n", bi.bi_mac.scc);
		cprintf("bi.bi_mac.id           = 0x%x\n", bi.bi_mac.id);
		cprintf("bi.bi_mac.memsize      = 0x%x\n", bi.bi_mac.memsize);
		cprintf("bi.bi_mac.serialmf     = 0x%x\n", bi.bi_mac.serialmf);
		cprintf("bi.bi_mac.serialhsk    = 0x%x\n", bi.bi_mac.serialhsk);
		cprintf("bi.bi_mac.serialgpi    = 0x%x\n", bi.bi_mac.serialgpi);
		cprintf("bi.bi_mac.printf       = 0x%x\n", bi.bi_mac.printf);
		cprintf("bi.bi_mac.printhsk     = 0x%x\n", bi.bi_mac.printhsk);
		cprintf("bi.bi_mac.printgpi     = 0x%x\n", bi.bi_mac.printgpi);
		cprintf("bi.bi_mac.cpuid        = 0x%x\n", bi.bi_mac.cpuid);
		cprintf("bi.bi_mac.rombase      = 0x%08x\n", bi.bi_mac.rombase);
		cprintf("bi.bi_mac.adbdelay     = 0x%x\n", bi.bi_mac.adbdelay);
		cprintf("bi.bi_mac.timedbra     = 0x%x\n", bi.bi_mac.timedbra);
	} else {
		struct bi2_record	*bi2_rec;
		int					chunk = 0;

		bi2_rec = (struct bi2_record *)v2_data;
		while(bi2_rec->tag != V2_BI_LAST) {
			l1 = *(unsigned long *)bi2_rec->data;
			switch(bi2_rec->tag) {
				case V2_BI_MACHTYPE:
					cprintf("BI_MACHTYPE           = 0x%x\n", l1);
					break;
				case V2_BI_CPUTYPE:
					cprintf("BI_CPUTYPE            = 0x%x\n", l1);
					break;
				case V2_BI_FPUTYPE:
					cprintf("BI_FPUTYPE            = 0x%x\n", l1);
					break;
				case V2_BI_MMUTYPE:
					cprintf("BI_MMUTYPE            = 0x%x\n", l1);
					break;
				case V2_BI_MEMCHUNK:
					cprintf("BI_MEMCHUNK[%d].addr   = 0x%08x\n", chunk, l1);
					cprintf("BI_MEMCHUNK[%d].size   = 0x%08x\n", chunk, *(unsigned long *)(&bi2_rec->data[4]));
					++chunk;
					break;
				case V2_BI_RAMDISK:
					cprintf("BI_RAMDISK.addr       = 0x%08x\n", l1);
					cprintf("BI_RAMDISK.size       = 0x%08x\n", *(unsigned long *)(&bi2_rec->data[4]));
					break;
				case V2_BI_COMMAND_LINE:
					cprintf("BI_COMMAND_LINE       = %s\n", (char *)bi2_rec->data);
					break;
				case V2_BI_MAC_MODEL:
					cprintf("BI_MAC_MODEL          = 0x%x\n", l1);
					break;
				case V2_BI_MAC_VADDR:
					cprintf("BI_MAC_VADDR          = 0x%x\n", l1);
					break;
				case V2_BI_MAC_VDEPTH:
					cprintf("BI_MAC_VDEPTH         = 0x%x\n", l1);
					break;
				case V2_BI_MAC_VROW:
					cprintf("BI_MAC_VROW           = 0x%x\n", l1);
					break;
				case V2_BI_MAC_VDIM:
					cprintf("BI_MAC_VDIM           = 0x%08x\n", l1);
					break;
				case V2_BI_MAC_VLOGICAL:
					cprintf("BI_MAC_VLOGICAL       = 0x%08x\n", l1);
					break;
				case V2_BI_MAC_SCCBASE:
					cprintf("BI_MAC_SCCBASE        = 0x%08x\n", l1);
					break;
				case V2_BI_MAC_BTIME:
					cprintf("BI_MAC_BTIME          = 0x%x\n", l1);
					break;
				case V2_BI_MAC_GMTBIAS:
					cprintf("BI_MAC_GMTBIAS        = 0x%x\n", l1);
					break;
				case V2_BI_MAC_MEMSIZE:
					cprintf("BI_MAC_MEMSIZE        = 0x%x\n", l1);
					break;
				case V2_BI_MAC_CPUID:
					cprintf("BI_MAC_CPUID          = 0x%x\n", l1);
					break;
				case V2_BI_MAC_ROMBASE:
					cprintf("BI_MAC_ROMBASE        = 0x%x\n", l1);
					break;
				default:
					cprintf("*** Unknown tag       = 0x%04hd\n", (unsigned short)bi2_rec->tag);
					break;
			}
			
			bi2_rec = (struct bi2_record *)((unsigned long)bi2_rec + bi2_rec->size);
			
		}

	}

}

/*
 *	boot_dump_segment_info
 *
 */
void boot_dump_segment_info (
	unsigned long				start,
	struct Kernel_Image_Info 	*info,
	struct Kernel_Offsets		*kernel_offsets,
	unsigned long				kernel_major,
	unsigned long				kernel_minor)
{
//#pragma unused(kernel_minor)

	int			i;
	
	if (bi.ramdisk_size)
		cprintf ("RAM disk at %#0.8lx, ends at %#0.8lx, size is %d K\n",
			start + kernel_offsets->ramdisk.start,
			start + kernel_offsets->ramdisk.start + kernel_offsets->ramdisk.size,
			kernel_offsets->ramdisk.size / 1024);
	if (info->type == KERNEL_ELF) {
		for (i = 0; i < info->exec_elf.e_phnum; i++) {
			cprintf ("Kernel segment %d at %#lx, size %ld\n", i,
				start + info->phdrs_elf[i].p_vaddr - PAGE_SIZE,
				info->phdrs_elf[i].p_memsz);
		}
  	} else {
		cprintf ("\nKernel text at %#lx, code size %d\n", start, info->exec.a_text);
		cprintf ("Kernel data at %#lx, data size %d\n", start + info->exec.a_text, info->exec.a_data );
		cprintf ("Kernel bss  at %#lx, bss  size %d\n", start + info->exec.a_text + info->exec.a_data, info->exec.a_bss );
	}
	cprintf ("Kernel size is %#lx\n", kernel_offsets->kernel.size);

	cprintf ("\nboot_info is at %#lx", start + kernel_offsets->boot_info);
	if (kernel_major < 2)
		cprintf ("\nboot_info size is %#lx\n", sizeof(bi));
	else
		cprintf ("\nboot_info size is dynamic\n");

	if (kernel_offsets->ramdisk.size > 0) {
		cprintf ("\nramdisk logical target %#lx", PhysicalToLogical(bi.ramdisk_addr));
		cprintf ("\nramdisk physical at %#lx", bi.ramdisk_addr);
		cprintf ("\nramdisk physical top at %#lx\n", bi.ramdisk_addr + kernel_offsets->ramdisk.size);
	}
//	cprintf ("ramdisk lower limit is %#lx\n", memptr + memreq - rd_size - PAGE_SIZE);
//	cprintf ("ramdisk src top is %#lx\n", memptr + memreq - PAGE_SIZE);
//	cprintf ("\nmover code is at %#lx\n", memptr + memreq - COPY_AND_GO_SIZE);
}

/*
 *	set_kernel_bootinfo
 *
 *	Copy bootinfo, version 1 or 2, to *dst (end of kernel image)
 *	Version 1 bootinfo is a single structure while version 2
 *	are {tag,size,data} blocks appended to the kernel image.
 */
void
set_kernel_bootinfo(char *dst, unsigned long major_version, unsigned long minor_version)
{
//#pragma unused(minor_version)

	unsigned long	l1, l2, l3;
	struct mem_info	mi;

	if (major_version < 2) {
		memcpy(dst, &bi, sizeof(bi));
	} else {
		l1 = bi.machtype;
		dst = add_v2_boot_record(dst, V2_BI_MACHTYPE, sizeof(l1), &l1);
		switch(bi.cputype & CPU_MASK) {
			case CPU_68020:
				l1 = V2_CPU_68020;
				l2 = V2_MMU_68851;
				break;
			case CPU_68030:
				l1 = V2_CPU_68030;
				l2 = V2_MMU_68030;
				break;
			case CPU_68040:
				l1 = V2_CPU_68040;
				l2 = V2_MMU_68040;
				break;
			case CPU_68060:
				l1 = V2_CPU_68060;
				l2 = V2_MMU_68060;
				break;
			default:
				l1 = 0;
				l2 = 0;
				break;
		}
		switch(bi.cputype & FPU_MASK) {
			case FPU_68881:
				l3 = V2_FPU_68881;
				break;
			case FPU_68882:
				l3 = V2_FPU_68882;
				break;
			case FPU_68040:
				l3 = V2_FPU_68040;
				break;
			case FPU_68060:
				l3 = V2_FPU_68060;
				break;
			default:
				l3 = 0;
				break;
		}
		dst = add_v2_boot_record(dst, V2_BI_CPUTYPE, sizeof(l1), &l1);
		dst = add_v2_boot_record(dst, V2_BI_FPUTYPE, sizeof(l3), &l3);
		dst = add_v2_boot_record(dst, V2_BI_MMUTYPE, sizeof(l2), &l2);

#if 1
		for(l1 = 0; l1 < bi.num_memory; l1++) {
			dst = add_v2_boot_record(dst, V2_BI_MEMCHUNK, sizeof(struct mem_info), &bi.memory[l1]);
		}
#else
		dst = add_v2_boot_record(dst, V2_BI_MEMCHUNK, sizeof(bi.memory), bi.memory);
#endif

		if (bi.ramdisk_size) {
			mi.addr = bi.ramdisk_addr;
			mi.size = bi.ramdisk_size * 1024L;
			dst = add_v2_boot_record(dst, V2_BI_RAMDISK, sizeof(mi), &mi);
		}
		dst = add_v2_boot_record(dst, V2_BI_COMMAND_LINE, sizeof(bi.command_line), bi.command_line);
		l1 = bi.bi_mac.id;
		dst = add_v2_boot_record(dst, V2_BI_MAC_MODEL, sizeof(l1), &l1);
		l1 = bi.bi_mac.videoaddr;
		dst = add_v2_boot_record(dst, V2_BI_MAC_VADDR, sizeof(l1), &l1);
		l1 = bi.bi_mac.videodepth;
		dst = add_v2_boot_record(dst, V2_BI_MAC_VDEPTH, sizeof(l1), &l1);
		l1 = bi.bi_mac.videorow;
		dst = add_v2_boot_record(dst, V2_BI_MAC_VROW, sizeof(l1), &l1);
		l1 = bi.bi_mac.dimensions;
		dst = add_v2_boot_record(dst, V2_BI_MAC_VDIM, sizeof(l1), &l1);
		l1 = bi.bi_mac.videological;
		dst = add_v2_boot_record(dst, V2_BI_MAC_VLOGICAL, sizeof(l1), &l1);
		l1 = bi.bi_mac.scc;
		dst = add_v2_boot_record(dst, V2_BI_MAC_SCCBASE, sizeof(l1), &l1);
		l1 = bi.bi_mac.boottime;
		dst = add_v2_boot_record(dst, V2_BI_MAC_BTIME, sizeof(l1), &l1);
		l1 = bi.bi_mac.gmtbias;
		dst = add_v2_boot_record(dst, V2_BI_MAC_GMTBIAS, sizeof(l1), &l1);
		l1 = bi.bi_mac.memsize;
		dst = add_v2_boot_record(dst, V2_BI_MAC_MEMSIZE, sizeof(l1), &l1);
		l1 = bi.bi_mac.cpuid;
		dst = add_v2_boot_record(dst, V2_BI_MAC_CPUID, sizeof(l1), &l1);
		l1 = (unsigned long) LMGetROMBase();
		dst = add_v2_boot_record(dst, V2_BI_MAC_ROMBASE, sizeof(l1), &l1);

/** Not yet - Pass dynamic v2.0 machine info to the kernel
		// VIA1
		if (MacHasHardware(gestaltHasVIA1)) {
			l1 = LogicalToPhysical((unsigned long) *((unsigned long *)0x01D4));
			dst = add_v2_boot_record(dst, V2_BI_MAC_VIA1BASE, sizeof(l1), &l1);
		}

		// VIA2/RBV/OSS
		l2 = -1;
		if (MacHasHardware(gestaltHasVIA2))
			l2 = V2_VIA2_VIA;
		else if (MacHasHardware(gestaltHasRBV))
			l2 = V2_VIA2_RBV;
		else if (MacHasHardware(gestaltHasOSS))
			l2 = V2_VIA2_OSS;
		if (l2 != -1) {
			l1 = LogicalToPhysical((unsigned long) *((unsigned long *)0x0CEC));
			dst = add_v2_boot_record(dst, V2_BI_MAC_VIA2BASE, sizeof(l1), &l1);
			dst = add_v2_boot_record(dst, V2_BI_MAC_VIA2TYPE, sizeof(l2), &l2);
		}

		// ADB
		l1 = V2_ADB_II;
		if (MacHasHardware(gestaltHasSWIMIOP))	// Assume ADB IOP if SWIM IOP
			l1 = V2_ADB_IOP;
		else {
			l2 = lmUnivROMBits;
			if ( l2 != -1 ) {
				l2 &= 0x07000000;
				l2 >>= 24L;
				if (l2 != 0) {
					if ( (l2 == 1) || (l2 == 2) )
						l1 = V2_ADB_IISI;
					else if (l2 == 3)
						l1 = V2_ADB_CUDA;
				}
				else {
					if ( lmPMgrBase != -1)
						l1 = V2_ADB_PB1;
				}
			}
		}
		dst = add_v2_boot_record(dst, V2_BI_MAC_ADBTYPE, sizeof(l1), &l1);

		// ASC
		l1 = lmASCBase;
		if (l1 != -1) {
			l1 = LogicalToPhysical(l1);
			dst = add_v2_boot_record(dst, V2_BI_MAC_ASCBASE, sizeof(l1), &l1);
		}

		// SCSI 5380
		if (MacHasHardware(gestaltHasSCSI)) {
			l1 = LogicalToPhysical(lmSCSIBase);
			dst = add_v2_boot_record(dst, V2_BI_MAC_SCSI5380, sizeof(l1), &l1);
		}

		// SCSI 5380 DMA
		if (MacHasHardware(gestaltHasSCSIDMA)) {
			l1 = LogicalToPhysical(lmSCSIBase);
			dst = add_v2_boot_record(dst, V2_BI_MAC_SCSIDMA, sizeof(l1), &l1);
		}

		// SCSI 5396, internal or external
		if ( MacHasHardware(gestaltHasSCSI961) || MacHasHardware(gestaltHasSCSI962) ) {
			l1 = LogicalToPhysical(lmSCSIBase);
			dst = add_v2_boot_record(dst, V2_BI_MAC_SCSI5396, sizeof(l1), &l1);
		}

		// ATA/IDE
		if (lmHWCfgFlags & 0x0080) {
			if (lmPMgrBase != -1)
				l1 = V2_IDE_PB;
			else
				l1 = V2_IDE_QUADRA;
			dst = add_v2_boot_record(dst, V2_BI_MAC_IDETYPE, sizeof(l1), &l1);
			l1 = V2_IDE_BASE;
			dst = add_v2_boot_record(dst, V2_BI_MAC_IDEBASE, sizeof(l1), &l1);
		}

		// NuBus
		if (Gestalt(gestaltNuBusConnectors, &l1) == noErr) {
			if (l1 != 0) {
				l2 = V2_NUBUS_NORMAL;
				dst = add_v2_boot_record(dst, V2_BI_MAC_NUBUS, sizeof(l2), &l2);
				dst = add_v2_boot_record(dst, V2_BI_MAC_SLOTMASK, sizeof(l1), &l1);
			}
		}

		// SCC
		if (MacHasHardware(gestaltHasSCCIOP))
			l1 = V2_SCC_NORMAL;
		else
			l2 = V2_SCC_IOP;
		dst = add_v2_boot_record(dst, V2_BI_MAC_SCCTYPE, sizeof(l1), &l1);

** End "Not yet" dynamic boot info */

		l1 = 0;
		dst = add_v2_boot_record(dst, V2_BI_LAST, 0, &l1);
	}
}

/*
 *	add_v2_boot_record
 *	
 */
static char *
add_v2_boot_record(char *dst, unsigned short tag,
	unsigned short in_data_size, void *in_data)
{
	struct bi2_record	*rec;

	rec = (struct bi2_record *)dst;
	rec->tag = tag;
	rec->size = in_data_size + sizeof(struct bi2_record);
	BlockMoveData(in_data, rec->data, in_data_size);

	return (dst + sizeof(struct bi2_record) + in_data_size);
}

/*
 *	get_v2_boot_record
 *	
 */
static unsigned short
get_v2_boot_record(char *start, unsigned short tag, void *out_data)
{
	unsigned short	size = 0;

	while( ((struct bi2_record *)start)->tag != V2_BI_LAST ) {
		if ( ((struct bi2_record *)start)->tag == tag) {
			BlockMoveData( ((struct bi2_record *)start)->data, out_data, 
					((struct bi2_record *)start)->size - sizeof(struct bi2_record) );
			size = ((struct bi2_record *)start)->size;
			break;
		}
		start += ((struct bi2_record *)start)->size;
	}
	
	return size;
}


