/*
**  MMU040.c
**
**	This file is subject to the terms and conditions of the GNU General Public
**	License.  See the file COPYING in the main directory of this archive
**	for more details.
**
*/

#include <stdio.h>
#include <Gestalt.h>
#include "MMU.h"
#include "MMU040.h"
#include "penguin_prototypes.h"

/* Prototypes */
static u32			byteCount = 0;

static void			record040TTMapping (u32 tt);
static u32			getTableIndex (void *table, int index);

/*
 *	isCpu040
 *
 */
int isCpu040 (void)
{
	long		cpuType;
	OSErr		err;
	
	err = Gestalt (gestaltProcessorType, &cpuType);
	if (err) {
		cprintf ("This computer does not recognize the Gestalt selector `gestaltProcessorType`.\n");
		cprintf ("Are you sure it's a new-enough system software for this?\n");
		return 0;
	}
	
	if (gestalt68040 == cpuType) 	return 1;
	else							return 0;
}

/*
 *	init040Mmu
 *
 */
void init040Mmu (void)
{
	gMemoryMappings.index = 0;
	gMemoryMappings.cacheEmitter = NULL;
}

/*
 *	calculate040MmuMappings
 *
 */
void calculate040MmuMappings (void)
{	
	u32 	 logAddr, phys;
	u32		 len, physAttr;
	u32		 tt;
	
	logAddr = 0;
	logTo040Phys (logAddr, &phys, &len, &physAttr);

	/*
	 *	Look at the transparent translation registers
	 */
	get040ITT0 (&tt);
	record040TTMapping (tt);
	
	get040ITT1 (&tt);
	record040TTMapping (tt);
	
	get040DTT0 (&tt);
	record040TTMapping (tt);
	
	get040DTT1 (&tt);
	record040TTMapping (tt);
	
	/*
	 *	Walk through all of logical address spaces
	 */
	do {
		u32		 tPhysAttr, tLen;
		u32		 tLogAddr;
		u32		 tPhys;
		
		tLogAddr = logAddr;
		tLen = 0;
		
		do {
			tLogAddr += len;
			tLen += len;
			
			logTo040Phys (tLogAddr, &tPhys, &len, &tPhysAttr);
			if (tPhysAttr != physAttr)
				break;
			if ((phys != NOT_MAPPED) && (tPhys != phys + tLen))
				break;
		} while (1);
		
		recordMemoryMapping (logAddr, phys, tLen, physAttr);

		physAttr = tPhysAttr;
		phys = tPhys;
		logAddr += tLen;
	} while (logAddr);
	
	cprintf ("MMU 040 read a total %d bytes in the MMU Tables!\n", byteCount);
}

/*
 *	record040TTMapping
 *
 */
void record040TTMapping (u32 tt)
{
	u32	logAddr, physAddr, len, attr;
	
	if ((tt & 0x00008000) == 0)
		return;
	
	logAddr = tt & 0xFF000000 & (~((tt & 0x00FF0000 ) << 8));
	physAddr = logAddr;
	len = (tt & 0x00FF0000) << 8;
	len += 0x01000000 -1;
	attr = (tt & 0x00000604) | 1;
	
	recordMemoryMapping (logAddr, physAddr, len, attr);
}

/*
 *	logTo040Phys
 *
 */
void logTo040Phys (u32 logAddr, u32 *physAddr, u32 *physLen, u32 *physAttr)
{
	u32		 rootIndex;
	u32		 ptrIndex;
	u32		 pageIndex;
	u32		 tc;
	u32		*rootTable, *ptrTable, *pageTable, *pageAddr;
	u32		 pageSize8k;
	u32		 pAttr, t;
	u32		 len;
	u32		 rootEntry, tableEntry, pageEntry;
	
	
	get040TCReg (&tc);
	pageSize8k = tc & 0x4000;
	
	rootIndex = (((u32)logAddr) & 0xFE000000) >> 25;
	ptrIndex  = (((u32)logAddr) & 0x01FC0000) >> 18;
	if (pageSize8k)	pageIndex = (((u32)logAddr) & 0x0003E000) >> 13;
	else			pageIndex = (((u32)logAddr) & 0x0003F000) >> 12;
	
	get040RootPtr (&rootTable);
	
	/*
	 *	First, Check the root table
	 */
	rootEntry = getTableIndex (rootTable, rootIndex);
	switch (rootEntry & 0x0003) {
		case 0x00:
		case 0x01:
			/*
			 *	If the root index is empty ... no lower page tables
			 */
			len = 32 * 1024 * 1024;	//  32Meg
			while (++ rootIndex < 128) {
				rootEntry = getTableIndex (rootTable, rootIndex);
				if ((rootEntry & 0x0003) != 0)
					break;
				len += 32 * 1024 * 1024;
			}
			*physAddr = (u32) NOT_MAPPED;
			*physAttr = 0;
			*physLen = len;
			return;
			break;

		case 0x02:
		case 0x03:
			ptrTable = (u32 *) (rootEntry & 0xFFFFFE00);
			break;
			cprintf ("Unsupported Indirect Page table\n");
			*physAddr = (u32) NOT_MAPPED;
			*physAttr = -1;
			*physLen = -1;
			return;
			break;
	}
	
	tableEntry = getTableIndex (ptrTable, ptrIndex);
	switch (tableEntry & 0x0003) {
		case 0x00:
		case 0x01:
			/*
			 *	If the root index is empty ... no lower page tables
			 */
			len = 256 * 1024;	//  256K
			while (++ ptrIndex < 128) {
				tableEntry = getTableIndex (ptrTable, ptrIndex);
				if ((tableEntry & 0x0003) != 0)
					break;
				len += 256 * 1024;
			}
			*physAddr = (u32) NOT_MAPPED;
			*physAttr = 0;
			*physLen = len;
			return;
			break;

		case 0x02:
		case 0x03:
			if (pageSize8k)	pageTable = (u32 *) (tableEntry & 0xFFFFF80);
			else			pageTable = (u32 *) (tableEntry & 0xFFFFF00);
			break;
	}
	
	pageEntry = getTableIndex (pageTable, pageIndex);
	if (pageSize8k)	pageAddr = (u32 *) (pageEntry & 0xFFFFE000);
	else			pageAddr = (u32 *) (pageEntry & 0xFFFFF000);

	pAttr = pageEntry & 0x000004FF;
	
	if (pageSize8k)	len = 8 * 1024;
	else			len = 4 * 1024;
	
	while (++pageIndex < 64) {
		pageEntry = getTableIndex (pageTable, pageIndex);
		t = pageEntry & 0x4FF;
		if (t != pAttr)
			break;
		
		if (pageSize8k)	len += 8 * 1024;
		else			len += 4 * 1024;
	}
	
	*physAddr = (u32) pageAddr;
	*physAttr = pAttr;
	*physLen = len;
}

/*
 *	getTableIndex
 *
 */
u32 getTableIndex (void *table, int index)
{
	u32		*addr;
	u32		 data;
	
	/*
	 *	Establish the table
	 */
	addr = (u32 *) table;
	data = addr[0];
	addReadData (table, data);
	tableReadData (table);
	
	/*
	 *	Fetch the real data
	 */
	data = addr[index];
	addReadData (&addr[index], data);
	
	byteCount += 4;
	
	return data;
}
#if 0
/*
 *	get040TCReg
 *
 */
void ASMFUNC (get040TCReg, u32 *tc)
{
BEGINASM
	movea.l	8(A6),a0
	dc.w	0x4E7A, 0x0003		/*	movec	%tc, %d0	*/
	move.l	d0,(a0)
ENDASM
}


/*
 *	get040RootPtr
 *
 */
void ASMFUNC (get040RootPtr, void *root)
{
BEGINASM
	movea.l	8(A6),a0
	dc.w	0x4E7A, 0x0807		/*	movec	%supervisor root pointer, %d0	*/
	move.l	d0,(a0)
ENDASM
}

/*
 *	get040ITT0
 *
 */
void ASMFUNC (get040ITT0, void *tt)
{
BEGINASM
	movea.l	8(A6),a0
	dc.w	0x4E7A, 0x0004		/*	movec	%itt0, %d0	*/
	move.l	d0,(a0)
ENDASM
}

/*
 *	get040ITT1
 *
 */
void ASMFUNC (get040ITT1, void *tt)
{
BEGINASM
	movea.l	8(A6),a0
	dc.w	0x4E7A, 0x0005		/*	movec	%itt1, %d0	*/
	move.l	d0,(a0)
ENDASM
}

/*
 *	get040DTT0
 *
 */
void ASMFUNC (get040DTT0, void *tt)
{
BEGINASM
	movea.l	8(A6),a0
	dc.w	0x4E7A, 0x0006		/*	movec	%dtt0, %d0	*/
	move.l	d0,(a0)
ENDASM
}

/*
 *	get040DTT1
 *
 */
void ASMFUNC (get040DTT1, void *tt)
{
BEGINASM
	movea.l	8(A6),a0
	dc.w	0x4E7A, 0x0007		/*	movec	%dtt1, %d0	*/
	move.l	d0,(a0)
ENDASM
}

#endif
