package PLSQL;
#------------------------------------------------------------------------------
# Project  : Oracle to PostgreSQL database schema converter
# Name     : Ora2Pg.pm
# Language : Perl
# Authors  : Gilles Darold, gilles@darold.net
# Copyright: Copyright (c) 2000-2009 : Gilles Darold - All rights reserved -
# Function : Main module used to export Oracle database schema to PostgreSQL
# Usage    : See documentation in this file with perldoc.
#------------------------------------------------------------------------------
# This program is free software; you can redistribute it and/or modify it under
# the same terms as Perl itself.
#------------------------------------------------------------------------------

use vars qw($VERSION);
use POSIX qw(locale_h);

#set locale to LC_NUMERIC C
setlocale(LC_NUMERIC,"C");


$VERSION = '5.5';


=head1 NAME

PSQL - Oracle to PostgreSQL procedural language converter


=head1 SYNOPSIS

	This external perl module is used to convert PLSQL code to PLPGSQL.
	It is in an external code to allow easy editing and modification.
	This converter is a work in progress and need your help.

	It is called internally by Ora2Pg.pm when you set PLSQL_PGSQL 
	configuration option to 1.
=cut

=head2 plsql_to_plpgsql

This function return a PLSQL code translated to PLPGSQL code

=cut

sub plsql_to_plpgsql
{
	my $str = shift;

	#--------------------------------------------
	# PL/SQL to PL/PGSQL code conversion
	# Feel free to add your contribution here.
	#--------------------------------------------
	# Change NVL to COALESCE
	$str =~ s/NVL[\s\t]*\(/coalesce(/igs;
	# Change trunc() to date_trunc('day', field)
	# Trunc is replaced with date_trunc if we find date in the name of
	# the value because Oracle have the same trunc function on number
	# and date type
	$str =~ s/trunc\(([^\)]*date[^\)]*)\)/date_trunc('day', $1)/igs;
	# Change SYSDATE to 'now'
	$str =~ s/SYSDATE/CURRENT_TIMESTAMP/igs;
	# remove FROM DUAL
	$str =~ s/FROM DUAL//igs;
	# Remove leading : on Oracle variable
	$str =~ s/([^\w]+):(\w+)/$1$2/igs;
	# Change nextval on sequence
	# Oracle's sequence grammar is sequence_name.nextval.
	# Postgres's sequence grammar is nextval('sequence_name'). 
	$str =~ s/(\w+)\.nextval/nextval('$1')/isg;
	# Oracle MINUS can be replaced by EXCEPT as is
	$str =~ s/\bMINUS\b/EXCEPT/igs;

	$str =~ s/DBMS_OUTPUT\.(put_line|put|new_line)*([^\)]+\);)/&raise_output($2)/igse;

	# Substitution to replace type of sql variable in PLSQL code
	$str =~ s/\bPROCEDURE\b/FUNCTION/igs;
	$str =~ s/\bNVARCHAR2\b/text/igs;
	$str =~ s/\bVARCHAR2\b/text/igs;
	$str =~ s/\bNCLOB\b/text/igs;
	$str =~ s/\bCLOB\b/text/igs;
	$str =~ s/\bLONG\b/text/igs;
	$str =~ s/\bNCHAR\b/text/igs;
	$str =~ s/\bBLOB\b/bytea/igs;
	$str =~ s/\bRAW\b/bytea/igs;
	$str =~ s/\bLONG RAW\b/bytea/igs;
	$str =~ s/\bNUMBER\b/numeric/igs;
	$str =~ s/\bDEFAULT NULL\b//igs;

	return $str;
}

# Function used to rewrite dbms_output.put, dbms_output.put_line and
# dbms_output.new_line by a plpgsql code
sub raise_output
{
	my $str = shift;

	$str =~ s/\|\|[\s\t]*'([^']*)'/$1/gs;

	my @params = ();
	while ($str =~ s/\|\|[\s\t]*([^\s\t]+)/\%/s) {
		push(@params, $1);
	}
	$str =~ s/'//gs;
	$str =~ s/\(//s;
	$str =~ s/\);//s;
	$str =~ s/[\r\s]+/ /gs;
	my $ret = "RAISE NOTICE '$str'";
	if ($#params >= 0) {
		$ret .= ', ' . join(', ', @params);
	}

	return $ret . ';';
}

1;

__END__


=head1 AUTHOR

Gilles Darold <gilles@darold.net>


=head1 COPYRIGHT

Copyright (c) 2000-2009 Gilles Darold - All rights reserved.

This program is free software; you can redistribute it and/or modify it under
the same terms as Perl itself.


=head1 BUGS

This perl module is in the same state as my knowledge regarding database,
it can move and not be compatible with older version so I will do my best
to give you official support for Ora2Pg. Your volontee to help construct
it and your contribution are welcome.


=head1 SEE ALSO

L<Ora2Pg>

=cut

