/* srttool-2 - Various manipulations of srtfiles
 * Copyright (C) 2002-2006 Arne Driescher <driescher@users.sf.net>
 * Copyright (C) 2006 Olivier Rolland <billl@users.sf.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <getopt.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include <stdio.h>
#include <math.h>

#define TRUE  1
#define FALSE 0

#define MAX_TEXT_LEN 4096

typedef struct
{
  unsigned int hour;
  unsigned int min;
  unsigned int sec;
  unsigned int msec;
} srt_time_t;

typedef struct
{
  unsigned int number;
  srt_time_t start_time;
  srt_time_t stop_time;
  char text[MAX_TEXT_LEN];
} srt_tag_t;

static unsigned int verbose;
static unsigned int renumber;
static unsigned int adjust;
static unsigned int subst;
static unsigned int strip;
static unsigned int help;
static unsigned int dos;

static char input_file[FILENAME_MAX];
static char output_file[FILENAME_MAX];
static char basename_file[FILENAME_MAX];

static double offset = 0.0;
static double factor = 1.0;

static unsigned int first_tag_number = 1;
static unsigned int last_tag_number = ~0;

static srt_time_t adjustment;

static unsigned int
parse_shift (const char *value)
{
  if (sscanf (value, "%lf", &offset) != 1)
  {
    fprintf (stderr, "Invalid parameter for option -d\n");

    return FALSE;
  }

  if (verbose)
    fprintf (stderr, "Using time offset of %lf\n", offset);

  return TRUE;
}

static unsigned int
parse_cut (const char *value)
{
  if (sscanf (value, "%u,%u", &first_tag_number, &last_tag_number) == 0)
  {
    fprintf (stderr, "Invalid parameter for option -c\n");

    return FALSE;
  }

  if (first_tag_number == 0)
  {
    first_tag_number = 1;
    if (verbose)
      fprintf (stderr, "Setting first tag to 1\n");
  }

  if (verbose)
    fprintf (stderr, "Writing from %d to %d entries. \n", first_tag_number, last_tag_number);

  return TRUE;
}

static unsigned int
parse_adjust (const char *value)
{
  if (sscanf (value, "%u:%u:%u,%u", &adjustment.hour, 
        &adjustment.min, &adjustment.sec, &adjustment.msec) != 4)
  {
    fprintf (stderr, "Invalid parameter for option -a\n");

    return FALSE;
  }

  if (verbose)
    fprintf (stderr, "Adjusting to %02d:%02d:%02d,%.3d as start time\n",
        adjustment.hour, adjustment.min, adjustment.sec, adjustment.msec);

  adjust = TRUE;

  return TRUE;
}

static unsigned int
parse_expand (const char *value)
{
  double expansion;

  if (sscanf (value, "%lf", &expansion) != 1)
  {
    fprintf (stderr, "Invalid parameter for option -e\n");

    return FALSE;
  }

  /* complain about an adjustment of more than 60 seconds */
  /*
  if (fabs (expansion > 60))
  {
    fprintf (stderr, "Invalid parameter for option -e\n");

    return FALSE;
  }
  */
  /* calculate the resulting scaling factor */
  factor = (3600.0 + expansion) / 3600.0;

  if (verbose)
    fprintf (stderr, "Using %f seconds for hour expansion\n", expansion);

  return TRUE;
}

static unsigned int
parse_format (const char *value)
{
  if (strcmp (value, "dos") == 0)
    dos = TRUE;
  else if (strcmp (value, "unix") == 0)
    dos = FALSE;
  else
  {
    fprintf (stderr, "Invalid parameter for option -f\n");

    return FALSE;
  }

  if (verbose)
    fprintf (stdout, "Using %s format\n", dos ? "dos" : "unix");

  return TRUE;
}

static int
read_srt_line (FILE *file, char *line, unsigned int len)
{
  if (!fgets (line, len - 1, file))
  {
    if (!feof (file))
    {
      fprintf (stderr, "Unknown error while reading (%s)\n", strerror (errno));
      return -1;
    }

    return 1;
  }

  len = strlen (line);
  if (len >= 1 && line[len - 1] == '\n')
  {
    if (len >= 2 && line[len - 2] == '\r')
      line[len - 2] = 0;
    else
      line[len - 1] = 0;
  }

  return 0;
}

static int
read_srt_time (FILE *file, srt_time_t *start_time, srt_time_t *stop_time)
{
  char line[MAX_TEXT_LEN];

  line[0] = 0;
  if (read_srt_line (file, line, MAX_TEXT_LEN) < 0)
    return -1;

  if (sscanf (line, "%u:%u:%u,%u --> %u:%u:%u,%u",
      &start_time->hour, &start_time->min, &start_time->sec, &start_time->msec,
      &stop_time->hour, &stop_time->min, &stop_time->sec, &stop_time->msec) != 8)
  {
    fprintf (stderr, "Cannot properly convert time stamp (%s)\n", strerror (errno));
    return -1;
  }

  return 0;
}

static int
read_srt_text (FILE *file, char *text)
{
  char line[MAX_TEXT_LEN];

  text[0] = 0;
  while (read_srt_line (file, line, MAX_TEXT_LEN) == 0)
  {
    if (line[0] == 0)
      break;

    strcat (text, line);

    if (dos)
      strcat (text, "\r\n");
    else
      strcat (text, "\n");
  }

  if (text[0] == 0)
  {
    fprintf (stderr, "Empty text or missing newline\n");
    return -1;
  }

  if (dos)
    strcat (text, "\r\n");
  else
    strcat (text, "\n");

  return 0;
}

static int
read_srt_tag (FILE *file, srt_tag_t *tag)
{
  /* read the current number */
  if (fscanf (file, "%u\n", &tag->number) != 1)
  {
    if (!feof (file))
      fprintf (stderr, "Cannot read tag number (%s)\n", strerror (errno));

    return -1;
  }

  /* read the time stamp */
  if (read_srt_time (file, &tag->start_time, &tag->stop_time) < 0)
    return -1;

  /* read the subtitle */
  if (read_srt_text (file, tag->text) < 0)
    return -1;

  return 0;
}

static int
write_srt_line (FILE *file, const char *line)
{
  char *str;

  str = (char *) line;
  if (strip)
  {
    while (*str && isspace (*str))
      str ++;
  }

  fprintf (file, "%s", str);

  return 0;
}

static int
write_srt_time (FILE *file, srt_time_t *start_time, srt_time_t *stop_time)
{
  if (fprintf (file, "%02u:%02u:%02u,%.3u --> %02u:%02u:%02u,%.3u",
      start_time->hour, start_time->min, start_time->sec, start_time->msec,
      stop_time->hour, stop_time->min, stop_time->sec, stop_time->msec) < 0)
  {
    fprintf (stderr, "Cannot properly write time stamp (%s)\n", strerror (errno));
    return -1;
  }

  if (dos)
    fprintf (file, "\r");
  fprintf (file, "\n");

  return 0;
}

static int
write_srt_text (FILE *file, char *text)
{
  if (strlen (text) > 0)
  {
    /* write the text */
    if (write_srt_line (file, text) < 0)
    {
      fprintf (stderr, "Cannot write text (%s)\n", strerror (errno));
      return -1;
    }
  }
  else
  {
    /* empty text fields are not allowed. */
    if (verbose)
      fprintf (stderr, "Empty text line detected\n");

    /* Lets write some stupid text instead. */
    if (write_srt_line (file, "(empty line detected)\n") < 0)
    {
      fprintf (stderr, "Cannot write stupid text (%s)\n", strerror (errno));
      return -1;
    }
  }

  return 0;
}

static int
write_srt_tag (FILE *file, srt_tag_t *tag)
{
  /* write the tag number */
  if (fprintf (file, "%d", tag->number) < 0)
  {
    fprintf (stderr, "Cannot properly write tag number (%s)\n", strerror (errno));
    return -1;
  }

  if (dos)
    fprintf (file, "\r");
  fprintf (file, "\n");

  /* write out the new time stamp */
  if (write_srt_time (file, &tag->start_time, &tag->stop_time) < 0)
    return -1;

  /* write out the subtitle */
  if (write_srt_text (file, tag->text) < 0)
    return -1;

  return 0;
}

static double
srt_time2sec (const srt_time_t *time_stamp)
{
  /* convert time stamp to seconds */
  return time_stamp->hour * 3600 + time_stamp->min * 60 +
    time_stamp->sec + time_stamp->msec / 1000.0;
}

static srt_time_t
sec2srt_time (double t)
{
  srt_time_t time_stamp;

  /* convert seconds to time stamp */
  time_stamp.hour = t / (3600);
  time_stamp.min = (t - 3600 * time_stamp.hour) / 60;
  time_stamp.sec = (t - 3600 * time_stamp.hour - 60 * time_stamp.min);
  time_stamp.msec = (unsigned int) rint ((t - 3600 * time_stamp.hour -
        60 * time_stamp.min - time_stamp.sec) * 1000.0);

  return time_stamp;
}

static void
adjust_srt_time (srt_time_t *time_stamp)
{
  double t;

  t = srt_time2sec (time_stamp);

  /*  adjust time according to offset */
  t += offset;

  /*  don't allow negative time */
  if (t < 0)
    t = 0;

  /*  scale the time */
  t *= factor;

  /*  writeout start and end time of this title */
  *time_stamp = sec2srt_time (t);
}

static int
substitute_txt (srt_tag_t *tag)
{
  char filename[FILENAME_MAX];

  FILE *sub_file;

  if (sscanf (tag->text, "%[^\r\n]", filename) != 1)
  {
    fprintf (stderr, "Cannot read substitution filename (%s)\n", strerror (errno));
    return -1;
  }

  if (verbose)
    fprintf (stderr, "Reading substitution from file '%s'\n", filename);

  sub_file = fopen (filename, "r");
  if (!sub_file)
  {
    fprintf (stderr, "Cannot open file '%s' for substitution (%s)\n", filename, strerror (errno));
    return -1;
  }

  if (read_srt_text (sub_file, tag->text) < 0)
  {
    fclose (sub_file);
    return -1;
  }

  fclose (sub_file);

  return 0;
}

static int
extract_txt (srt_tag_t *tag)
{
  char filename[FILENAME_MAX];

  FILE *ext_file;

  /*  construct the file name from basename and tag number */
  snprintf (filename, FILENAME_MAX - 9, "%s%04d.txt", basename_file, tag->number);

  if (verbose)
    fprintf (stderr, "Extracting tag number %d to file '%s'\n", tag->number, filename);

  ext_file = fopen (filename, "w");
  if (!ext_file)
  {
    fprintf (stderr, "Cannot open file '%s' for extracting (%s)\n", filename, strerror (errno));
    return -1;
  }

  /*  write the text to the file */
  if (write_srt_line (ext_file, tag->text) < 0)
  {
    fprintf (stderr, "Unknown error while extracting (%s)\n", strerror (errno));
    fclose (ext_file);
    return -1;
  }

  fclose (ext_file);

  /*  replace the tag.text by its filename */
  snprintf (tag->text, MAX_TEXT_LEN, "%s\n\n", filename);

  return 0;
}

static void
usage (const char *name)
{
  fprintf (stdout, "Usage:\n");
  fprintf (stdout, "  %s [OPTION...]\n", name);
  fprintf (stdout, "\n");
  fprintf (stdout, "Help Options:\n");
  fprintf (stdout, "  -h, --help                     Show help options\n");
  fprintf (stdout, "\n");
  fprintf (stdout, "Application Options:\n");
  fprintf (stdout, "  -v, --verbose                  Verbose\n");
  fprintf (stdout, "  -r, --renumber                 Renumber all entries\n");
  fprintf (stdout, "  -s, --subst                    Substitute filename after time stamps by its\n");
  fprintf (stdout, "                                 content\n");
  fprintf (stdout, "  -w, --strip                    Remove leading white space in text lines\n");
  fprintf (stdout, "  -d, --shift=<time>             Shift all timestamps by <time> seconds\n");
  fprintf (stdout, "  -c, --cut=<first[,last]>       Write only entries from first to last (default:\n");
  fprintf (stdout, "                                 all entries) to output\n");
  fprintf (stdout, "  -a, --adjust=<hh:mm:ss,ms>     Adjust all time stamps so that the first tag\n");
  fprintf (stdout, "                                 begins at hh:mm:ss,ms\n");
  fprintf (stdout, "  -e, --expand=<seconds>         Expand the subtitle hour by <seconds>\n");
  fprintf (stdout, "  -f, --format=<format>          Write output file in unix (default) or dos\n");
  fprintf (stdout, "                                 format\n");
  fprintf (stdout, "  -x, --extract=<filename>       Reverse -s operation: produce basename.srtx and\n");
  fprintf (stdout, "                                 basenameXXXX.txt files\n");
  fprintf (stdout, "  -i, --input=<filename>         Use filename for input (default: stdin)\n");
  fprintf (stdout, "  -o, --output=<filename>        Use filename for output (default: stdout)\n");
  fprintf (stdout, "\n");
}

static const char *shortopts = "hvrswd:c:a:e:f:x:i:o:";
static const struct option longopts[] =
{
  { "help",     no_argument,       NULL, 'h' },
  { "verbose",  no_argument,       NULL, 'v' },
  { "renumber", no_argument,       NULL, 'r' },
  { "subst",    no_argument,       NULL, 's' },
  { "strip",    no_argument,       NULL, 'w' },
  { "shift",    required_argument, NULL, 'd' }, 
  { "cut",      required_argument, NULL, 'c' },
  { "adjust",   required_argument, NULL, 'a' },
  { "expand",   required_argument, NULL, 'e' },
  { "format",   required_argument, NULL, 'f' },
  { "extract",  required_argument, NULL, 'x' },
  { "input",    required_argument, NULL, 'i' },
  { "output",   required_argument, NULL, 'o' },
  { NULL,       0,                 NULL, 0   }
};

int
main (int argc, char *argv[])
{
  srt_tag_t input_tag, output_tag;

  unsigned int input_counter, output_counter;
  int c, optidx;

  while ((c = getopt_long (argc, argv, shortopts, longopts, &optidx)) != EOF)
  {
    switch (c)
    {
      case 'h':
        help = TRUE;
        break;
      case 'v':
        verbose = TRUE;
        break;
      case 'r':
        renumber = TRUE;
        break;
      case 's':
        subst = TRUE;
        break;
      case 'w':
        strip = TRUE;
        break;
      case 'd':
        parse_shift (optarg);
        break;
      case 'c':
        parse_cut (optarg);
        break;
      case 'a':
        parse_adjust (optarg);
        break;
      case 'e':
        parse_expand (optarg);
        break;
      case 'f':
        parse_format (optarg);
        break;
      case 'x':
        strncpy (basename_file, optarg, FILENAME_MAX);
        break;
      case 'i':
        strncpy (input_file, optarg, FILENAME_MAX);
        break;
      case 'o':
        strncpy (output_file, optarg, FILENAME_MAX);
        break;
      default:
        /* TODO */
        return EXIT_FAILURE;
    }
  }

  if (help)
  {
    usage (argv[0]);
    return EXIT_SUCCESS;
  }

  if (subst && strlen (basename_file) > 0)
  {
    fprintf (stderr, "-s and -x option cannot be used together\n");
    return EXIT_FAILURE;
  }

  if (strlen (input_file) > 0)
  {
    if (!freopen (input_file, "r", stdin))
    {
      fprintf (stderr, "Cannot open file %s for reading (%s)\n", input_file, strerror (errno));
      return EXIT_FAILURE;
    }

    if (verbose)
      fprintf (stderr, "Using %s for input\n", input_file);
  }

  if (strlen (output_file) > 0)
  {
    if (!freopen (output_file, "w", stdout))
    {
      fprintf (stderr, "Cannot open file %s for writing (%s)\n", input_file, strerror (errno));
      return EXIT_FAILURE;
    }

    if (verbose)
      fprintf (stderr, "Using %s for output\n", output_file);
  }

  input_counter = output_counter = 0;

  while (read_srt_tag (stdin, &input_tag) == 0)
  {
    input_counter ++;

    /*  only handle tags in given input range */
    if (input_tag.number < first_tag_number ||
        input_tag.number > last_tag_number)
      continue;
    
    /*  copy the tag */
    output_tag = input_tag;

    /*  adjust tag number if -r option is given */
    if (renumber)
      output_tag.number = output_counter + 1;

    /*  adjust time so that tag starts at time given by -a option */
    if (adjust && output_counter == 0)
    {
      /*  simply add the correct adjustment to time_offset */
      offset += srt_time2sec (&adjustment) -
        srt_time2sec (&output_tag.start_time);
    }

    /*  add/sub the delay given in -d and -a option */
    /*  and scale the time line according to -e option */
    adjust_srt_time (&output_tag.start_time);
    adjust_srt_time (&output_tag.stop_time);

    if (subst)
    {
      /*  substitute the filename in next line by its content */
      substitute_txt (&output_tag);
    }
    else
    {
      if (strlen (basename_file) > 0)
      {
        /*  write out the text input to extract_basenameXXXX.txt */
        /*  and replace the tag.text by the filename */
        extract_txt (&output_tag);
      }
    }

    /*  write out the tag */
    if (write_srt_tag (stdout, &output_tag) < 0)
      return EXIT_FAILURE;

    output_counter ++;
  }

  if (!feof (stdin))
    return EXIT_FAILURE;

  return EXIT_SUCCESS;
}

