/* OGMRip - A library for DVD ripping and encoding
 * Copyright (C) 2004-2007 Olivier Rolland <billl@users.sf.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "ogmrip-lavc.h"
#include "ogmrip-mplayer.h"
#include "ogmrip-version.h"
#include "ogmrip-plugin.h"
#include "ogmrip-file.h"

#include "ogmjob-exec.h"

#include <stdio.h>
#include <glib/gi18n-lib.h>

#define OGMRIP_LAVC_VERY_HIGH_OPTIONS "vcodec=mpeg4:autoaspect:mbd=2:vb_strategy=1:last_pred=3:mv0:preme=2:qns=2"
#define OGMRIP_LAVC_HIGH_OPTIONS      "vcodec=mpeg4:autoaspect:mbd=2:vb_strategy=1:last_pred=2:vqcomp=0.6"
#define OGMRIP_LAVC_FAST_OPTIONS      "vcodec=mpeg4:autoaspect:mbd=2"

#define OGMRIP_LAVC_GET_PRIVATE(o) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((o), OGMRIP_TYPE_LAVC, OGMRipLavcPriv))

struct _OGMRipLavcPriv
{
  guint header;
  guint cmp, precmp, subcmp;
  gint dia, predia;
};

static gint ogmrip_lavc_run (OGMJobSpawn *spawn);

static gchar **
ogmrip_lavc_command (OGMRipVideo *video, const gchar *input, const gchar *output, const gchar *logf)
{
  OGMDvdTitle *title;
  GPtrArray *argv;
  GString *options;

  gint bitrate, vid, pass, dia, predia, header, threads;
  guint cmp, precmp, subcmp;

  g_return_val_if_fail (OGMRIP_IS_VIDEO (video), NULL);

  if (!output)
    output = ogmrip_codec_get_output (OGMRIP_CODEC (video));
  g_return_val_if_fail (output != NULL, NULL);

  title = ogmrip_codec_get_input (OGMRIP_CODEC (video));
  g_return_val_if_fail (title != NULL, NULL);

  pass = ogmrip_video_get_pass (video);
  if (pass > 0 && !logf)
    logf = ogmrip_video_get_log (video);
  g_return_val_if_fail (pass == 0 || logf != NULL, NULL);

  argv = ogmrip_mencoder_video_command (video, title, pass == 1 ? "/dev/null" : output);

  g_ptr_array_add (argv, g_strdup ("-ovc"));
  g_ptr_array_add (argv, g_strdup ("lavc"));

  switch (ogmrip_video_get_quality (video))
  {
    case OGMRIP_QUALITY_VERY_HIGH:
      options = g_string_new (OGMRIP_LAVC_VERY_HIGH_OPTIONS);
      break;
    case OGMRIP_QUALITY_HIGH:
      options = g_string_new (OGMRIP_LAVC_HIGH_OPTIONS);
      break;
    default:
      options = g_string_new (OGMRIP_LAVC_FAST_OPTIONS);
      break;
  }

#if MPLAYER_CHECK_VERSION(1,0,0,6)
  if (ogmrip_video_get_turbo (video))
    g_string_append (options, ":turbo");
#endif /* MPLAYER_CHECK_VERSION(1,0,0,6) */

  if (ogmrip_video_get_qpel (video))
    g_string_append (options, ":qpel");
  if (ogmrip_video_get_4mv (video))
    g_string_append (options, ":v4mv");
  if (ogmrip_video_get_trellis (video))
    g_string_append (options, ":trell:cbp");
  if (ogmrip_video_get_grayscale (video))
    g_string_append (options, ":gray");

  ogmrip_lavc_get_cmp (OGMRIP_LAVC (video), &cmp, &precmp, &subcmp);
  g_string_append_printf (options, ":precmp=%u:subcmp=%u:cmp=%u", precmp, subcmp, cmp);

  ogmrip_lavc_get_dia (OGMRIP_LAVC (video), &dia, &predia);
  g_string_append_printf (options, ":dia=%d:predia=%d", dia, predia);

  header = ogmrip_lavc_get_header (OGMRIP_LAVC (video));
  g_string_append_printf (options, ":vglobal=%d", header);

  g_string_append_printf (options, ":vmax_b_frames=%d", ogmrip_video_get_max_b_frames (video));

  bitrate = ogmrip_video_get_bitrate (video);
  if (bitrate > 0)
    g_string_append_printf (options, ":vbitrate=%u", bitrate);
  else
    g_string_append_printf (options, ":vqscale=%.0lf", ogmrip_video_get_quantizer (video));

  if (pass)
  {
    g_string_append_printf (options, ":vpass=%u", pass);
    g_ptr_array_add (argv, g_strdup ("-passlogfile"));
    g_ptr_array_add (argv, g_strdup (logf));
  }

  threads = ogmrip_video_get_threads (video);
  if (threads > 1)
    g_string_append_printf (options, ":threads=%u", CLAMP (threads, 1, 8));

  g_ptr_array_add (argv, g_strdup ("-lavcopts"));
  g_ptr_array_add (argv, g_string_free (options, FALSE));

  vid = ogmdvd_title_get_nr (title);

#if MPLAYER_CHECK_VERSION(1,0,0,1)
  g_ptr_array_add (argv, g_strdup_printf ("dvd://%d", vid + 1));
#else /* MPLAYER_CHECK_VERSION(1,0,0,1) */
  g_ptr_array_add (argv, g_strdup ("-dvd"));
  g_ptr_array_add (argv, g_strdup_printf ("%d", vid + 1));
#endif /* MPLAYER_CHECK_VERSION(1,0,0,1) */

  g_ptr_array_add (argv, NULL);

  return (gchar **) g_ptr_array_free (argv, FALSE);
}

G_DEFINE_TYPE (OGMRipLavc, ogmrip_lavc, OGMRIP_TYPE_VIDEO)

static void
ogmrip_lavc_class_init (OGMRipLavcClass *klass)
{
  OGMJobSpawnClass *spawn_class;

  spawn_class = OGMJOB_SPAWN_CLASS (klass);

  spawn_class->run = ogmrip_lavc_run;

  g_type_class_add_private (klass, sizeof (OGMRipLavcPriv));
}

static void
ogmrip_lavc_init (OGMRipLavc *lavc)
{
  lavc->priv = OGMRIP_LAVC_GET_PRIVATE (lavc);
  lavc->priv->cmp = 2;
  lavc->priv->precmp = 2;
  lavc->priv->subcmp = 2;
  lavc->priv->dia = 2;
  lavc->priv->predia = 2;
}

static gint
ogmrip_lavc_run (OGMJobSpawn *spawn)
{
  OGMJobSpawn *child;
  gchar **argv;
  gint result;

  argv = ogmrip_lavc_command (OGMRIP_VIDEO (spawn), NULL, NULL, NULL);
  if (!argv)
    return OGMJOB_RESULT_ERROR;

  child = ogmjob_exec_newv (argv);
  ogmjob_exec_add_watch_full (OGMJOB_EXEC (child), (OGMJobWatch) ogmrip_mencoder_codec_watch, spawn, TRUE, FALSE, FALSE);
  ogmjob_container_add (OGMJOB_CONTAINER (spawn), child);
  g_object_unref (child);

  result = OGMJOB_SPAWN_CLASS (ogmrip_lavc_parent_class)->run (spawn);

  ogmjob_container_remove (OGMJOB_CONTAINER (spawn), child);

  return result;
}

/**
 * ogmrip_lavc_new:
 * @title: An #OGMDvdTitle
 * @output: The output file
 *
 * Creates a new #OGMRipLavc
 *
 * Returns: The new #OGMRipLavc
 */
OGMJobSpawn *
ogmrip_lavc_new (OGMDvdTitle *title, const gchar *output)
{
  g_return_val_if_fail (title != NULL, NULL);
  g_return_val_if_fail (output && *output, NULL);

  return g_object_new (OGMRIP_TYPE_LAVC, "input", title, "output", output, NULL);
}

/**
 * ogmrip_lavc_set_cmp:
 * @lavc: An #OGMRipLavc
 * @cmp: The comparison function for full pel motion estimation
 * @precmp: The comparison function for motion estimation pre pass
 * @subcmp: The comparison function for sub pel motion estimation
 *
 * Sets the comparison function for full pel, pre pass and sub pel motion estimation
 */
void
ogmrip_lavc_set_cmp (OGMRipLavc *lavc, guint cmp, guint precmp, guint subcmp)
{
  g_return_if_fail (OGMRIP_IS_LAVC (lavc));

  lavc->priv->cmp = MIN (cmp, 2000);
  lavc->priv->precmp = MIN (precmp, 2000);
  lavc->priv->subcmp = MIN (subcmp, 2000);
}

/**
 * ogmrip_lavc_get_cmp:
 * @lavc: An #OGMRipLavc
 * @cmp: A pointer to store the comparison function for full pel motion estimation
 * @precmp: A pointer to store the comparison function for motion estimation pre pass
 * @subcmp: A pointer to store the comparison function for sub pel motion estimation
 *
 * Gets the comparison function for full pel, pre pass and sub pel motion estimation
 */
void
ogmrip_lavc_get_cmp (OGMRipLavc *lavc, guint *cmp, guint *precmp, guint *subcmp)
{
  g_return_if_fail (OGMRIP_IS_LAVC (lavc));
  g_return_if_fail (precmp != NULL);
  g_return_if_fail (subcmp != NULL);
  g_return_if_fail (cmp != NULL);

  *cmp = lavc->priv->cmp;
  *precmp = lavc->priv->precmp;
  *subcmp = lavc->priv->subcmp;
}

/**
 * ogmrip_lavc_set_dia:
 * @lavc: An #OGMRipLavc
 * @dia: The diamond type and size for full pel motion estimation
 * @predia: The diamond type and size for motion estimation pre-pass
 *
 * Sets the diamond type and size for full pel and pre pass motion estimation
 */
void
ogmrip_lavc_set_dia (OGMRipLavc *lavc, gint dia, gint predia)
{
  g_return_if_fail (OGMRIP_IS_LAVC (lavc));

  lavc->priv->dia = CLAMP (dia, -99, 6);
  lavc->priv->predia = CLAMP (predia, -99, 6);
}

/**
 * ogmrip_lavc_get_dia:
 * @lavc: An #OGMRipLavc
 * @dia: A pointer to store the diamond type and size for full pel motion estimation
 * @predia: A pointer to store the diamond type and size for motion estimation pre-pass
 *
 * Gets the diamond type and size for full pel and pre pass motion estimation
 */
void
ogmrip_lavc_get_dia (OGMRipLavc *lavc, gint *dia, gint *predia)
{
  g_return_if_fail (OGMRIP_IS_LAVC (lavc));
  g_return_if_fail (predia != NULL);
  g_return_if_fail (dia != NULL);

  *dia = lavc->priv->dia;
  *predia = lavc->priv->predia;
}

/**
 * ogmrip_lavc_set_header:
 * @lavc: An #OGMRipLavc
 * @header: The #OGMRipLavcHeaderType
 *
 * Sets the global video header type.
 */
void
ogmrip_lavc_set_header (OGMRipLavc *lavc, OGMRipLavcHeaderType header)
{
  g_return_if_fail (OGMRIP_IS_LAVC (lavc));

  lavc->priv->header = CLAMP (header, OGMRIP_LAVC_HEADER_AUTO, OGMRIP_LAVC_HEADER_COMBINE);
}

/**
 * ogmrip_lavc_get_header:
 * @lavc: An #OGMRipLavc
 *
 * Gets the global video header type.
 *
 * Returns: The current #OGMRipLavcHeaderType, or -1
 */
gint
ogmrip_lavc_get_header (OGMRipLavc *lavc)
{
  g_return_val_if_fail (OGMRIP_IS_LAVC (lavc), -1);

  return lavc->priv->header;
}

static OGMRipPluginVideoCodec lavc_plugin =
{
  NULL,
  G_TYPE_NONE,
  "lavc",
  N_("Lavc Mpeg-4"),
  OGMRIP_FORMAT_MPEG4,
  G_MAXINT,
  8
};

OGMRipPluginVideoCodec *
ogmrip_init_plugin (void)
{
  lavc_plugin.type = OGMRIP_TYPE_LAVC;

  return &lavc_plugin;
}

