#include "loader.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifdef _WIN32
# ifndef WIN32_LEAN_AND_MEAN
#   define WIN32_LEAN_AND_MEAN 1
# endif
# ifndef NOGDI
#   define NOGDI 1
# endif
# include <windows.h>
#elif defined(__APPLE__)
# include <AvailabilityMacros.h>
# ifndef MAC_OS_X_VERSION_10_3
#   include <mach-o/dyld.h>
# else
#   include <dlfcn.h>
# endif
#else
# define __USE_GNU 1
# define _GNU_SOURCE
# include <dlfcn.h>
#endif

LHandle
lLoadLibrary(const char *filename)
{
#ifdef __APPLE__
  char *framework = NULL;

  /* 1. Try loading bare framework name */

# if defined(MAC_OS_X_VERSION_10_3)
  LHandle ret = (LHandle) dlopen(filename, RTLD_LAZY);
# else
  LHandle ret = (LHandle) NSAddImage(filename, NSADDIMAGE_OPTION_NONE);
# endif
  if (ret)
    return ret;

  /* 2. Try loading framework in Name.framework/Name */

  framework = (char *) malloc(strlen(filename)*2+12);
  sprintf(framework,"%s.framework/%s",filename,filename);

# if defined(MAC_OS_X_VERSION_10_3)
  ret = (LHandle) dlopen(framework, RTLD_LAZY);
# else
  ret = (LHandle) NSAddImage(framework, NSADDIMAGE_OPTION_NONE);
# endif

  free(framework);
  return ret;

#elif _WIN32
  return (LHandle) LoadLibrary(filename);
#else
  return (LHandle) dlopen(filename, RTLD_LAZY);
#endif
}

int
lFreeLibrary(LHandle handle)
{
#ifdef _WIN32
  return FreeLibrary((HMODULE)handle) ? 1 : 0;
#elif defined(__APPLE__) && !defined(MAC_OS_X_VERSION_10_3)
  return 1;
#else
  return dlclose(handle) ? 0 : 1;
#endif
}

void *
lGetProcAddress(LHandle handle, const char *name)
{
#ifdef _WIN32
  return handle ? GetProcAddress((HMODULE)handle, name) : NULL;
#elif defined(__APPLE__) && !defined(MAC_OS_X_VERSION_10_3)
  return NSIsSymbolNameDefined(name) ? NSAddressOfSymbol(NSLookupAndBindSymbol(name)) : NULL;
#else
  return dlsym(handle, name);
#endif
}

const char *
lGetModuleFileName(LHandle handle, void *function)
{
#ifdef _WIN32
  static char buffer[4096];
  if (GetModuleFileName((HMODULE)handle, buffer, 4095))
  {
    buffer[4095] = '\0';
    return buffer;
  }
#elif defined(__APPLE__) && !defined(MAC_OS_X_VERSION_10_3)
#else
  Dl_info info;
  if (dladdr(function,&info))
    return info.dli_fname;
#endif
  return NULL;
}

/* Entry Points */

PFNCGSETLOCKINGPOLICYPROC cgSetLockingPolicy = NULL;
PFNCGGETLOCKINGPOLICYPROC cgGetLockingPolicy = NULL;
PFNCGSETSEMANTICCASEPOLICYPROC cgSetSemanticCasePolicy = NULL;
PFNCGGETSEMANTICCASEPOLICYPROC cgGetSemanticCasePolicy = NULL;
PFNCGCREATECONTEXTPROC cgCreateContext = NULL;
PFNCGSETCONTEXTBEHAVIORPROC cgSetContextBehavior = NULL;
PFNCGGETCONTEXTBEHAVIORPROC cgGetContextBehavior = NULL;
PFNCGGETBEHAVIORSTRINGPROC cgGetBehaviorString = NULL;
PFNCGGETBEHAVIORPROC cgGetBehavior = NULL;
PFNCGDESTROYCONTEXTPROC cgDestroyContext = NULL;
PFNCGISCONTEXTPROC cgIsContext = NULL;
PFNCGGETLASTLISTINGPROC cgGetLastListing = NULL;
PFNCGSETLASTLISTINGPROC cgSetLastListing = NULL;
PFNCGSETAUTOCOMPILEPROC cgSetAutoCompile = NULL;
PFNCGGETAUTOCOMPILEPROC cgGetAutoCompile = NULL;
PFNCGSETPARAMETERSETTINGMODEPROC cgSetParameterSettingMode = NULL;
PFNCGGETPARAMETERSETTINGMODEPROC cgGetParameterSettingMode = NULL;
PFNCGSETCOMPILERINCLUDESTRINGPROC cgSetCompilerIncludeString = NULL;
PFNCGSETCOMPILERINCLUDEFILEPROC cgSetCompilerIncludeFile = NULL;
PFNCGSETCOMPILERINCLUDECALLBACKPROC cgSetCompilerIncludeCallback = NULL;
PFNCGGETCOMPILERINCLUDECALLBACKPROC cgGetCompilerIncludeCallback = NULL;
PFNCGCREATEPROGRAMPROC cgCreateProgram = NULL;
PFNCGCREATEPROGRAMFROMFILEPROC cgCreateProgramFromFile = NULL;
PFNCGCOPYPROGRAMPROC cgCopyProgram = NULL;
PFNCGDESTROYPROGRAMPROC cgDestroyProgram = NULL;
PFNCGGETFIRSTPROGRAMPROC cgGetFirstProgram = NULL;
PFNCGGETNEXTPROGRAMPROC cgGetNextProgram = NULL;
PFNCGGETPROGRAMCONTEXTPROC cgGetProgramContext = NULL;
PFNCGISPROGRAMPROC cgIsProgram = NULL;
PFNCGCOMPILEPROGRAMPROC cgCompileProgram = NULL;
PFNCGISPROGRAMCOMPILEDPROC cgIsProgramCompiled = NULL;
PFNCGGETPROGRAMSTRINGPROC cgGetProgramString = NULL;
PFNCGGETPROGRAMPROFILEPROC cgGetProgramProfile = NULL;
PFNCGGETPROGRAMOPTIONSPROC cgGetProgramOptions = NULL;
PFNCGSETPROGRAMPROFILEPROC cgSetProgramProfile = NULL;
PFNCGGETPROGRAMINPUTPROC cgGetProgramInput = NULL;
PFNCGGETPROGRAMOUTPUTPROC cgGetProgramOutput = NULL;
PFNCGSETPASSPROGRAMPARAMETERSPROC cgSetPassProgramParameters = NULL;
PFNCGUPDATEPROGRAMPARAMETERSPROC cgUpdateProgramParameters = NULL;
PFNCGUPDATEPASSPARAMETERSPROC cgUpdatePassParameters = NULL;
PFNCGCREATEPARAMETERPROC cgCreateParameter = NULL;
PFNCGCREATEPARAMETERARRAYPROC cgCreateParameterArray = NULL;
PFNCGCREATEPARAMETERMULTIDIMARRAYPROC cgCreateParameterMultiDimArray = NULL;
PFNCGDESTROYPARAMETERPROC cgDestroyParameter = NULL;
PFNCGCONNECTPARAMETERPROC cgConnectParameter = NULL;
PFNCGDISCONNECTPARAMETERPROC cgDisconnectParameter = NULL;
PFNCGGETCONNECTEDPARAMETERPROC cgGetConnectedParameter = NULL;
PFNCGGETNUMCONNECTEDTOPARAMETERSPROC cgGetNumConnectedToParameters = NULL;
PFNCGGETCONNECTEDTOPARAMETERPROC cgGetConnectedToParameter = NULL;
PFNCGGETNAMEDPARAMETERPROC cgGetNamedParameter = NULL;
PFNCGGETNAMEDPROGRAMPARAMETERPROC cgGetNamedProgramParameter = NULL;
PFNCGGETFIRSTPARAMETERPROC cgGetFirstParameter = NULL;
PFNCGGETNEXTPARAMETERPROC cgGetNextParameter = NULL;
PFNCGGETFIRSTLEAFPARAMETERPROC cgGetFirstLeafParameter = NULL;
PFNCGGETNEXTLEAFPARAMETERPROC cgGetNextLeafParameter = NULL;
PFNCGGETFIRSTSTRUCTPARAMETERPROC cgGetFirstStructParameter = NULL;
PFNCGGETNAMEDSTRUCTPARAMETERPROC cgGetNamedStructParameter = NULL;
PFNCGGETFIRSTDEPENDENTPARAMETERPROC cgGetFirstDependentParameter = NULL;
PFNCGGETARRAYPARAMETERPROC cgGetArrayParameter = NULL;
PFNCGGETARRAYDIMENSIONPROC cgGetArrayDimension = NULL;
PFNCGGETARRAYTYPEPROC cgGetArrayType = NULL;
PFNCGGETARRAYSIZEPROC cgGetArraySize = NULL;
PFNCGGETARRAYTOTALSIZEPROC cgGetArrayTotalSize = NULL;
PFNCGSETARRAYSIZEPROC cgSetArraySize = NULL;
PFNCGSETMULTIDIMARRAYSIZEPROC cgSetMultiDimArraySize = NULL;
PFNCGGETPARAMETERPROGRAMPROC cgGetParameterProgram = NULL;
PFNCGGETPARAMETERCONTEXTPROC cgGetParameterContext = NULL;
PFNCGISPARAMETERPROC cgIsParameter = NULL;
PFNCGGETPARAMETERNAMEPROC cgGetParameterName = NULL;
PFNCGGETPARAMETERTYPEPROC cgGetParameterType = NULL;
PFNCGGETPARAMETERBASETYPEPROC cgGetParameterBaseType = NULL;
PFNCGGETPARAMETERCLASSPROC cgGetParameterClass = NULL;
PFNCGGETPARAMETERROWSPROC cgGetParameterRows = NULL;
PFNCGGETPARAMETERCOLUMNSPROC cgGetParameterColumns = NULL;
PFNCGGETPARAMETERNAMEDTYPEPROC cgGetParameterNamedType = NULL;
PFNCGGETPARAMETERSEMANTICPROC cgGetParameterSemantic = NULL;
PFNCGGETPARAMETERRESOURCEPROC cgGetParameterResource = NULL;
PFNCGGETPARAMETERBASERESOURCEPROC cgGetParameterBaseResource = NULL;
PFNCGGETPARAMETERRESOURCEINDEXPROC cgGetParameterResourceIndex = NULL;
PFNCGGETPARAMETERVARIABILITYPROC cgGetParameterVariability = NULL;
PFNCGGETPARAMETERDIRECTIONPROC cgGetParameterDirection = NULL;
PFNCGISPARAMETERREFERENCEDPROC cgIsParameterReferenced = NULL;
PFNCGISPARAMETERUSEDPROC cgIsParameterUsed = NULL;
PFNCGGETPARAMETERVALUESPROC cgGetParameterValues = NULL;
PFNCGSETPARAMETERVALUEDRPROC cgSetParameterValuedr = NULL;
PFNCGSETPARAMETERVALUEDCPROC cgSetParameterValuedc = NULL;
PFNCGSETPARAMETERVALUEFRPROC cgSetParameterValuefr = NULL;
PFNCGSETPARAMETERVALUEFCPROC cgSetParameterValuefc = NULL;
PFNCGSETPARAMETERVALUEIRPROC cgSetParameterValueir = NULL;
PFNCGSETPARAMETERVALUEICPROC cgSetParameterValueic = NULL;
PFNCGGETPARAMETERVALUEDRPROC cgGetParameterValuedr = NULL;
PFNCGGETPARAMETERVALUEDCPROC cgGetParameterValuedc = NULL;
PFNCGGETPARAMETERVALUEFRPROC cgGetParameterValuefr = NULL;
PFNCGGETPARAMETERVALUEFCPROC cgGetParameterValuefc = NULL;
PFNCGGETPARAMETERVALUEIRPROC cgGetParameterValueir = NULL;
PFNCGGETPARAMETERVALUEICPROC cgGetParameterValueic = NULL;
PFNCGGETPARAMETERDEFAULTVALUEDRPROC cgGetParameterDefaultValuedr = NULL;
PFNCGGETPARAMETERDEFAULTVALUEDCPROC cgGetParameterDefaultValuedc = NULL;
PFNCGGETPARAMETERDEFAULTVALUEFRPROC cgGetParameterDefaultValuefr = NULL;
PFNCGGETPARAMETERDEFAULTVALUEFCPROC cgGetParameterDefaultValuefc = NULL;
PFNCGGETPARAMETERDEFAULTVALUEIRPROC cgGetParameterDefaultValueir = NULL;
PFNCGGETPARAMETERDEFAULTVALUEICPROC cgGetParameterDefaultValueic = NULL;
PFNCGGETSTRINGPARAMETERVALUEPROC cgGetStringParameterValue = NULL;
PFNCGSETSTRINGPARAMETERVALUEPROC cgSetStringParameterValue = NULL;
PFNCGGETPARAMETERORDINALNUMBERPROC cgGetParameterOrdinalNumber = NULL;
PFNCGISPARAMETERGLOBALPROC cgIsParameterGlobal = NULL;
PFNCGGETPARAMETERINDEXPROC cgGetParameterIndex = NULL;
PFNCGSETPARAMETERVARIABILITYPROC cgSetParameterVariability = NULL;
PFNCGSETPARAMETERSEMANTICPROC cgSetParameterSemantic = NULL;
PFNCGSETPARAMETER1FPROC cgSetParameter1f = NULL;
PFNCGSETPARAMETER2FPROC cgSetParameter2f = NULL;
PFNCGSETPARAMETER3FPROC cgSetParameter3f = NULL;
PFNCGSETPARAMETER4FPROC cgSetParameter4f = NULL;
PFNCGSETPARAMETER1DPROC cgSetParameter1d = NULL;
PFNCGSETPARAMETER2DPROC cgSetParameter2d = NULL;
PFNCGSETPARAMETER3DPROC cgSetParameter3d = NULL;
PFNCGSETPARAMETER4DPROC cgSetParameter4d = NULL;
PFNCGSETPARAMETER1IPROC cgSetParameter1i = NULL;
PFNCGSETPARAMETER2IPROC cgSetParameter2i = NULL;
PFNCGSETPARAMETER3IPROC cgSetParameter3i = NULL;
PFNCGSETPARAMETER4IPROC cgSetParameter4i = NULL;
PFNCGSETPARAMETER1IVPROC cgSetParameter1iv = NULL;
PFNCGSETPARAMETER2IVPROC cgSetParameter2iv = NULL;
PFNCGSETPARAMETER3IVPROC cgSetParameter3iv = NULL;
PFNCGSETPARAMETER4IVPROC cgSetParameter4iv = NULL;
PFNCGSETPARAMETER1FVPROC cgSetParameter1fv = NULL;
PFNCGSETPARAMETER2FVPROC cgSetParameter2fv = NULL;
PFNCGSETPARAMETER3FVPROC cgSetParameter3fv = NULL;
PFNCGSETPARAMETER4FVPROC cgSetParameter4fv = NULL;
PFNCGSETPARAMETER1DVPROC cgSetParameter1dv = NULL;
PFNCGSETPARAMETER2DVPROC cgSetParameter2dv = NULL;
PFNCGSETPARAMETER3DVPROC cgSetParameter3dv = NULL;
PFNCGSETPARAMETER4DVPROC cgSetParameter4dv = NULL;
PFNCGSETMATRIXPARAMETERIRPROC cgSetMatrixParameterir = NULL;
PFNCGSETMATRIXPARAMETERDRPROC cgSetMatrixParameterdr = NULL;
PFNCGSETMATRIXPARAMETERFRPROC cgSetMatrixParameterfr = NULL;
PFNCGSETMATRIXPARAMETERICPROC cgSetMatrixParameteric = NULL;
PFNCGSETMATRIXPARAMETERDCPROC cgSetMatrixParameterdc = NULL;
PFNCGSETMATRIXPARAMETERFCPROC cgSetMatrixParameterfc = NULL;
PFNCGGETMATRIXPARAMETERIRPROC cgGetMatrixParameterir = NULL;
PFNCGGETMATRIXPARAMETERDRPROC cgGetMatrixParameterdr = NULL;
PFNCGGETMATRIXPARAMETERFRPROC cgGetMatrixParameterfr = NULL;
PFNCGGETMATRIXPARAMETERICPROC cgGetMatrixParameteric = NULL;
PFNCGGETMATRIXPARAMETERDCPROC cgGetMatrixParameterdc = NULL;
PFNCGGETMATRIXPARAMETERFCPROC cgGetMatrixParameterfc = NULL;
PFNCGGETMATRIXPARAMETERORDERPROC cgGetMatrixParameterOrder = NULL;
PFNCGGETNAMEDSUBPARAMETERPROC cgGetNamedSubParameter = NULL;
PFNCGGETTYPESTRINGPROC cgGetTypeString = NULL;
PFNCGGETTYPEPROC cgGetType = NULL;
PFNCGGETNAMEDUSERTYPEPROC cgGetNamedUserType = NULL;
PFNCGGETNUMUSERTYPESPROC cgGetNumUserTypes = NULL;
PFNCGGETUSERTYPEPROC cgGetUserType = NULL;
PFNCGGETNUMPARENTTYPESPROC cgGetNumParentTypes = NULL;
PFNCGGETPARENTTYPEPROC cgGetParentType = NULL;
PFNCGISPARENTTYPEPROC cgIsParentType = NULL;
PFNCGISINTERFACETYPEPROC cgIsInterfaceType = NULL;
PFNCGGETRESOURCESTRINGPROC cgGetResourceString = NULL;
PFNCGGETRESOURCEPROC cgGetResource = NULL;
PFNCGGETENUMSTRINGPROC cgGetEnumString = NULL;
PFNCGGETENUMPROC cgGetEnum = NULL;
PFNCGGETPROFILESTRINGPROC cgGetProfileString = NULL;
PFNCGGETPROFILEPROC cgGetProfile = NULL;
PFNCGGETNUMSUPPORTEDPROFILESPROC cgGetNumSupportedProfiles = NULL;
PFNCGGETSUPPORTEDPROFILEPROC cgGetSupportedProfile = NULL;
PFNCGISPROFILESUPPORTEDPROC cgIsProfileSupported = NULL;
PFNCGGETPROFILEPROPERTYPROC cgGetProfileProperty = NULL;
PFNCGGETPARAMETERCLASSSTRINGPROC cgGetParameterClassString = NULL;
PFNCGGETPARAMETERCLASSENUMPROC cgGetParameterClassEnum = NULL;
PFNCGGETDOMAINSTRINGPROC cgGetDomainString = NULL;
PFNCGGETDOMAINPROC cgGetDomain = NULL;
PFNCGGETPROGRAMDOMAINPROC cgGetProgramDomain = NULL;
PFNCGGETERRORPROC cgGetError = NULL;
PFNCGGETFIRSTERRORPROC cgGetFirstError = NULL;
PFNCGGETERRORSTRINGPROC cgGetErrorString = NULL;
PFNCGGETLASTERRORSTRINGPROC cgGetLastErrorString = NULL;
PFNCGSETERRORCALLBACKPROC cgSetErrorCallback = NULL;
PFNCGGETERRORCALLBACKPROC cgGetErrorCallback = NULL;
PFNCGSETERRORHANDLERPROC cgSetErrorHandler = NULL;
PFNCGGETERRORHANDLERPROC cgGetErrorHandler = NULL;
PFNCGGETSTRINGPROC cgGetString = NULL;
PFNCGCREATEEFFECTPROC cgCreateEffect = NULL;
PFNCGCREATEEFFECTFROMFILEPROC cgCreateEffectFromFile = NULL;
PFNCGCOPYEFFECTPROC cgCopyEffect = NULL;
PFNCGDESTROYEFFECTPROC cgDestroyEffect = NULL;
PFNCGGETEFFECTCONTEXTPROC cgGetEffectContext = NULL;
PFNCGISEFFECTPROC cgIsEffect = NULL;
PFNCGGETFIRSTEFFECTPROC cgGetFirstEffect = NULL;
PFNCGGETNEXTEFFECTPROC cgGetNextEffect = NULL;
PFNCGCREATEPROGRAMFROMEFFECTPROC cgCreateProgramFromEffect = NULL;
PFNCGGETFIRSTTECHNIQUEPROC cgGetFirstTechnique = NULL;
PFNCGGETNEXTTECHNIQUEPROC cgGetNextTechnique = NULL;
PFNCGGETNAMEDTECHNIQUEPROC cgGetNamedTechnique = NULL;
PFNCGGETTECHNIQUENAMEPROC cgGetTechniqueName = NULL;
PFNCGISTECHNIQUEPROC cgIsTechnique = NULL;
PFNCGVALIDATETECHNIQUEPROC cgValidateTechnique = NULL;
PFNCGISTECHNIQUEVALIDATEDPROC cgIsTechniqueValidated = NULL;
PFNCGGETTECHNIQUEEFFECTPROC cgGetTechniqueEffect = NULL;
PFNCGGETFIRSTPASSPROC cgGetFirstPass = NULL;
PFNCGGETNAMEDPASSPROC cgGetNamedPass = NULL;
PFNCGGETNEXTPASSPROC cgGetNextPass = NULL;
PFNCGISPASSPROC cgIsPass = NULL;
PFNCGGETPASSNAMEPROC cgGetPassName = NULL;
PFNCGGETPASSTECHNIQUEPROC cgGetPassTechnique = NULL;
PFNCGGETPASSPROGRAMPROC cgGetPassProgram = NULL;
PFNCGSETPASSSTATEPROC cgSetPassState = NULL;
PFNCGRESETPASSSTATEPROC cgResetPassState = NULL;
PFNCGGETFIRSTSTATEASSIGNMENTPROC cgGetFirstStateAssignment = NULL;
PFNCGGETNAMEDSTATEASSIGNMENTPROC cgGetNamedStateAssignment = NULL;
PFNCGGETNEXTSTATEASSIGNMENTPROC cgGetNextStateAssignment = NULL;
PFNCGISSTATEASSIGNMENTPROC cgIsStateAssignment = NULL;
PFNCGCALLSTATESETCALLBACKPROC cgCallStateSetCallback = NULL;
PFNCGCALLSTATEVALIDATECALLBACKPROC cgCallStateValidateCallback = NULL;
PFNCGCALLSTATERESETCALLBACKPROC cgCallStateResetCallback = NULL;
PFNCGGETSTATEASSIGNMENTPASSPROC cgGetStateAssignmentPass = NULL;
PFNCGGETSAMPLERSTATEASSIGNMENTPARAMETERPROC cgGetSamplerStateAssignmentParameter = NULL;
PFNCGGETFLOATSTATEASSIGNMENTVALUESPROC cgGetFloatStateAssignmentValues = NULL;
PFNCGGETINTSTATEASSIGNMENTVALUESPROC cgGetIntStateAssignmentValues = NULL;
PFNCGGETBOOLSTATEASSIGNMENTVALUESPROC cgGetBoolStateAssignmentValues = NULL;
PFNCGGETSTRINGSTATEASSIGNMENTVALUEPROC cgGetStringStateAssignmentValue = NULL;
PFNCGGETPROGRAMSTATEASSIGNMENTVALUEPROC cgGetProgramStateAssignmentValue = NULL;
PFNCGGETTEXTURESTATEASSIGNMENTVALUEPROC cgGetTextureStateAssignmentValue = NULL;
PFNCGGETSAMPLERSTATEASSIGNMENTVALUEPROC cgGetSamplerStateAssignmentValue = NULL;
PFNCGGETSTATEASSIGNMENTINDEXPROC cgGetStateAssignmentIndex = NULL;
PFNCGGETNUMDEPENDENTSTATEASSIGNMENTPARAMETERSPROC cgGetNumDependentStateAssignmentParameters = NULL;
PFNCGGETDEPENDENTSTATEASSIGNMENTPARAMETERPROC cgGetDependentStateAssignmentParameter = NULL;
PFNCGGETNUMDEPENDENTPROGRAMARRAYSTATEASSIGNMENTPARAMETERSPROC cgGetNumDependentProgramArrayStateAssignmentParameters = NULL;
PFNCGGETDEPENDENTPROGRAMARRAYSTATEASSIGNMENTPARAMETERPROC cgGetDependentProgramArrayStateAssignmentParameter = NULL;
PFNCGGETCONNECTEDSTATEASSIGNMENTPARAMETERPROC cgGetConnectedStateAssignmentParameter = NULL;
PFNCGGETSTATEASSIGNMENTSTATEPROC cgGetStateAssignmentState = NULL;
PFNCGGETSAMPLERSTATEASSIGNMENTSTATEPROC cgGetSamplerStateAssignmentState = NULL;
PFNCGCREATESTATEPROC cgCreateState = NULL;
PFNCGCREATEARRAYSTATEPROC cgCreateArrayState = NULL;
PFNCGSETSTATECALLBACKSPROC cgSetStateCallbacks = NULL;
PFNCGSETSTATELATESTPROFILEPROC cgSetStateLatestProfile = NULL;
PFNCGGETSTATESETCALLBACKPROC cgGetStateSetCallback = NULL;
PFNCGGETSTATERESETCALLBACKPROC cgGetStateResetCallback = NULL;
PFNCGGETSTATEVALIDATECALLBACKPROC cgGetStateValidateCallback = NULL;
PFNCGGETSTATELATESTPROFILEPROC cgGetStateLatestProfile = NULL;
PFNCGGETSTATECONTEXTPROC cgGetStateContext = NULL;
PFNCGGETSTATETYPEPROC cgGetStateType = NULL;
PFNCGGETSTATENAMEPROC cgGetStateName = NULL;
PFNCGGETNAMEDSTATEPROC cgGetNamedState = NULL;
PFNCGGETFIRSTSTATEPROC cgGetFirstState = NULL;
PFNCGGETNEXTSTATEPROC cgGetNextState = NULL;
PFNCGISSTATEPROC cgIsState = NULL;
PFNCGADDSTATEENUMERANTPROC cgAddStateEnumerant = NULL;
PFNCGCREATESAMPLERSTATEPROC cgCreateSamplerState = NULL;
PFNCGCREATEARRAYSAMPLERSTATEPROC cgCreateArraySamplerState = NULL;
PFNCGGETNAMEDSAMPLERSTATEPROC cgGetNamedSamplerState = NULL;
PFNCGGETFIRSTSAMPLERSTATEPROC cgGetFirstSamplerState = NULL;
PFNCGGETFIRSTSAMPLERSTATEASSIGNMENTPROC cgGetFirstSamplerStateAssignment = NULL;
PFNCGGETNAMEDSAMPLERSTATEASSIGNMENTPROC cgGetNamedSamplerStateAssignment = NULL;
PFNCGSETSAMPLERSTATEPROC cgSetSamplerState = NULL;
PFNCGGETNAMEDEFFECTPARAMETERPROC cgGetNamedEffectParameter = NULL;
PFNCGGETFIRSTLEAFEFFECTPARAMETERPROC cgGetFirstLeafEffectParameter = NULL;
PFNCGGETFIRSTEFFECTPARAMETERPROC cgGetFirstEffectParameter = NULL;
PFNCGGETEFFECTPARAMETERBYSEMANTICPROC cgGetEffectParameterBySemantic = NULL;
PFNCGGETFIRSTTECHNIQUEANNOTATIONPROC cgGetFirstTechniqueAnnotation = NULL;
PFNCGGETFIRSTPASSANNOTATIONPROC cgGetFirstPassAnnotation = NULL;
PFNCGGETFIRSTPARAMETERANNOTATIONPROC cgGetFirstParameterAnnotation = NULL;
PFNCGGETFIRSTPROGRAMANNOTATIONPROC cgGetFirstProgramAnnotation = NULL;
PFNCGGETFIRSTEFFECTANNOTATIONPROC cgGetFirstEffectAnnotation = NULL;
PFNCGGETNEXTANNOTATIONPROC cgGetNextAnnotation = NULL;
PFNCGGETNAMEDTECHNIQUEANNOTATIONPROC cgGetNamedTechniqueAnnotation = NULL;
PFNCGGETNAMEDPASSANNOTATIONPROC cgGetNamedPassAnnotation = NULL;
PFNCGGETNAMEDPARAMETERANNOTATIONPROC cgGetNamedParameterAnnotation = NULL;
PFNCGGETNAMEDPROGRAMANNOTATIONPROC cgGetNamedProgramAnnotation = NULL;
PFNCGGETNAMEDEFFECTANNOTATIONPROC cgGetNamedEffectAnnotation = NULL;
PFNCGISANNOTATIONPROC cgIsAnnotation = NULL;
PFNCGGETANNOTATIONNAMEPROC cgGetAnnotationName = NULL;
PFNCGGETANNOTATIONTYPEPROC cgGetAnnotationType = NULL;
PFNCGGETFLOATANNOTATIONVALUESPROC cgGetFloatAnnotationValues = NULL;
PFNCGGETINTANNOTATIONVALUESPROC cgGetIntAnnotationValues = NULL;
PFNCGGETSTRINGANNOTATIONVALUEPROC cgGetStringAnnotationValue = NULL;
PFNCGGETSTRINGANNOTATIONVALUESPROC cgGetStringAnnotationValues = NULL;
PFNCGGETBOOLANNOTATIONVALUESPROC cgGetBoolAnnotationValues = NULL;
PFNCGGETBOOLEANANNOTATIONVALUESPROC cgGetBooleanAnnotationValues = NULL;
PFNCGGETNUMDEPENDENTANNOTATIONPARAMETERSPROC cgGetNumDependentAnnotationParameters = NULL;
PFNCGGETDEPENDENTANNOTATIONPARAMETERPROC cgGetDependentAnnotationParameter = NULL;
PFNCGEVALUATEPROGRAMPROC cgEvaluateProgram = NULL;
PFNCGSETEFFECTNAMEPROC cgSetEffectName = NULL;
PFNCGGETEFFECTNAMEPROC cgGetEffectName = NULL;
PFNCGGETNAMEDEFFECTPROC cgGetNamedEffect = NULL;
PFNCGCREATEEFFECTPARAMETERPROC cgCreateEffectParameter = NULL;
PFNCGCREATETECHNIQUEPROC cgCreateTechnique = NULL;
PFNCGCREATEEFFECTPARAMETERARRAYPROC cgCreateEffectParameterArray = NULL;
PFNCGCREATEEFFECTPARAMETERMULTIDIMARRAYPROC cgCreateEffectParameterMultiDimArray = NULL;
PFNCGCREATEPASSPROC cgCreatePass = NULL;
PFNCGCREATESTATEASSIGNMENTPROC cgCreateStateAssignment = NULL;
PFNCGCREATESTATEASSIGNMENTINDEXPROC cgCreateStateAssignmentIndex = NULL;
PFNCGCREATESAMPLERSTATEASSIGNMENTPROC cgCreateSamplerStateAssignment = NULL;
PFNCGSETFLOATSTATEASSIGNMENTPROC cgSetFloatStateAssignment = NULL;
PFNCGSETINTSTATEASSIGNMENTPROC cgSetIntStateAssignment = NULL;
PFNCGSETBOOLSTATEASSIGNMENTPROC cgSetBoolStateAssignment = NULL;
PFNCGSETSTRINGSTATEASSIGNMENTPROC cgSetStringStateAssignment = NULL;
PFNCGSETPROGRAMSTATEASSIGNMENTPROC cgSetProgramStateAssignment = NULL;
PFNCGSETSAMPLERSTATEASSIGNMENTPROC cgSetSamplerStateAssignment = NULL;
PFNCGSETTEXTURESTATEASSIGNMENTPROC cgSetTextureStateAssignment = NULL;
PFNCGSETFLOATARRAYSTATEASSIGNMENTPROC cgSetFloatArrayStateAssignment = NULL;
PFNCGSETINTARRAYSTATEASSIGNMENTPROC cgSetIntArrayStateAssignment = NULL;
PFNCGSETBOOLARRAYSTATEASSIGNMENTPROC cgSetBoolArrayStateAssignment = NULL;
PFNCGCREATETECHNIQUEANNOTATIONPROC cgCreateTechniqueAnnotation = NULL;
PFNCGCREATEPASSANNOTATIONPROC cgCreatePassAnnotation = NULL;
PFNCGCREATEPARAMETERANNOTATIONPROC cgCreateParameterAnnotation = NULL;
PFNCGCREATEPROGRAMANNOTATIONPROC cgCreateProgramAnnotation = NULL;
PFNCGCREATEEFFECTANNOTATIONPROC cgCreateEffectAnnotation = NULL;
PFNCGSETINTANNOTATIONPROC cgSetIntAnnotation = NULL;
PFNCGSETFLOATANNOTATIONPROC cgSetFloatAnnotation = NULL;
PFNCGSETBOOLANNOTATIONPROC cgSetBoolAnnotation = NULL;
PFNCGSETSTRINGANNOTATIONPROC cgSetStringAnnotation = NULL;
PFNCGGETNUMSTATEENUMERANTSPROC cgGetNumStateEnumerants = NULL;
PFNCGGETSTATEENUMERANTPROC cgGetStateEnumerant = NULL;
PFNCGGETSTATEENUMERANTNAMEPROC cgGetStateEnumerantName = NULL;
PFNCGGETSTATEENUMERANTVALUEPROC cgGetStateEnumerantValue = NULL;
PFNCGGETPARAMETEREFFECTPROC cgGetParameterEffect = NULL;
PFNCGGETTYPECLASSPROC cgGetTypeClass = NULL;
PFNCGGETTYPEBASEPROC cgGetTypeBase = NULL;
PFNCGGETTYPESIZESPROC cgGetTypeSizes = NULL;
PFNCGGETMATRIXSIZEPROC cgGetMatrixSize = NULL;
PFNCGGETNUMPROGRAMDOMAINSPROC cgGetNumProgramDomains = NULL;
PFNCGGETPROFILEDOMAINPROC cgGetProfileDomain = NULL;
PFNCGCOMBINEPROGRAMSPROC cgCombinePrograms = NULL;
PFNCGCOMBINEPROGRAMS2PROC cgCombinePrograms2 = NULL;
PFNCGCOMBINEPROGRAMS3PROC cgCombinePrograms3 = NULL;
PFNCGCOMBINEPROGRAMS4PROC cgCombinePrograms4 = NULL;
PFNCGCOMBINEPROGRAMS5PROC cgCombinePrograms5 = NULL;
PFNCGGETPROGRAMDOMAINPROFILEPROC cgGetProgramDomainProfile = NULL;
PFNCGGETPROGRAMDOMAINPROGRAMPROC cgGetProgramDomainProgram = NULL;
PFNCGCREATEOBJPROC cgCreateObj = NULL;
PFNCGCREATEOBJFROMFILEPROC cgCreateObjFromFile = NULL;
PFNCGDESTROYOBJPROC cgDestroyObj = NULL;
PFNCGGETPARAMETERRESOURCESIZEPROC cgGetParameterResourceSize = NULL;
PFNCGGETPARAMETERRESOURCETYPEPROC cgGetParameterResourceType = NULL;
PFNCGGETPARAMETERRESOURCENAMEPROC cgGetParameterResourceName = NULL;
PFNCGGETPARAMETERBUFFERINDEXPROC cgGetParameterBufferIndex = NULL;
PFNCGGETPARAMETERBUFFEROFFSETPROC cgGetParameterBufferOffset = NULL;
PFNCGCREATEBUFFERPROC cgCreateBuffer = NULL;
PFNCGSETBUFFERDATAPROC cgSetBufferData = NULL;
PFNCGSETBUFFERSUBDATAPROC cgSetBufferSubData = NULL;
PFNCGSETPROGRAMBUFFERPROC cgSetProgramBuffer = NULL;
PFNCGMAPBUFFERPROC cgMapBuffer = NULL;
PFNCGUNMAPBUFFERPROC cgUnmapBuffer = NULL;
PFNCGDESTROYBUFFERPROC cgDestroyBuffer = NULL;
PFNCGGETPROGRAMBUFFERPROC cgGetProgramBuffer = NULL;
PFNCGGETBUFFERSIZEPROC cgGetBufferSize = NULL;
PFNCGGETPROGRAMBUFFERMAXSIZEPROC cgGetProgramBufferMaxSize = NULL;
PFNCGGETPROGRAMBUFFERMAXINDEXPROC cgGetProgramBufferMaxIndex = NULL;
PFNCGGETEFFECTPARAMETERBUFFERPROC cgGetEffectParameterBuffer = NULL;
PFNCGSETEFFECTPARAMETERBUFFERPROC cgSetEffectParameterBuffer = NULL;

/* Implementation */

LHandle lHandle = NULL;

int
lInit(const char *filename)
{
  lHandle = lLoadLibrary(filename);
  if (!lHandle)
    return 0;

  cgSetLockingPolicy = (PFNCGSETLOCKINGPOLICYPROC) lGetProcAddress(lHandle, "cgSetLockingPolicy");
  cgGetLockingPolicy = (PFNCGGETLOCKINGPOLICYPROC) lGetProcAddress(lHandle, "cgGetLockingPolicy");
  cgSetSemanticCasePolicy = (PFNCGSETSEMANTICCASEPOLICYPROC) lGetProcAddress(lHandle, "cgSetSemanticCasePolicy");
  cgGetSemanticCasePolicy = (PFNCGGETSEMANTICCASEPOLICYPROC) lGetProcAddress(lHandle, "cgGetSemanticCasePolicy");
  cgCreateContext = (PFNCGCREATECONTEXTPROC) lGetProcAddress(lHandle, "cgCreateContext");
  cgSetContextBehavior = (PFNCGSETCONTEXTBEHAVIORPROC) lGetProcAddress(lHandle, "cgSetContextBehavior");
  cgGetContextBehavior = (PFNCGGETCONTEXTBEHAVIORPROC) lGetProcAddress(lHandle, "cgGetContextBehavior");
  cgGetBehaviorString = (PFNCGGETBEHAVIORSTRINGPROC) lGetProcAddress(lHandle, "cgGetBehaviorString");
  cgGetBehavior = (PFNCGGETBEHAVIORPROC) lGetProcAddress(lHandle, "cgGetBehavior");
  cgDestroyContext = (PFNCGDESTROYCONTEXTPROC) lGetProcAddress(lHandle, "cgDestroyContext");
  cgIsContext = (PFNCGISCONTEXTPROC) lGetProcAddress(lHandle, "cgIsContext");
  cgGetLastListing = (PFNCGGETLASTLISTINGPROC) lGetProcAddress(lHandle, "cgGetLastListing");
  cgSetLastListing = (PFNCGSETLASTLISTINGPROC) lGetProcAddress(lHandle, "cgSetLastListing");
  cgSetAutoCompile = (PFNCGSETAUTOCOMPILEPROC) lGetProcAddress(lHandle, "cgSetAutoCompile");
  cgGetAutoCompile = (PFNCGGETAUTOCOMPILEPROC) lGetProcAddress(lHandle, "cgGetAutoCompile");
  cgSetParameterSettingMode = (PFNCGSETPARAMETERSETTINGMODEPROC) lGetProcAddress(lHandle, "cgSetParameterSettingMode");
  cgGetParameterSettingMode = (PFNCGGETPARAMETERSETTINGMODEPROC) lGetProcAddress(lHandle, "cgGetParameterSettingMode");
  cgSetCompilerIncludeString = (PFNCGSETCOMPILERINCLUDESTRINGPROC) lGetProcAddress(lHandle, "cgSetCompilerIncludeString");
  cgSetCompilerIncludeFile = (PFNCGSETCOMPILERINCLUDEFILEPROC) lGetProcAddress(lHandle, "cgSetCompilerIncludeFile");
  cgSetCompilerIncludeCallback = (PFNCGSETCOMPILERINCLUDECALLBACKPROC) lGetProcAddress(lHandle, "cgSetCompilerIncludeCallback");
  cgGetCompilerIncludeCallback = (PFNCGGETCOMPILERINCLUDECALLBACKPROC) lGetProcAddress(lHandle, "cgGetCompilerIncludeCallback");
  cgCreateProgram = (PFNCGCREATEPROGRAMPROC) lGetProcAddress(lHandle, "cgCreateProgram");
  cgCreateProgramFromFile = (PFNCGCREATEPROGRAMFROMFILEPROC) lGetProcAddress(lHandle, "cgCreateProgramFromFile");
  cgCopyProgram = (PFNCGCOPYPROGRAMPROC) lGetProcAddress(lHandle, "cgCopyProgram");
  cgDestroyProgram = (PFNCGDESTROYPROGRAMPROC) lGetProcAddress(lHandle, "cgDestroyProgram");
  cgGetFirstProgram = (PFNCGGETFIRSTPROGRAMPROC) lGetProcAddress(lHandle, "cgGetFirstProgram");
  cgGetNextProgram = (PFNCGGETNEXTPROGRAMPROC) lGetProcAddress(lHandle, "cgGetNextProgram");
  cgGetProgramContext = (PFNCGGETPROGRAMCONTEXTPROC) lGetProcAddress(lHandle, "cgGetProgramContext");
  cgIsProgram = (PFNCGISPROGRAMPROC) lGetProcAddress(lHandle, "cgIsProgram");
  cgCompileProgram = (PFNCGCOMPILEPROGRAMPROC) lGetProcAddress(lHandle, "cgCompileProgram");
  cgIsProgramCompiled = (PFNCGISPROGRAMCOMPILEDPROC) lGetProcAddress(lHandle, "cgIsProgramCompiled");
  cgGetProgramString = (PFNCGGETPROGRAMSTRINGPROC) lGetProcAddress(lHandle, "cgGetProgramString");
  cgGetProgramProfile = (PFNCGGETPROGRAMPROFILEPROC) lGetProcAddress(lHandle, "cgGetProgramProfile");
  cgGetProgramOptions = (PFNCGGETPROGRAMOPTIONSPROC) lGetProcAddress(lHandle, "cgGetProgramOptions");
  cgSetProgramProfile = (PFNCGSETPROGRAMPROFILEPROC) lGetProcAddress(lHandle, "cgSetProgramProfile");
  cgGetProgramInput = (PFNCGGETPROGRAMINPUTPROC) lGetProcAddress(lHandle, "cgGetProgramInput");
  cgGetProgramOutput = (PFNCGGETPROGRAMOUTPUTPROC) lGetProcAddress(lHandle, "cgGetProgramOutput");
  cgSetPassProgramParameters = (PFNCGSETPASSPROGRAMPARAMETERSPROC) lGetProcAddress(lHandle, "cgSetPassProgramParameters");
  cgUpdateProgramParameters = (PFNCGUPDATEPROGRAMPARAMETERSPROC) lGetProcAddress(lHandle, "cgUpdateProgramParameters");
  cgUpdatePassParameters = (PFNCGUPDATEPASSPARAMETERSPROC) lGetProcAddress(lHandle, "cgUpdatePassParameters");
  cgCreateParameter = (PFNCGCREATEPARAMETERPROC) lGetProcAddress(lHandle, "cgCreateParameter");
  cgCreateParameterArray = (PFNCGCREATEPARAMETERARRAYPROC) lGetProcAddress(lHandle, "cgCreateParameterArray");
  cgCreateParameterMultiDimArray = (PFNCGCREATEPARAMETERMULTIDIMARRAYPROC) lGetProcAddress(lHandle, "cgCreateParameterMultiDimArray");
  cgDestroyParameter = (PFNCGDESTROYPARAMETERPROC) lGetProcAddress(lHandle, "cgDestroyParameter");
  cgConnectParameter = (PFNCGCONNECTPARAMETERPROC) lGetProcAddress(lHandle, "cgConnectParameter");
  cgDisconnectParameter = (PFNCGDISCONNECTPARAMETERPROC) lGetProcAddress(lHandle, "cgDisconnectParameter");
  cgGetConnectedParameter = (PFNCGGETCONNECTEDPARAMETERPROC) lGetProcAddress(lHandle, "cgGetConnectedParameter");
  cgGetNumConnectedToParameters = (PFNCGGETNUMCONNECTEDTOPARAMETERSPROC) lGetProcAddress(lHandle, "cgGetNumConnectedToParameters");
  cgGetConnectedToParameter = (PFNCGGETCONNECTEDTOPARAMETERPROC) lGetProcAddress(lHandle, "cgGetConnectedToParameter");
  cgGetNamedParameter = (PFNCGGETNAMEDPARAMETERPROC) lGetProcAddress(lHandle, "cgGetNamedParameter");
  cgGetNamedProgramParameter = (PFNCGGETNAMEDPROGRAMPARAMETERPROC) lGetProcAddress(lHandle, "cgGetNamedProgramParameter");
  cgGetFirstParameter = (PFNCGGETFIRSTPARAMETERPROC) lGetProcAddress(lHandle, "cgGetFirstParameter");
  cgGetNextParameter = (PFNCGGETNEXTPARAMETERPROC) lGetProcAddress(lHandle, "cgGetNextParameter");
  cgGetFirstLeafParameter = (PFNCGGETFIRSTLEAFPARAMETERPROC) lGetProcAddress(lHandle, "cgGetFirstLeafParameter");
  cgGetNextLeafParameter = (PFNCGGETNEXTLEAFPARAMETERPROC) lGetProcAddress(lHandle, "cgGetNextLeafParameter");
  cgGetFirstStructParameter = (PFNCGGETFIRSTSTRUCTPARAMETERPROC) lGetProcAddress(lHandle, "cgGetFirstStructParameter");
  cgGetNamedStructParameter = (PFNCGGETNAMEDSTRUCTPARAMETERPROC) lGetProcAddress(lHandle, "cgGetNamedStructParameter");
  cgGetFirstDependentParameter = (PFNCGGETFIRSTDEPENDENTPARAMETERPROC) lGetProcAddress(lHandle, "cgGetFirstDependentParameter");
  cgGetArrayParameter = (PFNCGGETARRAYPARAMETERPROC) lGetProcAddress(lHandle, "cgGetArrayParameter");
  cgGetArrayDimension = (PFNCGGETARRAYDIMENSIONPROC) lGetProcAddress(lHandle, "cgGetArrayDimension");
  cgGetArrayType = (PFNCGGETARRAYTYPEPROC) lGetProcAddress(lHandle, "cgGetArrayType");
  cgGetArraySize = (PFNCGGETARRAYSIZEPROC) lGetProcAddress(lHandle, "cgGetArraySize");
  cgGetArrayTotalSize = (PFNCGGETARRAYTOTALSIZEPROC) lGetProcAddress(lHandle, "cgGetArrayTotalSize");
  cgSetArraySize = (PFNCGSETARRAYSIZEPROC) lGetProcAddress(lHandle, "cgSetArraySize");
  cgSetMultiDimArraySize = (PFNCGSETMULTIDIMARRAYSIZEPROC) lGetProcAddress(lHandle, "cgSetMultiDimArraySize");
  cgGetParameterProgram = (PFNCGGETPARAMETERPROGRAMPROC) lGetProcAddress(lHandle, "cgGetParameterProgram");
  cgGetParameterContext = (PFNCGGETPARAMETERCONTEXTPROC) lGetProcAddress(lHandle, "cgGetParameterContext");
  cgIsParameter = (PFNCGISPARAMETERPROC) lGetProcAddress(lHandle, "cgIsParameter");
  cgGetParameterName = (PFNCGGETPARAMETERNAMEPROC) lGetProcAddress(lHandle, "cgGetParameterName");
  cgGetParameterType = (PFNCGGETPARAMETERTYPEPROC) lGetProcAddress(lHandle, "cgGetParameterType");
  cgGetParameterBaseType = (PFNCGGETPARAMETERBASETYPEPROC) lGetProcAddress(lHandle, "cgGetParameterBaseType");
  cgGetParameterClass = (PFNCGGETPARAMETERCLASSPROC) lGetProcAddress(lHandle, "cgGetParameterClass");
  cgGetParameterRows = (PFNCGGETPARAMETERROWSPROC) lGetProcAddress(lHandle, "cgGetParameterRows");
  cgGetParameterColumns = (PFNCGGETPARAMETERCOLUMNSPROC) lGetProcAddress(lHandle, "cgGetParameterColumns");
  cgGetParameterNamedType = (PFNCGGETPARAMETERNAMEDTYPEPROC) lGetProcAddress(lHandle, "cgGetParameterNamedType");
  cgGetParameterSemantic = (PFNCGGETPARAMETERSEMANTICPROC) lGetProcAddress(lHandle, "cgGetParameterSemantic");
  cgGetParameterResource = (PFNCGGETPARAMETERRESOURCEPROC) lGetProcAddress(lHandle, "cgGetParameterResource");
  cgGetParameterBaseResource = (PFNCGGETPARAMETERBASERESOURCEPROC) lGetProcAddress(lHandle, "cgGetParameterBaseResource");
  cgGetParameterResourceIndex = (PFNCGGETPARAMETERRESOURCEINDEXPROC) lGetProcAddress(lHandle, "cgGetParameterResourceIndex");
  cgGetParameterVariability = (PFNCGGETPARAMETERVARIABILITYPROC) lGetProcAddress(lHandle, "cgGetParameterVariability");
  cgGetParameterDirection = (PFNCGGETPARAMETERDIRECTIONPROC) lGetProcAddress(lHandle, "cgGetParameterDirection");
  cgIsParameterReferenced = (PFNCGISPARAMETERREFERENCEDPROC) lGetProcAddress(lHandle, "cgIsParameterReferenced");
  cgIsParameterUsed = (PFNCGISPARAMETERUSEDPROC) lGetProcAddress(lHandle, "cgIsParameterUsed");
  cgGetParameterValues = (PFNCGGETPARAMETERVALUESPROC) lGetProcAddress(lHandle, "cgGetParameterValues");
  cgSetParameterValuedr = (PFNCGSETPARAMETERVALUEDRPROC) lGetProcAddress(lHandle, "cgSetParameterValuedr");
  cgSetParameterValuedc = (PFNCGSETPARAMETERVALUEDCPROC) lGetProcAddress(lHandle, "cgSetParameterValuedc");
  cgSetParameterValuefr = (PFNCGSETPARAMETERVALUEFRPROC) lGetProcAddress(lHandle, "cgSetParameterValuefr");
  cgSetParameterValuefc = (PFNCGSETPARAMETERVALUEFCPROC) lGetProcAddress(lHandle, "cgSetParameterValuefc");
  cgSetParameterValueir = (PFNCGSETPARAMETERVALUEIRPROC) lGetProcAddress(lHandle, "cgSetParameterValueir");
  cgSetParameterValueic = (PFNCGSETPARAMETERVALUEICPROC) lGetProcAddress(lHandle, "cgSetParameterValueic");
  cgGetParameterValuedr = (PFNCGGETPARAMETERVALUEDRPROC) lGetProcAddress(lHandle, "cgGetParameterValuedr");
  cgGetParameterValuedc = (PFNCGGETPARAMETERVALUEDCPROC) lGetProcAddress(lHandle, "cgGetParameterValuedc");
  cgGetParameterValuefr = (PFNCGGETPARAMETERVALUEFRPROC) lGetProcAddress(lHandle, "cgGetParameterValuefr");
  cgGetParameterValuefc = (PFNCGGETPARAMETERVALUEFCPROC) lGetProcAddress(lHandle, "cgGetParameterValuefc");
  cgGetParameterValueir = (PFNCGGETPARAMETERVALUEIRPROC) lGetProcAddress(lHandle, "cgGetParameterValueir");
  cgGetParameterValueic = (PFNCGGETPARAMETERVALUEICPROC) lGetProcAddress(lHandle, "cgGetParameterValueic");
  cgGetParameterDefaultValuedr = (PFNCGGETPARAMETERDEFAULTVALUEDRPROC) lGetProcAddress(lHandle, "cgGetParameterDefaultValuedr");
  cgGetParameterDefaultValuedc = (PFNCGGETPARAMETERDEFAULTVALUEDCPROC) lGetProcAddress(lHandle, "cgGetParameterDefaultValuedc");
  cgGetParameterDefaultValuefr = (PFNCGGETPARAMETERDEFAULTVALUEFRPROC) lGetProcAddress(lHandle, "cgGetParameterDefaultValuefr");
  cgGetParameterDefaultValuefc = (PFNCGGETPARAMETERDEFAULTVALUEFCPROC) lGetProcAddress(lHandle, "cgGetParameterDefaultValuefc");
  cgGetParameterDefaultValueir = (PFNCGGETPARAMETERDEFAULTVALUEIRPROC) lGetProcAddress(lHandle, "cgGetParameterDefaultValueir");
  cgGetParameterDefaultValueic = (PFNCGGETPARAMETERDEFAULTVALUEICPROC) lGetProcAddress(lHandle, "cgGetParameterDefaultValueic");
  cgGetStringParameterValue = (PFNCGGETSTRINGPARAMETERVALUEPROC) lGetProcAddress(lHandle, "cgGetStringParameterValue");
  cgSetStringParameterValue = (PFNCGSETSTRINGPARAMETERVALUEPROC) lGetProcAddress(lHandle, "cgSetStringParameterValue");
  cgGetParameterOrdinalNumber = (PFNCGGETPARAMETERORDINALNUMBERPROC) lGetProcAddress(lHandle, "cgGetParameterOrdinalNumber");
  cgIsParameterGlobal = (PFNCGISPARAMETERGLOBALPROC) lGetProcAddress(lHandle, "cgIsParameterGlobal");
  cgGetParameterIndex = (PFNCGGETPARAMETERINDEXPROC) lGetProcAddress(lHandle, "cgGetParameterIndex");
  cgSetParameterVariability = (PFNCGSETPARAMETERVARIABILITYPROC) lGetProcAddress(lHandle, "cgSetParameterVariability");
  cgSetParameterSemantic = (PFNCGSETPARAMETERSEMANTICPROC) lGetProcAddress(lHandle, "cgSetParameterSemantic");
  cgSetParameter1f = (PFNCGSETPARAMETER1FPROC) lGetProcAddress(lHandle, "cgSetParameter1f");
  cgSetParameter2f = (PFNCGSETPARAMETER2FPROC) lGetProcAddress(lHandle, "cgSetParameter2f");
  cgSetParameter3f = (PFNCGSETPARAMETER3FPROC) lGetProcAddress(lHandle, "cgSetParameter3f");
  cgSetParameter4f = (PFNCGSETPARAMETER4FPROC) lGetProcAddress(lHandle, "cgSetParameter4f");
  cgSetParameter1d = (PFNCGSETPARAMETER1DPROC) lGetProcAddress(lHandle, "cgSetParameter1d");
  cgSetParameter2d = (PFNCGSETPARAMETER2DPROC) lGetProcAddress(lHandle, "cgSetParameter2d");
  cgSetParameter3d = (PFNCGSETPARAMETER3DPROC) lGetProcAddress(lHandle, "cgSetParameter3d");
  cgSetParameter4d = (PFNCGSETPARAMETER4DPROC) lGetProcAddress(lHandle, "cgSetParameter4d");
  cgSetParameter1i = (PFNCGSETPARAMETER1IPROC) lGetProcAddress(lHandle, "cgSetParameter1i");
  cgSetParameter2i = (PFNCGSETPARAMETER2IPROC) lGetProcAddress(lHandle, "cgSetParameter2i");
  cgSetParameter3i = (PFNCGSETPARAMETER3IPROC) lGetProcAddress(lHandle, "cgSetParameter3i");
  cgSetParameter4i = (PFNCGSETPARAMETER4IPROC) lGetProcAddress(lHandle, "cgSetParameter4i");
  cgSetParameter1iv = (PFNCGSETPARAMETER1IVPROC) lGetProcAddress(lHandle, "cgSetParameter1iv");
  cgSetParameter2iv = (PFNCGSETPARAMETER2IVPROC) lGetProcAddress(lHandle, "cgSetParameter2iv");
  cgSetParameter3iv = (PFNCGSETPARAMETER3IVPROC) lGetProcAddress(lHandle, "cgSetParameter3iv");
  cgSetParameter4iv = (PFNCGSETPARAMETER4IVPROC) lGetProcAddress(lHandle, "cgSetParameter4iv");
  cgSetParameter1fv = (PFNCGSETPARAMETER1FVPROC) lGetProcAddress(lHandle, "cgSetParameter1fv");
  cgSetParameter2fv = (PFNCGSETPARAMETER2FVPROC) lGetProcAddress(lHandle, "cgSetParameter2fv");
  cgSetParameter3fv = (PFNCGSETPARAMETER3FVPROC) lGetProcAddress(lHandle, "cgSetParameter3fv");
  cgSetParameter4fv = (PFNCGSETPARAMETER4FVPROC) lGetProcAddress(lHandle, "cgSetParameter4fv");
  cgSetParameter1dv = (PFNCGSETPARAMETER1DVPROC) lGetProcAddress(lHandle, "cgSetParameter1dv");
  cgSetParameter2dv = (PFNCGSETPARAMETER2DVPROC) lGetProcAddress(lHandle, "cgSetParameter2dv");
  cgSetParameter3dv = (PFNCGSETPARAMETER3DVPROC) lGetProcAddress(lHandle, "cgSetParameter3dv");
  cgSetParameter4dv = (PFNCGSETPARAMETER4DVPROC) lGetProcAddress(lHandle, "cgSetParameter4dv");
  cgSetMatrixParameterir = (PFNCGSETMATRIXPARAMETERIRPROC) lGetProcAddress(lHandle, "cgSetMatrixParameterir");
  cgSetMatrixParameterdr = (PFNCGSETMATRIXPARAMETERDRPROC) lGetProcAddress(lHandle, "cgSetMatrixParameterdr");
  cgSetMatrixParameterfr = (PFNCGSETMATRIXPARAMETERFRPROC) lGetProcAddress(lHandle, "cgSetMatrixParameterfr");
  cgSetMatrixParameteric = (PFNCGSETMATRIXPARAMETERICPROC) lGetProcAddress(lHandle, "cgSetMatrixParameteric");
  cgSetMatrixParameterdc = (PFNCGSETMATRIXPARAMETERDCPROC) lGetProcAddress(lHandle, "cgSetMatrixParameterdc");
  cgSetMatrixParameterfc = (PFNCGSETMATRIXPARAMETERFCPROC) lGetProcAddress(lHandle, "cgSetMatrixParameterfc");
  cgGetMatrixParameterir = (PFNCGGETMATRIXPARAMETERIRPROC) lGetProcAddress(lHandle, "cgGetMatrixParameterir");
  cgGetMatrixParameterdr = (PFNCGGETMATRIXPARAMETERDRPROC) lGetProcAddress(lHandle, "cgGetMatrixParameterdr");
  cgGetMatrixParameterfr = (PFNCGGETMATRIXPARAMETERFRPROC) lGetProcAddress(lHandle, "cgGetMatrixParameterfr");
  cgGetMatrixParameteric = (PFNCGGETMATRIXPARAMETERICPROC) lGetProcAddress(lHandle, "cgGetMatrixParameteric");
  cgGetMatrixParameterdc = (PFNCGGETMATRIXPARAMETERDCPROC) lGetProcAddress(lHandle, "cgGetMatrixParameterdc");
  cgGetMatrixParameterfc = (PFNCGGETMATRIXPARAMETERFCPROC) lGetProcAddress(lHandle, "cgGetMatrixParameterfc");
  cgGetMatrixParameterOrder = (PFNCGGETMATRIXPARAMETERORDERPROC) lGetProcAddress(lHandle, "cgGetMatrixParameterOrder");
  cgGetNamedSubParameter = (PFNCGGETNAMEDSUBPARAMETERPROC) lGetProcAddress(lHandle, "cgGetNamedSubParameter");
  cgGetTypeString = (PFNCGGETTYPESTRINGPROC) lGetProcAddress(lHandle, "cgGetTypeString");
  cgGetType = (PFNCGGETTYPEPROC) lGetProcAddress(lHandle, "cgGetType");
  cgGetNamedUserType = (PFNCGGETNAMEDUSERTYPEPROC) lGetProcAddress(lHandle, "cgGetNamedUserType");
  cgGetNumUserTypes = (PFNCGGETNUMUSERTYPESPROC) lGetProcAddress(lHandle, "cgGetNumUserTypes");
  cgGetUserType = (PFNCGGETUSERTYPEPROC) lGetProcAddress(lHandle, "cgGetUserType");
  cgGetNumParentTypes = (PFNCGGETNUMPARENTTYPESPROC) lGetProcAddress(lHandle, "cgGetNumParentTypes");
  cgGetParentType = (PFNCGGETPARENTTYPEPROC) lGetProcAddress(lHandle, "cgGetParentType");
  cgIsParentType = (PFNCGISPARENTTYPEPROC) lGetProcAddress(lHandle, "cgIsParentType");
  cgIsInterfaceType = (PFNCGISINTERFACETYPEPROC) lGetProcAddress(lHandle, "cgIsInterfaceType");
  cgGetResourceString = (PFNCGGETRESOURCESTRINGPROC) lGetProcAddress(lHandle, "cgGetResourceString");
  cgGetResource = (PFNCGGETRESOURCEPROC) lGetProcAddress(lHandle, "cgGetResource");
  cgGetEnumString = (PFNCGGETENUMSTRINGPROC) lGetProcAddress(lHandle, "cgGetEnumString");
  cgGetEnum = (PFNCGGETENUMPROC) lGetProcAddress(lHandle, "cgGetEnum");
  cgGetProfileString = (PFNCGGETPROFILESTRINGPROC) lGetProcAddress(lHandle, "cgGetProfileString");
  cgGetProfile = (PFNCGGETPROFILEPROC) lGetProcAddress(lHandle, "cgGetProfile");
  cgGetNumSupportedProfiles = (PFNCGGETNUMSUPPORTEDPROFILESPROC) lGetProcAddress(lHandle, "cgGetNumSupportedProfiles");
  cgGetSupportedProfile = (PFNCGGETSUPPORTEDPROFILEPROC) lGetProcAddress(lHandle, "cgGetSupportedProfile");
  cgIsProfileSupported = (PFNCGISPROFILESUPPORTEDPROC) lGetProcAddress(lHandle, "cgIsProfileSupported");
  cgGetProfileProperty = (PFNCGGETPROFILEPROPERTYPROC) lGetProcAddress(lHandle, "cgGetProfileProperty");
  cgGetParameterClassString = (PFNCGGETPARAMETERCLASSSTRINGPROC) lGetProcAddress(lHandle, "cgGetParameterClassString");
  cgGetParameterClassEnum = (PFNCGGETPARAMETERCLASSENUMPROC) lGetProcAddress(lHandle, "cgGetParameterClassEnum");
  cgGetDomainString = (PFNCGGETDOMAINSTRINGPROC) lGetProcAddress(lHandle, "cgGetDomainString");
  cgGetDomain = (PFNCGGETDOMAINPROC) lGetProcAddress(lHandle, "cgGetDomain");
  cgGetProgramDomain = (PFNCGGETPROGRAMDOMAINPROC) lGetProcAddress(lHandle, "cgGetProgramDomain");
  cgGetError = (PFNCGGETERRORPROC) lGetProcAddress(lHandle, "cgGetError");
  cgGetFirstError = (PFNCGGETFIRSTERRORPROC) lGetProcAddress(lHandle, "cgGetFirstError");
  cgGetErrorString = (PFNCGGETERRORSTRINGPROC) lGetProcAddress(lHandle, "cgGetErrorString");
  cgGetLastErrorString = (PFNCGGETLASTERRORSTRINGPROC) lGetProcAddress(lHandle, "cgGetLastErrorString");
  cgSetErrorCallback = (PFNCGSETERRORCALLBACKPROC) lGetProcAddress(lHandle, "cgSetErrorCallback");
  cgGetErrorCallback = (PFNCGGETERRORCALLBACKPROC) lGetProcAddress(lHandle, "cgGetErrorCallback");
  cgSetErrorHandler = (PFNCGSETERRORHANDLERPROC) lGetProcAddress(lHandle, "cgSetErrorHandler");
  cgGetErrorHandler = (PFNCGGETERRORHANDLERPROC) lGetProcAddress(lHandle, "cgGetErrorHandler");
  cgGetString = (PFNCGGETSTRINGPROC) lGetProcAddress(lHandle, "cgGetString");
  cgCreateEffect = (PFNCGCREATEEFFECTPROC) lGetProcAddress(lHandle, "cgCreateEffect");
  cgCreateEffectFromFile = (PFNCGCREATEEFFECTFROMFILEPROC) lGetProcAddress(lHandle, "cgCreateEffectFromFile");
  cgCopyEffect = (PFNCGCOPYEFFECTPROC) lGetProcAddress(lHandle, "cgCopyEffect");
  cgDestroyEffect = (PFNCGDESTROYEFFECTPROC) lGetProcAddress(lHandle, "cgDestroyEffect");
  cgGetEffectContext = (PFNCGGETEFFECTCONTEXTPROC) lGetProcAddress(lHandle, "cgGetEffectContext");
  cgIsEffect = (PFNCGISEFFECTPROC) lGetProcAddress(lHandle, "cgIsEffect");
  cgGetFirstEffect = (PFNCGGETFIRSTEFFECTPROC) lGetProcAddress(lHandle, "cgGetFirstEffect");
  cgGetNextEffect = (PFNCGGETNEXTEFFECTPROC) lGetProcAddress(lHandle, "cgGetNextEffect");
  cgCreateProgramFromEffect = (PFNCGCREATEPROGRAMFROMEFFECTPROC) lGetProcAddress(lHandle, "cgCreateProgramFromEffect");
  cgGetFirstTechnique = (PFNCGGETFIRSTTECHNIQUEPROC) lGetProcAddress(lHandle, "cgGetFirstTechnique");
  cgGetNextTechnique = (PFNCGGETNEXTTECHNIQUEPROC) lGetProcAddress(lHandle, "cgGetNextTechnique");
  cgGetNamedTechnique = (PFNCGGETNAMEDTECHNIQUEPROC) lGetProcAddress(lHandle, "cgGetNamedTechnique");
  cgGetTechniqueName = (PFNCGGETTECHNIQUENAMEPROC) lGetProcAddress(lHandle, "cgGetTechniqueName");
  cgIsTechnique = (PFNCGISTECHNIQUEPROC) lGetProcAddress(lHandle, "cgIsTechnique");
  cgValidateTechnique = (PFNCGVALIDATETECHNIQUEPROC) lGetProcAddress(lHandle, "cgValidateTechnique");
  cgIsTechniqueValidated = (PFNCGISTECHNIQUEVALIDATEDPROC) lGetProcAddress(lHandle, "cgIsTechniqueValidated");
  cgGetTechniqueEffect = (PFNCGGETTECHNIQUEEFFECTPROC) lGetProcAddress(lHandle, "cgGetTechniqueEffect");
  cgGetFirstPass = (PFNCGGETFIRSTPASSPROC) lGetProcAddress(lHandle, "cgGetFirstPass");
  cgGetNamedPass = (PFNCGGETNAMEDPASSPROC) lGetProcAddress(lHandle, "cgGetNamedPass");
  cgGetNextPass = (PFNCGGETNEXTPASSPROC) lGetProcAddress(lHandle, "cgGetNextPass");
  cgIsPass = (PFNCGISPASSPROC) lGetProcAddress(lHandle, "cgIsPass");
  cgGetPassName = (PFNCGGETPASSNAMEPROC) lGetProcAddress(lHandle, "cgGetPassName");
  cgGetPassTechnique = (PFNCGGETPASSTECHNIQUEPROC) lGetProcAddress(lHandle, "cgGetPassTechnique");
  cgGetPassProgram = (PFNCGGETPASSPROGRAMPROC) lGetProcAddress(lHandle, "cgGetPassProgram");
  cgSetPassState = (PFNCGSETPASSSTATEPROC) lGetProcAddress(lHandle, "cgSetPassState");
  cgResetPassState = (PFNCGRESETPASSSTATEPROC) lGetProcAddress(lHandle, "cgResetPassState");
  cgGetFirstStateAssignment = (PFNCGGETFIRSTSTATEASSIGNMENTPROC) lGetProcAddress(lHandle, "cgGetFirstStateAssignment");
  cgGetNamedStateAssignment = (PFNCGGETNAMEDSTATEASSIGNMENTPROC) lGetProcAddress(lHandle, "cgGetNamedStateAssignment");
  cgGetNextStateAssignment = (PFNCGGETNEXTSTATEASSIGNMENTPROC) lGetProcAddress(lHandle, "cgGetNextStateAssignment");
  cgIsStateAssignment = (PFNCGISSTATEASSIGNMENTPROC) lGetProcAddress(lHandle, "cgIsStateAssignment");
  cgCallStateSetCallback = (PFNCGCALLSTATESETCALLBACKPROC) lGetProcAddress(lHandle, "cgCallStateSetCallback");
  cgCallStateValidateCallback = (PFNCGCALLSTATEVALIDATECALLBACKPROC) lGetProcAddress(lHandle, "cgCallStateValidateCallback");
  cgCallStateResetCallback = (PFNCGCALLSTATERESETCALLBACKPROC) lGetProcAddress(lHandle, "cgCallStateResetCallback");
  cgGetStateAssignmentPass = (PFNCGGETSTATEASSIGNMENTPASSPROC) lGetProcAddress(lHandle, "cgGetStateAssignmentPass");
  cgGetSamplerStateAssignmentParameter = (PFNCGGETSAMPLERSTATEASSIGNMENTPARAMETERPROC) lGetProcAddress(lHandle, "cgGetSamplerStateAssignmentParameter");
  cgGetFloatStateAssignmentValues = (PFNCGGETFLOATSTATEASSIGNMENTVALUESPROC) lGetProcAddress(lHandle, "cgGetFloatStateAssignmentValues");
  cgGetIntStateAssignmentValues = (PFNCGGETINTSTATEASSIGNMENTVALUESPROC) lGetProcAddress(lHandle, "cgGetIntStateAssignmentValues");
  cgGetBoolStateAssignmentValues = (PFNCGGETBOOLSTATEASSIGNMENTVALUESPROC) lGetProcAddress(lHandle, "cgGetBoolStateAssignmentValues");
  cgGetStringStateAssignmentValue = (PFNCGGETSTRINGSTATEASSIGNMENTVALUEPROC) lGetProcAddress(lHandle, "cgGetStringStateAssignmentValue");
  cgGetProgramStateAssignmentValue = (PFNCGGETPROGRAMSTATEASSIGNMENTVALUEPROC) lGetProcAddress(lHandle, "cgGetProgramStateAssignmentValue");
  cgGetTextureStateAssignmentValue = (PFNCGGETTEXTURESTATEASSIGNMENTVALUEPROC) lGetProcAddress(lHandle, "cgGetTextureStateAssignmentValue");
  cgGetSamplerStateAssignmentValue = (PFNCGGETSAMPLERSTATEASSIGNMENTVALUEPROC) lGetProcAddress(lHandle, "cgGetSamplerStateAssignmentValue");
  cgGetStateAssignmentIndex = (PFNCGGETSTATEASSIGNMENTINDEXPROC) lGetProcAddress(lHandle, "cgGetStateAssignmentIndex");
  cgGetNumDependentStateAssignmentParameters = (PFNCGGETNUMDEPENDENTSTATEASSIGNMENTPARAMETERSPROC) lGetProcAddress(lHandle, "cgGetNumDependentStateAssignmentParameters");
  cgGetDependentStateAssignmentParameter = (PFNCGGETDEPENDENTSTATEASSIGNMENTPARAMETERPROC) lGetProcAddress(lHandle, "cgGetDependentStateAssignmentParameter");
  cgGetNumDependentProgramArrayStateAssignmentParameters = (PFNCGGETNUMDEPENDENTPROGRAMARRAYSTATEASSIGNMENTPARAMETERSPROC) lGetProcAddress(lHandle, "cgGetNumDependentProgramArrayStateAssignmentParameters");
  cgGetDependentProgramArrayStateAssignmentParameter = (PFNCGGETDEPENDENTPROGRAMARRAYSTATEASSIGNMENTPARAMETERPROC) lGetProcAddress(lHandle, "cgGetDependentProgramArrayStateAssignmentParameter");
  cgGetConnectedStateAssignmentParameter = (PFNCGGETCONNECTEDSTATEASSIGNMENTPARAMETERPROC) lGetProcAddress(lHandle, "cgGetConnectedStateAssignmentParameter");
  cgGetStateAssignmentState = (PFNCGGETSTATEASSIGNMENTSTATEPROC) lGetProcAddress(lHandle, "cgGetStateAssignmentState");
  cgGetSamplerStateAssignmentState = (PFNCGGETSAMPLERSTATEASSIGNMENTSTATEPROC) lGetProcAddress(lHandle, "cgGetSamplerStateAssignmentState");
  cgCreateState = (PFNCGCREATESTATEPROC) lGetProcAddress(lHandle, "cgCreateState");
  cgCreateArrayState = (PFNCGCREATEARRAYSTATEPROC) lGetProcAddress(lHandle, "cgCreateArrayState");
  cgSetStateCallbacks = (PFNCGSETSTATECALLBACKSPROC) lGetProcAddress(lHandle, "cgSetStateCallbacks");
  cgSetStateLatestProfile = (PFNCGSETSTATELATESTPROFILEPROC) lGetProcAddress(lHandle, "cgSetStateLatestProfile");
  cgGetStateSetCallback = (PFNCGGETSTATESETCALLBACKPROC) lGetProcAddress(lHandle, "cgGetStateSetCallback");
  cgGetStateResetCallback = (PFNCGGETSTATERESETCALLBACKPROC) lGetProcAddress(lHandle, "cgGetStateResetCallback");
  cgGetStateValidateCallback = (PFNCGGETSTATEVALIDATECALLBACKPROC) lGetProcAddress(lHandle, "cgGetStateValidateCallback");
  cgGetStateLatestProfile = (PFNCGGETSTATELATESTPROFILEPROC) lGetProcAddress(lHandle, "cgGetStateLatestProfile");
  cgGetStateContext = (PFNCGGETSTATECONTEXTPROC) lGetProcAddress(lHandle, "cgGetStateContext");
  cgGetStateType = (PFNCGGETSTATETYPEPROC) lGetProcAddress(lHandle, "cgGetStateType");
  cgGetStateName = (PFNCGGETSTATENAMEPROC) lGetProcAddress(lHandle, "cgGetStateName");
  cgGetNamedState = (PFNCGGETNAMEDSTATEPROC) lGetProcAddress(lHandle, "cgGetNamedState");
  cgGetFirstState = (PFNCGGETFIRSTSTATEPROC) lGetProcAddress(lHandle, "cgGetFirstState");
  cgGetNextState = (PFNCGGETNEXTSTATEPROC) lGetProcAddress(lHandle, "cgGetNextState");
  cgIsState = (PFNCGISSTATEPROC) lGetProcAddress(lHandle, "cgIsState");
  cgAddStateEnumerant = (PFNCGADDSTATEENUMERANTPROC) lGetProcAddress(lHandle, "cgAddStateEnumerant");
  cgCreateSamplerState = (PFNCGCREATESAMPLERSTATEPROC) lGetProcAddress(lHandle, "cgCreateSamplerState");
  cgCreateArraySamplerState = (PFNCGCREATEARRAYSAMPLERSTATEPROC) lGetProcAddress(lHandle, "cgCreateArraySamplerState");
  cgGetNamedSamplerState = (PFNCGGETNAMEDSAMPLERSTATEPROC) lGetProcAddress(lHandle, "cgGetNamedSamplerState");
  cgGetFirstSamplerState = (PFNCGGETFIRSTSAMPLERSTATEPROC) lGetProcAddress(lHandle, "cgGetFirstSamplerState");
  cgGetFirstSamplerStateAssignment = (PFNCGGETFIRSTSAMPLERSTATEASSIGNMENTPROC) lGetProcAddress(lHandle, "cgGetFirstSamplerStateAssignment");
  cgGetNamedSamplerStateAssignment = (PFNCGGETNAMEDSAMPLERSTATEASSIGNMENTPROC) lGetProcAddress(lHandle, "cgGetNamedSamplerStateAssignment");
  cgSetSamplerState = (PFNCGSETSAMPLERSTATEPROC) lGetProcAddress(lHandle, "cgSetSamplerState");
  cgGetNamedEffectParameter = (PFNCGGETNAMEDEFFECTPARAMETERPROC) lGetProcAddress(lHandle, "cgGetNamedEffectParameter");
  cgGetFirstLeafEffectParameter = (PFNCGGETFIRSTLEAFEFFECTPARAMETERPROC) lGetProcAddress(lHandle, "cgGetFirstLeafEffectParameter");
  cgGetFirstEffectParameter = (PFNCGGETFIRSTEFFECTPARAMETERPROC) lGetProcAddress(lHandle, "cgGetFirstEffectParameter");
  cgGetEffectParameterBySemantic = (PFNCGGETEFFECTPARAMETERBYSEMANTICPROC) lGetProcAddress(lHandle, "cgGetEffectParameterBySemantic");
  cgGetFirstTechniqueAnnotation = (PFNCGGETFIRSTTECHNIQUEANNOTATIONPROC) lGetProcAddress(lHandle, "cgGetFirstTechniqueAnnotation");
  cgGetFirstPassAnnotation = (PFNCGGETFIRSTPASSANNOTATIONPROC) lGetProcAddress(lHandle, "cgGetFirstPassAnnotation");
  cgGetFirstParameterAnnotation = (PFNCGGETFIRSTPARAMETERANNOTATIONPROC) lGetProcAddress(lHandle, "cgGetFirstParameterAnnotation");
  cgGetFirstProgramAnnotation = (PFNCGGETFIRSTPROGRAMANNOTATIONPROC) lGetProcAddress(lHandle, "cgGetFirstProgramAnnotation");
  cgGetFirstEffectAnnotation = (PFNCGGETFIRSTEFFECTANNOTATIONPROC) lGetProcAddress(lHandle, "cgGetFirstEffectAnnotation");
  cgGetNextAnnotation = (PFNCGGETNEXTANNOTATIONPROC) lGetProcAddress(lHandle, "cgGetNextAnnotation");
  cgGetNamedTechniqueAnnotation = (PFNCGGETNAMEDTECHNIQUEANNOTATIONPROC) lGetProcAddress(lHandle, "cgGetNamedTechniqueAnnotation");
  cgGetNamedPassAnnotation = (PFNCGGETNAMEDPASSANNOTATIONPROC) lGetProcAddress(lHandle, "cgGetNamedPassAnnotation");
  cgGetNamedParameterAnnotation = (PFNCGGETNAMEDPARAMETERANNOTATIONPROC) lGetProcAddress(lHandle, "cgGetNamedParameterAnnotation");
  cgGetNamedProgramAnnotation = (PFNCGGETNAMEDPROGRAMANNOTATIONPROC) lGetProcAddress(lHandle, "cgGetNamedProgramAnnotation");
  cgGetNamedEffectAnnotation = (PFNCGGETNAMEDEFFECTANNOTATIONPROC) lGetProcAddress(lHandle, "cgGetNamedEffectAnnotation");
  cgIsAnnotation = (PFNCGISANNOTATIONPROC) lGetProcAddress(lHandle, "cgIsAnnotation");
  cgGetAnnotationName = (PFNCGGETANNOTATIONNAMEPROC) lGetProcAddress(lHandle, "cgGetAnnotationName");
  cgGetAnnotationType = (PFNCGGETANNOTATIONTYPEPROC) lGetProcAddress(lHandle, "cgGetAnnotationType");
  cgGetFloatAnnotationValues = (PFNCGGETFLOATANNOTATIONVALUESPROC) lGetProcAddress(lHandle, "cgGetFloatAnnotationValues");
  cgGetIntAnnotationValues = (PFNCGGETINTANNOTATIONVALUESPROC) lGetProcAddress(lHandle, "cgGetIntAnnotationValues");
  cgGetStringAnnotationValue = (PFNCGGETSTRINGANNOTATIONVALUEPROC) lGetProcAddress(lHandle, "cgGetStringAnnotationValue");
  cgGetStringAnnotationValues = (PFNCGGETSTRINGANNOTATIONVALUESPROC) lGetProcAddress(lHandle, "cgGetStringAnnotationValues");
  cgGetBoolAnnotationValues = (PFNCGGETBOOLANNOTATIONVALUESPROC) lGetProcAddress(lHandle, "cgGetBoolAnnotationValues");
  cgGetBooleanAnnotationValues = (PFNCGGETBOOLEANANNOTATIONVALUESPROC) lGetProcAddress(lHandle, "cgGetBooleanAnnotationValues");
  cgGetNumDependentAnnotationParameters = (PFNCGGETNUMDEPENDENTANNOTATIONPARAMETERSPROC) lGetProcAddress(lHandle, "cgGetNumDependentAnnotationParameters");
  cgGetDependentAnnotationParameter = (PFNCGGETDEPENDENTANNOTATIONPARAMETERPROC) lGetProcAddress(lHandle, "cgGetDependentAnnotationParameter");
  cgEvaluateProgram = (PFNCGEVALUATEPROGRAMPROC) lGetProcAddress(lHandle, "cgEvaluateProgram");
  cgSetEffectName = (PFNCGSETEFFECTNAMEPROC) lGetProcAddress(lHandle, "cgSetEffectName");
  cgGetEffectName = (PFNCGGETEFFECTNAMEPROC) lGetProcAddress(lHandle, "cgGetEffectName");
  cgGetNamedEffect = (PFNCGGETNAMEDEFFECTPROC) lGetProcAddress(lHandle, "cgGetNamedEffect");
  cgCreateEffectParameter = (PFNCGCREATEEFFECTPARAMETERPROC) lGetProcAddress(lHandle, "cgCreateEffectParameter");
  cgCreateTechnique = (PFNCGCREATETECHNIQUEPROC) lGetProcAddress(lHandle, "cgCreateTechnique");
  cgCreateEffectParameterArray = (PFNCGCREATEEFFECTPARAMETERARRAYPROC) lGetProcAddress(lHandle, "cgCreateEffectParameterArray");
  cgCreateEffectParameterMultiDimArray = (PFNCGCREATEEFFECTPARAMETERMULTIDIMARRAYPROC) lGetProcAddress(lHandle, "cgCreateEffectParameterMultiDimArray");
  cgCreatePass = (PFNCGCREATEPASSPROC) lGetProcAddress(lHandle, "cgCreatePass");
  cgCreateStateAssignment = (PFNCGCREATESTATEASSIGNMENTPROC) lGetProcAddress(lHandle, "cgCreateStateAssignment");
  cgCreateStateAssignmentIndex = (PFNCGCREATESTATEASSIGNMENTINDEXPROC) lGetProcAddress(lHandle, "cgCreateStateAssignmentIndex");
  cgCreateSamplerStateAssignment = (PFNCGCREATESAMPLERSTATEASSIGNMENTPROC) lGetProcAddress(lHandle, "cgCreateSamplerStateAssignment");
  cgSetFloatStateAssignment = (PFNCGSETFLOATSTATEASSIGNMENTPROC) lGetProcAddress(lHandle, "cgSetFloatStateAssignment");
  cgSetIntStateAssignment = (PFNCGSETINTSTATEASSIGNMENTPROC) lGetProcAddress(lHandle, "cgSetIntStateAssignment");
  cgSetBoolStateAssignment = (PFNCGSETBOOLSTATEASSIGNMENTPROC) lGetProcAddress(lHandle, "cgSetBoolStateAssignment");
  cgSetStringStateAssignment = (PFNCGSETSTRINGSTATEASSIGNMENTPROC) lGetProcAddress(lHandle, "cgSetStringStateAssignment");
  cgSetProgramStateAssignment = (PFNCGSETPROGRAMSTATEASSIGNMENTPROC) lGetProcAddress(lHandle, "cgSetProgramStateAssignment");
  cgSetSamplerStateAssignment = (PFNCGSETSAMPLERSTATEASSIGNMENTPROC) lGetProcAddress(lHandle, "cgSetSamplerStateAssignment");
  cgSetTextureStateAssignment = (PFNCGSETTEXTURESTATEASSIGNMENTPROC) lGetProcAddress(lHandle, "cgSetTextureStateAssignment");
  cgSetFloatArrayStateAssignment = (PFNCGSETFLOATARRAYSTATEASSIGNMENTPROC) lGetProcAddress(lHandle, "cgSetFloatArrayStateAssignment");
  cgSetIntArrayStateAssignment = (PFNCGSETINTARRAYSTATEASSIGNMENTPROC) lGetProcAddress(lHandle, "cgSetIntArrayStateAssignment");
  cgSetBoolArrayStateAssignment = (PFNCGSETBOOLARRAYSTATEASSIGNMENTPROC) lGetProcAddress(lHandle, "cgSetBoolArrayStateAssignment");
  cgCreateTechniqueAnnotation = (PFNCGCREATETECHNIQUEANNOTATIONPROC) lGetProcAddress(lHandle, "cgCreateTechniqueAnnotation");
  cgCreatePassAnnotation = (PFNCGCREATEPASSANNOTATIONPROC) lGetProcAddress(lHandle, "cgCreatePassAnnotation");
  cgCreateParameterAnnotation = (PFNCGCREATEPARAMETERANNOTATIONPROC) lGetProcAddress(lHandle, "cgCreateParameterAnnotation");
  cgCreateProgramAnnotation = (PFNCGCREATEPROGRAMANNOTATIONPROC) lGetProcAddress(lHandle, "cgCreateProgramAnnotation");
  cgCreateEffectAnnotation = (PFNCGCREATEEFFECTANNOTATIONPROC) lGetProcAddress(lHandle, "cgCreateEffectAnnotation");
  cgSetIntAnnotation = (PFNCGSETINTANNOTATIONPROC) lGetProcAddress(lHandle, "cgSetIntAnnotation");
  cgSetFloatAnnotation = (PFNCGSETFLOATANNOTATIONPROC) lGetProcAddress(lHandle, "cgSetFloatAnnotation");
  cgSetBoolAnnotation = (PFNCGSETBOOLANNOTATIONPROC) lGetProcAddress(lHandle, "cgSetBoolAnnotation");
  cgSetStringAnnotation = (PFNCGSETSTRINGANNOTATIONPROC) lGetProcAddress(lHandle, "cgSetStringAnnotation");
  cgGetNumStateEnumerants = (PFNCGGETNUMSTATEENUMERANTSPROC) lGetProcAddress(lHandle, "cgGetNumStateEnumerants");
  cgGetStateEnumerant = (PFNCGGETSTATEENUMERANTPROC) lGetProcAddress(lHandle, "cgGetStateEnumerant");
  cgGetStateEnumerantName = (PFNCGGETSTATEENUMERANTNAMEPROC) lGetProcAddress(lHandle, "cgGetStateEnumerantName");
  cgGetStateEnumerantValue = (PFNCGGETSTATEENUMERANTVALUEPROC) lGetProcAddress(lHandle, "cgGetStateEnumerantValue");
  cgGetParameterEffect = (PFNCGGETPARAMETEREFFECTPROC) lGetProcAddress(lHandle, "cgGetParameterEffect");
  cgGetTypeClass = (PFNCGGETTYPECLASSPROC) lGetProcAddress(lHandle, "cgGetTypeClass");
  cgGetTypeBase = (PFNCGGETTYPEBASEPROC) lGetProcAddress(lHandle, "cgGetTypeBase");
  cgGetTypeSizes = (PFNCGGETTYPESIZESPROC) lGetProcAddress(lHandle, "cgGetTypeSizes");
  cgGetMatrixSize = (PFNCGGETMATRIXSIZEPROC) lGetProcAddress(lHandle, "cgGetMatrixSize");
  cgGetNumProgramDomains = (PFNCGGETNUMPROGRAMDOMAINSPROC) lGetProcAddress(lHandle, "cgGetNumProgramDomains");
  cgGetProfileDomain = (PFNCGGETPROFILEDOMAINPROC) lGetProcAddress(lHandle, "cgGetProfileDomain");
  cgCombinePrograms = (PFNCGCOMBINEPROGRAMSPROC) lGetProcAddress(lHandle, "cgCombinePrograms");
  cgCombinePrograms2 = (PFNCGCOMBINEPROGRAMS2PROC) lGetProcAddress(lHandle, "cgCombinePrograms2");
  cgCombinePrograms3 = (PFNCGCOMBINEPROGRAMS3PROC) lGetProcAddress(lHandle, "cgCombinePrograms3");
  cgCombinePrograms4 = (PFNCGCOMBINEPROGRAMS4PROC) lGetProcAddress(lHandle, "cgCombinePrograms4");
  cgCombinePrograms5 = (PFNCGCOMBINEPROGRAMS5PROC) lGetProcAddress(lHandle, "cgCombinePrograms5");
  cgGetProgramDomainProfile = (PFNCGGETPROGRAMDOMAINPROFILEPROC) lGetProcAddress(lHandle, "cgGetProgramDomainProfile");
  cgGetProgramDomainProgram = (PFNCGGETPROGRAMDOMAINPROGRAMPROC) lGetProcAddress(lHandle, "cgGetProgramDomainProgram");
  cgCreateObj = (PFNCGCREATEOBJPROC) lGetProcAddress(lHandle, "cgCreateObj");
  cgCreateObjFromFile = (PFNCGCREATEOBJFROMFILEPROC) lGetProcAddress(lHandle, "cgCreateObjFromFile");
  cgDestroyObj = (PFNCGDESTROYOBJPROC) lGetProcAddress(lHandle, "cgDestroyObj");
  cgGetParameterResourceSize = (PFNCGGETPARAMETERRESOURCESIZEPROC) lGetProcAddress(lHandle, "cgGetParameterResourceSize");
  cgGetParameterResourceType = (PFNCGGETPARAMETERRESOURCETYPEPROC) lGetProcAddress(lHandle, "cgGetParameterResourceType");
  cgGetParameterResourceName = (PFNCGGETPARAMETERRESOURCENAMEPROC) lGetProcAddress(lHandle, "cgGetParameterResourceName");
  cgGetParameterBufferIndex = (PFNCGGETPARAMETERBUFFERINDEXPROC) lGetProcAddress(lHandle, "cgGetParameterBufferIndex");
  cgGetParameterBufferOffset = (PFNCGGETPARAMETERBUFFEROFFSETPROC) lGetProcAddress(lHandle, "cgGetParameterBufferOffset");
  cgCreateBuffer = (PFNCGCREATEBUFFERPROC) lGetProcAddress(lHandle, "cgCreateBuffer");
  cgSetBufferData = (PFNCGSETBUFFERDATAPROC) lGetProcAddress(lHandle, "cgSetBufferData");
  cgSetBufferSubData = (PFNCGSETBUFFERSUBDATAPROC) lGetProcAddress(lHandle, "cgSetBufferSubData");
  cgSetProgramBuffer = (PFNCGSETPROGRAMBUFFERPROC) lGetProcAddress(lHandle, "cgSetProgramBuffer");
  cgMapBuffer = (PFNCGMAPBUFFERPROC) lGetProcAddress(lHandle, "cgMapBuffer");
  cgUnmapBuffer = (PFNCGUNMAPBUFFERPROC) lGetProcAddress(lHandle, "cgUnmapBuffer");
  cgDestroyBuffer = (PFNCGDESTROYBUFFERPROC) lGetProcAddress(lHandle, "cgDestroyBuffer");
  cgGetProgramBuffer = (PFNCGGETPROGRAMBUFFERPROC) lGetProcAddress(lHandle, "cgGetProgramBuffer");
  cgGetBufferSize = (PFNCGGETBUFFERSIZEPROC) lGetProcAddress(lHandle, "cgGetBufferSize");
  cgGetProgramBufferMaxSize = (PFNCGGETPROGRAMBUFFERMAXSIZEPROC) lGetProcAddress(lHandle, "cgGetProgramBufferMaxSize");
  cgGetProgramBufferMaxIndex = (PFNCGGETPROGRAMBUFFERMAXINDEXPROC) lGetProcAddress(lHandle, "cgGetProgramBufferMaxIndex");
  cgGetEffectParameterBuffer = (PFNCGGETEFFECTPARAMETERBUFFERPROC) lGetProcAddress(lHandle, "cgGetEffectParameterBuffer");
  cgSetEffectParameterBuffer = (PFNCGSETEFFECTPARAMETERBUFFERPROC) lGetProcAddress(lHandle, "cgSetEffectParameterBuffer");

  return 1;
}

void
lList(const char *prefix, FILE *out)
{
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetLockingPolicy", cgSetLockingPolicy ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetLockingPolicy", cgGetLockingPolicy ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetSemanticCasePolicy", cgSetSemanticCasePolicy ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetSemanticCasePolicy", cgGetSemanticCasePolicy ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgCreateContext", cgCreateContext ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetContextBehavior", cgSetContextBehavior ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetContextBehavior", cgGetContextBehavior ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetBehaviorString", cgGetBehaviorString ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetBehavior", cgGetBehavior ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgDestroyContext", cgDestroyContext ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgIsContext", cgIsContext ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetLastListing", cgGetLastListing ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetLastListing", cgSetLastListing ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetAutoCompile", cgSetAutoCompile ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetAutoCompile", cgGetAutoCompile ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetParameterSettingMode", cgSetParameterSettingMode ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetParameterSettingMode", cgGetParameterSettingMode ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetCompilerIncludeString", cgSetCompilerIncludeString ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetCompilerIncludeFile", cgSetCompilerIncludeFile ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetCompilerIncludeCallback", cgSetCompilerIncludeCallback ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetCompilerIncludeCallback", cgGetCompilerIncludeCallback ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgCreateProgram", cgCreateProgram ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgCreateProgramFromFile", cgCreateProgramFromFile ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgCopyProgram", cgCopyProgram ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgDestroyProgram", cgDestroyProgram ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetFirstProgram", cgGetFirstProgram ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetNextProgram", cgGetNextProgram ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetProgramContext", cgGetProgramContext ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgIsProgram", cgIsProgram ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgCompileProgram", cgCompileProgram ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgIsProgramCompiled", cgIsProgramCompiled ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetProgramString", cgGetProgramString ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetProgramProfile", cgGetProgramProfile ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetProgramOptions", cgGetProgramOptions ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetProgramProfile", cgSetProgramProfile ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetProgramInput", cgGetProgramInput ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetProgramOutput", cgGetProgramOutput ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetPassProgramParameters", cgSetPassProgramParameters ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgUpdateProgramParameters", cgUpdateProgramParameters ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgUpdatePassParameters", cgUpdatePassParameters ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgCreateParameter", cgCreateParameter ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgCreateParameterArray", cgCreateParameterArray ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgCreateParameterMultiDimArray", cgCreateParameterMultiDimArray ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgDestroyParameter", cgDestroyParameter ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgConnectParameter", cgConnectParameter ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgDisconnectParameter", cgDisconnectParameter ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetConnectedParameter", cgGetConnectedParameter ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetNumConnectedToParameters", cgGetNumConnectedToParameters ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetConnectedToParameter", cgGetConnectedToParameter ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetNamedParameter", cgGetNamedParameter ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetNamedProgramParameter", cgGetNamedProgramParameter ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetFirstParameter", cgGetFirstParameter ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetNextParameter", cgGetNextParameter ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetFirstLeafParameter", cgGetFirstLeafParameter ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetNextLeafParameter", cgGetNextLeafParameter ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetFirstStructParameter", cgGetFirstStructParameter ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetNamedStructParameter", cgGetNamedStructParameter ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetFirstDependentParameter", cgGetFirstDependentParameter ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetArrayParameter", cgGetArrayParameter ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetArrayDimension", cgGetArrayDimension ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetArrayType", cgGetArrayType ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetArraySize", cgGetArraySize ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetArrayTotalSize", cgGetArrayTotalSize ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetArraySize", cgSetArraySize ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetMultiDimArraySize", cgSetMultiDimArraySize ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetParameterProgram", cgGetParameterProgram ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetParameterContext", cgGetParameterContext ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgIsParameter", cgIsParameter ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetParameterName", cgGetParameterName ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetParameterType", cgGetParameterType ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetParameterBaseType", cgGetParameterBaseType ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetParameterClass", cgGetParameterClass ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetParameterRows", cgGetParameterRows ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetParameterColumns", cgGetParameterColumns ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetParameterNamedType", cgGetParameterNamedType ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetParameterSemantic", cgGetParameterSemantic ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetParameterResource", cgGetParameterResource ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetParameterBaseResource", cgGetParameterBaseResource ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetParameterResourceIndex", cgGetParameterResourceIndex ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetParameterVariability", cgGetParameterVariability ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetParameterDirection", cgGetParameterDirection ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgIsParameterReferenced", cgIsParameterReferenced ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgIsParameterUsed", cgIsParameterUsed ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetParameterValues", cgGetParameterValues ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetParameterValuedr", cgSetParameterValuedr ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetParameterValuedc", cgSetParameterValuedc ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetParameterValuefr", cgSetParameterValuefr ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetParameterValuefc", cgSetParameterValuefc ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetParameterValueir", cgSetParameterValueir ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetParameterValueic", cgSetParameterValueic ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetParameterValuedr", cgGetParameterValuedr ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetParameterValuedc", cgGetParameterValuedc ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetParameterValuefr", cgGetParameterValuefr ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetParameterValuefc", cgGetParameterValuefc ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetParameterValueir", cgGetParameterValueir ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetParameterValueic", cgGetParameterValueic ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetParameterDefaultValuedr", cgGetParameterDefaultValuedr ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetParameterDefaultValuedc", cgGetParameterDefaultValuedc ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetParameterDefaultValuefr", cgGetParameterDefaultValuefr ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetParameterDefaultValuefc", cgGetParameterDefaultValuefc ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetParameterDefaultValueir", cgGetParameterDefaultValueir ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetParameterDefaultValueic", cgGetParameterDefaultValueic ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetStringParameterValue", cgGetStringParameterValue ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetStringParameterValue", cgSetStringParameterValue ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetParameterOrdinalNumber", cgGetParameterOrdinalNumber ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgIsParameterGlobal", cgIsParameterGlobal ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetParameterIndex", cgGetParameterIndex ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetParameterVariability", cgSetParameterVariability ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetParameterSemantic", cgSetParameterSemantic ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetParameter1f", cgSetParameter1f ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetParameter2f", cgSetParameter2f ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetParameter3f", cgSetParameter3f ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetParameter4f", cgSetParameter4f ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetParameter1d", cgSetParameter1d ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetParameter2d", cgSetParameter2d ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetParameter3d", cgSetParameter3d ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetParameter4d", cgSetParameter4d ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetParameter1i", cgSetParameter1i ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetParameter2i", cgSetParameter2i ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetParameter3i", cgSetParameter3i ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetParameter4i", cgSetParameter4i ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetParameter1iv", cgSetParameter1iv ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetParameter2iv", cgSetParameter2iv ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetParameter3iv", cgSetParameter3iv ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetParameter4iv", cgSetParameter4iv ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetParameter1fv", cgSetParameter1fv ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetParameter2fv", cgSetParameter2fv ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetParameter3fv", cgSetParameter3fv ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetParameter4fv", cgSetParameter4fv ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetParameter1dv", cgSetParameter1dv ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetParameter2dv", cgSetParameter2dv ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetParameter3dv", cgSetParameter3dv ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetParameter4dv", cgSetParameter4dv ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetMatrixParameterir", cgSetMatrixParameterir ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetMatrixParameterdr", cgSetMatrixParameterdr ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetMatrixParameterfr", cgSetMatrixParameterfr ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetMatrixParameteric", cgSetMatrixParameteric ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetMatrixParameterdc", cgSetMatrixParameterdc ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetMatrixParameterfc", cgSetMatrixParameterfc ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetMatrixParameterir", cgGetMatrixParameterir ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetMatrixParameterdr", cgGetMatrixParameterdr ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetMatrixParameterfr", cgGetMatrixParameterfr ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetMatrixParameteric", cgGetMatrixParameteric ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetMatrixParameterdc", cgGetMatrixParameterdc ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetMatrixParameterfc", cgGetMatrixParameterfc ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetMatrixParameterOrder", cgGetMatrixParameterOrder ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetNamedSubParameter", cgGetNamedSubParameter ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetTypeString", cgGetTypeString ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetType", cgGetType ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetNamedUserType", cgGetNamedUserType ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetNumUserTypes", cgGetNumUserTypes ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetUserType", cgGetUserType ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetNumParentTypes", cgGetNumParentTypes ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetParentType", cgGetParentType ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgIsParentType", cgIsParentType ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgIsInterfaceType", cgIsInterfaceType ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetResourceString", cgGetResourceString ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetResource", cgGetResource ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetEnumString", cgGetEnumString ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetEnum", cgGetEnum ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetProfileString", cgGetProfileString ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetProfile", cgGetProfile ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetNumSupportedProfiles", cgGetNumSupportedProfiles ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetSupportedProfile", cgGetSupportedProfile ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgIsProfileSupported", cgIsProfileSupported ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetProfileProperty", cgGetProfileProperty ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetParameterClassString", cgGetParameterClassString ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetParameterClassEnum", cgGetParameterClassEnum ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetDomainString", cgGetDomainString ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetDomain", cgGetDomain ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetProgramDomain", cgGetProgramDomain ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetError", cgGetError ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetFirstError", cgGetFirstError ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetErrorString", cgGetErrorString ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetLastErrorString", cgGetLastErrorString ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetErrorCallback", cgSetErrorCallback ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetErrorCallback", cgGetErrorCallback ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetErrorHandler", cgSetErrorHandler ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetErrorHandler", cgGetErrorHandler ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetString", cgGetString ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgCreateEffect", cgCreateEffect ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgCreateEffectFromFile", cgCreateEffectFromFile ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgCopyEffect", cgCopyEffect ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgDestroyEffect", cgDestroyEffect ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetEffectContext", cgGetEffectContext ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgIsEffect", cgIsEffect ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetFirstEffect", cgGetFirstEffect ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetNextEffect", cgGetNextEffect ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgCreateProgramFromEffect", cgCreateProgramFromEffect ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetFirstTechnique", cgGetFirstTechnique ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetNextTechnique", cgGetNextTechnique ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetNamedTechnique", cgGetNamedTechnique ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetTechniqueName", cgGetTechniqueName ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgIsTechnique", cgIsTechnique ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgValidateTechnique", cgValidateTechnique ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgIsTechniqueValidated", cgIsTechniqueValidated ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetTechniqueEffect", cgGetTechniqueEffect ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetFirstPass", cgGetFirstPass ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetNamedPass", cgGetNamedPass ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetNextPass", cgGetNextPass ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgIsPass", cgIsPass ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetPassName", cgGetPassName ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetPassTechnique", cgGetPassTechnique ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetPassProgram", cgGetPassProgram ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetPassState", cgSetPassState ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgResetPassState", cgResetPassState ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetFirstStateAssignment", cgGetFirstStateAssignment ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetNamedStateAssignment", cgGetNamedStateAssignment ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetNextStateAssignment", cgGetNextStateAssignment ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgIsStateAssignment", cgIsStateAssignment ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgCallStateSetCallback", cgCallStateSetCallback ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgCallStateValidateCallback", cgCallStateValidateCallback ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgCallStateResetCallback", cgCallStateResetCallback ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetStateAssignmentPass", cgGetStateAssignmentPass ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetSamplerStateAssignmentParameter", cgGetSamplerStateAssignmentParameter ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetFloatStateAssignmentValues", cgGetFloatStateAssignmentValues ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetIntStateAssignmentValues", cgGetIntStateAssignmentValues ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetBoolStateAssignmentValues", cgGetBoolStateAssignmentValues ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetStringStateAssignmentValue", cgGetStringStateAssignmentValue ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetProgramStateAssignmentValue", cgGetProgramStateAssignmentValue ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetTextureStateAssignmentValue", cgGetTextureStateAssignmentValue ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetSamplerStateAssignmentValue", cgGetSamplerStateAssignmentValue ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetStateAssignmentIndex", cgGetStateAssignmentIndex ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetNumDependentStateAssignmentParameters", cgGetNumDependentStateAssignmentParameters ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetDependentStateAssignmentParameter", cgGetDependentStateAssignmentParameter ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetNumDependentProgramArrayStateAssignmentParameters", cgGetNumDependentProgramArrayStateAssignmentParameters ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetDependentProgramArrayStateAssignmentParameter", cgGetDependentProgramArrayStateAssignmentParameter ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetConnectedStateAssignmentParameter", cgGetConnectedStateAssignmentParameter ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetStateAssignmentState", cgGetStateAssignmentState ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetSamplerStateAssignmentState", cgGetSamplerStateAssignmentState ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgCreateState", cgCreateState ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgCreateArrayState", cgCreateArrayState ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetStateCallbacks", cgSetStateCallbacks ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetStateLatestProfile", cgSetStateLatestProfile ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetStateSetCallback", cgGetStateSetCallback ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetStateResetCallback", cgGetStateResetCallback ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetStateValidateCallback", cgGetStateValidateCallback ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetStateLatestProfile", cgGetStateLatestProfile ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetStateContext", cgGetStateContext ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetStateType", cgGetStateType ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetStateName", cgGetStateName ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetNamedState", cgGetNamedState ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetFirstState", cgGetFirstState ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetNextState", cgGetNextState ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgIsState", cgIsState ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgAddStateEnumerant", cgAddStateEnumerant ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgCreateSamplerState", cgCreateSamplerState ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgCreateArraySamplerState", cgCreateArraySamplerState ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetNamedSamplerState", cgGetNamedSamplerState ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetFirstSamplerState", cgGetFirstSamplerState ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetFirstSamplerStateAssignment", cgGetFirstSamplerStateAssignment ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetNamedSamplerStateAssignment", cgGetNamedSamplerStateAssignment ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetSamplerState", cgSetSamplerState ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetNamedEffectParameter", cgGetNamedEffectParameter ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetFirstLeafEffectParameter", cgGetFirstLeafEffectParameter ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetFirstEffectParameter", cgGetFirstEffectParameter ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetEffectParameterBySemantic", cgGetEffectParameterBySemantic ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetFirstTechniqueAnnotation", cgGetFirstTechniqueAnnotation ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetFirstPassAnnotation", cgGetFirstPassAnnotation ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetFirstParameterAnnotation", cgGetFirstParameterAnnotation ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetFirstProgramAnnotation", cgGetFirstProgramAnnotation ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetFirstEffectAnnotation", cgGetFirstEffectAnnotation ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetNextAnnotation", cgGetNextAnnotation ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetNamedTechniqueAnnotation", cgGetNamedTechniqueAnnotation ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetNamedPassAnnotation", cgGetNamedPassAnnotation ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetNamedParameterAnnotation", cgGetNamedParameterAnnotation ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetNamedProgramAnnotation", cgGetNamedProgramAnnotation ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetNamedEffectAnnotation", cgGetNamedEffectAnnotation ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgIsAnnotation", cgIsAnnotation ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetAnnotationName", cgGetAnnotationName ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetAnnotationType", cgGetAnnotationType ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetFloatAnnotationValues", cgGetFloatAnnotationValues ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetIntAnnotationValues", cgGetIntAnnotationValues ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetStringAnnotationValue", cgGetStringAnnotationValue ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetStringAnnotationValues", cgGetStringAnnotationValues ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetBoolAnnotationValues", cgGetBoolAnnotationValues ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetBooleanAnnotationValues", cgGetBooleanAnnotationValues ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetNumDependentAnnotationParameters", cgGetNumDependentAnnotationParameters ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetDependentAnnotationParameter", cgGetDependentAnnotationParameter ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgEvaluateProgram", cgEvaluateProgram ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetEffectName", cgSetEffectName ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetEffectName", cgGetEffectName ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetNamedEffect", cgGetNamedEffect ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgCreateEffectParameter", cgCreateEffectParameter ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgCreateTechnique", cgCreateTechnique ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgCreateEffectParameterArray", cgCreateEffectParameterArray ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgCreateEffectParameterMultiDimArray", cgCreateEffectParameterMultiDimArray ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgCreatePass", cgCreatePass ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgCreateStateAssignment", cgCreateStateAssignment ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgCreateStateAssignmentIndex", cgCreateStateAssignmentIndex ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgCreateSamplerStateAssignment", cgCreateSamplerStateAssignment ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetFloatStateAssignment", cgSetFloatStateAssignment ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetIntStateAssignment", cgSetIntStateAssignment ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetBoolStateAssignment", cgSetBoolStateAssignment ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetStringStateAssignment", cgSetStringStateAssignment ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetProgramStateAssignment", cgSetProgramStateAssignment ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetSamplerStateAssignment", cgSetSamplerStateAssignment ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetTextureStateAssignment", cgSetTextureStateAssignment ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetFloatArrayStateAssignment", cgSetFloatArrayStateAssignment ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetIntArrayStateAssignment", cgSetIntArrayStateAssignment ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetBoolArrayStateAssignment", cgSetBoolArrayStateAssignment ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgCreateTechniqueAnnotation", cgCreateTechniqueAnnotation ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgCreatePassAnnotation", cgCreatePassAnnotation ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgCreateParameterAnnotation", cgCreateParameterAnnotation ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgCreateProgramAnnotation", cgCreateProgramAnnotation ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgCreateEffectAnnotation", cgCreateEffectAnnotation ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetIntAnnotation", cgSetIntAnnotation ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetFloatAnnotation", cgSetFloatAnnotation ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetBoolAnnotation", cgSetBoolAnnotation ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetStringAnnotation", cgSetStringAnnotation ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetNumStateEnumerants", cgGetNumStateEnumerants ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetStateEnumerant", cgGetStateEnumerant ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetStateEnumerantName", cgGetStateEnumerantName ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetStateEnumerantValue", cgGetStateEnumerantValue ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetParameterEffect", cgGetParameterEffect ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetTypeClass", cgGetTypeClass ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetTypeBase", cgGetTypeBase ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetTypeSizes", cgGetTypeSizes ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetMatrixSize", cgGetMatrixSize ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetNumProgramDomains", cgGetNumProgramDomains ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetProfileDomain", cgGetProfileDomain ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgCombinePrograms", cgCombinePrograms ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgCombinePrograms2", cgCombinePrograms2 ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgCombinePrograms3", cgCombinePrograms3 ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgCombinePrograms4", cgCombinePrograms4 ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgCombinePrograms5", cgCombinePrograms5 ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetProgramDomainProfile", cgGetProgramDomainProfile ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetProgramDomainProgram", cgGetProgramDomainProgram ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgCreateObj", cgCreateObj ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgCreateObjFromFile", cgCreateObjFromFile ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgDestroyObj", cgDestroyObj ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetParameterResourceSize", cgGetParameterResourceSize ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetParameterResourceType", cgGetParameterResourceType ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetParameterResourceName", cgGetParameterResourceName ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetParameterBufferIndex", cgGetParameterBufferIndex ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetParameterBufferOffset", cgGetParameterBufferOffset ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgCreateBuffer", cgCreateBuffer ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetBufferData", cgSetBufferData ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetBufferSubData", cgSetBufferSubData ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetProgramBuffer", cgSetProgramBuffer ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgMapBuffer", cgMapBuffer ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgUnmapBuffer", cgUnmapBuffer ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgDestroyBuffer", cgDestroyBuffer ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetProgramBuffer", cgGetProgramBuffer ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetBufferSize", cgGetBufferSize ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetProgramBufferMaxSize", cgGetProgramBufferMaxSize ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetProgramBufferMaxIndex", cgGetProgramBufferMaxIndex ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgGetEffectParameterBuffer", cgGetEffectParameterBuffer ? "found" : "missing");
  fprintf(out, "%s%-40s ... %s\n", prefix, "cgSetEffectParameterBuffer", cgSetEffectParameterBuffer ? "found" : "missing");
}

void
lDestroy(void)
{
  if (lHandle)
  {
    lFreeLibrary(lHandle);
    lHandle = NULL;
  }
}
