/* Nestra mapper.c */
/* Public Domain */

#include "mapper.h"
#include "consts.h"

/* The mapper works via a table (MAPTABLE) which is an array of pointers. */
/* Each pointer defines where reads of its respective 4K bank of memory   */
/* are redirected to.  Writes into the rom area generate calls to one of  */
/* the functions defined below, which emulate various mapper chips.       */

extern void MAPPER_NONE;
extern void MAPPER_MMC1;
extern void MAPPER_UNROM;
extern void MAPPER_CHROM;
extern void MAPPER_MMC3;
extern void MAPPER_MMC2;
extern void MAPPER_CLRDRMS;
extern void MAPPER_AOROM;

void *Mapper[12]={
	&MAPPER_NONE, /* 0 */
	&MAPPER_MMC1, /* 1 */
	&MAPPER_UNROM,/* 2 */
	&MAPPER_CHROM,/* 3 */
	&MAPPER_MMC3, /* 4 */
	0, /* 5 */
	0, /* 6 */
	&MAPPER_AOROM,/* 7 */
	0, /* 8 */
	&MAPPER_MMC2, /* 9 */
	0, /* 10 */
	&MAPPER_CLRDRMS, /* 11 */
	/* More Mappers Go Here */
};

/* ROM mapper table (pointers) */
u_char *MAPTABLE[17];

/* some globals */
extern unsigned char *ROM_BASE;
extern unsigned char *VROM_BASE;
extern unsigned char vram[];
extern unsigned int ROM_PAGES;
extern unsigned int VROM_PAGES;
extern unsigned int ROM_MASK;
extern unsigned int VROM_MASK;
extern unsigned int VROM_MASK_1k;
extern unsigned int hvmirror;
extern unsigned int osmirror;
extern unsigned int nomirror;
extern unsigned int mapmirror;
extern unsigned int irqflag;
extern unsigned int CLOCK;
extern signed int CTNI;
extern void(*drawimage)();
#define LAST_PAGE (ROM_PAGES-1)
#define LAST_HALF_PAGE ((ROM_PAGES<<1)-1)

void init_none()
{
  MAPTABLE[8]=MAPTABLE[9]=MAPTABLE[10]=MAPTABLE[11]=
  MAPTABLE[12]=MAPTABLE[13]=MAPTABLE[14]=MAPTABLE[15]=ROM_BASE-0x8000;

  /* for 16K roms */
  if(ROM_PAGES<2) MAPTABLE[12]=MAPTABLE[13]=MAPTABLE[14]=MAPTABLE[15]=ROM_BASE-0xC000;
  
  mapmirror=0;
}

/* defaults for MMC1 (Zelda, etc) */
void init_mmc1(){
  /* First page is mapped to first page of rom */
  /* (first page of rom is at 0x8000, hence the -0x8000 offset) */
  MAPTABLE[8]=MAPTABLE[9]=MAPTABLE[10]=MAPTABLE[11]=ROM_BASE-0x8000;

  /* Last page is mapped to last page of rom */
  MAPTABLE[12]=MAPTABLE[13]=MAPTABLE[14]=MAPTABLE[15]=ROM_BASE-0x8000;
  if(ROM_PAGES>2)
  MAPTABLE[12]=MAPTABLE[13]=MAPTABLE[14]=MAPTABLE[15]=ROM_BASE;
  if(ROM_PAGES>4)
  MAPTABLE[12]=MAPTABLE[13]=MAPTABLE[14]=MAPTABLE[15]=ROM_BASE+0x10000;
  if(ROM_PAGES>8)
  MAPTABLE[12]=MAPTABLE[13]=MAPTABLE[14]=MAPTABLE[15]=ROM_BASE+0x30000;

  mapmirror=1;  
}

mmc1(addr,val)
{
  static int mmc1reg[4];
  static int mmc1shc[4];
  val&=0xff;
  /*printf("Mapper MMC1:%4x,%2x\n",addr,val);/**/
  /*printf("Mapper: %4x,%2x stack at %x, shift %d,%d,%d,%d\n",addr,val,STACKPTR,
    mmc1shc[0],mmc1shc[1],mmc1shc[2],mmc1shc[3]);/**/

  if(val&0x80) {
    /* Reset Mapper */
    mmc1reg[0]|=12;mmc1shc[0]=0;
    /* When the mapper is reset, does it affect just one register or all?
       I think just one, but I am unsure of the exact effect of a reset. */
    /*mmc1reg[1]=0;*/mmc1shc[1]=0;
    /*mmc1reg[2]=0;*/mmc1shc[2]=0;
    /*mmc1reg[3]=0;*/mmc1shc[3]=0;
    /*mmc1reg[1]=mmc1reg[2]=mmc1reg[3]=0;*/
    nomirror=0; /* MMC1 always has mirroring */
    /*printf("Mapper: MMC1 reset\n");/**/
  }
  else {mmc1reg[(addr>>13)&3]>>=1;mmc1reg[(addr>>13)&3]|=(val&1)<<4;mmc1shc[(addr>>13)&3]++;
    if(mmc1shc[(addr>>13)&3]%5==0)
    {
      /*printf("mapper reg %d set to %d (clock %d)\n",(addr>>13)&3,mmc1reg[(addr>>13)&3],CLOCK); /* debug info*/
      if(((addr>>13)&3)==0){
        drawimage(CLOCK*3);
        hvmirror=mmc1reg[0]&1;
        osmirror=(~mmc1reg[0]&2)>>1;
        /*printf("hvmirror:%d osmirror:%d\n",hvmirror,osmirror);*/
      }
      else if(VROM_PAGES)
      {
        drawimage(CLOCK*3);
        if(((addr>>13)&3)==1)
          memcpy(VRAM,VROM_BASE+(mmc1reg[1]&31)*0x1000,4096<<((mmc1reg[0]&16)==0));
        if(((addr>>13)&3)==2&&(mmc1reg[0]&16)==16)
          memcpy(VRAM+0x1000,VROM_BASE+(mmc1reg[2]&31)*0x1000,4096);
      }
      
      /* Set Map Table */
      
      if(mmc1reg[0]&8) /* Swap 16K rom */
      {
        if(mmc1reg[0]&4)
        { /* Swap $8000 */
          MAPTABLE[8]=MAPTABLE[9]=MAPTABLE[10]=MAPTABLE[11]=ROM_BASE+((mmc1reg[3]&ROM_MASK)<<14)-0x8000;
          MAPTABLE[12]=MAPTABLE[13]=MAPTABLE[14]=MAPTABLE[15]=ROM_BASE+(ROM_PAGES<<14)-0x10000; /* Last Page hardwired */
          if(ROM_PAGES>16) /* 512K roms */
          {
            MAPTABLE[8]=MAPTABLE[9]=MAPTABLE[10]=MAPTABLE[11]=ROM_BASE+(mmc1reg[3]<<14)+((mmc1reg[1]&16)<<14)-0x8000;
            MAPTABLE[12]=MAPTABLE[13]=MAPTABLE[14]=MAPTABLE[15]=ROM_BASE+262144+((mmc1reg[1]&16)<<14)-0x10000; /* Last Page semi-hardwired */
          }
        }
        else
        { /* Swap $C000 */
          MAPTABLE[8]=MAPTABLE[9]=MAPTABLE[10]=MAPTABLE[11]=ROM_BASE-0x8000; /* First page hardwired */
          MAPTABLE[12]=MAPTABLE[13]=MAPTABLE[14]=MAPTABLE[15]=ROM_BASE+((mmc1reg[3]&ROM_MASK)<<14)+(((mmc1reg[1]&16)<<14)&((ROM_PAGES>16)<<18))-0xC000;
        }
      }
      else /* Swap 32K rom */
      {
        MAPTABLE[8]=MAPTABLE[9]=MAPTABLE[10]=MAPTABLE[11]=
        MAPTABLE[12]=MAPTABLE[13]=MAPTABLE[14]=MAPTABLE[15]=
        ROM_BASE+((mmc1reg[3]&ROM_MASK&(~1))<<14)+(((mmc1reg[1]&16)<<14)&((ROM_PAGES>16)<<18))-0x8000;
      }
    }
  }
}

init_unrom() {init_mmc1();mapmirror=0;}

unrom(addr,val)
{
  val&=0xff;
  val&=7;
  MAPTABLE[8]=MAPTABLE[9]=MAPTABLE[10]=MAPTABLE[11]=ROM_BASE+16384*val-0x8000;
}

init_crom() {init_none();}

crom(addr,val)
{
  val&=VROM_MASK;
  memcpy(VRAM,VROM_BASE+val*8192,8192);
}

/* defaults for MMC3 (SMB2, etc) */
void init_mmc3(){
  /* First page is mapped to second-to-last page of rom */
  MAPTABLE[8]=MAPTABLE[9]=ROM_BASE-0x4000;
  if(ROM_PAGES>2)
  MAPTABLE[8]=MAPTABLE[9]=ROM_BASE+0x4000;
  if(ROM_PAGES>4)
  MAPTABLE[8]=MAPTABLE[9]=ROM_BASE+0x14000;
  if(ROM_PAGES>8)
  MAPTABLE[8]=MAPTABLE[9]=ROM_BASE+0x34000;
  if(ROM_PAGES>16)
  MAPTABLE[8]=MAPTABLE[9]=ROM_BASE+0x74000;

  MAPTABLE[10]=MAPTABLE[11]=ROM_BASE+(ROM_PAGES<<14)-0xE000;

  /* Last page is mapped to last page of rom */
  MAPTABLE[12]=MAPTABLE[13]=MAPTABLE[14]=MAPTABLE[15]=ROM_BASE-0x8000;
  if(ROM_PAGES>2)
  MAPTABLE[12]=MAPTABLE[13]=MAPTABLE[14]=MAPTABLE[15]=ROM_BASE;
  if(ROM_PAGES>4)
  MAPTABLE[12]=MAPTABLE[13]=MAPTABLE[14]=MAPTABLE[15]=ROM_BASE+0x10000;
  if(ROM_PAGES>8)
  MAPTABLE[12]=MAPTABLE[13]=MAPTABLE[14]=MAPTABLE[15]=ROM_BASE+0x30000;
  if(ROM_PAGES>16)
  MAPTABLE[12]=MAPTABLE[13]=MAPTABLE[14]=MAPTABLE[15]=ROM_BASE+0x70000;

  mapmirror=1;
}

mmc3(addr,val)
{  
  static unsigned char mmc3cmd;
  static unsigned char irqval;
  static unsigned char irqenabled=0;

  val&=0xff;
  /*printf("Mapper MMC3:%4x,%2x (%d)\n",addr,val,CLOCK);/**/
  if(addr==0x8000) mmc3cmd=val;
  if(addr==0x8001)
  {
    if((mmc3cmd&7)<6) drawimage(CLOCK*3);

    if((mmc3cmd&0x87)==0) /* Switch first 2k video ROM segment */
      memcpy(VRAM,VROM_BASE+(val&VROM_MASK_1k&(~1))*1024,2048);
    if((mmc3cmd&0x87)==1) /* Switch second 2k video ROM segment */
      memcpy(VRAM+0x800,VROM_BASE+(val&VROM_MASK_1k&(~1))*1024,2048);
    if((mmc3cmd&0x87)==0x80) /* Switch first 2k video ROM segment to alternate address */
      memcpy(VRAM+0x1000,VROM_BASE+(val&VROM_MASK_1k&(~1))*1024,2048);
    if((mmc3cmd&0x87)==0x81) /* Switch second 2k video ROM segment to alternate address */
      memcpy(VRAM+0x1800,VROM_BASE+(val&VROM_MASK_1k&(~1))*1024,2048);

    if((mmc3cmd&0x87)==2) /* Switch first 1k video ROM segment */
      memcpy(VRAM+4096,VROM_BASE+(val&VROM_MASK_1k)*1024,1024);
    if((mmc3cmd&0x87)==3) /* Switch second 1k video ROM segment */
      memcpy(VRAM+4096+1024,VROM_BASE+(val&VROM_MASK_1k)*1024,1024);
    if((mmc3cmd&0x87)==4) /* Switch third 1k video ROM segment */
      memcpy(VRAM+4096+2048,VROM_BASE+(val&VROM_MASK_1k)*1024,1024);
    if((mmc3cmd&0x87)==5) /* Switch fourth 1k video ROM segment */
      memcpy(VRAM+4096+3072,VROM_BASE+(val&VROM_MASK_1k)*1024,1024);
    if((mmc3cmd&0x87)==0x82) /* Switch first 1k video ROM segment to alt addr */
      memcpy(VRAM,VROM_BASE+(val&VROM_MASK_1k)*1024,1024);
    if((mmc3cmd&0x87)==0x83) /* Switch second 1k video ROM segment to alt addr */
      memcpy(VRAM+1024,VROM_BASE+(val&VROM_MASK_1k)*1024,1024);
    if((mmc3cmd&0x87)==0x84) /* Switch third 1k video ROM segment to alt addr */
      memcpy(VRAM+2048,VROM_BASE+(val&VROM_MASK_1k)*1024,1024);
    if((mmc3cmd&0x87)==0x85) /* Switch fourth 1k video ROM segment to alt addr */
      memcpy(VRAM+3072,VROM_BASE+(val&VROM_MASK_1k)*1024,1024);

    if((mmc3cmd&0x47)==0x06) /* Switch $8000 */
      MAPTABLE[8]=MAPTABLE[9]=ROM_BASE+8192*(val&LAST_HALF_PAGE)-0x8000;
    if((mmc3cmd&0x47)==0x46) /* Switch $C000 */
      MAPTABLE[12]=MAPTABLE[13]=ROM_BASE+8192*(val&LAST_HALF_PAGE)-0xC000;

    if((mmc3cmd&7)==7) /* Switch $A000 */
      MAPTABLE[10]=MAPTABLE[11]=ROM_BASE+8192*(val&LAST_HALF_PAGE)-0xA000;

  }
  if(addr==0xA000)
  {
    hvmirror=val&1;
  }
  if(addr==0xC000)
  {
    irqval=val;
    if(irqenabled)
    {
      if(CLOCK>=VBL)
      {
        irqflag=1;
        CTNI=-((89342-CLOCK*3+341*(irqval))/3);
      }
      else
      {
        if((CLOCK*3+(irqval+1)*341)<81840)
        {
          irqflag=1;
          CTNI=-((341*(irqval+1)-((CLOCK*3)%341))/3);
        }
        /*printf("clock %d irqval %d set\n",CLOCK,irqval);*/
      }
    }
  }

  if(addr==0xE000)
  {
    irqenabled=0;
    irqflag=0;
    CTNI=-((VBL-CLOCK+29781)%29781);
  }

  if(addr==0xE001)
  {
    irqenabled=1;
    if(CLOCK>=VBL)
    {
      irqflag=1;
      CTNI=-((89342-CLOCK*3+341*(irqval))/3);
      /*printf("%d %d \n",CLOCK,CTNI);*/
    }
    else
    {
      if((CLOCK*3+(irqval+1)*341)<81840)
      {
        irqflag=1;
        CTNI=-((341*(irqval+1)-((CLOCK*3)%341))/3);
        /*printf("%d %d \n",CLOCK,CTNI);*/
      }
    }
  }
}

/* MMC2 for PunchOut contributed by Kristoffer Brnemyr */
/* This doesn't seem to work right, so mmc2_latch is not enabled in this version */
int mmc2_latch1;
int mmc2_latch1hi;
int mmc2_latch1low;
int mmc2_latch2;
int mmc2_latch2hi;
int mmc2_latch2low;
int mmc2_init=0;
void init_mmc2()
{
	int i;

	/* the first page is at the start of the rom */
	MAPTABLE[8]=MAPTABLE[9]=ROM_BASE-0x8000;
	/* the last three pages are at the end of the rom */
	MAPTABLE[10]=MAPTABLE[11]=
	MAPTABLE[12]=MAPTABLE[13]=
	MAPTABLE[14]=MAPTABLE[15]=ROM_BASE+0x10000;
	mmc2_latch1=1;
	mmc2_latch2=1;
	mmc2_init=1;
}

void mmc2_latch(int addr)
{
	if(mmc2_init==0)
		return;
	if(addr>=0x2fd0 && addr<=0x2fdf) {
		mmc2_latch1=0;
		memcpy(VRAM,VROM_BASE+(mmc2_latch1low*0x1000),0x1000);
	}
	else if(addr>=0x2fe0 && addr<=0x2fef) {
		mmc2_latch1=1;
		memcpy(VRAM,VROM_BASE+(mmc2_latch1hi*0x1000),0x1000);
	}
	if(addr>=0x3fd0 && addr<=0x3fdf) {
		mmc2_latch2=0;
		memcpy(VRAM,VROM_BASE+0x1000+(mmc2_latch2low*0x1000),0x1000);
	}
	else if(addr>=0x3fe0 && addr<=0x3fef) {
		mmc2_latch2=1;
		memcpy(VRAM,VROM_BASE+0x1000+(mmc2_latch2hi*0x1000),0x1000);
	}
}

mmc2(addr,val)
{
	if(addr >= 0xA000 && addr <= 0xAFFF) {
		/* switch $8000 */
		MAPTABLE[8]=MAPTABLE[9]=ROM_BASE-0x8000+(val*0x2000);
		return;
	}

	if(addr>=0xB000 && addr<=0xBFFF) {
		/* switch ppu $0000 */
		mmc2_latch1low=val;
		if(mmc2_latch1==0)
			memcpy(VRAM,VROM_BASE+(val*0x1000),0x1000);
	}
	if(addr>=0xC000 && addr<=0xCFFF) {
		/* switch ppu $0000 */
		mmc2_latch1hi=val;
		if(mmc2_latch1==1)
			memcpy(VRAM,VROM_BASE+(val*0x1000),0x1000);
	}
	if(addr>=0xD000 && addr<=0xDFFF) {
		/* switch ppu $1000 */
		mmc2_latch1low=val;
		if(mmc2_latch2==0) 
			memcpy(VRAM+0x1000,VROM_BASE+(val*0x1000),0x1000);
	}
	if (addr>=0xE000 && addr<=0xEFFF) {
		/* switch ppu $1000 */
		mmc2_latch1hi=val;
		if(mmc2_latch2==1)
			memcpy(VRAM+0x1000,VROM_BASE+(val*0x1000),0x1000);
	}
	if(addr>=0xF000 && addr<=0xFFFF) {
		if(val==0)
			hvmirror=0;
		else
			hvmirror=1;
	}
}

/* Mapper 11 (Colordreams) */
clrdrms(addr,val)
{
	unsigned char romaddr=0,vromaddr=0;

	romaddr=val&0x0f;
	if(romaddr>ROM_PAGES) {
		romaddr=val&1;
		if(romaddr<ROM_PAGES)
			if(val&2)
				romaddr|=2;
	}
	vromaddr=(val>>4);
	if(vromaddr>VROM_PAGES)
		vromaddr=val&16;
	MAPTABLE[8]=MAPTABLE[9]=
	MAPTABLE[10]=MAPTABLE[11]=
	MAPTABLE[12]=MAPTABLE[13]=
	MAPTABLE[14]=MAPTABLE[15]=ROM_BASE-0x8000+(romaddr*0x8000);
	memcpy(VRAM,VROM_BASE+(vromaddr*0x2000),0x2000);
}

void init_clrdrms()
{
	MAPTABLE[8]=MAPTABLE[9]=MAPTABLE[10]=MAPTABLE[11]=
	MAPTABLE[12]=MAPTABLE[13]=MAPTABLE[14]=MAPTABLE[15]=ROM_BASE-0x8000;
}

init_aorom() {
  mapmirror=osmirror=1;
  MAPTABLE[8]=MAPTABLE[9]=MAPTABLE[10]=MAPTABLE[11]=
  MAPTABLE[12]=MAPTABLE[13]=MAPTABLE[14]=MAPTABLE[15]=ROM_BASE-0x8000;
}

aorom(addr,val)
{
  static mirrorbit=0;
  char temp[1024];
  /*val&=0xff;printf("Mapper AOROM:%4x,%2x (%d)\n",addr,val,CLOCK);/**/
  val&=0x1f;
  if((val>>4)!=mirrorbit)
  {
    drawimage(CLOCK*3);
    memcpy(temp,VRAM+0x2000,1024);
    memcpy(VRAM+0x2000,VRAM+0x2400,1024);
    memcpy(VRAM+0x2400,temp,1024);
    mirrorbit=(val>>4);
  }
  val&=0x0f;val&=ROM_PAGES-1;
  MAPTABLE[8]=MAPTABLE[9]=MAPTABLE[10]=MAPTABLE[11]=
  MAPTABLE[12]=MAPTABLE[13]=MAPTABLE[14]=MAPTABLE[15]=ROM_BASE+(val<<15)-0x8000;
}


void *MapperInit[12]={
	&init_none,   /* 0 */
	&init_mmc1,   /* 1 */
	&init_unrom,  /* 2 */
	&init_crom,   /* 3 */
	&init_mmc3,   /* 4 */
	0, /* 5 */
	0, /* 6 */
	&init_aorom,  /* 7 */
	0, /* 8 */
	&init_mmc2,   /* 9 */
	0, /* 10 */
	&init_clrdrms,/* 11 */
	/* More Mappers Go Here */
};
